/* 
 * EDDGridAggregateExistingDimension Copyright 2008, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.*;

import java.util.ArrayList;


/** 
 * This class represents a grid dataset created by aggregating 
 * for first existing dimension of other datasets.
 * Each child holds a contiguous group of values for the aggregated dimension.
 * 
 * @author Bob Simons (bob.simons@noaa.gov) 2008-02-04
 */
public class EDDGridAggregateExistingDimension extends EDDGrid { 

    protected EDDGrid childDatasets[];
    protected int childStopsAt[]; //the last valid axisVar0 index for each childDataset
    protected boolean ensureAxisValuesAreEqual;

    /**
     * This constructs an EDDGridAggregateExistingDimension based on the information in an .xml file.
     * Only the attributes from the first dataset are used for the composite
     * dataset.
     * 
     * @param xmlReader with the &lt;erddapDatasets&gt;&lt;dataset type="EDDGridAggregateExistingDimension"&gt; 
     *    having just been read.  
     * @return an EDDGridAggregateExistingDimension.
     *    When this returns, xmlReader will have just read &lt;erddapDatasets&gt;&lt;/dataset&gt; .
     * @throws Throwable if trouble
     */
    public static EDDGridAggregateExistingDimension fromXml(SimpleXMLReader xmlReader) throws Throwable {

        if (verbose) String2.log("\n*** constructing EDDGridAggregateExistingDimension(xmlReader)...");
        String tDatasetID = xmlReader.attributeValue("datasetID"); 

        //data to be obtained while reading xml
        EDDGrid firstChild = null;
        StringArray tSourceUrls = new StringArray();
        String tAccessibleTo = null;
        StringArray tOnChange = new StringArray();
        boolean tEnsureAxisValuesAreEqual = true;

        //process the tags
        String startOfTags = xmlReader.allTags();
        int startOfTagsN = xmlReader.stackSize();
        int startOfTagsLength = startOfTags.length();

        while (true) {
            xmlReader.nextTag();
            String tags = xmlReader.allTags();
            String content = xmlReader.content();
            if (reallyVerbose) String2.log("  tags=" + tags + content);
            if (xmlReader.stackSize() == startOfTagsN) 
                break; //the </dataset> tag
            String localTags = tags.substring(startOfTagsLength);

            //try to make the tag names as consistent, descriptive and readable as possible
            if (localTags.equals("<dataset>")) {
                if (firstChild == null) {
                    EDD edd = EDD.fromXml(xmlReader.attributeValue("type"), xmlReader);
                    if (edd instanceof EDDGrid) {
                        firstChild = (EDDGrid)edd;
                    } else {
                        throw new RuntimeException("Datasets.xml error: " +
                            "The dataset defined in an " +
                            "EDDGridAggregateExistingDimension must be a subclass of EDDGrid.");
                    }
                } else {
                    throw new RuntimeException("Datasets.xml error: " +
                        "There can be only one <dataset> defined within an " +
                        "EDDGridAggregateExistingDimension <dataset>.");
                }

            } 
            else if (localTags.equals( "<sourceUrl>")) {}
            else if (localTags.equals("</sourceUrl>")) 
                tSourceUrls.add(content);

            else if (localTags.equals( "<ensureAxisValuesAreEqual>")) {}
            else if (localTags.equals("</ensureAxisValuesAreEqual>")) 
                tEnsureAxisValuesAreEqual = String2.parseBoolean(content); 

            else if (localTags.equals( "<accessibleTo>")) {}
            else if (localTags.equals("</accessibleTo>")) tAccessibleTo = content;

            //onChange
            else if (localTags.equals( "<onChange>")) {}
            else if (localTags.equals("</onChange>")) 
                tOnChange.add(content); 

            else xmlReader.unexpectedTagException();
        }

        //make the main dataset based on the information gathered
        return new EDDGridAggregateExistingDimension(tDatasetID, tAccessibleTo, tOnChange,
            firstChild, tSourceUrls.toArray(), tEnsureAxisValuesAreEqual);

    }

    /**
     * The constructor.
     * The axisVariables must be the same and in the same
     * order for each dataVariable.
     *
     * @param tDatasetID
     * @param tAccessibleTo is a space separated list of 0 or more
     *    roles which will have access to this dataset.
     *    <br>If null, everyone will have access to this dataset (even if not logged in).
     *    <br>If "", no one will have access to this dataset.
     * @param tOnChange 0 or more actions (starting with "http://" or "mailto:")
     *    to be done whenever the dataset changes significantly
     * @param firstChild
     * @param tSourceUrls the sourceUrls for the other siblings
     * @param tEnsureAxisValuesAreEqual if true (recommended), this ensures
     *   that the axis sourceValues for axisVar 1+ are almostEqual.
     *   (Setting this to false is not recommended, but is useful if the, 
     *   for example, lat and lon values vary slightly and you 
     *   are willing to accept the initial values as the correct values.)   
     *   Actually, the tests are always done but if this is false and the test fails,
     *   the error is just logged and doesn't throw an exception.
     * @throws Throwable if trouble
     */
    public EDDGridAggregateExistingDimension(String tDatasetID, 
        String tAccessibleTo, StringArray tOnChange, 
        EDDGrid firstChild, String tSourceUrls[], boolean tEnsureAxisValuesAreEqual) throws Throwable {

        if (verbose) String2.log(
            "\n*** constructing EDDGridAggregateExistingDimension " + tDatasetID + 
            " ensureEqual=" + ensureAxisValuesAreEqual); 
        long constructionStartMillis = System.currentTimeMillis();
        String errorInMethod = "Error in EDDGridGridAggregateExistingDimension(" + 
            tDatasetID + ") constructor:\n";
            
        //save some of the parameters
        className = "EDDGridAggregateExistingDimension"; 
        datasetID = tDatasetID;
        setAccessibleTo(tAccessibleTo);
        onChange = tOnChange;
        ensureAxisValuesAreEqual = tEnsureAxisValuesAreEqual;
        int nChildren = tSourceUrls.length + 1;
        childDatasets = new EDDGrid[nChildren];
        childDatasets[0] = firstChild;
        PrimitiveArray cumSV = firstChild.axisVariables()[0].sourceValues();
        childStopsAt = new int[nChildren];
        childStopsAt[0] = cumSV.size() - 1;

        //create the siblings
        for (int i = 0; i < nChildren - 1; i++) {
            EDDGrid sibling = firstChild.sibling(tSourceUrls[i], 
                ensureAxisValuesAreEqual? 1 : Integer.MAX_VALUE, true);
            childDatasets[i + 1] = sibling;
            cumSV.append(sibling.axisVariables()[0].destinationValues());
            childStopsAt[i + 1] = cumSV.size() - 1;
        }


        //create the aggregate dataset
        setReloadEveryNMinutes(firstChild.getReloadEveryNMinutes());
      
        addGlobalAttributes = firstChild.addGlobalAttributes();
        sourceGlobalAttributes = firstChild.sourceGlobalAttributes();
        combinedGlobalAttributes = firstChild.combinedGlobalAttributes();
        //??? sourceUrl = addGlobalAttributes.get("sourceUrl") + " and similar urls";
        //addGlobalAttributes.set("sourceUrl", sourceUrl); 
        //combinedGlobalAttributes.set("sourceUrl", sourceUrl); 
        //and clear searchString of children?

        int nAv = firstChild.axisVariables.length;
        axisVariables = new EDVGridAxis[nAv];
        System.arraycopy(firstChild.axisVariables, 0, axisVariables, 0, nAv);
        lonIndex = firstChild.lonIndex;
        latIndex = firstChild.latIndex;
        altIndex = firstChild.altIndex;
        timeIndex = firstChild.timeIndex;
        EDVGridAxis av0 = axisVariables[0];
        String sn = av0.sourceName();
        Attributes sa = av0.sourceAttributes();
        Attributes aa = av0.addAttributes();
        if      (lonIndex  == 0) axisVariables[0] = new EDVLonGridAxis(sn, sa, aa, cumSV);
        else if (latIndex  == 0) axisVariables[0] = new EDVLatGridAxis(sn, sa, aa, cumSV);
        else if (altIndex  == 0) axisVariables[0] = new EDVAltGridAxis(sn, sa, aa, cumSV, 
            ((EDVAltGridAxis)av0).metersPerSourceUnit());
        else if (timeIndex == 0) axisVariables[0] = new EDVTimeGridAxis(sn, sa, aa, cumSV);
        else {axisVariables[0] = new EDVGridAxis(sn, av0.destinationName(), sa, aa, cumSV);
              axisVariables[0].setActualRangeFromDestinationMinMax();
        }


        int nDv = firstChild.dataVariables.length;
        dataVariables = new EDV[nDv];
        System.arraycopy(firstChild.dataVariables, 0, dataVariables, 0, nDv);

        //ensure the setup is valid
        ensureValid();

        //finally
        if (verbose) String2.log(
            (reallyVerbose? "\n" + toString() : "") +
            "\n*** EDDGridAggregateExistingDimension " + datasetID() + " constructor finished. TIME=" + 
            (System.currentTimeMillis() - constructionStartMillis) + "\n"); 

    }

    /**
     * This makes a sibling dataset, based on the new sourceUrl.
     *
     * @throws Throwable always (since this class doesn't support sibling())
     */
    public EDDGrid sibling(String tSourceUrl, int ensureAxisValuesAreEqual, 
        boolean shareInfo) throws Throwable {
        throw new SimpleException( 
            "Error: EDDGridAggregateExistingDimension doesn't support method=\"sibling\".");
    }

    /** 
     * This gets data (not yet standardized) from the data 
     * source for this EDDGrid.     
     * 
     * @param tDataVariables
     * @param tConstraints
     * @return a PrimitiveArray[] where the first axisVariables.length elements
     *   are the axisValues and the next tDataVariables.length elements
     *   are the dataValues.
     *   Both the axisValues and dataValues are straight from the source,
     *   not modified.
     * @throws Throwable if trouble
     */
    public PrimitiveArray[] getSourceData(EDV tDataVariables[], IntArray tConstraints) 
        throws Throwable {

        //parcel first dimension's constraints to appropriate datasets
        int nChildren = childStopsAt.length;
        int nAv = axisVariables.length;
        int nDv = tDataVariables.length;
        PrimitiveArray[] cumResults = null;
        int index = tConstraints.get(0);
        int stride = tConstraints.get(1);
        int stop = tConstraints.get(2);
        //find currentDataset (associated with index)
        int currentStart = index;
        int currentDataset = 0;  
        while (index > childStopsAt[currentDataset])
            currentDataset++;
        IntArray childConstraints = (IntArray)tConstraints.clone();

        //walk through the requested index values
        while (index <= stop) {
            //find nextDataset (associated with next iteration's index)
            int nextDataset = currentDataset; 
            while (nextDataset < nChildren && index + stride > childStopsAt[nextDataset])
                nextDataset++; //ok if >= nDatasets

            //get a chunk of data related to current chunk of indexes?
            if (nextDataset != currentDataset ||   //next iteration will be a different dataset
                index + stride > stop) {           //this is last iteration
                //get currentStart:stride:index
                int currentDatasetStartsAt = currentDataset == 0? 0 : childStopsAt[currentDataset - 1] + 1;
                childConstraints.set(0, currentStart - currentDatasetStartsAt);
                childConstraints.set(2, index - currentDatasetStartsAt);
                if (verbose) String2.log("  currentDataset=" + currentDataset +
                    "  datasetStartsAt=" + currentDatasetStartsAt + 
                    "  localStart=" + childConstraints.get(0) +
                    "  localStop=" + childConstraints.get(2));
                PrimitiveArray[] tResults = childDatasets[currentDataset].getSourceData(
                    tDataVariables, childConstraints);
                //childDataset has already checked that axis values are as *it* expects          
                if (cumResults == null) {
                    if (!ensureAxisValuesAreEqual) {
                        //make axis values exactly as expected by aggregate dataset
                        for (int av = 1; av < nAv; av++)
                            tResults[av] = axisVariables[av].sourceValues().subset(
                                childConstraints.get(av * 3 + 0),
                                childConstraints.get(av * 3 + 1),
                                childConstraints.get(av * 3 + 2));
                    }
                    cumResults = tResults;
                } else {
                    cumResults[0].append(tResults[0]);
                    for (int dv = 0; dv < nDv; dv++)
                        cumResults[nAv + dv].append(tResults[nAv + dv]);
                }

                currentDataset = nextDataset;
                currentStart = index + stride;            
            }

            //increment index
            index += stride;
        }

        return cumResults;
    }



    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void test() throws Throwable {

        String2.log("\n****************** EDDGridAggregateExistingDimension.test() *****************\n");
        verbose = true;
        reallyVerbose = true;
        GridDataAccessor.verbose = true;
        GridDataAccessor.reallyVerbose = true;
        String name, tName, userDapQuery, results, expected, error;

        //one time
        //String2.log(generateDatasetsXml("http://dods.ndbc.noaa.gov:8080/opendap/cwind/41002/41002c1989.nc"));

        //*** NDBC  is also IMPORTANT UNIQUE TEST of >1 variable in a file
        EDDGrid gridDataset = (EDDGrid)oneFromDatasetXml("ndbcCWind41002");       

        //min max
        EDV edv = gridDataset.findAxisVariableByDestinationName("longitude");
        Test.ensureEqual(edv.destinationMin(), -75.42, "");
        Test.ensureEqual(edv.destinationMax(), -75.42, "");

        String ndbcDapQuery = "wind_speed[1:5][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className(), ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_speed\n" +
"UTC, degrees_north, degrees_east, m s-1\n" +
"1989-06-13T16:10:00Z, 32.27, -75.42, 15.7\n" +
"1989-06-13T16:20:00Z, 32.27, -75.42, 15.0\n" +
"1989-06-13T16:30:00Z, 32.27, -75.42, 14.4\n" +
"1989-06-13T16:40:00Z, 32.27, -75.42, 14.0\n" +
"1989-06-13T16:50:00Z, 32.27, -75.42, 13.6\n";
        Test.ensureEqual(results, expected, "results=" + results);

        ndbcDapQuery = "wind_speed[1:5][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className(), ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory + tName, true);
        int dataPo = results.indexOf("data:");
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
        expected = 
"netcdf ndbc.nc {\n" +
" dimensions:\n" +
"   time = 5;\n" +   // (has coord.var)\n" +   //changed when switched to netcdf-java 4.0, 2009-02-23
"   latitude = 1;\n" +   // (has coord.var)\n" +
"   longitude = 1;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   int time(time=5);\n" +
"     :_CoordinateAxisType = \"Time\";\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);

//BUG???!!!
//"     :actual_range = 6.137568E8, 7.258458E8; // double\n" +  //this changes depending on how many files aggregated

expected = 
":axis = \"T\";\n" +
"     :ioos_category = \"Time\";\n" +
"     :long_name = \"Time\";\n" +
"     :short_name = \"time\";\n" +
"     :standard_name = \"time\";\n" +
"     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
"     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
"   float latitude(latitude=1);\n" +
"     :_CoordinateAxisType = \"Lat\";\n" +
"     :actual_range = 32.27f, 32.27f; // float\n" +
"     :axis = \"Y\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Latitude\";\n" +
"     :short_name = \"latitude\";\n" +
"     :standard_name = \"latitude\";\n" +
"     :units = \"degrees_north\";\n" +
"   float longitude(longitude=1);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = -75.42f, -75.42f; // float\n" +
"     :axis = \"X\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :short_name = \"longitude\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"   float wind_speed(time=5, latitude=1, longitude=1);\n" +
"     :_FillValue = 99.0f; // float\n" +
"     :ioos_category = \"Wind\";\n" +
"     :long_name = \"Wind Speed\";\n" +
"     :missing_value = 99.0f; // float\n" +
"     :short_name = \"wspd\";\n" +
"     :standard_name = \"wind_speed\";\n" +
"     :units = \"m s-1\";\n" +
"\n" +
" :cdm_data_type = \"Grid\";\n" +
" :comment = \"S HATTERAS - 250 NM East of Charleston, SC\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
" :history = \"" + today + " http://dods.ndbc.noaa.gov:8080/opendap/cwind/41002/41002c1989.nc\n" +
today + " " + EDStatic.erddapUrl + //in tests, always non-https url
    "/griddap/ndbcCWind41002.nc?wind_speed[1:5][0][0]\";\n" +
" :infoUrl = \"http://www.ndbc.noaa.gov/cwind.shtml\";\n" +
" :institution = \"NOAA NDBC\";\n" +
" :license = \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
" :location = \"32.27 N 75.42 W \";\n" +
" :quality = \"Automated QC checks with manual editing and comprehensive monthly QC\";\n" +
" :sourceUrl = \"http://dods.ndbc.noaa.gov:8080/opendap/cwind/41002/41002c1989.nc\";\n" +
" :standard_name_vocabulary = \"CF\";\n" +
" :station = \"41002\";\n" +
" :summary = \"These continuous wind measurements from the NOAA National Data Buoy Center (NDBC) stations are 10-minute average values of wind speed (in m/s) and direction (in degrees clockwise from North).\";\n" +
" :title = \"Wind Data from NDBC 41002\";\n" +
" :url = \"http://dods.ndbc.noaa.gov\";\n" +
" data:\n" +
"time =\n" +
"  {613757400, 613758000, 613758600, 613759200, 613759800}\n" +
"latitude =\n" +
"  {32.27}\n" +
"longitude =\n" +
"  {-75.42}\n" +
"wind_speed =\n" +
"  {\n" +
"    {\n" +
"      {15.7}\n" +
"    },\n" +
"    {\n" +
"      {15.0}\n" +
"    },\n" +
"    {\n" +
"      {14.4}\n" +
"    },\n" +
"    {\n" +
"      {14.0}\n" +
"    },\n" +
"    {\n" +
"      {13.6}\n" +
"    }\n" +
"  }\n" +
"}\n";
        int po = results.indexOf(":axis =");
        Test.ensureEqual(results.substring(po), expected, "results=" + results);

        ndbcDapQuery = "wind_direction[1:5][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_direction\n" +
"UTC, degrees_north, degrees_east, degrees_true\n" +
"1989-06-13T16:10:00Z, 32.27, -75.42, 234\n" +
"1989-06-13T16:20:00Z, 32.27, -75.42, 233\n" +
"1989-06-13T16:30:00Z, 32.27, -75.42, 233\n" +
"1989-06-13T16:40:00Z, 32.27, -75.42, 235\n" +
"1989-06-13T16:50:00Z, 32.27, -75.42, 235\n";
        Test.ensureEqual(results, expected, "results=" + results);

        ndbcDapQuery = "wind_speed[1:5][0][0],wind_direction[1:5][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "3", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_speed, wind_direction\n" +
"UTC, degrees_north, degrees_east, m s-1, degrees_true\n" +
"1989-06-13T16:10:00Z, 32.27, -75.42, 15.7, 234\n" +
"1989-06-13T16:20:00Z, 32.27, -75.42, 15.0, 233\n" +
"1989-06-13T16:30:00Z, 32.27, -75.42, 14.4, 233\n" +
"1989-06-13T16:40:00Z, 32.27, -75.42, 14.0, 235\n" +
"1989-06-13T16:50:00Z, 32.27, -75.42, 13.6, 235\n";
        Test.ensureEqual(results, expected, "results=" + results);

        ndbcDapQuery = "wind_direction[1:5][0][0],wind_speed[1:5][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "4", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_direction, wind_speed\n" +
"UTC, degrees_north, degrees_east, degrees_true, m s-1\n" +
"1989-06-13T16:10:00Z, 32.27, -75.42, 234, 15.7\n" +
"1989-06-13T16:20:00Z, 32.27, -75.42, 233, 15.0\n" +
"1989-06-13T16:30:00Z, 32.27, -75.42, 233, 14.4\n" +
"1989-06-13T16:40:00Z, 32.27, -75.42, 235, 14.0\n" +
"1989-06-13T16:50:00Z, 32.27, -75.42, 235, 13.6\n";
        Test.ensureEqual(results, expected, "results=" + results);

        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "4", ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory + tName, true);
        dataPo = results.indexOf("data:");
        results = results.substring(dataPo);
        expected = 
"data:\n" +
"time =\n" +
"  {613757400, 613758000, 613758600, 613759200, 613759800}\n" +
"latitude =\n" +
"  {32.27}\n" +
"longitude =\n" +
"  {-75.42}\n" +
"wind_direction =\n" +
"  {\n" +
"    {\n" +
"      {234}\n" +
"    },\n" +
"    {\n" +
"      {233}\n" +
"    },\n" +
"    {\n" +
"      {233}\n" +
"    },\n" +
"    {\n" +
"      {235}\n" +
"    },\n" +
"    {\n" +
"      {235}\n" +
"    }\n" +
"  }\n" +
"wind_speed =\n" +
"  {\n" +
"    {\n" +
"      {15.7}\n" +
"    },\n" +
"    {\n" +
"      {15.0}\n" +
"    },\n" +
"    {\n" +
"      {14.4}\n" +
"    },\n" +
"    {\n" +
"      {14.0}\n" +
"    },\n" +
"    {\n" +
"      {13.6}\n" +
"    }\n" +
"  }\n" +
"}\n";
        Test.ensureEqual(results, expected, "results=" + results);

        //test seam of two datasets
        ndbcDapQuery = "wind_direction[22232:22239][0][0],wind_speed[22232:22239][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "5", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_direction, wind_speed\n" +
"UTC, degrees_north, degrees_east, degrees_true, m s-1\n" +
"1989-12-05T05:20:00Z, 32.27, -75.42, 232, 23.7\n" +
"1989-12-05T05:30:00Z, 32.27, -75.42, 230, 24.1\n" +
"1989-12-05T05:40:00Z, 32.27, -75.42, 225, 23.5\n" +
"1989-12-05T05:50:00Z, 32.27, -75.42, 233, 23.3\n" +
"1992-04-28T23:00:00Z, 32.27, -75.42, 335, 29.4\n" +
"1992-04-28T23:10:00Z, 32.27, -75.42, 335, 30.5\n" +
"1992-04-28T23:20:00Z, 32.27, -75.42, 330, 32.3\n" +
"1992-04-28T23:30:00Z, 32.27, -75.42, 331, 33.2\n";
        Test.ensureEqual(results, expected, "results=" + results);

        //test seam of two datasets   with stride
        ndbcDapQuery = "wind_direction[22232:2:22239][0][0],wind_speed[22232:2:22239][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "6", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_direction, wind_speed\n" +
"UTC, degrees_north, degrees_east, degrees_true, m s-1\n" +
"1989-12-05T05:20:00Z, 32.27, -75.42, 232, 23.7\n" +
"1989-12-05T05:40:00Z, 32.27, -75.42, 225, 23.5\n" +
"1992-04-28T23:00:00Z, 32.27, -75.42, 335, 29.4\n" +
"1992-04-28T23:20:00Z, 32.27, -75.42, 330, 32.3\n";
        Test.ensureEqual(results, expected, "results=" + results);
        // */

        //test last data value (it causes a few problems -- including different axis values)
        ndbcDapQuery = "wind_direction[(2008-02-06T18:50:00):1:(2008-02-06T20:50:00)][0][0]," +
                           "wind_speed[(2008-02-06T18:50:00):1:(2008-02-06T20:50:00)][0][0]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, ndbcDapQuery, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "7", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time, latitude, longitude, wind_direction, wind_speed\n" +
"UTC, degrees_north, degrees_east, degrees_true, m s-1\n" +
"2008-02-06T18:50:00Z, 32.27, -75.42, 194, 8.6\n" +
"2008-02-06T19:00:00Z, 32.27, -75.42, 192, 8.9\n" +
"2008-02-06T19:10:00Z, 32.27, -75.42, 190, 9.3\n" +
"2008-02-06T19:20:00Z, 32.27, -75.42, 193, 9.7\n" +
"2008-02-06T19:30:00Z, 32.27, -75.42, 195, 9.5\n" +
"2008-02-06T19:40:00Z, 32.27, -75.42, 194, 9.1\n" +
"2008-02-06T19:50:00Z, 32.27, -75.42, 193, 10.0\n" +
"2008-02-06T20:00:00Z, 32.27, -75.42, 192, 10.4\n" +
"2008-02-06T20:10:00Z, 32.27, -75.42, 193, 10.3\n" +
"2008-02-06T20:20:00Z, 32.27, -75.42, 194, 10.2\n" +
"2008-02-06T20:30:00Z, 32.27, -75.42, 193, 10.5\n" +
"2008-02-06T20:40:00Z, 32.27, -75.42, 192, 10.4\n" +
"2008-02-06T20:50:00Z, 32.27, -75.42, 190, 10.5\n";
        Test.ensureEqual(results, expected, "results=" + results);


        String2.log("\n*** EDDGridAggregateExistingDimension.test finished.");

    }


}
