/* 
 * EDDGridFromDap Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.FloatArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.ShortArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.ResourceBundle2;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;

/** The Java DAP classes.  */
import dods.dap.*;

import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.sgt.SgtGraph;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.*;

import java.io.FileWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Enumeration;

/**
 * Get netcdf-X.X.XX.jar from http://www.unidata.ucar.edu/software/netcdf-java/index.htm
 * and copy it to <context>/WEB-INF/lib renamed as netcdf-latest.jar.
 * Get slf4j-jdk14.jar from 
 * ftp://ftp.unidata.ucar.edu/pub/netcdf-java/slf4j-jdk14.jar
 * and copy it to <context>/WEB-INF/lib.
 * Put both of these .jar files in the classpath for the compiler and for Java.
 */
import ucar.nc2.*;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dods.*;
import ucar.nc2.util.*;
import ucar.ma2.*;  //only Array is needed; all other ucar is for testing netcdf-java

/** 
 * This class represents a grid dataset from an opendap DAP source.
 * 
 * @author Bob Simons (bob.simons@noaa.gov) 2007-06-04
 */
public class EDDGridFromDap extends EDDGrid { 

    /** Indicates if data can be transmitted in a compressed form.
     * It is unlikely anyone would want to change this. */
    public static boolean acceptDeflate = true;


    /**
     * This constructs an EDDGridFromDap based on the information in an .xml file.
     * 
     * @param xmlReader with the &lt;erddapDatasets&gt;&lt;dataset type="EDDGridFromDap"&gt; 
     *    having just been read.  
     * @return an EDDGridFromDap.
     *    When this returns, xmlReader will have just read &lt;erddapDatasets&gt;&lt;/dataset&gt; .
     * @throws Throwable if trouble
     */
    public static EDDGridFromDap fromXml(SimpleXMLReader xmlReader) throws Throwable {

        //data to be obtained (or not)
        if (verbose) String2.log("\n*** constructing EDDGridFromDap(xmlReader)...");
        String tDatasetID = xmlReader.attributeValue("datasetID"); 
        Attributes tGlobalAttributes = null;
        double tAltitudeMetersPerSourceUnit = 1; 
        String tAccessibleTo = null;
        StringArray tOnChange = new StringArray();
        ArrayList tAxisVariables = new ArrayList();
        ArrayList tDataVariables = new ArrayList();
        int tReloadEveryNMinutes = DEFAULT_RELOAD_EVERY_N_MINUTES;
        String tSourceUrl = null;

        //process the tags
        String startOfTags = xmlReader.allTags();
        int startOfTagsN = xmlReader.stackSize();
        int startOfTagsLength = startOfTags.length();

        while (true) {
            xmlReader.nextTag();
            String tags = xmlReader.allTags();
            String content = xmlReader.content();
            //if (reallyVerbose) String2.log("  tags=" + tags + content);
            if (xmlReader.stackSize() == startOfTagsN) 
                break; //the </dataset> tag
            String localTags = tags.substring(startOfTagsLength);

            //try to make the tag names as consistent, descriptive and readable as possible
            if (localTags.equals("<addAttributes>"))
                tGlobalAttributes = getAttributesFromXml(xmlReader);
            else if (localTags.equals( "<altitudeMetersPerSourceUnit>")) {}
            else if (localTags.equals("</altitudeMetersPerSourceUnit>")) 
                tAltitudeMetersPerSourceUnit = String2.parseDouble(content); 
            else if (localTags.equals( "<axisVariable>")) tAxisVariables.add(getSDAVVariableFromXml(xmlReader));           
            else if (localTags.equals( "<dataVariable>")) tDataVariables.add(getSDADVariableFromXml(xmlReader));           
            else if (localTags.equals( "<accessibleTo>")) {}
            else if (localTags.equals("</accessibleTo>")) tAccessibleTo = content;
            else if (localTags.equals( "<reloadEveryNMinutes>")) {}
            else if (localTags.equals("</reloadEveryNMinutes>")) tReloadEveryNMinutes = String2.parseInt(content); 
            else if (localTags.equals( "<sourceUrl>")) {}
            else if (localTags.equals("</sourceUrl>")) tSourceUrl = content; 

            //onChange
            else if (localTags.equals( "<onChange>")) {}
            else if (localTags.equals("</onChange>")) 
                tOnChange.add(content); 

            else xmlReader.unexpectedTagException();
        }
        int nav = tAxisVariables.size();
        Object ttAxisVariables[][] = nav == 0? null : new Object[nav][];
        for (int i = 0; i < tAxisVariables.size(); i++)
            ttAxisVariables[i] = (Object[])tAxisVariables.get(i);

        int ndv = tDataVariables.size();
        Object ttDataVariables[][] = new Object[ndv][];
        for (int i = 0; i < tDataVariables.size(); i++)
            ttDataVariables[i] = (Object[])tDataVariables.get(i);

        return new EDDGridFromDap(tDatasetID, tAccessibleTo,
            tOnChange, tGlobalAttributes,
            tAltitudeMetersPerSourceUnit,
            ttAxisVariables,
            ttDataVariables,
            tReloadEveryNMinutes, tSourceUrl);
    }

    /**
     * The constructor.
     * The axisVariables must be the same and in the same
     * order for each dataVariable.
     *
     * <p>Yes, lots of detailed information must be supplied here
     * that is sometimes available in metadata. If it is in metadata,
     * make a subclass that extracts info from metadata and calls this 
     * constructor.
     *
     * @param tDatasetID is a very short string identifier 
     *   (required: just safe characters: A-Z, a-z, 0-9, _, -, or .)
     *   for this dataset. See EDD.datasetID().
     * @param tAccessibleTo is a comma separated list of 0 or more
     *    roles which will have access to this dataset.
     *    <br>If null, everyone will have access to this dataset (even if not logged in).
     *    <br>If "", no one will have access to this dataset.
     * @param tOnChange 0 or more actions (starting with "http://" or "mailto:")
     *    to be done whenever the dataset changes significantly
     * @param tAddGlobalAttributes are global attributes which will
     *   be added to (and take precedence over) the data source's global attributes.
     *   This may be null if you have nothing to add.
     *   The combined global attributes must include:
     *   <ul>
     *   <li> "title" - the short (&lt; 80 characters) description of the dataset 
     *   <li> "summary" - the longer description of the dataset.
     *      It may have newline characters (usually at &lt;= 72 chars per line).
     *   <li> "institution" - the source of the data 
     *      (best if &lt; 50 characters so it fits in a graph's legend).
     *   <li> "infoUrl" - the url with information about this data set 
     *   </ul>
     *   Special case: value="null" causes that item to be removed from combinedGlobalAttributes.
     *   Special case: if addGlobalAttributes name="license" value="[standard]",
     *     the EDStatic.standardLicense will be used.
     * @param tAltMetersPerSourceUnit the factor needed to convert the source
     *    alt values to/from meters above sea level.
     *    If there is no altitude variable, this is irrelevant.
     * @param tAxisVariables is an Object[nAxisVariables][3]: 
     *    <br>[0]=String sourceName (the name of the data variable in the dataset source),
     *    <br>[1]=String destinationName (the name to be presented to the ERDDAP user, 
     *        or null to use the sourceName),
     *    <br>[2]=Attributes addAttributes (at ERD, this must have "ioos_category" -
     *        a category from EDV.ioosCategories, 
     *        although they are added automatically for lon, lat, alt, and time). 
     *        Special case: value="null" causes that item to be removed from combinedAttributes.
     *    <br>If there are longitude, latitude, altitude, or time variables,
     *        they must have that name as the destinationName (or sourceName) 
     *        to be identified as such.
     *    <br>Or, use tAxisVariables=null if the axis variables need no addAttributes
     *        and the longitude,latitude,altitude,time variables (if present) 
     *        all have their correct names in the source.
     *    <br>The order of variables you define must match the
     *       order in the source.
     *    <br>A time variable must have "units" specified in addAttributes (read first)
     *       or sourceAttributes.  "units" must be either <ul>
     *      <li> udunits string (containing " since ")
     *        describing how to interpret numeric values 
     *        (e.g., "seconds since 1970-01-01T00:00:00Z"), or
     *      <li>(currently not supported here except for sourceFixedValues in the sourceName) 
     *         a java.text.SimpleDateFormat string describing how to interpret string times  
     *        (see http://java.sun.com/j2se/1.4.2/docs/api/java/text/SimpleDateFormat.html).   
     *      </ul>
     * @param tDataVariables is an Object[nDataVariables][3]: 
     *    <br>[0]=String sourceName (the name of the data variable in the dataset source),
     *    <br>[1]=String destinationName (the name to be presented to the ERDDAP user, 
     *        or null to use the sourceName),
     *    <br>[2]=Attributes addAttributes (at ERD, this must have "ioos_category" 
     *        - a category from EDV.ioosCategories). 
     *        Special case: value="null" causes that item to be removed from combinedAttributes.
     *    <br>All dataVariables must share the same axis variables.
     *    <br>The order of variables you define doesn't have to match the
     *       order in the source.
     * @param tReloadEveryNMinutes indicates how often the source should
     *    be checked for new data (use Integer.MAX_VALUE for never).
     * @param tSourceUrl the url to which .das or .dds or ... can be added
     * @throws Throwable if trouble
     */
    public EDDGridFromDap(
        String tDatasetID, String tAccessibleTo, StringArray tOnChange,
        Attributes tAddGlobalAttributes,
        double tAltMetersPerSourceUnit, 
        Object tAxisVariables[][],
        Object tDataVariables[][],
        int tReloadEveryNMinutes,
        String tSourceUrl) throws Throwable {

        if (verbose) String2.log(
            "\n*** constructing EDDGridFromDap " + tDatasetID); 
        long constructionStartMillis = System.currentTimeMillis();
        String errorInMethod = "Error in EDDGridFromDap(" + 
            tDatasetID + ") constructor:\n";
            
        //save some of the parameters
        className = "EDDGridFromDap"; 
        datasetID = tDatasetID;
        setAccessibleTo(tAccessibleTo);
        onChange = tOnChange;
        if (tAddGlobalAttributes == null)
            tAddGlobalAttributes = new Attributes();
        addGlobalAttributes = tAddGlobalAttributes.add("cdm_data_type", "Grid");
        String tLicense = addGlobalAttributes.getString("license");
        if (tLicense != null && tLicense.equals("[standard]"))
            addGlobalAttributes.set("license", EDStatic.standardLicense);
        addGlobalAttributes.set("sourceUrl", tSourceUrl);
        setReloadEveryNMinutes(tReloadEveryNMinutes);
        sourceUrl = tSourceUrl;
      
        //open the connection to the opendap source
        //Design decision: this doesn't use ucar.nc2.dt.GridDataSet 
        //  because GridDataSet determines axes via _CoordinateAxisType (or similar) metadata
        //  which most datasets we use don't have yet.
        //  One could certainly write another class that did use ucar.nc2.dt.GridDataSet.
        DConnect dConnect = new DConnect(sourceUrl, acceptDeflate, 1, 1);

        //this is common point of failure, so make error more descriptive
        DAS das;
        try {
            das = dConnect.getDAS(OpendapHelper.DEFAULT_TIMEOUT);
        } catch (Throwable t) {
            throw new SimpleException("Error while getting DAS from " + sourceUrl + ".das.", t);
        }
        DDS dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);

        //get global attributes
        sourceGlobalAttributes = new Attributes();
        OpendapHelper.getAttributes(das, "GLOBAL", sourceGlobalAttributes);
        combinedGlobalAttributes = new Attributes(addGlobalAttributes, sourceGlobalAttributes); //order is important
        combinedGlobalAttributes.removeValue("null");

        //create dataVariables[]
        dataVariables = new EDV[tDataVariables.length];
        for (int dv = 0; dv < tDataVariables.length; dv++) {
            String tDataSourceName = (String)tDataVariables[dv][0];
            String tDataDestName   = (String)tDataVariables[dv][1];
            Attributes tDataSourceAttributes = new Attributes();

            OpendapHelper.getAttributes(das, tDataSourceName, tDataSourceAttributes);

            //get the 
            BaseType bt = dds.getVariable(tDataSourceName);  //throws Throwable if not found
            DArray mainDArray;
            if (bt instanceof DGrid) 
                mainDArray = (DArray)((DGrid)bt).getVar(0); //first element is always main array
            else if (bt instanceof DArray) 
                mainDArray = (DArray)bt;
            else throw new RuntimeException("dataVariable=" + tDataSourceName + " must be a DGrid or a DArray (" + 
                bt.toString() + ").");

            //look at the dimensions
            PrimitiveVector pv = mainDArray.getPrimitiveVector(); //just gets the data type
            //if (reallyVerbose) String2.log(tDataSourceName + " pv=" + pv);
            String dvSourceDataType = PrimitiveArray.elementTypeToString( 
                OpendapHelper.getElementType(pv));
            int numDimensions = mainDArray.numDimensions();
            if (dv == 0) {
                axisVariables = new EDVGridAxis[numDimensions];
            } else {
                Test.ensureEqual(numDimensions, axisVariables.length,
                    errorInMethod + "nDimensions was different for " +
                    "dataVariable#0=" + axisVariables[0].destinationName() + " and " +
                    "dataVariable#" + dv + "=" + tDataSourceName + ".");
            }
            for (int av = 0; av < numDimensions; av++) {

                DArrayDimension dad = mainDArray.getDimension(av);
                String tSourceAxisName = dad.getName();

               //ensure this dimension's name is the same as for the other dataVariables
                //(or as specified in tAxisVariables()[0])
                if (tAxisVariables == null) {
                    if (dv > 0)
                        Test.ensureEqual(tSourceAxisName, axisVariables[av].sourceName(),
                            errorInMethod + "Observed dimension name doesn't equal " +
                            "expected dimension name for dimension #" + av + 
                            " dataVariable#" + dv + "=" + tDataSourceName + 
                            " (compared to dataVariable#0).");
                } else {
                    Test.ensureEqual(tSourceAxisName, (String)tAxisVariables[av][0],
                        errorInMethod + "Observed dimension name doesn't equal " +
                        "expected dimension name for dimension #" + av + 
                        " dataVariable#" + dv + "=" + tDataSourceName + ".");
                    }
 
                //if dv!=0, nothing new to do, so continue
                if (dv != 0) 
                    continue;

                //do dv==0 things: create axisVariables[av]
                Attributes tSourceAttributes = new Attributes();
                PrimitiveArray tSourceValues = null;
                try {
                    dds.getVariable(tSourceAxisName); //throws NoSuchVariableException
                    OpendapHelper.getAttributes(das, tSourceAxisName, tSourceAttributes);
                    tSourceValues = OpendapHelper.getPrimitiveArray(dConnect, "?" + tSourceAxisName);
                    if (reallyVerbose) {
                        int nsv = tSourceValues.size();
                        String2.log("    " + tSourceAxisName + 
                            " source values #0=" + tSourceValues.getDouble(0) +
                            " #" + (nsv-1) + "=" + tSourceValues.getDouble(nsv - 1));
                    }
                } catch (NoSuchVariableException nsve) {
                    //this occurs if no corresponding variable; ignore it
                    //make tSourceValues 0..dimensionSize-1
                    int dadSize1 = dad.getSize() - 1;
                    tSourceValues = av > 0 && dadSize1 < 32000? 
                        new ShortArray(0, dadSize1) :
                        new IntArray(0, dadSize1);
                    tSourceAttributes.add("units", "count"); //"count" is udunits;  "index" isn't, but better?
                    if (reallyVerbose) String2.log("    " + tSourceAxisName + 
                        " not found.  So made from indices 0 - " + dadSize1);
                } //but other exceptions aren't caught

                Attributes tAddAttributes = tAxisVariables == null?
                    new Attributes() : (Attributes)tAxisVariables[av][2];
                String tDestinationAxisName = tAxisVariables == null? null : 
                    (String)tAxisVariables[av][1];
                if (tDestinationAxisName == null || tDestinationAxisName.trim().length() == 0)
                    tDestinationAxisName = tSourceAxisName;

                //String2.log(tSourceAxisName + " pa=" + tSourceValues);             
                //is this the lon axis?
                if (EDV.LON_NAME.equals(tDestinationAxisName)) {
                    lonIndex = av;
                    axisVariables[av] = new EDVLonGridAxis(tSourceAxisName, 
                        tSourceAttributes, tAddAttributes, tSourceValues);

                //is this the lat axis?
                } else if (EDV.LAT_NAME.equals(tDestinationAxisName)) {
                    latIndex = av;
                    axisVariables[av] = new EDVLatGridAxis(tSourceAxisName, 
                        tSourceAttributes, tAddAttributes, tSourceValues);

                //is this the alt axis?
                } else if (EDV.ALT_NAME.equals(tDestinationAxisName)) {
                    altIndex = av;
                    axisVariables[av] = new EDVAltGridAxis(tSourceAxisName, 
                        tSourceAttributes, tAddAttributes, tSourceValues,
                        tAltMetersPerSourceUnit);

                //is this the time axis?
                } else if (EDV.TIME_NAME.equals(tDestinationAxisName)) {
                    timeIndex = av;
                    axisVariables[av] = new EDVTimeGridAxis(tSourceAxisName, 
                        tSourceAttributes, tAddAttributes, tSourceValues);

                //it is some other axis variable
                } else {
                    axisVariables[av] = new EDVGridAxis(
                        tSourceAxisName, tDestinationAxisName,
                        tSourceAttributes, tAddAttributes, tSourceValues);
                    axisVariables[av].setActualRangeFromDestinationMinMax();
                }
            }

            //create the EDVGridData
            dataVariables[dv] = new EDV(
                tDataSourceName, tDataDestName, 
                tDataSourceAttributes, (Attributes)tDataVariables[dv][2],
                dvSourceDataType, 
                Double.NaN, Double.NaN);  //hard to get min and max
            dataVariables[dv].extractAndSetActualRange();

        }

        //ensure the setup is valid
        ensureValid();

        //finally
        if (verbose) String2.log(
            (reallyVerbose? "\n" + toString() : "") +
            "\n*** EDDGridFromDap " + datasetID() + " constructor finished. TIME=" + 
            (System.currentTimeMillis() - constructionStartMillis) + "\n"); 

    }

    /**
     * This makes a sibling dataset, based on the new sourceUrl.
     *
     * @param tSourceUrl
     * @param ensureAxisValuesAreEqual If Integer.MAX_VALUE, no axis sourceValue tests are performed. 
     *    If 0, this tests if sourceValues for axis-variable #0+ are same.
     *    If 1, this tests if sourceValues for axis-variable #1+ are same.
     *    (This is useful if the, for example, lat and lon values vary slightly and you 
     *    are willing to accept the initial values as the correct values.)
     *    Actually, the tests are always done but this determines whether
     *    the error is just logged or whether it throws an exception.
     * @param shareInfo if true, this ensures that the sibling's 
     *    axis and data variables are basically the same as this datasets,
     *    and then makes the new dataset point to the this instance's data structures
     *    to save memory. (AxisVariable #0 isn't duplicated.)
     *    Saving memory is important if there are 1000's of siblings in ERDDAP.
     * @return EDDGrid
     * @throws Throwable if trouble  (e.g., try to shareInfo, but datasets not similar)
     */
    public EDDGrid sibling(String tSourceUrl, int ensureAxisValuesAreEqual, boolean shareInfo) throws Throwable {
        if (verbose) String2.log("EDDGridFromDap.sibling " + tSourceUrl);

        int nAv = axisVariables.length;
        Object tAxisVariables[][] = new Object[nAv][3];
        for (int av = 0; av < nAv; av++) {
            tAxisVariables[av][0] = axisVariables[av].sourceName();
            tAxisVariables[av][1] = axisVariables[av].destinationName();
            tAxisVariables[av][2] = axisVariables[av].addAttributes();
        }

        int nDv = dataVariables.length;
        Object tDataVariables[][] = new Object[nDv][3];
        for (int dv = 0; dv < nDv; dv++) {
            tDataVariables[dv][0] = dataVariables[dv].sourceName();
            tDataVariables[dv][1] = dataVariables[dv].destinationName();
            tDataVariables[dv][2] = dataVariables[dv].addAttributes();
        }

        EDDGridFromDap newEDDGrid = new EDDGridFromDap(
            datasetID, 
            String2.toSSVString(accessibleTo),
            shareInfo? onChange : (StringArray)onChange.clone(), 
            addGlobalAttributes,
            altIndex < 0? 1 : ((EDVAltGridAxis)axisVariables[altIndex]).metersPerSourceUnit(),  
            tAxisVariables,
            tDataVariables,
            getReloadEveryNMinutes(),
            tSourceUrl);

        //if shareInfo, point to same internal data
        if (shareInfo) {

            //ensure similar
            boolean testAV0 = false;
            String results = similar(newEDDGrid, ensureAxisValuesAreEqual, false); 
            if (results.length() > 0)
                throw new SimpleException("Error in EDDGrid.sibling: " + results);

            //shareInfo
            for (int av = 1; av < nAv; av++) //not av0
                newEDDGrid.axisVariables()[av] = axisVariables[av];
            newEDDGrid.dataVariables = dataVariables;

            //shareInfo  (the EDDGrid variables)
            newEDDGrid.axisVariableSourceNames      = axisVariableSourceNames(); //() makes the array
            newEDDGrid.axisVariableDestinationNames = axisVariableDestinationNames();

            //shareInfo  (the EDD variables)
            newEDDGrid.dataVariableSourceNames      = dataVariableSourceNames();
            newEDDGrid.dataVariableDestinationNames = dataVariableDestinationNames();
            newEDDGrid.title                        = title();
            newEDDGrid.summary                      = summary();
            newEDDGrid.institution                  = institution();
            newEDDGrid.infoUrl                      = infoUrl();
            newEDDGrid.cdmDataType                  = cdmDataType();
            newEDDGrid.searchString                 = searchString();
            //not sourceUrl, which will be different
            newEDDGrid.sourceGlobalAttributes       = sourceGlobalAttributes();
            newEDDGrid.addGlobalAttributes          = addGlobalAttributes();
            newEDDGrid.combinedGlobalAttributes     = combinedGlobalAttributes();

        }

        return newEDDGrid;
    }

    /** 
     * This gets data (not yet standardized) from the data 
     * source for this EDDGrid.     
     * 
     * @param tDataVariables
     * @param tConstraints
     * @return a PrimitiveArray[] where the first axisVariables.length elements
     *   are the axisValues and the next tDataVariables.length elements
     *   are the dataValues.
     *   Both the axisValues and dataValues are straight from the source,
     *   not modified.
     * @throws Throwable if trouble
     */
    public PrimitiveArray[] getSourceData(EDV tDataVariables[], IntArray tConstraints) 
        throws Throwable {

        //build String form of the constraint
        //String errorInMethod = "Error in EDDGridFromDap.getSourceData for " + datasetID() + ": "; 
        String constraint = buildDapArrayQuery(tConstraints);

        DConnect dConnect = new DConnect(sourceUrl, acceptDeflate, 1, 1);
        PrimitiveArray results[] = new PrimitiveArray[axisVariables.length + tDataVariables.length];
        for (int dv = 0; dv < tDataVariables.length; dv++) {
            //???why not get all the dataVariables at once?
            //thredds has (and other servers may have) limits to the size of a given request
            //so breaking into parts avoids the problem.

            //get the data
            PrimitiveArray pa[] = null;
            try {
                pa = OpendapHelper.getPrimitiveArrays(dConnect, 
                    "?" + tDataVariables[dv].sourceName() + constraint);
            } catch (Throwable t) {
                requestReloadASAP(); 
                throw new WaitThenTryAgainException(EDStatic.waitThenTryAgain, t); 
            }

            if (pa.length == 1) {
                //it's a DArray
                if (dv == 0) {
                    //GridDataAccessor compares observed and expected axis values
                    int av3 = 0;
                    for (int av = 0; av < axisVariables.length; av++) {
                        results[av] = axisVariables[av].sourceValues().subset(
                            tConstraints.get(av3), 
                            tConstraints.get(av3 + 1), 
                            tConstraints.get(av3 + 2));
                        av3 += 3;
                    }
                }

            } else if (pa.length == axisVariables.length + 1) {
                //it's a DGrid;  test the axes
                if (dv == 0) {
                    //GridDataAccessor compares observed and expected axis values
                    for (int av = 0; av < axisVariables.length; av++) {
                        results[av] = pa[av + 1];
                    }
                } else if (pa.length != 1) {
                    for (int av = 0; av < axisVariables.length; av++) {
                        String tError = results[av].almostEqual(pa[av + 1]); 
                        if (tError.length() > 0) {
                            requestReloadASAP(); 
                            throw new WaitThenTryAgainException(
                                EDStatic.waitThenTryAgain +
                                "\nDetails: The axis values for dataVariable=0,axis=" + av +  
                                ")\ndon't equal the axis values for dataVariable=" + dv + ",axis=" + av + ".\n" +
                                tError);
                        }
                    }
                }

            } else {
                requestReloadASAP(); 
                throw new WaitThenTryAgainException(EDStatic.waitThenTryAgain + 
                    "\nDetails: An unexpected data structure was returned from the source (size observed=" + 
                    pa.length + ", expected=" + (axisVariables.length + 1) + ").");
            }

            //store the grid data 
            results[axisVariables.length + dv] = pa[0];
        }
        return results;
    }

    /** 
     * This generates a rough draft of the datasets.xml entry for an EDDGridFromDap.
     * The XML can then be edited by hand and added to the datasets.xml file.
     *
     * @param url the base url for the dataset (no extension), e.g., 
     *   "http://thredds1.pfeg.noaa.gov/thredds/dodsC/satellite/BA/ssta/5day"
     * @throws Throwable if trouble
     */
    public static String generateDatasetsXml(String url) 
        throws Throwable {

        String2.log("EDDGridFromDap.generateDatasetsXml " + url);
        StringBuffer sb = new StringBuffer();

        //get DConnect
        DConnect dConnect = new DConnect(url, acceptDeflate, 1, 1);
        DAS das;
        try {
            das = dConnect.getDAS(OpendapHelper.DEFAULT_TIMEOUT);
        } catch (Throwable t) {
            throw new SimpleException("Error while getting DAS from " + url + ".das.", t);
        }
        DDS dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);

        //create tables to hold info
        Table axisVars = new Table();
        Table dataVars = new Table();

        //get global attributes and ensure required entries are present (perhaps as "???")
        OpendapHelper.getAttributes(das, "GLOBAL", axisVars.globalAttributes());
        addDummyRequiredGlobalAttributesForDatasetsXml(axisVars.globalAttributes(), "Grid");

        //some data sources have grids with different axis variables
        //so need to write out grids and their axis variables (in case, human needs to separate them)
        sb.append(directionsForDatasetsXml());
        sb.append(
            "<!-- If the multidimensional variables from a data source don't all use the same dimensions,\n" +
            "you need to separate the data source into more than one dataset.\n" +            
            "The multidimensional variables for this data source are:\n");

        //read through the variables[]
        Enumeration vars = dds.getVariables();
        while (vars.hasMoreElements()) {
            BaseType bt = (BaseType)vars.nextElement();

            //ensure it is of correct type
            DArray mainDArray;
            if (bt instanceof DGrid) 
                mainDArray = (DArray)((DGrid)bt).getVariables().nextElement(); //first element is always main array
            else if (bt instanceof DArray) 
                mainDArray = (DArray)bt;
            else continue;

            if (mainDArray.numDimensions() == 1 && 
                mainDArray.getDimension(0).getName().equals(bt.getName()))
                continue; //it's a coordinate variable, so skip it

            //investigate the dimensions
            String dName = bt.getName();
            int numDimensions = mainDArray.numDimensions();
            sb.append(dName + " " + numDimensions + "[");
            for (int av = 0; av < numDimensions; av++) {

                DArrayDimension dad = mainDArray.getDimension(av);
                String aName = dad.getName();
                sb.append(aName + (av < numDimensions - 1? ", " : "]\n"));

                //it's a new axis?
                if (axisVars.findColumnNumber(aName) < 0) {
                    Attributes aAtts = new Attributes();
                    try {
                        OpendapHelper.getAttributes(das, aName, aAtts);
                    } catch (Throwable t) {
                        //e.g., ignore exception for dimension without corresponding coordinate variable
                    }
                    addDummyRequiredVariableAttributesForDatasetsXml(aAtts, aName, false);
                    axisVars.addColumn(axisVars.nColumns(), aName, new DoubleArray(), aAtts); //type doesn't matter
                }
            }

            //add to dataVars
            Attributes dAtts = new Attributes();
            OpendapHelper.getAttributes(das, dName, dAtts);
            addDummyRequiredVariableAttributesForDatasetsXml(dAtts, dName, true);
            dataVars.addColumn(dataVars.nColumns(), dName, new DoubleArray(), dAtts);
        }
        sb.append("-->\n");

        //write the information
        sb.append(
"    <dataset type=\"EDDGridFromDap\" datasetID=\"???\">\n" +
"        <sourceUrl>" + url + "</sourceUrl>\n" +
"        <reloadEveryNMinutes>???60</reloadEveryNMinutes>\n" +  //???
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n");
        sb.append(attsForDatasetsXml(axisVars.globalAttributes(), "        "));

        sb.append(variablesForDatasetsXml(axisVars, "axisVariable", false));
        sb.append(variablesForDatasetsXml(dataVars, "dataVariable", false));
        sb.append(
"    </dataset>\n");
        return sb.toString();
    }

    /** 
     * This is for use by Bob at ERD -- others don't need it.
     * This generates a rough draft of the datasets.xml entry for an EDDGridFromDap
     * for the datasets served by ERD's Thredds server.
     * The XML can then be edited by hand and added to the datasets.xml file.
     *
     * @param search1 e.g., Satellite/aggregsat
     * @param search2 e.g., satellite
     * @throws Throwable if trouble
     */
    public static String generateErdThreddsDatasetXml(String search1, String search2) 
        throws Throwable {

        String2.log("EDDGridFromDap.generateErdThreddsDatasetXml");

        //read DataSet.properties
        ResourceBundle2 dataSetRB2 = new ResourceBundle2("gov.noaa.pfel.coastwatch.DataSet");

        //read the main catalog
        String baseUrl = "http://oceanwatch.pfeg.noaa.gov/thredds/";
        String mainCat = SSR.getUrlResponseString(baseUrl + "catalog.html");
        int mainCatPo = 0;
        int mainCount = 0;

        StringBuffer sb = new StringBuffer();
        sb.append(directionsForDatasetsXml());

        while (true) {
            //search for and extract from... 
            //<a href="Satellite/aggregsatMH/chla/catalog.html"><tt>Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality/</tt></a></td>
            String mainSearch = "<a href=\"" + search1;
            mainCatPo = mainCat.indexOf(mainSearch, mainCatPo + 1);
            if (mainCatPo < 0) // || mainCount++ >= 2) 
                break;

            int q1 = mainCatPo + 8;
            int q2 = mainCat.indexOf('"', q1 + 1);
            int tt1 = mainCat.indexOf("<tt>", mainCatPo);
            int tt2 = mainCat.indexOf("/</tt>", mainCatPo);
            String subCatUrl = mainCat.substring(q1 + 1, q2);
            String twoLetter = mainCat.substring(mainCatPo + mainSearch.length(), mainCatPo + mainSearch.length() + 2);
            String fourLetter = mainCat.substring(mainCatPo + mainSearch.length() + 3, mainCatPo + mainSearch.length() + 7);
            String title = mainCat.substring(tt1 + 4, tt2);
            String longName = EDV.suggestLongName("",
                dataSetRB2.getString(twoLetter + fourLetter + "StandardName", ""));
            String2.log(twoLetter + fourLetter + " = " + title);
       
            //read the sub catalog
            String subCat = SSR.getUrlResponseString(baseUrl + subCatUrl);
            int subCatPo = 0;

            while (true) {
                //search for and extract from... 
                //?dataset=satellite/MH/chla/5day">
                String subSearch = "?dataset=" + search2 + "/" + twoLetter + "/" + fourLetter + "/";
                subCatPo = subCat.indexOf(subSearch, subCatPo + 1);
                if (subCatPo < 0) 
                    break;
                int sq = subCat.indexOf('"', subCatPo);
                Test.ensureTrue(sq >= 0, "subSearch close quote not found.");
                String timePeriod = subCat.substring(subCatPo + subSearch.length(), sq);
                String reload = title.indexOf("Science Quality") >= 0? "10080" : //weekly
                    timePeriod.equals("hday")? "60" : "360"; //hourly or 6hourly
                int tpLength = timePeriod.length();
                String niceTimePeriod = 
                    timePeriod.equals("hday")? "Hourly" :
                    timePeriod.equals("mday")? "Monthly Composite" :
                        timePeriod.substring(0, tpLength - 3) + " Day Composite";
                String2.log("  " + timePeriod + " => " + niceTimePeriod);

                sb.append(
"    <dataset type=\"EDDGridFromDap\" datasetID=\"erd" + twoLetter + fourLetter + timePeriod + "\">\n" +
"        <!-- outside of ERD, use http://oceanwatch.pfeg.noaa.gov -->\n" +
"        <sourceUrl>http://192.168.31.18/thredds/dodsC/satellite/" + twoLetter + 
                "/" + fourLetter + "/" + timePeriod + "</sourceUrl>\n" +
"        <reloadEveryNMinutes>" + reload + "</reloadEveryNMinutes>\n" +
"        <addAttributes> \n" +
"            <att name=\"infoUrl\">http://coastwatch.pfeg.noaa.gov/infog/" + 
                twoLetter + "_" + fourLetter + "_las.html</att>\n" +
"            <att name=\"title\">" + title + " (" + niceTimePeriod + ")</att>\n" +
"            <att name=\"cwhdf_version\" />\n" +
"            <att name=\"cols\" />  \n" +
"            <att name=\"et_affine\" />\n" +
"            <att name=\"gctp_datum\" />\n" +
"            <att name=\"gctp_parm\" />\n" +
"            <att name=\"gctp_sys\" />\n" +
"            <att name=\"gctp_zone\" />\n" +
"            <att name=\"id\" />\n" +
"            <att name=\"pass_date\" />\n" +
"            <att name=\"polygon_latitude\" />\n" +
"            <att name=\"polygon_longitude\" />\n" +
"            <att name=\"rows\" />\n" +
"            <att name=\"start_time\" />\n" +
"            <att name=\"time_coverage_end\" />  \n" +
"            <att name=\"time_coverage_start\" />\n" +
"        </addAttributes>\n" +
"        <longitudeSourceName>lon</longitudeSourceName>\n" +
"        <latitudeSourceName>lat</latitudeSourceName>\n" +
"        <altitudeSourceName>altitude</altitudeSourceName>\n" +
"        <altitudeMetersPerSourceUnit>1</altitudeMetersPerSourceUnit>\n" +
"        <timeSourceName>time</timeSourceName>\n" +
"        <timeSourceFormat></timeSourceFormat> \n" +
"        <dataVariable>\n" +
"            <sourceName>" + twoLetter + fourLetter + "</sourceName>\n" +
"            <destinationName>???" + fourLetter + "</destinationName>\n" +
"            <addAttributes> \n" +
"                <att name=\"ioos_category\">???Temperature</att>\n" +
                    (longName.length() > 0? 
"                <att name=\"long_name\">" + longName + "</att>\n" : "") +
"                <att name=\"actual_range\" /> \n" +
"                <att name=\"numberOfObservations\" /> \n" +
"                <att name=\"percentCoverage\" />\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"    </dataset>\n" +
"\n");
            }
        }
        return sb.toString();
    }



    public static void testForCarleton() throws Throwable {
        //test for Charles Carleton   .nc request failed; others ok
        testVerboseOn();
        EDDGrid gridDataset = (EDDGridFromDap)oneFromDatasetXml("NCOM_Region7_2D"); //should work
        String tName = gridDataset.makeNewFileForDapQuery(null, null,
            "surf_el[(2008-06-12T00:00:00):1:(2008-06-12T00:00:00)][(10.0):100:(65.0)][(-150.0):100:(-100.0)]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Carleton", ".nc"); 
        String results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory + tName, true);
        String expected = 
"time =\n" +
"  {1.2132288E9}\n" +
"latitude =\n" +
"  {10.0, 22.5, 35.0, 47.5, 60.0}\n" +
"longitude =\n" +
"  {-150.0, -137.5, -125.0, -112.5, -100.0}\n" +
"surf_el =\n" +
"  {\n" +
"    {\n" +
"      {0.314, 0.203, 0.505, 0.495, 0.317},\n" +
"      {0.646, 0.6, 0.621, 0.547, -30.0},\n" +
"      {0.326, 0.487, 0.589, -30.0, -30.0},\n" +
"      {-0.34400000000000003, -0.044, 0.318, -30.0, -30.0},\n" +
"      {-30.0, -30.0, -30.0, -30.0, -30.0}\n" +
"    }\n" +
"  }\n" +
"}\n";
        Test.ensureTrue(results.endsWith(expected), "RESULTS=\n" + results);
    }


    public static void testForDave() throws Throwable {
        testVerboseOn();

        //tests for Dave    works, but datasets not always active
        //String tid = "cwAMchlaD1";
        String tid = "cwAMchlaD61";
        //String tid = "cwAMchlaG";
        //String tid = "cwAMchlaAnG";
        EDDGrid gridDataset = (EDDGridFromDap)oneFromDatasetXml(tid);

        String2.log("\n\n***** DDS");
        String tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_" + tid, ".dds"); 
        String2.log(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));

        String2.log("\n\n***** DAS ");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_" + tid, ".das"); 
        String2.log(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));

        String2.log("\n\n***** NCDUMP ");
        tName = gridDataset.makeNewFileForDapQuery(null, null, 
            "chlor_a[(2008-03-29T12:00:00):1:(2008-03-29T12:00:00)][(0.0):1:(0.0)][(16.995124378128825):100:(31.00905181853181)][(-99.01235553141787):100:(-78.99636386525192)]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_" + tid, ".nc"); 
        String2.log(NcHelper.dumpString(EDStatic.fullTestCacheDirectory + tName, true));        

        String2.log("\n\n***** PNGP ");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "chlor_a[0][][][]&.colorBar=Rainbow|C|Log|.04|10|", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_" + tid + "_Map", ".png"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

    }


    public static void testBasic() throws Throwable {
        testVerboseOn();
        EDDGridFromDap gridDataset;
        String name, tName, axisDapQuery, results, expected, error;
        int tPo;
        String userDapQuery  = "chlorophyll[(2007-02-06)][][(29):10:(50)][(225):10:(247)]";
        String graphDapQuery = "chlorophyll[0:10:200][][(29)][(225)]"; 
        String mapDapQuery   = "chlorophyll[200][][(29):(50)][(225):(247)]"; //stride irrelevant 
        StringArray destinationNames = new StringArray();
        IntArray constraints = new IntArray();

        //test regex for integer >=0 
        Test.ensureTrue("0".matches("[0-9]+"), "");
        Test.ensureTrue("123".matches("[0-9]+"), "");
        Test.ensureTrue(!"-1".matches("[0-9]+"), "");
        Test.ensureTrue(!"2.3".matches("[0-9]+"), "");


        gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdMHchla8day"); //should work
        //just comment out to work on some other test

        //test that bad metadata was removed
        //String2.log(
        //    "\n\naddAtt=" + gridDataset.addGlobalAttributes() +
        //    "\n\ncombinedAtt=" + gridDataset.combinedGlobalAttributes());
        Test.ensureEqual(gridDataset.combinedGlobalAttributes().getString("et_affine"), null, "");
        Test.ensureEqual(gridDataset.dataVariables()[0].combinedAttributes().getString("percentCoverage"), null, "");

        //test getUserQueryParts
        Test.ensureEqual(getUserQueryParts(null), new String[]{""}, "");
        Test.ensureEqual(getUserQueryParts(""), new String[]{""}, "");
        Test.ensureEqual(getUserQueryParts("  "), new String[]{"  "}, "");
        Test.ensureEqual(getUserQueryParts("ab%3dc"), new String[]{"ab=c"}, "");  
        Test.ensureEqual(getUserQueryParts("a&b&c"), new String[]{"a", "b", "c"}, "");
        Test.ensureEqual(getUserQueryParts("&&"), new String[]{"", "", ""}, "");
        Test.ensureEqual(getUserQueryParts("a&b=R%26D"), new String[]{"a", "b=R&D"}, "");  //& visible
        Test.ensureEqual(getUserQueryParts("a%26b=\"R%26D\""), new String[]{"a", "b=\"R&D\""}, ""); //& encoded
        Test.ensureEqual(getUserQueryParts("a%26b=\"R%26D\"%26c"), new String[]{"a", "b=\"R&D\"", "c"}, ""); //& encoded
        Test.ensureEqual(getUserQueryParts("a%26b%3dR-D"), new String[]{"a", "b=R-D"}, ""); 

        //*** test getUserQueryParts with invalid queries
        error = "";
        try {
            getUserQueryParts("a%26b=\"R%26D");  //unclosed "
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Query error: A closing doublequote is missing.", 
            "error=" + error);


        //*** test parseQuery with invalid queries
        error = "";
        try {
            gridDataset.parseDataDapQuery("zztop", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Error: destination variable name='zztop' wasn't found.", 
            "error=" + error);

        error = "";
        try {
            gridDataset.parseDataDapQuery("chlorophyll[][][]", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Query error: '[' expected at position=17 (for variable=chlorophyll, axis=3).", 
            "error=" + error);

        error = "";
        try {
            gridDataset.parseDataDapQuery("chlorophyll[(2007-02-06)[][(29):10:(50)][(225):10:(247)]", 
                destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Query error: Invalid requested axis stop=\"\" in constraint isn't an " +
            "integer >= 0 (for variable=chlorophyll, axis=0, constraint=[(2007-02-06)[]).", 
            "error=" + error);

        error = "";
        try {
            gridDataset.parseAxisDapQuery("zztop", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Error: destination variable name='zztop' wasn't found.", 
            "error=" + error);

        error = "";
        try {
            gridDataset.parseAxisDapQuery("latitude,chlorophyll", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Query error: A griddap axis variable query can't include a data variable (chlorophyll).", 
            "error=" + error);

        error = "";
        try {
            gridDataset.parseDataDapQuery("chlorophyll,latitude", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Query error: A griddap data variable query can't include an axis variable (latitude).", 
            "error=" + error);

        error = "";
        try {
            gridDataset.parseAxisDapQuery("latitude[", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureEqual(String2.split(error, '\n')[0],
            "SimpleException: Query error: ']' not found after position=8 (for variable=latitude, axis=2).", 
            "error=" + error);

        //test error message from dataset that doesn't load
        //EDDGrid tGrid = (EDDGrid)oneFromDatasetXml("erdAGtanm3day"); //should fail
        //String2.log("tGrid==null = " + (tGrid == null));
        //if (true) System.exit(0);

        //*** test valid parseDataDapQuery
        String iso = Calendar2.epochSecondsToIsoStringT(1.0260864E9);
        Test.ensureTrue(!gridDataset.isAxisDapQuery(userDapQuery), "");
        gridDataset.parseDataDapQuery(userDapQuery, destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toString(), "chlorophyll", "");
        Test.ensureEqual(constraints.toString(), "206, 1, 206, 0, 1, 0, 2856, 10, 3360, 5399, 10, 5927", "");

        String tDapQuery  = "chlorophyll[(2007-02-06)][][(29):10:(50)][last:1:last]"; //test last
        gridDataset.parseDataDapQuery(tDapQuery, destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toString(), "chlorophyll", "");
        Test.ensureEqual(constraints.toString(), "206, 1, 206, 0, 1, 0, 2856, 10, 3360, 8639, 1, 8639", "");

        Test.ensureTrue(gridDataset.isAxisDapQuery("time"), "");
        gridDataset.parseAxisDapQuery("time", destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toString(), "time", "");        
        Test.ensureEqual(constraints.toString(), "0, 1, 301", ""); //this will increase once in a while

        Test.ensureTrue(gridDataset.isAxisDapQuery("time["), "");
        gridDataset.parseAxisDapQuery("time[(2007-02-06)]", destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toString(), "time", "");
        Test.ensureEqual(constraints.toString(), "206, 1, 206", "");

        gridDataset.parseAxisDapQuery("longitude[ last : 1 : last ]", destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toString(), "longitude", "");
        Test.ensureEqual(constraints.toString(), "8639, 1, 8639", "");

        gridDataset.parseAxisDapQuery("longitude[ last ]", destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toString(), "longitude", "");
        Test.ensureEqual(constraints.toString(), "8639, 1, 8639", "");

        gridDataset.parseAxisDapQuery("longitude[ last - 20]", destinationNames, constraints, false);
        Test.ensureEqual(constraints.toString(), "8619, 1, 8619", "");

        gridDataset.parseAxisDapQuery("longitude[last+-20]", destinationNames, constraints, false);
        Test.ensureEqual(constraints.toString(), "8619, 1, 8619", "");

        gridDataset.parseAxisDapQuery("time[20:10:(2007-02-06)],altitude,longitude[last]", destinationNames, constraints, false);
        Test.ensureEqual(destinationNames.toNewlineString(), "time\naltitude\nlongitude\n", "");
        Test.ensureEqual(constraints.toString(), "20, 10, 206, 0, 1, 0, 8639, 1, 8639", "");

        //lon: incr=0.04166667   n=8640
        gridDataset.parseAxisDapQuery("longitude[(last-0.4166)]", destinationNames, constraints, false);
        Test.ensureEqual(constraints.toString(), "8629, 1, 8629", "");

        gridDataset.parseAxisDapQuery("longitude[(last+-0.4166)]", destinationNames, constraints, false);
        Test.ensureEqual(constraints.toString(), "8629, 1, 8629", "");

        //time: incr=8days  n=272    16days=16*86400=1382400
        gridDataset.parseAxisDapQuery("time[(last-1382400)]", destinationNames, constraints, false);
        Test.ensureEqual(constraints.toString(), "299, 1, 299", ""); //changes sometimes

        gridDataset.parseAxisDapQuery("time[(last+-1382400)]", destinationNames, constraints, false);
        Test.ensureEqual(constraints.toString(), "299, 1, 299", ""); //changes sometimes

        error = "";
        try {
            gridDataset.parseAxisDapQuery("latitude[last-2.0]", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureTrue(error.indexOf("The +/-index value in StartValue=last-2.0 isn't an integer.") >= 0, "error=" + error);

        error = "";
        try {
            gridDataset.parseAxisDapQuery("latitude[(last-2.0a)]", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureTrue(error.indexOf("The +/-value in StartValue=(last-2.0a) isn't valid.") >= 0, "error=" + error);

        error = "";
        try {
            gridDataset.parseAxisDapQuery("latitude[(last*2)]", destinationNames, constraints, false);  
        } catch (Throwable t) {
            error = MustBe.throwableToString(t);
        }
        Test.ensureTrue(error.indexOf("Unexpected character after 'last' in StartValue=(last*2).") >= 0, "error=" + error);

        //***test some edvga things
        EDVGridAxis edvga = gridDataset.axisVariables()[0];
        Test.ensureEqual(edvga.isEvenlySpaced(), false, "");
        Test.ensureEqual(edvga.averageSpacing(), 701820.5980066445, "");  //changes sometimes
        Test.ensureEqual(edvga.spacingDescription(), "8 days 02:57:01 (uneven)", ""); //changes sometimes

        edvga = gridDataset.axisVariables()[1];
        Test.ensureEqual(edvga.isEvenlySpaced(), true, "");
        Test.ensureEqual(edvga.averageSpacing(), Double.NaN, "");
        Test.ensureEqual(edvga.spacingDescription(), "(just one value)", "");

        edvga = gridDataset.axisVariables()[2];
        Test.ensureEqual(edvga.isEvenlySpaced(), true, "");
        Test.ensureEqual(edvga.averageSpacing(), 0.0416666635795323, "");  //not ideal, but true
        Test.ensureEqual(edvga.spacingDescription(), "0.04166666 (even)", "");



        //*** test dapInstructions
        //StringWriter sw = new StringWriter();
        //writeGeneralDapHtmlDocument(EDStatic.erddapUrl, sw);  //for testing, always non-https url
        //results = sw.toString();
        //String2.log(results);
        //expected = "Requests for Gridded Data in ";
        //Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        //expected = "In ERDDAP, time variables always have the name \"" + EDV.TIME_NAME + "\"";
        //Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);


        //*** test getting das for entire dataset
        String2.log("\n****************** EDDGridFromDap test entire dataset\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Entire", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
//"Attributes {\n" +
//"  time {\n" +
//"    Float64 actual_range 1.1886912e+9, 1.1886912e+9;\n"; //this will change sometimes
"    String axis \"T\";\n" +
"    Int32 fraction_digits 0;\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Centered Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  altitude {\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);


        expected =   //test that _FillValue and missing_value are as in sourceAtts
        //but complicated, because that's the value my Grid class uses.
"  chlorophyll {\n" +
"    Float32 _FillValue -9999999.0;\n" +
"    Float64 colorBarMaximum 30.0;\n" +
"    Float64 colorBarMinimum 0.03;\n" +
"    String colorBarScale \"Log\";\n" +
"    String coordsys \"geographic\";\n" +
"    Int32 fraction_digits 2;\n" +
"    String ioos_category \"Ocean Color\";\n" +
"    String long_name \"Concentration Of Chlorophyll In Sea Water\";\n" +
"    Float32 missing_value -9999999.0;\n" +    
"    String standard_name \"concentration_of_chlorophyll_in_sea_water\";\n" +
"    String units \"mg m-3\";\n" +
"  }\n" +
"  NC_GLOBAL {\n" +
"    String acknowledgement \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
"    String cdm_data_type \"Grid\";\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);


        //*** test getting dds for entire dataset
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Float64 time[time = 302];\n" +   //302 will change sometimes
"  Float64 altitude[altitude = 1];\n" +
"  Float64 latitude[latitude = 4320];\n" +
"  Float64 longitude[longitude = 8640];\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 chlorophyll[time = 302][altitude = 1][latitude = 4320][longitude = 8640];\n" +
"    MAPS:\n" +
"      Float64 time[time = 302];\n" +
"      Float64 altitude[altitude = 1];\n" +
"      Float64 latitude[latitude = 4320];\n" +
"      Float64 longitude[longitude = 8640];\n" +
"  } chlorophyll;\n" +
"} erdMHchla8day;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //*** test DAP data access form
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Entire", ".html"); 
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        //*** test getting das for 1 variable     das isn't affected by userDapQuery
        String2.log("\n****************** EDDGridFromDap test 1 variable\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "chlorophyll", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_1Variable", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected =            
//"Attributes {\n" +
//"  time {\n" +
//"    Float64 actual_range 1.17072e+9, 1.17072e+9;\n" + //changes sometimes
"    String axis \"T\";\n" +
"    Int32 fraction_digits 0;\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Centered Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  altitude {\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  NC_GLOBAL {\n" +
"    String acknowledgement \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
"    String cdm_data_type \"Grid\";\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);


        //*** test getting dds for 1 variable
        tName = gridDataset.makeNewFileForDapQuery(null, null, "chlorophyll", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_1Variable", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  GRID {\n" +
"    ARRAY:\n" +   //302 will change sometimes
"      Float32 chlorophyll[time = 302][altitude = 1][latitude = 4320][longitude = 8640];\n" +
"    MAPS:\n" +
"      Float64 time[time = 302];\n" +
"      Float64 altitude[altitude = 1];\n" +
"      Float64 latitude[latitude = 4320];\n" +
"      Float64 longitude[longitude = 8640];\n" +
"  } chlorophyll;\n" +
"} erdMHchla8day;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //********************************************** test getting axis data

        //.asc
        String2.log("\n*** EDDGridFromDap test get .ASC axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Axis", ".asc"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Float64 time[time = 3];\n" +
"  Float64 longitude[longitude = 1];\n" +
"} erdMHchla8day;\n" +
"---------------------------------------------\n" +
"Data:\n" +
"time[3]\n" +
"1.0260864E9, 1.0960704E9, 1.1661408E9\n" +
"longitude[1]\n" +
"359.9792\n";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);
 
        //.csv
        String2.log("\n*** EDDGridFromDap test get .CSV axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Axis", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"time, longitude\n" +
"UTC, degrees_east\n" +
"2002-07-08T00:00:00Z, 359.9792\n" +
"2004-09-25T00:00:00Z, NaN\n" +
"2006-12-15T00:00:00Z, NaN\n";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        //.csv  test of gridName.axisName notation
        String2.log("\n*** EDDGridFromDap test get .CSV axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "chlorophyll.time[0:100:200],chlorophyll.longitude[last]", 
            EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_AxisG.A", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"time, longitude\n" +
"UTC, degrees_east\n" +
"2002-07-08T00:00:00Z, 359.9792\n" +
"2004-09-25T00:00:00Z, NaN\n" +
"2006-12-15T00:00:00Z, NaN\n";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        //.das     which disregards userDapQuery
        String2.log("\n*** EDDGridFromDap test get .DAS axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Axis", ".das"); 
        results = String2.annotatedString(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));
        expected =  //see OpendapHelper.EOL definition for comments
//"Attributes {[10]\n" + 
//"  time {[10]\n" +
//"    Float64 actual_range 1.17072e+9, 1.17072e+9;[10]\n" + //this will change sometimes
"    String axis \"T\";[10]\n" +
"    Int32 fraction_digits 0;[10]\n" +
"    String ioos_category \"Time\";[10]\n" +
"    String long_name \"Centered Time\";[10]\n" +
"    String standard_name \"time\";[10]\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";[10]\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";[10]\n" +
"  }[10]\n" +
"  altitude {[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  NC_GLOBAL {[10]\n" +
"    String acknowledgement \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";[10]\n" +
"    String cdm_data_type \"Grid\";[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);

        //.dds
        String2.log("\n*** EDDGridFromDap test get .DDS axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Axis", ".dds"); 
        results = String2.annotatedString(new String((new ByteArray(
            EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"Dataset {[10]\n" +
"  Float64 time[time = 3];[10]\n" +
"  Float64 longitude[longitude = 1];[10]\n" +
"} erdMHchla8day;[10]\n" +
"[end]";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        //.dods
        String2.log("\n*** EDDGridFromDap test get .DODS axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_Axis", ".dods"); 
        results = String2.annotatedString(new String((new ByteArray(
            EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"Dataset {[10]\n" +
"  Float64 time[time = 3];[10]\n" +
"  Float64 longitude[longitude = 1];[10]\n" +
"} erdMHchla8day;[10]\n" +
"[10]\n" +
"Data:[10]\n" +
"[0][0][0][3][0][0][0][3]A[206][8221]k[0][0][0][0]A[208]U-@[0][0][0]A[209]`y`[0][0][0][0][0][0][1][0][0][0][1]@v[127][170][205][382][402][228][end]";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        //.json
        String2.log("\n*** EDDGridFromDap test get .JSON axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".json"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"time\", \"longitude\"],\n" +
"    \"columnTypes\": [\"String\", \"double\"],\n" +
"    \"columnUnits\": [\"UTC\", \"degrees_east\"],\n" +
"    \"rows\": [\n" +
"      [\"2002-07-08T00:00:00Z\", 359.9792],\n" +
"      [\"2004-09-25T00:00:00Z\", null],\n" +
"      [\"2006-12-15T00:00:00Z\", null]\n" +
"    ]\n" +
"  }\n" +
"}\n";
        Test.ensureEqual(results, expected, "results=" + results);

        //.json with jsonp
        String2.log("\n*** EDDGridFromDap test get .JSON axis data (with jsonp)\n");
        String jsonp = "Some encoded {}\n() ! text";
        tName = gridDataset.makeNewFileForDapQuery(null, null, 
            "time[0:100:200],longitude[last]" + "&.jsonp=" + SSR.percentEncode(jsonp), 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".json"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = jsonp + "(" +
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"time\", \"longitude\"],\n" +
"    \"columnTypes\": [\"String\", \"double\"],\n" +
"    \"columnUnits\": [\"UTC\", \"degrees_east\"],\n" +
"    \"rows\": [\n" +
"      [\"2002-07-08T00:00:00Z\", 359.9792],\n" +
"      [\"2004-09-25T00:00:00Z\", null],\n" +
"      [\"2006-12-15T00:00:00Z\", null]\n" +
"    ]\n" +
"  }\n" +
"}\n" +
")";
        Test.ensureEqual(results, expected, "results=" + results);

        //.mat
        //octave> load('c:/temp/griddap/EDDGridFromDap_Axis.mat');
        //octave> erdMHchla8day
        String matlabAxisQuery = "time[0:100:200],longitude[last]"; 
        String2.log("\n*** EDDGridFromDap test get .MAT axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, matlabAxisQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".mat"); 
        String2.log(".mat test file is " + EDStatic.fullTestCacheDirectory + tName);
        results = File2.hexDump(EDStatic.fullTestCacheDirectory + tName, 1000000);
        String2.log(results);
        expected = 
"4d 41 54 4c 41 42 20 35   2e 30 20 4d 41 54 2d 66   MATLAB 5.0 MAT-f |\n" +
"69 6c 65 2c 20 43 72 65   61 74 65 64 20 62 79 3a   ile, Created by: |\n" +
"20 67 6f 76 2e 6e 6f 61   61 2e 70 66 65 6c 2e 63    gov.noaa.pfel.c |\n" +
"6f 61 73 74 77 61 74 63   68 2e 4d 61 74 6c 61 62   oastwatch.Matlab |\n" +
//"2c 20 43 72 65 61 74 65   64 20 6f 6e 3a 20 54 75   , Created on: Tu |\n" +
//"65 20 4f 63 74 20 31 34   20 30 38 3a 35 36 3a 35   e Oct 14 08:56:5 |\n" +
//"34 20 32 30 30 38 20 20   20 20 20 20 20 20 20 20   4 2008           |\n" +
"20 20 20 20 00 00 00 00   00 00 00 00 01 00 4d 49                 MI |\n" +
"00 00 00 0e 00 00 01 18   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 02 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 0d                    |\n" +
"65 72 64 4d 48 63 68 6c   61 38 64 61 79 00 00 00   erdMHchla8day    |\n" +
"00 04 00 05 00 00 00 20   00 00 00 01 00 00 00 40                  @ |\n" +
"74 69 6d 65 00 00 00 00   00 00 00 00 00 00 00 00   time             |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"6c 6f 6e 67 69 74 75 64   65 00 00 00 00 00 00 00   longitude        |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"00 00 00 0e 00 00 00 48   00 00 00 06 00 00 00 08          H         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 03 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 18   41 ce 94 6b 00 00 00 00           A  k     |\n" +
"41 d0 55 2d 40 00 00 00   41 d1 60 79 60 00 00 00   A U-@   A `y`    |\n" +
"00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 08   40 76 7f aa cd 9e 83 e4           @v       |\n";
        Test.ensureEqual(
            results.substring(0, 71 * 4) + results.substring(71 * 7), //remove the creation dateTime
            expected, "RESULTS(" + EDStatic.fullTestCacheDirectory + tName + ")=\n" + results);

        //.nc
        String2.log("\n*** EDDGridFromDap test get .NC axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory  + tName, true);
        expected = 
"netcdf EDDGridFromDap_Axis.nc {\n" +
" dimensions:\n" +
"   time = 3;\n" +   // (has coord.var)\n" +    //changed when switched to netcdf-java 4.0, 2009-02-23
"   longitude = 1;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   double time(time=3);\n" +
"     :_CoordinateAxisType = \"Time\";\n" +
"     :actual_range = 1.0260864E9, 1.1661408E9; // double\n" + //up-to-date
"     :axis = \"T\";\n" +
"     :fraction_digits = 0; // int\n" +
"     :ioos_category = \"Time\";\n" +
"     :long_name = \"Centered Time\";\n" +
"     :standard_name = \"time\";\n" +
"     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
"     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
"   double longitude(longitude=1);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = 359.9792, 359.9792; // double\n" +
"     :axis = \"X\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 4; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :point_spacing = \"even\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"\n" +
" :acknowledgement = \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
" :cdm_data_type = \"Grid\";\n" +
" :composite = \"true\";\n" +
" :contributor_name = \"NASA GSFC (G. Feldman)\";\n" +
" :contributor_role = \"Source of level 2 data.\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
" :creator_email = \"dave.foley@noaa.gov\";\n" +
" :creator_name = \"NOAA CoastWatch, West Coast Node\";\n" +
" :creator_url = \"http://coastwatch.pfel.noaa.gov\";\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = 
" :infoUrl = \"http://coastwatch.pfeg.noaa.gov/infog/MH_chla_las.html\";\n" +
" :institution = \"NOAA CoastWatch, West Coast Node\";\n" +
" :keywords = \"EARTH SCIENCE > Oceans > Ocean Chemistry > Chlorophyll\";\n" +
" :keywords_vocabulary = \"GCMD Science Keywords\";\n" +
" :license = \"The data may be used and redistributed for free but is not intended for legal use, since it may contain inaccuracies. Neither the data Contributor, CoastWatch, NOAA, nor the United States Government, nor any of their employees or contractors, makes any warranty, express or implied, including warranties of merchantability and fitness for a particular purpose, or assumes any legal liability for the accuracy, completeness, or usefulness, of this information.\";\n" +
" :naming_authority = \"gov.noaa.pfel.coastwatch\";\n" +
" :origin = \"NASA GSFC (G. Feldman)\";\n" +
" :processing_level = \"3\";\n" +
" :project = \"CoastWatch (http://coastwatch.noaa.gov/)\";\n" +
" :projection = \"geographic\";\n" +
" :projection_type = \"mapped\";\n" +
" :references = \"Aqua/MODIS information: http://oceancolor.gsfc.nasa.gov/ . MODIS information: http://coastwatch.noaa.gov/modis_ocolor_overview.html .\";\n" +
" :satellite = \"Aqua\";\n" +
" :sensor = \"MODIS\";\n" +
" :source = \"satellite observation: Aqua, MODIS\";\n" +
" :sourceUrl = \"http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day\";\n" +
" :standard_name_vocabulary = \"CF-1.0\";\n" +
" :summary = \"NOAA CoastWatch distributes chlorophyll-a concentration data from NASA's Aqua Spacecraft.  Measurements are gathered by the Moderate Resolution Imaging Spectroradiometer (MODIS) carried aboard the spacecraft.   This is Science Quality data.\";\n" +
" :time_coverage_end = \"2006-12-15T00:00:00Z\";\n" +
" :time_coverage_start = \"2002-07-08T00:00:00Z\";\n" +
" :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
" :Westernmost_Easting = 359.9792; // double\n" +
" data:\n" +
"time =\n" +
"  {1.0260864E9, 1.0960704E9, 1.1661408E9}\n" +
"longitude =\n" +
"  {359.9792}\n" +
"}\n";
        tPo = results.indexOf(" :infoUrl");
        Test.ensureEqual(results.substring(tPo), expected, "RESULTS=\n" + results);
           

        //.ncHeader
        String2.log("\n*** EDDGridFromDap test get .NCHEADER axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".ncHeader"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"netcdf EDDGridFromDap_Axis.nc {\n" +
" dimensions:\n" +
"   time = 3;\n" +   // (has coord.var)\n" +   //changed when switched to netcdf-java 4.0, 2009-02-23
"   longitude = 1;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   double time(time=3);\n" +
"     :_CoordinateAxisType = \"Time\";\n" +
"     :actual_range = 1.0260864E9, 1.1661408E9; // double\n" +  //up-to-date
"     :axis = \"T\";\n" +
"     :fraction_digits = 0; // int\n" +
"     :ioos_category = \"Time\";\n" +
"     :long_name = \"Centered Time\";\n" +
"     :standard_name = \"time\";\n" +
"     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
"     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
"   double longitude(longitude=1);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = 359.9792, 359.9792; // double\n" +
"     :axis = \"X\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 4; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :point_spacing = \"even\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"\n" +
" :acknowledgement = \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = 
" :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
" :Westernmost_Easting = 359.9792; // double\n" +
" data:\n" +
"}\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

         //.tsv
        String2.log("\n*** EDDGridFromDap test get .TSV axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".tsv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"time\tlongitude\n" +
"UTC\tdegrees_east\n" +
"2002-07-08T00:00:00Z\t359.9792\n" +
"2004-09-25T00:00:00Z\tNaN\n" +
"2006-12-15T00:00:00Z\tNaN\n";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);


        //.xhtml   latitude
        String2.log("\n*** EDDGridFromDap test get .XHTML axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "latitude[0:10:40],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_LatAxis", ".xhtml"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n" +
"  \"http://www.w3.org/TR/xhtml1/xhtml1-transitional.dtd\">\n" +
"<html xmlns=\"http://www.w3.org/1999/xhtml\">\n" +
"<head>\n" +
"  <meta http-equiv=\"content-type\" content=\"text/html; charset=UTF-8\" />\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = 
"<th>latitude</th>\n" +
"<th>longitude</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<th>degrees_north</th>\n" +
"<th>degrees_east</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">-89.97916</td>\n" +
"<td>359.9792</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">-89.56249336420467</td>\n" +
"<td></td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">-89.14582672840935</td>\n" +
"<td></td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">-88.72916009261402</td>\n" +
"<td></td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">-88.3124934568187</td>\n" +
"<td></td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
        int po = results.indexOf("<th>latitude</th>");
        Test.ensureEqual(results.substring(po), expected, "RESULTS=\n" + results);

        //.htmlTable   latitude
        String2.log("\n*** EDDGridFromDap test get .htmlTable axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "latitude[0:10:40],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_LatAxis", ".htmlTable"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \n" +
"  \"http://www.w3.org/TR/html4/loose.dtd\"> \n" +
"<html>\n" +
"<head>\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = 
"<th>latitude</th>\n" +
"<th>longitude</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<th>degrees_north</th>\n" +
"<th>degrees_east</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>-89.97916</td>\n" +
"<td>359.9792</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>-89.56249336420467</td>\n" +
"<td>&nbsp;</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>-89.14582672840935</td>\n" +
"<td>&nbsp;</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>-88.72916009261402</td>\n" +
"<td>&nbsp;</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>-88.3124934568187</td>\n" +
"<td>&nbsp;</td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
        int po8 = results.indexOf("<th>latitude</th>");
        Test.ensureEqual(results.substring(po8), expected, "RESULTS=\n" + results);

        //.xhtml   time
        String2.log("\n*** EDDGridFromDap test get .XHTML axis data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "time[0:100:200],longitude[last]", 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Axis", ".xhtml"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        expected = 
"<tr>\n" +
"<th>time</th>\n" +
"<th>longitude</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<th>UTC</th>\n" +
"<th>degrees_east</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">2002-07-08T00:00:00Z</td>\n" +
"<td>359.9792</td>\n" +
"</tr>\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);
        expected = 
"<tr>\n" +
"<td nowrap=\"nowrap\">2006-12-15T00:00:00Z</td>\n" +
"<td></td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);


        //******************************************** test GridDataRandomAccessor
        //set up GridDataRandomAccessor
        GridDataAccessor gda = new GridDataAccessor(gridDataset, "", userDapQuery, true, true); //rowMajor toNaN
        GridDataRandomAccessor gdra = new GridDataRandomAccessor(gda);
        //maka a new rowMajor gda and test if same data
        gda = new GridDataAccessor(gridDataset, "", userDapQuery, true, true); //rowMajor toNaN
        int current[] = gda.totalIndex().getCurrent(); //the internal object that changes
        int count = 0;
        while (gda.increment()) {
            //String2.log(String2.toCSVString(current));  //to prove that access is rowMajor
            Test.ensureEqual(gda.getDataValueAsDouble(0), 
                gdra.getDataValueAsDouble(current, 0), "count=" + count);
            count++;
        }
        String2.log("Test of GridDataRandomAccess rowMajor succeeded. count=" + count);
        //maka a new columnMajor gda and test if same data
        gda = new GridDataAccessor(gridDataset, "", userDapQuery, false, true); //rowMajor toNaN
        current = gda.totalIndex().getCurrent(); //the internal object that changes
        count = 0;
        while (gda.increment()) {
            //String2.log(String2.toCSVString(current)); //to prove that access is columnMajor
            Test.ensureEqual(gda.getDataValueAsDouble(0), 
                gdra.getDataValueAsDouble(current, 0), "count=" + count);
            count++;
        }
        String2.log("Test of GridDataRandomAccess columnMajor succeeded. count=" + count);


        //********************************************** test getting grid data
        //.asc
        String2.log("\n*** EDDGridFromDap test get .ASC data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".asc"); 
        results = String2.annotatedString(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log("\n.asc results=\n" + results);
        expected = 
"Dataset {[10]\n" +
"  GRID {[10]\n" +
"    ARRAY:[10]\n" +
"      Float32 chlorophyll[time = 1][altitude = 1][latitude = 51][longitude = 53];[10]\n" +
"    MAPS:[10]\n" +
"      Float64 time[time = 1];[10]\n" +
"      Float64 altitude[altitude = 1];[10]\n" +
"      Float64 latitude[latitude = 51];[10]\n" +
"      Float64 longitude[longitude = 53];[10]\n" +
"  } chlorophyll;[10]\n" +
"} erdMHchla8day;[10]\n" +
"---------------------------------------------[10]\n" +
"chlorophyll.chlorophyll[1][1][51][53][10]\n" +
"[0][0][0], -9999999.0, -9999999.0, 0.096, 0.119, -9999999.0, 0.102, 0.09, 0.09,"; //missing values are sourceMissingValue
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = "[0][0][42], -9999999.0,";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);
        expected = 
"chlorophyll.time[1][10]\n" +
"1.17072E9[10]\n" +
"[10]\n" +
"chlorophyll.altitude[1][10]\n" +
"0.0[10]\n" +
"[10]\n" +
"chlorophyll.latitude[51][10]\n" +
"29.020831183144253, 29.437497818939576, 29.8541644547349, 30.27083109053021, 30.68749772632553, 31.104164362120855, 31.52083099791618, 31.9374976337115, 32.354164269506825, 32.77083090530215, 33.18749754109747, 33.604164176892795, 34.02083081268812, 34.43749744848344, 34.854164084278764, 35.27083072007409, 35.68749735586941, 36.104163991664734, 36.52083062746006, 36.93749726325538, 37.354163899050704, 37.77083053484603, 38.18749717064135, 38.60416380643669, 39.020830442232, 39.437497078027334, 39.85416371382264, 40.27083034961795, 40.68749698541329, 41.1041636212086, 41.52083025700394, 41.937496892799246, 42.35416352859458, 42.77083016438989, 43.18749680018523, 43.60416343598054, 44.020830071775876, 44.437496707571185, 44.85416334336652, 45.27082997916183, 45.68749661495717, 46.10416325075248, 46.520829886547816, 46.937496522343125, 47.35416315813846, 47.77082979393377, 48.18749642972911, 48.60416306552442, 49.020829701319755, 49.437496337115064, 49.8541629729104[10]\n" +
"[10]\n" +
"chlorophyll.longitude[53][10]\n" +
"224.97918749730292, 225.39585420255816, 225.8125209078134, 226.22918761306863, 226.64585431832387, 227.0625210235791, 227.47918772883435, 227.89585443408959, 228.31252113934482, 228.72918784460006, 229.1458545498553, 229.56252125511054, 229.97918796036578, 230.39585466562102, 230.81252137087625, 231.2291880761315, 231.64585478138673, 232.06252148664197, 232.4791881918972, 232.89585489715245, 233.31252160240768, 233.72918830766292, 234.14585501291813, 234.56252171817337, 234.9791884234286, 235.39585512868385, 235.8125218339391, 236.22918853919433, 236.64585524444956, 237.0625219497048, 237.47918865496004, 237.89585536021528, 238.31252206547052, 238.72918877072576, 239.145855475981, 239.56252218123623, 239.97918888649147, 240.3958555917467, 240.81252229700195, 241.2291890022572, 241.64585570751242, 242.06252241276766, 242.4791891180229, 242.89585582327814, 243.31252252853338, 243.72918923378862, 244.14585593904386, 244.5625226442991, 244.97918934955433, 245.39585605480957, 245.8125227600648, 246.22918946532005, 246.64585617057529[10]\n[end]";
        int po7 = results.indexOf("chlorophyll.time");
        Test.ensureEqual(results.substring(po7), expected, "RESULTS=\n" + results);

        //.csv
        String2.log("\n*** EDDGridFromDap test get .CSV data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected =  //missing values are "NaN"
"time, altitude, latitude, longitude, chlorophyll\n" +
"UTC, m, degrees_north, degrees_east, mg m-3\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 224.97918749730292, NaN\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.39585420255816, NaN\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.8125209078134, 0.096\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.22918761306863, 0.119\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.64585431832387, NaN\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 227.0625210235791, 0.102\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        expected = "2007-02-06T00:00:00Z, 0.0, 49.437496337115064, 232.06252148664197, 0.367\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        //.csv   test gridName.gridName notation
        String2.log("\n*** EDDGridFromDap test get .CSV data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, "chlorophyll." + userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_DotNotation", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"time, altitude, latitude, longitude, chlorophyll\n" +
"UTC, m, degrees_north, degrees_east, mg m-3\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 224.97918749730292, NaN\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.39585420255816, NaN\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.8125209078134, 0.096\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.22918761306863, 0.119\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.64585431832387, NaN\n" +
"2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 227.0625210235791, 0.102\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        expected = "2007-02-06T00:00:00Z, 0.0, 49.437496337115064, 232.06252148664197, 0.367\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        //.das
        String2.log("\n*** EDDGridFromDap test get .DAS data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".das"); 
        results = String2.annotatedString(new String((new ByteArray(
            EDStatic.fullTestCacheDirectory + tName)).toArray()));
        expected = 
//"Attributes {[10]\n" +
//"  time {[10]\n" +
//"    Float64 actual_range 1.17072e+9, 1.17072e+9;[10]\n" + //changes sometimes
"    String axis \"T\";[10]\n" +
"    Int32 fraction_digits 0;[10]\n" +
"    String ioos_category \"Time\";[10]\n" +
"    String long_name \"Centered Time\";[10]\n" +
"    String standard_name \"time\";[10]\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";[10]\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";[10]\n" +
"  }[10]\n" +
"  altitude {[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  NC_GLOBAL {[10]\n" +
"    String acknowledgement \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";[10]\n" +
"    String cdm_data_type \"Grid\";[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);

        //.dds
        String2.log("\n*** EDDGridFromDap test get .DDS data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".dds"); 
        results = String2.annotatedString(new String((new ByteArray(
            EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"Dataset {[10]\n" +
"  GRID {[10]\n" +
"    ARRAY:[10]\n" +
"      Float32 chlorophyll[time = 1][altitude = 1][latitude = 51][longitude = 53];[10]\n" +
"    MAPS:[10]\n" +
"      Float64 time[time = 1];[10]\n" +
"      Float64 altitude[altitude = 1];[10]\n" +
"      Float64 latitude[latitude = 51];[10]\n" +
"      Float64 longitude[longitude = 53];[10]\n" +
"  } chlorophyll;[10]\n" +
"} erdMHchla8day;[10]\n" +
"[end]";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        //.dods
        String2.log("\n*** EDDGridFromDap test get .DODS data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".dods"); 
        results = String2.annotatedString(new String((new ByteArray(
            EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"Dataset {[10]\n" +
"  GRID {[10]\n" +
"    ARRAY:[10]\n" +
"      Float32 chlorophyll[time = 1][altitude = 1][latitude = 51][longitude = 53];[10]\n" +
"    MAPS:[10]\n" +
"      Float64 time[time = 1];[10]\n" +
"      Float64 altitude[altitude = 1];[10]\n" +
"      Float64 latitude[latitude = 51];[10]\n" +
"      Float64 longitude[longitude = 53];[10]\n" +
"  } chlorophyll;[10]\n" +
"} erdMHchla8day;[10]\n" +
"[10]\n" +
"Data:[10]\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        results = String2.annotatedString(results);
        expected = 
"8][19][3][221]@n[8217][0]/}[16][197]@n[376]U[8222][231][29][173]@n[172][170][218]Q*[8226]@n" +
"[186][0]/[187]7}@n[199]U[8230]%De@n[212][170][218][65533]QM[end][end]";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        //.esriAscii 
        String2.log("\n*** EDDGridFromDap test get .esriAscii data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Map", //must be Map because .asc already used
            ".esriAscii"); 
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = //note that lon values have been shifted from 225 to -135
"ncols 53\n" +
"nrows 51\n" +
"xllcenter -135.02081250269708\n" +
"yllcenter 29.020831183144253\n" +
"cellsize 0.416666635795323\n" +
"nodata_value -9999999\n" +
"-9999999 -9999999 -9999999 -9999999 -9999999 -9999999 -9999999 -9999999 -9999999 -9999999";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        expected = //last row
"-9999999 0.253 0.245 0.267 0.275 0.301 0.396 0.384 0.545 -9999999 -9999999 -9999999 1.205\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        //.json
        String2.log("\n*** EDDGridFromDap test get .JSON data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".json"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = //missing values are "null"
"{\n" +
"  \"table\": {\n" +
"    \"columnNames\": [\"time\", \"altitude\", \"latitude\", \"longitude\", \"chlorophyll\"],\n" +
"    \"columnTypes\": [\"String\", \"double\", \"double\", \"double\", \"float\"],\n" +
"    \"columnUnits\": [\"UTC\", \"m\", \"degrees_north\", \"degrees_east\", \"mg m-3\"],\n" +
"    \"rows\": [\n" +
"      [\"2007-02-06T00:00:00Z\", 0.0, 29.020831183144253, 224.97918749730292, null],\n" +
"      [\"2007-02-06T00:00:00Z\", 0.0, 29.020831183144253, 225.39585420255816, null],\n" +
"      [\"2007-02-06T00:00:00Z\", 0.0, 29.020831183144253, 225.8125209078134, 0.096],\n" +
"      [\"2007-02-06T00:00:00Z\", 0.0, 29.020831183144253, 226.22918761306863, 0.119],\n" +
"      [\"2007-02-06T00:00:00Z\", 0.0, 29.020831183144253, 226.64585431832387, null],\n" +
"      [\"2007-02-06T00:00:00Z\", 0.0, 29.020831183144253, 227.0625210235791, 0.102],\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        expected = 
"      [\"2007-02-06T00:00:00Z\", 0.0, 49.8541629729104, 246.64585617057529, null]\n" +
"    ]\n" +
"  }\n" +
"}\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        //.mat
        //octave> load('c:/temp/griddap/EDDGridFromDap_Data.mat');
        //octave> erdMHchla8day
        String2.log("\n*** EDDGridFromDap test get .MAT data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".mat"); 
        results = File2.hexDump(EDStatic.fullTestCacheDirectory + tName, 1000000);

        //String2.log(results);
        expected = 
"4d 41 54 4c 41 42 20 35   2e 30 20 4d 41 54 2d 66   MATLAB 5.0 MAT-f |\n" +
"69 6c 65 2c 20 43 72 65   61 74 65 64 20 62 79 3a   ile, Created by: |\n" +
"20 67 6f 76 2e 6e 6f 61   61 2e 70 66 65 6c 2e 63    gov.noaa.pfel.c |\n" +
"6f 61 73 74 77 61 74 63   68 2e 4d 61 74 6c 61 62   oastwatch.Matlab |\n" +
"20 20 20 20 00 00 00 00   00 00 00 00 01 00 4d 49                 MI |\n" +
"00 00 00 0e 00 00 2f 98   00 00 00 06 00 00 00 08         /          |\n" +
"00 00 00 02 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 0d                    |\n" +
"65 72 64 4d 48 63 68 6c   61 38 64 61 79 00 00 00   erdMHchla8day    |\n" +
"00 04 00 05 00 00 00 20   00 00 00 01 00 00 00 a0                    |\n" +
"74 69 6d 65 00 00 00 00   00 00 00 00 00 00 00 00   time             |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"61 6c 74 69 74 75 64 65   00 00 00 00 00 00 00 00   altitude         |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"6c 61 74 69 74 75 64 65   00 00 00 00 00 00 00 00   latitude         |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"6c 6f 6e 67 69 74 75 64   65 00 00 00 00 00 00 00   longitude        |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"63 68 6c 6f 72 6f 70 68   79 6c 6c 00 00 00 00 00   chlorophyll      |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 08   41 d1 71 f1 40 00 00 00           A q @    |\n" +
"00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 08   00 00 00 00 00 00 00 00                    |\n" +
"00 00 00 0e 00 00 01 c8   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 33 00 00 00 01   00 00 00 01 00 00 00 00      3             |\n" +
"00 00 00 09 00 00 01 98   40 3d 05 55 31 42 57 70           @= U1BWp |\n" +
"40 3d 6f ff db 68 6a b4   40 3d da aa 85 8e 7d f8   @=o  hj @=    }  |\n" +
"40 3e 45 55 2f b4 91 38   40 3e af ff d9 da a4 7c   @>EU/  8@>     | |\n" +
"40 3f 1a aa 84 00 b7 c0   40 3f 85 55 2e 26 cb 04   @?      @? U.&   |\n" +
"40 3f ef ff d8 4c de 48   40 40 2d 55 41 39 78 c6   @?   L H@@-UA9x  |\n" +
"40 40 62 aa 96 4c 82 68   40 40 97 ff eb 5f 8c 0a   @@b  L h@@   _   |\n" +
"40 40 cd 55 40 72 95 ac   40 41 02 aa 95 85 9f 4e   @@ U@r  @A     N |";
        results = results.substring(0, 71 * 4) + results.substring(71 * 7); //remove the creation dateTime
        results = results.substring(0, expected.length()); //remove the creation dateTime
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        //.nc
        String2.log("\n*** EDDGridFromDap test get .NC data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory  + tName, true);
        expected = 
"netcdf EDDGridFromDap_Data.nc {\n" +
" dimensions:\n" +
"   time = 1;\n" +   // (has coord.var)\n" +   //changed when switched to netcdf-java 4.0, 2009-02-23
"   altitude = 1;\n" +   // (has coord.var)\n" +
"   latitude = 51;\n" +   // (has coord.var)\n" +
"   longitude = 53;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   double time(time=1);\n" +
"     :_CoordinateAxisType = \"Time\";\n" +
"     :actual_range = 1.17072E9, 1.17072E9; // double\n" +
"     :axis = \"T\";\n" +
"     :fraction_digits = 0; // int\n" +
"     :ioos_category = \"Time\";\n" +
"     :long_name = \"Centered Time\";\n" +
"     :standard_name = \"time\";\n" +
"     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
"     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
"   double altitude(altitude=1);\n" +
"     :_CoordinateAxisType = \"Height\";\n" +
"     :_CoordinateZisPositive = \"up\";\n" +
"     :actual_range = 0.0, 0.0; // double\n" +
"     :axis = \"Z\";\n" +
"     :fraction_digits = 0; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Altitude\";\n" +
"     :positive = \"up\";\n" +
"     :standard_name = \"altitude\";\n" +
"     :units = \"m\";\n" +
"   double latitude(latitude=51);\n" +
"     :_CoordinateAxisType = \"Lat\";\n" +
"     :actual_range = 29.020831183144253, 49.8541629729104; // double\n" +
"     :axis = \"Y\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 4; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Latitude\";\n" +
"     :point_spacing = \"even\";\n" +
"     :standard_name = \"latitude\";\n" +
"     :units = \"degrees_north\";\n" +
"   double longitude(longitude=53);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = 224.97918749730292, 246.64585617057529; // double\n" +
"     :axis = \"X\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 4; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :point_spacing = \"even\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"   float chlorophyll(time=1, altitude=1, latitude=51, longitude=53);\n" +
"     :_FillValue = -9999999.0f; // float\n" +
"     :colorBarMaximum = 30.0; // double\n" +
"     :colorBarMinimum = 0.03; // double\n" +
"     :colorBarScale = \"Log\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 2; // int\n" +
"     :ioos_category = \"Ocean Color\";\n" +
"     :long_name = \"Concentration Of Chlorophyll In Sea Water\";\n" +
"     :missing_value = -9999999.0f; // float\n" +
"     :standard_name = \"concentration_of_chlorophyll_in_sea_water\";\n" +
"     :units = \"mg m-3\";\n" +
"\n" +
" :acknowledgement = \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
" :cdm_data_type = \"Grid\";\n" +
" :composite = \"true\";\n" +
" :contributor_name = \"NASA GSFC (G. Feldman)\";\n" +
" :contributor_role = \"Source of level 2 data.\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
" :creator_email = \"dave.foley@noaa.gov\";\n" +
" :creator_name = \"NOAA CoastWatch, West Coast Node\";\n" +
" :creator_url = \"http://coastwatch.pfel.noaa.gov\";\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = //note original missing values
" :infoUrl = \"http://coastwatch.pfeg.noaa.gov/infog/MH_chla_las.html\";\n" +
" :institution = \"NOAA CoastWatch, West Coast Node\";\n" +
" :keywords = \"EARTH SCIENCE > Oceans > Ocean Chemistry > Chlorophyll\";\n" +
" :keywords_vocabulary = \"GCMD Science Keywords\";\n" +
" :license = \"The data may be used and redistributed for free but is not intended for legal use, since it may contain inaccuracies. Neither the data Contributor, CoastWatch, NOAA, nor the United States Government, nor any of their employees or contractors, makes any warranty, express or implied, including warranties of merchantability and fitness for a particular purpose, or assumes any legal liability for the accuracy, completeness, or usefulness, of this information.\";\n" +
" :naming_authority = \"gov.noaa.pfel.coastwatch\";\n" +
" :Northernmost_Northing = 49.8541629729104; // double\n" +
" :origin = \"NASA GSFC (G. Feldman)\";\n" +
" :processing_level = \"3\";\n" +
" :project = \"CoastWatch (http://coastwatch.noaa.gov/)\";\n" +
" :projection = \"geographic\";\n" +
" :projection_type = \"mapped\";\n" +
" :references = \"Aqua/MODIS information: http://oceancolor.gsfc.nasa.gov/ . MODIS information: http://coastwatch.noaa.gov/modis_ocolor_overview.html .\";\n" +
" :satellite = \"Aqua\";\n" +
" :sensor = \"MODIS\";\n" +
" :source = \"satellite observation: Aqua, MODIS\";\n" +
" :sourceUrl = \"http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day\";\n" +
" :Southernmost_Northing = 29.020831183144253; // double\n" +
" :standard_name_vocabulary = \"CF-1.0\";\n" +
" :summary = \"NOAA CoastWatch distributes chlorophyll-a concentration data from NASA's Aqua Spacecraft.  Measurements are gathered by the Moderate Resolution Imaging Spectroradiometer (MODIS) carried aboard the spacecraft.   This is Science Quality data.\";\n" +
" :time_coverage_end = \"2007-02-06T00:00:00Z\";\n" +
" :time_coverage_start = \"2007-02-06T00:00:00Z\";\n" +
" :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
" :Westernmost_Easting = 224.97918749730292; // double\n" +
" data:\n" +
"time =\n" +
"  {1.17072E9}\n" +
"altitude =\n" +
"  {0.0}\n" +
"latitude =\n" +
"  {29.020831183144253, 29.437497818939576, 29.8541644547349, 30.27083109053021, 30.68749772632553, 31.104164362120855, 31.52083099791618, 31.9374976337115, 32.354164269506825, 32.77083090530215, 33.18749754109747, 33.604164176892795, 34.02083081268812, 34.43749744848344, 34.854164084278764, 35.27083072007409, 35.68749735586941, 36.104163991664734, 36.52083062746006, 36.93749726325538, 37.354163899050704, 37.77083053484603, 38.18749717064135, 38.60416380643669, 39.020830442232, 39.437497078027334, 39.85416371382264, 40.27083034961795, 40.68749698541329, 41.1041636212086, 41.52083025700394, 41.937496892799246, 42.35416352859458, 42.77083016438989, 43.18749680018523, 43.60416343598054, 44.020830071775876, 44.437496707571185, 44.85416334336652, 45.27082997916183, 45.68749661495717, 46.10416325075248, 46.520829886547816, 46.937496522343125, 47.35416315813846, 47.77082979393377, 48.18749642972911, 48.60416306552442, 49.020829701319755, 49.437496337115064, 49.8541629729104}\n" +
"longitude =\n" +
"  {224.97918749730292, 225.39585420255816, 225.8125209078134, 226.22918761306863, 226.64585431832387, 227.0625210235791, 227.47918772883435, 227.89585443408959, 228.31252113934482, 228.72918784460006, 229.1458545498553, 229.56252125511054, 229.97918796036578, 230.39585466562102, 230.81252137087625, 231.2291880761315, 231.64585478138673, 232.06252148664197, 232.4791881918972, 232.89585489715245, 233.31252160240768, 233.72918830766292, 234.14585501291813, 234.56252171817337, 234.9791884234286, 235.39585512868385, 235.8125218339391, 236.22918853919433, 236.64585524444956, 237.0625219497048, 237.47918865496004, 237.89585536021528, 238.31252206547052, 238.72918877072576, 239.145855475981, 239.56252218123623, 239.97918888649147, 240.3958555917467, 240.81252229700195, 241.2291890022572, 241.64585570751242, 242.06252241276766, 242.4791891180229, 242.89585582327814, 243.31252252853338, 243.72918923378862, 244.14585593904386, 244.5625226442991, 244.97918934955433, 245.39585605480957, 245.8125227600648, 246.22918946532005, 246.64585617057529}\n" +
"chlorophyll =\n" +
"  {\n" +
"    {\n" +
"      {\n" +
"        {-9999999.0, -9999999.0, 0.096, 0.119, -9999999.0, 0.102, 0.09, 0.09, 0.084, 0.103, -9999999.0, 0.1, 0.071, 0.067, -9999999.0, 0.071, 0.074, -9999999.0, -9999999.0, -9999999.0, 0.074, -9999999.0, 0.09, -9999999.0, -9999999.0, -9999999.0, 0.094, -9999999.0, 0.078, 0.077, 0.093, -9999999.0, -9999999.0, 0.127, 0.189, 0.152, 0.188, 0.151, 0.136, 0.163, -9999999.0, 0.253, 0.245, 0.267, 0.275, 0.301, 0.396, 0.384, 0.545, -9999999.0, -9999999.0, -9999999.0, 1.205},\n";
        tPo = results.indexOf(" :infoUrl");
        Test.ensureEqual(results.substring(tPo, tPo + expected.length()), expected, "RESULTS=\n" + results);


        //.ncHeader
        String2.log("\n*** EDDGridFromDap test get .NCHEADER data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".ncHeader"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        //if (true) System.exit(1);
        expected = 
"netcdf EDDGridFromDap_Data.nc {\n" +
" dimensions:\n" +
"   time = 1;\n" +   // (has coord.var)\n" +   //changed when switched to netcdf-java 4.0, 2009-02-23
"   altitude = 1;\n" +   // (has coord.var)\n" +
"   latitude = 51;\n" +   // (has coord.var)\n" +
"   longitude = 53;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   double time(time=1);\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);

        expected =  //test that sourceMissingValue is intact
         //(but complicated because that is the mv I use in Grid)
         //test that actual_range has been removed
"   float chlorophyll(time=1, altitude=1, latitude=51, longitude=53);\n" +
"     :_FillValue = -9999999.0f; // float\n" +
"     :colorBarMaximum = 30.0; // double\n" +
"     :colorBarMinimum = 0.03; // double\n" +
"     :colorBarScale = \"Log\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 2; // int\n" +
"     :ioos_category = \"Ocean Color\";\n" +
"     :long_name = \"Concentration Of Chlorophyll In Sea Water\";\n" +
"     :missing_value = -9999999.0f; // float\n" +
"     :standard_name = \"concentration_of_chlorophyll_in_sea_water\";\n" +
"     :units = \"mg m-3\";\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        expected = 
" :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
" :Westernmost_Easting = 224.97918749730292; // double\n" +  //note updated value
" data:\n" +
"}\n";
        int po10 = results.indexOf(" :title");
        Test.ensureEqual(results.substring(po10), expected, "RESULTS=\n" + results);

        //.tsv
        String2.log("\n*** EDDGridFromDap test get .TSV data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".tsv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"time\taltitude\tlatitude\tlongitude\tchlorophyll\n" +
"UTC\tm\tdegrees_north\tdegrees_east\tmg m-3\n" +
"2007-02-06T00:00:00Z\t0.0\t29.020831183144253\t224.97918749730292\tNaN\n" +
"2007-02-06T00:00:00Z\t0.0\t29.020831183144253\t225.39585420255816\tNaN\n" +
"2007-02-06T00:00:00Z\t0.0\t29.020831183144253\t225.8125209078134\t0.096\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        expected = "2007-02-06T00:00:00Z\t0.0\t49.8541629729104\t246.64585617057529\tNaN\n";   
        Test.ensureTrue(results.indexOf(expected) > 0, "RESULTS=\n" + results);

        //.xhtml
        String2.log("\n*** EDDGridFromDap test get .XHTMLTABLE data\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Data", ".xhtml"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n" +
"  \"http://www.w3.org/TR/xhtml1/xhtml1-transitional.dtd\">\n" +
"<html xmlns=\"http://www.w3.org/1999/xhtml\">\n" +
"<head>\n";
        Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
        expected = 
"<th>time</th>\n" +
"<th>altitude</th>\n" +
"<th>latitude</th>\n" +
"<th>longitude</th>\n" +
"<th>chlorophyll</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<th>UTC</th>\n" +
"<th>m</th>\n" +
"<th>degrees_north</th>\n" +
"<th>degrees_east</th>\n" +
"<th>mg m-3</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">2007-02-06T00:00:00Z</td>\n" +
"<td>0.0</td>\n" +
"<td>29.020831183144253</td>\n" +
"<td>224.97918749730292</td>\n" +
"<td></td>\n" +   //missing value is ""
"</tr>";
        int po11 = results.indexOf("<th>time</th>");
        Test.ensureEqual(results.substring(po11, po11 + expected.length()), expected, "RESULTS=\n" + results.substring(0, 1000));
        expected = 
"<tr>\n" +
"<td nowrap=\"nowrap\">2007-02-06T00:00:00Z</td>\n" +
"<td>0.0</td>\n" +
"<td>49.8541629729104</td>\n" +
"<td>246.64585617057529</td>\n" +
"<td></td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, 
            "RESULTS=\n" + results.substring(results.length() - expected.length()));

        //test " in attributes
        try {
            EDDGrid tedg = (EDDGridFromDap)oneFromDatasetXml("erdSGchla8day");
            String2.log("\n***raw references=" + tedg.addGlobalAttributes.getString("references"));
            tName = tedg.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                tedg.className() + "Quotes", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            int po9 = results.indexOf("The 4th Pacific");
            String2.log("*** in results: " + results.substring(po9 - 10, po9 + 15));
            expected = " Proceedings of \"\"The 4th Pacific";
            Test.ensureTrue(results.indexOf(expected) < 0, "\nresults=\n" + results);
            expected = " Proceedings of \\\"The 4th Pacific";
            Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. Press ^C to stop or Enter to continue..."); 
        }


        //test loading other datasets
        oneFromDatasetXml("erdAGssta1day");
        //oneFromDatasetXml("erdAGssta8day"); 
        //oneFromDatasetXml("erdAGsstamday"); 
        //oneFromDatasetXml("erdPHssta1day"); 
        //oneFromDatasetXml("erdPHssta8day");
        //oneFromDatasetXml("erdPHsstamday"); 


    }


    public static void testGraphics() throws Throwable {
        testVerboseOn();
        EDDGridFromDap gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdMHchla8day"); 
        String graphDapQuery = "chlorophyll[0:10:200][][(29)][(225)]"; 
        String mapDapQuery   = "chlorophyll[200][][(29):(50)][(225):(247)]"; //stride irrelevant 
        String tName;

        //*** test getting graphs
        String2.log("\n****************** EDDGridFromDap test get graphs\n");

        tName = gridDataset.makeNewFileForDapQuery(null, null, graphDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Graph", ".png"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, graphDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_GraphS", ".smallPng"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, graphDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_GraphL", ".largePng"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, graphDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_GraphS", ".smallPdf"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, graphDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Graph", ".pdf"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, graphDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_GraphL", ".largePdf"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);


        //*** test getting colored surface graph
        //String2.log("\n****************** EDDGridFromDap test get colored surface graph\n");
        //not working yet    time axis is hard to work with
        //String tempDapQuery = "chlorophyll[0:10:20][][(29):1:(50)][(225):1:(225)]";
        //tName = gridDataset.makeNewFileForDapQuery(null, null, tempDapQuery, 
        //    EDStatic.fullTestCacheDirectory, gridDataset.className() + "_CSGraph", ".png"); 
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);


        //*** test getting maps
        String2.log("\n****************** EDDGridFromDap test get maps\n");
        tName = gridDataset.makeNewFileForDapQuery(null, null, mapDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Map", ".png"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, mapDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_MapTP", ".transparentPng"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, mapDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Map", ".pdf"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = gridDataset.makeNewFileForDapQuery(null, null, mapDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Map", ".geotif"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);


        tName = gridDataset.makeNewFileForDapQuery(null, null, mapDapQuery, 
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_Map", ".kml"); 
        //results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log("results=" + results);
        //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        EDDGrid agssta8 = (EDDGridFromDap)oneFromDatasetXml("erdAGssta8day"); //should work
//        tName = agssta8.makeNewFileForDapQuery(null, null, dataIndexExample, 
//            EDStatic.fullTestCacheDirectory, agssta8.className() + "std1", ".csv"); 
//        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
//        String2.log(results);

/* currently not an option (after changes to kml in late 2009)
        //animated map   (works, just not usually an active test)
        if (true) {
            tName = agssta8.makeNewFileForDapQuery(null, null, "sst[200:2:220][][(29):10:(50)][(225):10:(247)]",
                EDStatic.fullTestCacheDirectory, agssta8.className() + "_Animated", ".kml"); 
            String results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            String2.log("results=" + results);
            SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        }
*/
        //this is cool looking (but you have to double click on file in windows explorer to see it)
        tName = agssta8.makeNewFileForDapQuery(null, null, EDStatic.EDDGridMapExample, 
            EDStatic.fullTestCacheDirectory, agssta8.className() + "_ExampleMap", ".geotif"); 
        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName); //doesn't display it or show error message
 
        //give graphs some time to be displayed (other test() may delete the files)
        Math2.sleep(2000);
    }


    public static void testOpendap() throws Throwable {
        try {
            String2.log("\n****************** EDDGridFromDap test opendap\n" +
                "!!!THIS READS DATA FROM SERVER RUNNING ON COASTWATCH: erdMHchla8day on " + 
                EDStatic.erddapUrl + "!!!"); //in tests, always non-https url
            testVerboseOn();
            String results, expected, tName;
            int tPo;
            String userDapQuery  = "chlorophyll[(2007-02-06)][][(29):10:(50)][(225):10:(247)]";
            String graphDapQuery = "chlorophyll[0:10:200][][(29)][(225)]"; 
            String mapDapQuery   = "chlorophyll[200][][(29):(50)][(225):(247)]"; //stride irrelevant 
            StringArray destinationNames = new StringArray();
            IntArray constraints = new IntArray();

            //get das and dds
            String threddsUrl = "http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day";
            String erddapUrl  = EDStatic.erddapUrl + "/griddap/erdMHchla8day"; //in tests, always non-https url
            DConnect threddsConnect = new DConnect(threddsUrl, true, 1, 1);
            DConnect erddapConnect  = new DConnect(erddapUrl,  true, 1, 1); //in tests, always non-https url
            DAS das = erddapConnect.getDAS(OpendapHelper.DEFAULT_TIMEOUT);
            DDS dds = erddapConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);
            PrimitiveArray tpas[], epas[];

            //get global attributes
            Attributes attributes = new Attributes();
            OpendapHelper.getAttributes(das, "GLOBAL", attributes);
            Test.ensureEqual(attributes.getString("contributor_name"), "NASA GSFC (G. Feldman)", "");
            Test.ensureEqual(attributes.getString("keywords"), "EARTH SCIENCE > Oceans > Ocean Chemistry > Chlorophyll", "");

            //get attributes for a dimension 
            attributes.clear();
            OpendapHelper.getAttributes(das, "latitude", attributes);
            Test.ensureEqual(attributes.getString("coordsys"), "geographic", "");
            Test.ensureEqual(attributes.get("fraction_digits"), new IntArray(new int[]{4}), ""); //test if stored in correct form

            //get attributes for grid variable
            attributes.clear();
            OpendapHelper.getAttributes(das, "chlorophyll", attributes);
            Test.ensureEqual(attributes.getString("standard_name"), "concentration_of_chlorophyll_in_sea_water", "");
            Test.ensureEqual(attributes.getString("units"), "mg m-3", "");

            //test get dimension data - all
            String threddsQuery = "lat";
            String erddapQuery  = "latitude";
            String2.log("\nFrom thredds:\n" + String2.annotatedString(new String(
                gov.noaa.pfel.coastwatch.util.SSR.getUrlResponseBytes(threddsUrl + ".asc?" + threddsQuery))));
            String2.log("\nFrom erddap:\n" + String2.annotatedString(new String(
                gov.noaa.pfel.coastwatch.util.SSR.getUrlResponseBytes(erddapUrl + ".asc?" + erddapQuery)))); //in tests, always non-https url
            tpas = OpendapHelper.getPrimitiveArrays(threddsConnect, "?" + threddsQuery);
            epas = OpendapHelper.getPrimitiveArrays(erddapConnect, "?" + erddapQuery);
            Test.ensureEqual(tpas[0].size(), 4320, ""); 
            Test.ensureEqual(epas[0].size(), 4320, "");
            Test.ensureEqual(tpas[0].getDouble(0), -89.97916, ""); 
            Test.ensureEqual(epas[0].getDouble(0), -89.97916, "");
            Test.ensureEqual(tpas[0].getDouble(3000), 35.0208307385969, ""); 
            Test.ensureEqual(epas[0].getDouble(3000), 35.0208307385969, ""); 

            //test get dimension data - part
            threddsQuery = "lat[10:2:20]";
            erddapQuery  = "latitude[10:2:20]";
            String2.log("\nFrom thredds:\n" + String2.annotatedString(new String(
                gov.noaa.pfel.coastwatch.util.SSR.getUrlResponseBytes(threddsUrl + ".asc?" + threddsQuery))));
            String2.log("\nFrom erddap:\n" + String2.annotatedString(new String(
                gov.noaa.pfel.coastwatch.util.SSR.getUrlResponseBytes(erddapUrl + ".asc?" + erddapQuery)))); //in tests, always non-https url
            tpas = OpendapHelper.getPrimitiveArrays(threddsConnect, "?" + threddsQuery);
            epas = OpendapHelper.getPrimitiveArrays(erddapConnect, "?" + erddapQuery);
            Test.ensureEqual(tpas[0].toString(), "-89.56249336420467, -89.4791600370456, -89.39582670988653, -89.31249338272748, -89.22916005556841, -89.14582672840935", ""); 
            Test.ensureEqual(epas[0].toString(), "-89.56249336420467, -89.4791600370456, -89.39582670988653, -89.31249338272748, -89.22916005556841, -89.14582672840935", ""); 

            //get grid data
            //chlorophyll[177][0][2080:20:2500][4500:20:4940]
            String threddsUserDapQuery = "MHchla[177][0][2080:2:2082][4940]";
            String griddapUserDapQuery = "chlorophyll[177][0][2080:2:2082][4940]";
            String2.log("\nFrom thredds:\n" + String2.annotatedString(new String(
                gov.noaa.pfel.coastwatch.util.SSR.getUrlResponseBytes(threddsUrl + ".asc?" + threddsUserDapQuery))));
            String2.log("\nFrom erddap:\n" + String2.annotatedString(new String(
                gov.noaa.pfel.coastwatch.util.SSR.getUrlResponseBytes(erddapUrl + ".asc?" + griddapUserDapQuery)))); //in tests, always non-https url

            //corresponding time varies, so just make sure they match
            tpas = OpendapHelper.getPrimitiveArrays(threddsConnect, "?" + threddsUserDapQuery);
            epas = OpendapHelper.getPrimitiveArrays(erddapConnect, "?" + griddapUserDapQuery);
            Test.ensureEqual(epas[1], tpas[1], ""); //time
            Test.ensureEqual(epas[2], tpas[2], ""); //alt
            Test.ensureEqual(epas[3], tpas[3], ""); //lat
            Test.ensureEqual(epas[4], tpas[4], ""); //lon
            Test.ensureEqual(epas[0], tpas[0], ""); //data
            String tTime = Calendar2.epochSecondsToIsoStringT(tpas[1].getDouble(0));
            float tData1 = tpas[0].getFloat(0);
            float tData2 = tpas[0].getFloat(1);

            //*** test that EDDGridFromDAP works via netcdf-java library
            String2.log("\n****************** EDDGridFromDap test netcdf-java\n");
            NetcdfFile nc = NetcdfDataset.openFile(EDStatic.erddapUrl + "/griddap/erdMHchla8day", null); //in tests, always non-https url
            try {
                results = nc.toString();
                results = NcHelper.decodeNcDump(results); //added with switch to netcdf-java 4.0
                String tUrl = String2.replaceAll(EDStatic.erddapUrl, "http:", "dods:"); //in tests, always non-https url
                expected = 
"netcdf " + tUrl + "/griddap/erdMHchla8day {\n" +
" dimensions:\n" +
"   time = 302;\n" +   // (has coord.var)\n" +  //changes sometimes
"   altitude = 1;\n" +   // (has coord.var)\n" +
"   latitude = 4320;\n" +   // (has coord.var)\n" +
"   longitude = 8640;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   double time(time=302);\n" +
"     :_CoordinateAxisType = \"Time\";\n" +
"     :actual_range = 1.0260864E9, 1.2373344E9; // double\n" +
"     :axis = \"T\";\n" +
"     :fraction_digits = 0; // int\n" +
"     :ioos_category = \"Time\";\n" +
"     :long_name = \"Centered Time\";\n" +
"     :standard_name = \"time\";\n" +
"     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
"     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
"   double altitude(altitude=1);\n" +
"     :_CoordinateAxisType = \"Height\";\n" +
"     :_CoordinateZisPositive = \"up\";\n" +
"     :actual_range = 0.0, 0.0; // double\n" +
"     :axis = \"Z\";\n" +
"     :fraction_digits = 0; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Altitude\";\n" +
"     :positive = \"up\";\n" +
"     :standard_name = \"altitude\";\n" +
"     :units = \"m\";\n" +
"   double latitude(latitude=4320);\n" +
"     :_CoordinateAxisType = \"Lat\";\n" +
"     :actual_range = -89.97916, 89.97916; // double\n" +
"     :axis = \"Y\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 4; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Latitude\";\n" +
"     :point_spacing = \"even\";\n" +
"     :standard_name = \"latitude\";\n" +
"     :units = \"degrees_north\";\n" +
"   double longitude(longitude=8640);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = 0.02083333, 359.9792; // double\n" +
"     :axis = \"X\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 4; // int\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :point_spacing = \"even\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"   float chlorophyll(time=302, altitude=1, latitude=4320, longitude=8640);\n" +
"     :_CoordinateAxes = \"time altitude latitude longitude \";\n" +
"     :_FillValue = -9999999.0f; // float\n" +
"     :colorBarMaximum = 30.0; // double\n" +
"     :colorBarMinimum = 0.03; // double\n" +
"     :colorBarScale = \"Log\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :fraction_digits = 2; // int\n" +
"     :ioos_category = \"Ocean Color\";\n" +
"     :long_name = \"Concentration Of Chlorophyll In Sea Water\";\n" +
"     :missing_value = -9999999.0f; // float\n" +
"     :standard_name = \"concentration_of_chlorophyll_in_sea_water\";\n" +
"     :units = \"mg m-3\";\n" +
"\n" +
" :acknowledgement = \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
" :cdm_data_type = \"Grid\";\n" +
" :composite = \"true\";\n" +
" :contributor_name = \"NASA GSFC (G. Feldman)\";\n" +
" :contributor_role = \"Source of level 2 data.\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
" :creator_email = \"dave.foley@noaa.gov\";\n" +
" :creator_name = \"NOAA CoastWatch, West Coast Node\";\n" +
" :creator_url = \"http://coastwatch.pfel.noaa.gov\";\n" +
" :date_created = \"2009-04-21Z\";\n" +
" :date_issued = \"2009-04-21Z\";\n" +
" :Easternmost_Easting = 359.9792; // double\n" +
" :geospatial_lat_max = 89.97916; // double\n" +
" :geospatial_lat_min = -89.97916; // double\n" +
" :geospatial_lat_resolution = 0.0416666635795323; // double\n" +
" :geospatial_lat_units = \"degrees_north\";\n" +
" :geospatial_lon_max = 359.9792; // double\n" +
" :geospatial_lon_min = 0.02083333; // double\n" +
" :geospatial_lon_resolution = 0.04166667052552379; // double\n" +
" :geospatial_lon_units = \"degrees_east\";\n" +
" :geospatial_vertical_max = 0.0; // double\n" +
" :geospatial_vertical_min = 0.0; // double\n" +
" :geospatial_vertical_positive = \"up\";\n" +
" :geospatial_vertical_units = \"m\";\n" +
" :history = \"NASA GSFC (G. Feldman)\n" +  //important test   re netcdf 4.0
"20";
                Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

                expected = 
" :satellite = \"Aqua\";\n" +
" :sensor = \"MODIS\";\n" +
" :source = \"satellite observation: Aqua, MODIS\";\n" +
" :sourceUrl = \"http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day\";\n" +
" :Southernmost_Northing = -89.97916; // double\n" +
" :standard_name_vocabulary = \"CF-1.0\";\n" +
" :summary = \"NOAA CoastWatch distributes chlorophyll-a concentration data from NASA's Aqua Spacecraft.  Measurements are gathered by the Moderate Resolution Imaging Spectroradiometer (MODIS) carried aboard the spacecraft.   This is Science Quality data.\";\n" +
" :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
" :Westernmost_Easting = 0.02083333; // double\n" +
"}\n";
                Test.ensureEqual(results.substring(results.indexOf(" :satellite =")), expected, "RESULTS=\n" + results);

                attributes.clear();
                NcHelper.getGlobalAttributes(nc, attributes);
                Test.ensureEqual(attributes.getString("contributor_name"), "NASA GSFC (G. Feldman)", "");
                Test.ensureEqual(attributes.getString("keywords"), "EARTH SCIENCE > Oceans > Ocean Chemistry > Chlorophyll", "");

                //get attributes for a dimension 
                Variable ncLat = nc.findVariable("latitude");
                attributes.clear();
                NcHelper.getVariableAttributes(ncLat, attributes);
                Test.ensureEqual(attributes.getString("coordsys"), "geographic", "");
                Test.ensureEqual(attributes.get("fraction_digits"), new IntArray(new int[]{4}), ""); //test if stored in correct form

                //get attributes for grid variable
                Variable ncChl = nc.findVariable("chlorophyll");
                attributes.clear();
                NcHelper.getVariableAttributes(ncChl, attributes);
                Test.ensureEqual(attributes.getString("standard_name"), "concentration_of_chlorophyll_in_sea_water", "");
                Test.ensureEqual(attributes.getString("units"), "mg m-3", "");

                //test get dimension data - all
                PrimitiveArray pa = NcHelper.getPrimitiveArray(ncLat);
                Test.ensureEqual(pa.getElementType(), double.class, "");
                Test.ensureEqual(pa.size(), 4320, "");
                Test.ensureEqual(pa.getDouble(0), -89.97916, "");
                Test.ensureEqual(pa.getDouble(4319), 89.97916, ""); 

                //test get dimension data - part
                pa = NcHelper.getPrimitiveArray(ncLat, 10, 20);
                Test.ensureEqual(pa.getElementType(), double.class, "");
                Test.ensureEqual(pa.size(), 11, "");
                Test.ensureEqual(pa.getDouble(0), -89.56249336420467, "");
                Test.ensureEqual(pa.getDouble(10), -89.14582672840935, ""); 

                //get grid data
                pa = NcHelper.get4DValues(ncChl, 4500, 2080, 0, 170, 190); //x,y,z,t1,t2
                Test.ensureEqual(pa.getElementType(), float.class, "");
                String2.log("pa=" + pa);
                Test.ensureEqual(pa.size(), 21, "");
                Test.ensureEqual(pa.getFloat(0), 0.113f, "");
                Test.ensureEqual(pa.getFloat(1), -9999999.0f, "");

            } finally {
                nc.close();
            }


            //*** test that EDDGridFromDap can treat itself as a datasource
            String2.log("\n*** EDDGridFromDap test can treat itself as a datasource\n");
            ArrayList tDataVariables = new ArrayList();

            EDDGrid eddGrid2 = new EDDGridFromDap(
                "erddapChlorophyll", //String tDatasetID, 
                null,
                null,
                null,  
                1, //tAltMetersPerSourceUnit
                null,  
                new Object[][] {
                    { //dataVariables[dvIndex][0=sourceName, 1=destName, 2=addAttributes]
                        "chlorophyll", null, null}},
                60, //int tReloadEveryNMinutes,
                erddapUrl); //sourceUrl); //in tests, always non-https url

            //.xhtml
            tName = eddGrid2.makeNewFileForDapQuery(null, null, griddapUserDapQuery, 
                EDStatic.fullTestCacheDirectory, eddGrid2.className() + "_Itself", ".xhtml"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n" +
"  \"http://www.w3.org/TR/xhtml1/xhtml1-transitional.dtd\">\n" +
"<html xmlns=\"http://www.w3.org/1999/xhtml\">\n" +
"<head>\n" +
"  <meta http-equiv=\"content-type\" content=\"text/html; charset=UTF-8\" />\n" +
"  <title>EDDGridFromDap_Itself</title>\n" +
"\n" +
"<style type=\"text/CSS\"> <!--\n" +
"  table.erd {border-collapse:collapse; border:1px solid gray; }\n" +
"  table.erd th, table.erd td {padding:2px; border:1px solid gray; }\n" +
"--> </style>\n" +
"</head>\n" +
"<body style=\"color:black; background:white; font-family:Arial,Helvetica,sans-serif; font-size:85%;\">\n" +
"<table bgcolor=\"#FFFFCC\" border=\"1\" cellpadding=\"2\" cellspacing=\"0\">\n" +
"<tr>\n" +
"<th>time</th>\n" +
"<th>altitude</th>\n" +
"<th>latitude</th>\n" +
"<th>longitude</th>\n" +
"<th>chlorophyll</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<th>UTC</th>\n" +
"<th>m</th>\n" +
"<th>degrees_north</th>\n" +
"<th>degrees_east</th>\n" +
"<th>mg m-3</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">" + tTime + "Z</td>\n" +
"<td>0.0</td>\n" +
"<td nowrap=\"nowrap\">-3.312499754572812</td>\n" +
"<td>205.8541857260875</td>\n" +
"<td>" + (tData1 == -9999999.0f? "" : "" + tData1) + "</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap=\"nowrap\">" + tTime + "Z</td>\n" +
"<td>0.0</td>\n" +
"<td nowrap=\"nowrap\">-3.2291664274137446</td>\n" +
"<td>205.8541857260875</td>\n" +
"<td>" + (tData2 == -9999999.0f? "" : "" + tData2) + "</td>\n" +
"</tr>\n" +
"</table>\n" +
"</body>\n" +
"</html>\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);


            //check error...
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing " + EDStatic.erddapUrl + //in tests, always non-https url
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * testGenerateDatasetsXml
     */
    public static void testGenerateDatasetsXml() throws Throwable {
        testVerboseOn();

        //don't test local dataset because of dns/numericIP problems
        String tUrl = "http://dapper.pmel.noaa.gov/dapper/oscar/world-unfilter.nc";

        //test local generateDatasetsXml
String expected1 = 
"<!-- Directions:\n" +
" * Read about this type of dataset in\n" +
"   http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html .\n" +
" * Read http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html#addAttributes\n" +
"   so that you understand about sourceAttributes and addAttributes.\n" +
" * All of the content below that starts with \"???\" is a guess. It must be edited.\n" +
" * All of the other tags and their content are based on information from the source.\n" +
" * For the att tags, you should either:\n" +
"    * Delete the att tag (so ERDDAP will use the unchanged source attribute).\n" +
"    * Change the att tag's value (because it isn't quite right).\n" +
"    * Or, remove the att tag's value, but leave the att tag\n" +
"      (so the source attribute will be removed by ERDDAP).\n" +
" * You can reorder data variables, but don't reorder axis variables.\n" +
" * The current IOOS category options are:\n" +
"      Bathymetry, Biology, Bottom Character, Contaminants, Currents, Dissolved \n" +
"      Nutrients, Dissolved O2, Ecology, Fish Abundance, Fish Species, Heat \n" +
"      Flux, Ice Distribution, Identifier, Location, Meteorology, Ocean Color, \n" +
"      Optical Properties, Other, Pathogens, Phytoplankton Species, Pressure, \n" +
"      Productivity, Salinity, Sea Level, Surface Waves, Taxonomy, Temperature, \n" +
"      Time, Unknown, Wind, Zooplankton Species, Zooplankton Abundance\n" +
" * For longitude, latitude, altitude (or depth), and time variables:\n" +
"   * If the sourceName isn't \"longitude\", \"latitude\", \"altitude\", or \"time\",\n" +
"     you need to specify \"longitude\", \"latitude\", \"altitude\", or \"time\"\n" +
"     with a destinationName tag.\n" +
"   * For EDDTable datasets: if possible, add an actual_range attribute\n" +
"     if one isn't already there.\n" +
"   * (Usually) remove all other attributes. They usually aren't needed.\n" +
"     Attributes will be added automatically.\n" +
"-->\n" +
"<!-- If the multidimensional variables from a data source don't all use the same dimensions,\n" +
"you need to separate the data source into more than one dataset.\n" +            
"The multidimensional variables for this data source are:\n" +
"u 4[time, depth, latitude, longitude]\n" +
"v 4[time, depth, latitude, longitude]\n" +
"u_anom 4[time, depth, latitude, longitude]\n" +
"v_anom 4[time, depth, latitude, longitude]\n" +
"-->\n" +
"    <dataset type=\"EDDGridFromDap\" datasetID=\"???\">\n" +
"        <sourceUrl>http://dapper.pmel.noaa.gov/dapper/oscar/world-unfilter.nc</sourceUrl>\n" +
"        <reloadEveryNMinutes>???60</reloadEveryNMinutes>\n" +
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n" +
"        <addAttributes>\n" +
"            <att name=\"ANOM_MEAN_PERIOD\">1993-01-01 to 2006-12-31</att>\n" +
"            <att name=\"cdm_data_type\">Grid</att>\n" +
"            <att name=\"company\">Earth &amp; Space Research, Seattle, WA</att>\n" +
"            <att name=\"contact\">Fabrice Bonjean (bonjean@esr.org) or John T. Gunn (gunn@esr.org)</att>\n" +
"            <att name=\"Conventions\">???COARDS, CF-1.0, Unidata Dataset Discovery v1.0</att>\n";

//"            <att name=\"CREATION_DATE\">15:31 22-JUL-2008</att>\n" +
//"            <att name=\"DATASUBTYPE\">unfiltered</att>\n" +
//"            <att name=\"DATATYPE\">5-Day Interval</att>\n" +
//"            <att name=\"date\">27-Apr-2007</att>\n" +
//"            <att name=\"description\">Sea Surface Velocity</att>\n" +
//"            <att name=\"GEORANGE\">20.5E to 419.5E, 69.5S to 69.5N</att>\n" +
//"            <att name=\"infoUrl\">???</att>\n" +
//"            <att name=\"institution\">???</att>\n" +
//"            <att name=\"license\">???[standard]</att>\n" +
//"            <att name=\"PERIOD\">1992-10-21 to 2008-07-22</att>\n" +

String expected2 = 
//"            <att name=\"
"reference1\">Bonjean F. and G.S.E. Lagerloef, 2002 ,\\&quot;Diagnostic model and analysis of the surface currents in the tropical Pacific ocean\\&quot;, J. Phys. Oceanogr., 32, 2,938-2,954</att>\n" +
"            <att name=\"source\">Gary Lagerloef, ESR (lager@esr.org) and Fabrice Bonjean (bonjean@esr.org)</att>\n" +
"            <att name=\"standard_name_vocabulary\">???CF-11</att>\n" +
"            <att name=\"summary\">???</att>\n" +
"            <att name=\"title\">???</att>\n" +
"            <att name=\"VARIABLE\">Ocean Surface Currents</att>\n" +
"            <att name=\"version\" type=\"float\">2006.0</att>\n" +
"        </addAttributes>\n" +
"        <axisVariable>\n" +
"            <sourceName>time</sourceName>\n" +
"            <destinationName>???time</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Time</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">day since 1992-10-05 00:00:00</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>depth</sourceName>\n" +
"            <destinationName>???depth</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Depth</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">meter</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>latitude</sourceName>\n" +
"            <destinationName>???latitude</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Latitude</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">degrees_north</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>longitude</sourceName>\n" +
"            <destinationName>???longitude</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Longitude</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">degrees_east</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>u</sourceName>\n" +
"            <destinationName>???u</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"generic_name\">u</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Ocean Surface Zonal Currents</att>\n" +
"                <att name=\"missing_value\" type=\"double\">NaN</att>\n" +
"                <att name=\"name\">u</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">meter/sec</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>v</sourceName>\n" +
"            <destinationName>???v</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"generic_name\">v</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Ocean Surface Meridional Currents</att>\n" +
"                <att name=\"missing_value\" type=\"double\">NaN</att>\n" +
"                <att name=\"name\">v</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">meter/sec</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>u_anom</sourceName>\n" +
"            <destinationName>???u_anom</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"generic_name\">u_anom</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Ocean Surface Zonal Currents Anomaly</att>\n" +
"                <att name=\"missing_value\" type=\"double\">NaN</att>\n" +
"                <att name=\"name\">u_anom</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">meter/sec</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>v_anom</sourceName>\n" +
"            <destinationName>???v_anom</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"generic_name\">v_anom</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Ocean Surface Meridional Currents Anomaly</att>\n" +
"                <att name=\"missing_value\" type=\"double\">NaN</att>\n" +
"                <att name=\"name\">v_anom</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">meter/sec</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"    </dataset>\n";

        try {
            String results = generateDatasetsXml(tUrl);
            Test.ensureEqual(results.substring(0, Math.min(results.length(), expected1.length())), 
                expected1, "results=\n" + results);

            int po = results.indexOf(expected2.substring(0, 10));
            Test.ensureEqual(results.substring(po, po + expected2.length()), expected2, "results=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError using generateDatasetsXml locally." +
                "\nPress ^C to stop or Enter to continue..."); 
        }


        //test BA ssta 5day used as demo in erddap
        tUrl = "http://thredds1.pfeg.noaa.gov/thredds/dodsC/satellite/BA/ssta/5day";

        try {
            String results = generateDatasetsXml(tUrl);

        //String2.log("results=\n" + results);

        //test local generateDatasetsXml
expected1 = 
"<!-- Directions:\n" +
" * Read about this type of dataset in\n" +
"   http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html .\n" +
" * Read http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html#addAttributes\n" +
"   so that you understand about sourceAttributes and addAttributes.\n" +
" * All of the content below that starts with \"???\" is a guess. It must be edited.\n" +
" * All of the other tags and their content are based on information from the source.\n" +
" * For the att tags, you should either:\n" +
"    * Delete the att tag (so ERDDAP will use the unchanged source attribute).\n" +
"    * Change the att tag's value (because it isn't quite right).\n" +
"    * Or, remove the att tag's value, but leave the att tag\n" +
"      (so the source attribute will be removed by ERDDAP).\n" +
" * You can reorder data variables, but don't reorder axis variables.\n" +
" * The current IOOS category options are:\n" +
"      Bathymetry, Biology, Bottom Character, Contaminants, Currents, Dissolved \n" +
"      Nutrients, Dissolved O2, Ecology, Fish Abundance, Fish Species, Heat \n" +
"      Flux, Ice Distribution, Identifier, Location, Meteorology, Ocean Color, \n" +
"      Optical Properties, Other, Pathogens, Phytoplankton Species, Pressure, \n" +
"      Productivity, Salinity, Sea Level, Surface Waves, Taxonomy, Temperature, \n" +
"      Time, Unknown, Wind, Zooplankton Species, Zooplankton Abundance\n" +
" * For longitude, latitude, altitude (or depth), and time variables:\n" +
"   * If the sourceName isn't \"longitude\", \"latitude\", \"altitude\", or \"time\",\n" +
"     you need to specify \"longitude\", \"latitude\", \"altitude\", or \"time\"\n" +
"     with a destinationName tag.\n" +
"   * For EDDTable datasets: if possible, add an actual_range attribute\n" +
"     if one isn't already there.\n" +
"   * (Usually) remove all other attributes. They usually aren't needed.\n" +
"     Attributes will be added automatically.\n" +
"-->\n" +
"<!-- If the multidimensional variables from a data source don't all use the same dimensions,\n" +
"you need to separate the data source into more than one dataset.\n" +
"The multidimensional variables for this data source are:\n" +
"BAssta 4[time, altitude, lat, lon]\n" +
"-->\n" +
"    <dataset type=\"EDDGridFromDap\" datasetID=\"???\">\n" +
"        <sourceUrl>http://thredds1.pfeg.noaa.gov/thredds/dodsC/satellite/BA/ssta/5day</sourceUrl>\n" +
"        <reloadEveryNMinutes>???60</reloadEveryNMinutes>\n" +
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n" +
"        <addAttributes>\n" +
"            <att name=\"acknowledgement\">NOAA NESDIS COASTWATCH, NOAA SWFSC ERD</att>\n" +
"            <att name=\"cdm_data_type\">Grid</att>\n" +
"            <att name=\"cols\" type=\"int\">3601</att>\n" +
"            <att name=\"composite\">true</att>\n" +
"            <att name=\"contributor_name\">Remote Sensing Systems Inc, JAXA, NASA, OSDPD, CoastWatch</att>\n" +
"            <att name=\"contributor_role\">Source of level 2 data.</att>\n" +
"            <att name=\"Conventions\">COARDS, CF-1.0, Unidata Dataset Discovery v1.0, CWHDF??????COARDS, CF-1.0, Unidata Dataset Discovery v1.0</att>\n";

expected2 = 
"\"axis\">T</att>\n" +
"                <att name=\"fraction_digits\" type=\"int\">0</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Centered Time</att>\n" +
"                <att name=\"standard_name\">time</att>\n" +
"                <att name=\"units\">seconds since 1970-01-01T00:00:00Z</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>altitude</sourceName>\n" +
"            <destinationName>???altitude</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"_CoordinateAxisType\">Height</att>\n" +
"                <att name=\"_CoordinateZisPositive\">up</att>\n" +
"                <att name=\"actual_range\" type=\"doubleList\">0.0 0.0</att>\n" +
"                <att name=\"axis\">Z</att>\n" +
"                <att name=\"fraction_digits\" type=\"int\">0</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Altitude</att>\n" +
"                <att name=\"positive\">up</att>\n" +
"                <att name=\"standard_name\">altitude</att>\n" +
"                <att name=\"units\">m</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>lat</sourceName>\n" +
"            <destinationName>???lat</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"_CoordinateAxisType\">Lat</att>\n" +
"                <att name=\"actual_range\" type=\"doubleList\">-75.0 75.0</att>\n" +
"                <att name=\"axis\">Y</att>\n" +
"                <att name=\"coordsys\">geographic</att>\n" +
"                <att name=\"fraction_digits\" type=\"int\">1</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Latitude</att>\n" +
"                <att name=\"point_spacing\">even</att>\n" +
"                <att name=\"standard_name\">latitude</att>\n" +
"                <att name=\"units\">degrees_north</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>lon</sourceName>\n" +
"            <destinationName>???lon</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"_CoordinateAxisType\">Lon</att>\n" +
"                <att name=\"actual_range\" type=\"doubleList\">0.0 360.0</att>\n" +
"                <att name=\"axis\">X</att>\n" +
"                <att name=\"coordsys\">geographic</att>\n" +
"                <att name=\"fraction_digits\" type=\"int\">1</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">Longitude</att>\n" +
"                <att name=\"point_spacing\">even</att>\n" +
"                <att name=\"standard_name\">longitude</att>\n" +
"                <att name=\"units\">degrees_east</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>BAssta</sourceName>\n" +
"            <destinationName>???BAssta</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"_FillValue\" type=\"float\">-9999999.0</att>\n";

            Test.ensureEqual(results.substring(0, Math.min(results.length(), expected1.length())), 
                expected1, "results=\n" + results);

            int po = results.indexOf(expected2.substring(0, 10));
            Test.ensureEqual(results.substring(po, po + expected2.length()), expected2, "results=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError using generateDatasetsXml locally." +
                "\nPress ^C to stop or Enter to continue..."); 
        }

        //test generateDatasetsXml on EDStatic.erddapUrl //in tests, always non-https url
        /*   not yet done. I need to make service able to return just xml (not a web page).
        try {
            String results = SSR.getUrlResponseString(
                EDStatic.erddapUrl + //in tests, always non-https url
                    "/generateDatasetsXml/EDDGridFromDap.html?" + 
                "sourceUrl=" + tUrl);
            Test.ensureEqual(results.substring(0, Math.min(results.length(), expected1.length())), 
                expected1, "results=\n" + results);

            int po = results.indexOf(expected2.substring(0, 10));
            Test.ensureEqual(results.substring(po), expected2, "results=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing generateDatasetsXml on " + 
                    EDStatic.erddapUrl + //in tests, always non-https url
                "\nPress ^C to stop or Enter to continue..."); 
        }
        */
    }


        //*** test standard examples
// idExample         = "erdAGssta8day";
// dimensionExample  = "latitude[0:10:100]";
// noHyperExample    = "sst";
// dataIndexExample  = "sst[656][0][0:100:1500][0:100:3600]";
// dataValueExample  = "sst[(1192924800)][0][(-75):100:(75)][(0):100:(360)]";
// dataTimeExample   = "sst[(2007-10-21T00:00:00)][0][(-75):100:(75)][(0):100:(360)]";
// graphExample      = "sst[(2007-07-01):(2007-10-21)][0][(29)][(225)]";
// mapExample        = "sst[(2007-10-21)][0][(-75):(75)][(0):(180)]";  

    
    public static void testScaleAddOffset() throws Throwable {
        //tests of scale_factor/scaleFactor and add_offset/addOffset
        //and tests of _FillValue with no missing_value
        testVerboseOn();
        int tPo;
        try {
            EDDGrid amsr = (EDDGridFromDap)oneFromDatasetXml("ncdcOisstAmsrAgg"); //should work
            String tName = amsr.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                amsr.className() + "amsr", ".das"); 
            String results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            String expected = 
"sst {\n" +
"    Float32 _FillValue -9.99;\n" +      //note affected by scaleFactor
"    Float64 colorBarMaximum 32.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Daily Sea Surface Temperature\";\n" +
"    String standard_name \"sea_surface_temperature\";\n" +
"    String units \"degree_C\";\n" +
"    Float32 valid_max 45.0;\n" +
"    Float32 valid_min -3.0;\n" +
"  }\n" +
"  anom {\n" +
"    Float32 _FillValue -9.99;\n" +
"    Float64 colorBarMaximum 3.0;\n" +
"    Float64 colorBarMinimum -3.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Daily Sea Surface Temperature Anomalies\";\n" +
"    String standard_name \"surface_temperature_anomaly\";\n" +
"    String units \"degree_C\";\n" +
"    Float32 valid_max 12.0;\n" +
"    Float32 valid_min -12.0;\n" +
"  }\n" +
"  err {\n" +
"    Float32 _FillValue -9.99;\n" +
"    Float64 colorBarMaximum 0.6;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String colorBarPalette \"WhiteRedBlack\";\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Estimated Error Standard Deviation of Analyzed_SST\";\n" +
"    String units \"degree_C\";\n" +
"    Float32 valid_max 10.0;\n" +
"    Float32 valid_min 0.0;\n" +
"  }\n" +
"  ice {\n" +
"    Float32 _FillValue -9.99;\n" +
"    Float64 colorBarMaximum 1.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String colorBarPalette \"BlackBlueWhite\";\n" +
"    String ioos_category \"Ice Distribution\";\n" +
"    String long_name \"Sea Ice Concentration\";\n" +
"    String standard_name \"sea_ice_area_fraction\";\n" +
"    String units \"percentage\";\n" +
"    Float32 valid_max 1.0;\n" +
"    Float32 valid_min 0.0;\n" +
"  }"; 
            tPo = results.indexOf("sst {");
            String tResults = results.substring(tPo, tPo + expected.length());
            Test.ensureEqual(tResults, expected, "tResults=" + tResults);

            //test mv as -9.99  (adjusted by scaleFactor)
            String amsrq = "sst[0][0][0:200:600][0:200:600]";
            tName = amsr.makeNewFileForDapQuery(null, null, amsrq, EDStatic.fullTestCacheDirectory, 
                amsr.className() + "amsr", ".asc"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
    "Dataset {\n" +
    "  GRID {\n" +
    "    ARRAY:\n" +
    "      Float32 sst[time = 1][altitude = 1][latitude = 4][longitude = 4];\n" + //note Float32
    "    MAPS:\n" +
    "      Float64 time[time = 1];\n" +
    "      Float32 altitude[altitude = 1];\n" +
    "      Float32 latitude[latitude = 4];\n" +
    "      Float32 longitude[longitude = 4];\n" +
    "  } sst;\n" +
    "} ncdcOisstAmsrAgg;\n" +
    "---------------------------------------------\n" +
    "sst.sst[1][1][4][4]\n" +
    "[0][0][0], -9.99, -9.99, -9.99, -9.99\n" +
    "[0][0][1], 11.65, 15.02, 12.44, 17.41\n" +
    "[0][0][2], -9.99, -9.99, 29.57, 28.96\n" +
    "[0][0][3], 9.5, -9.99, -9.99, -9.99\n" +
    "\n" +
    "sst.time[1]\n" +
    "1.0228896E9\n" +
    "\n" +
    "sst.altitude[1]\n" +
    "0.0\n" +
    "\n" +
    "sst.latitude[4]\n" +
    "-89.875, -39.875, 10.125, 60.125\n" +
    "\n" +
    "sst.longitude[4]\n" +
    "0.125, 50.125, 100.125, 150.125\n"; 
            Test.ensureEqual(results, expected, "results=" + results);

            //test mv as NaN (adjusted to -9.99 by scaleFactor, then converted to NaN (then null))
            tName = amsr.makeNewFileForDapQuery(null, null, amsrq, EDStatic.fullTestCacheDirectory, 
                amsr.className() + "amsr", ".json"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
    "{\n" +
    "  \"table\": {\n" +
    "    \"columnNames\": [\"time\", \"altitude\", \"latitude\", \"longitude\", \"sst\"],\n" +
    "    \"columnTypes\": [\"String\", \"float\", \"float\", \"float\", \"float\"],\n" +
    "    \"columnUnits\": [\"UTC\", \"m\", \"degrees_north\", \"degrees_east\", \"degree_C\"],\n" +
    "    \"rows\": [\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -89.875, 0.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -89.875, 50.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -89.875, 100.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -89.875, 150.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -39.875, 0.125, 11.65],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -39.875, 50.125, 15.02],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -39.875, 100.125, 12.44],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, -39.875, 150.125, 17.41],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 10.125, 0.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 10.125, 50.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 10.125, 100.125, 29.57],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 10.125, 150.125, 28.96],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 60.125, 0.125, 9.5],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 60.125, 50.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 60.125, 100.125, null],\n" +
    "      [\"2002-06-01T00:00:00Z\", 0.0, 60.125, 150.125, null]\n" +
    "    ]\n" +
    "  }\n" +
    "}\n"; 
            Test.ensureEqual(results, expected, "results=" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing ncdc dataset." +
                "\nPress ^C to stop or Enter to continue..."); 
        }

    }     

    public static void testOneTime() throws Throwable {

        //gridDataset = (EDDGridFromDap)oneFromDatasetXml("pmelOscar"); 
        //if (true) System.exit(0);

        //one time stuff
        //gridDataset = (EDDGridFromDap)oneFromDatasetXml("ncdcOisstAmsrAgg"); 
        //tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
        //    gridDataset.className() + "ncdc", ".das"); 
        //String results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);

        //soda
        EDDGrid gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdSoda202d"); //should work
        String tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_soda202d", ".das"); 
        String results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);

        String sodaq = "time[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)]";
        tName = gridDataset.makeNewFileForDapQuery(null, null, sodaq, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_soda202dqt", ".json"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);

        sodaq =
"temp[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]"
//+ ",salt[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]" 
//+ ",u[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]" 
//+ ",v[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]" 
//+ ",w[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]" 
//+ ",utrans[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]" 
//+ ",vtrans[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]" 
//+ ",CFC11[(2001-12-15T00:00:00):100:(2001-12-15T00:00:00)][(-5.01):1:(-5.01)][(-75.25):100:(89.25)][(0.25):100:(359.75)]"
;
//The error: 	GridDataAccessor.increment: partialResults[0] was not as expected. 
//The other primitiveArray has a different value #0 (1008374400 != 623)
        tName = gridDataset.makeNewFileForDapQuery(null, null, sodaq, EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_soda202dq", ".json"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);


        gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdSoda202s"); //should work
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_soda202s", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);

        gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdSoda203d"); //should work
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_soda203d", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);

        gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdSoda203s"); //should work
        tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            gridDataset.className() + "_soda203s", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);

    }

    /**
     * This does important tests with pmelOscar (which has descending lat axis values AND
     * wierd rage: longitude is 20 .. 419!!!).
     *
     * @throws Throwable if trouble
     */
    public static void testPmelOscar(boolean doGraphicsTests) throws Throwable {
        String2.log("\n*** test for pmelOscar");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
        try {
        EDDGrid eddGrid = (EDDGrid)oneFromDatasetXml("pmelOscar"); 
        EDVGridAxis edvga;

        //***test some edvga things
        //time commented out: it changes too often
        //EDVGridAxis edvga = eddGrid.axisVariables()[0];  
        //Test.ensureEqual(edvga.isEvenlySpaced(), false, "");
        //Test.ensureEqual(edvga.averageSpacing(), 438329.1311754685, "");  //changes sometimes
        //Test.ensureEqual(edvga.spacingDescription(), "5 days 01:45:29 (uneven)", "");

        edvga = eddGrid.axisVariables()[1];
        Test.ensureEqual(edvga.isEvenlySpaced(), true, "");
        Test.ensureEqual(edvga.averageSpacing(), Double.NaN, "");
        Test.ensureEqual(edvga.spacingDescription(), "(just one value)", "");

        edvga = eddGrid.axisVariables()[2];
        Test.ensureEqual(edvga.isEvenlySpaced(), true, "");
        Test.ensureEqual(edvga.averageSpacing(), -1, "");  
        Test.ensureEqual(edvga.spacingDescription(), "-1.0 (even)", "");

        //time
        results = eddGrid.axisVariables[0].sliderCsvValues();
        expected = "\"1992-10-21\", \"1992-11-16\", \"1992-12-11\", \"1993-01-06\",";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);
        //last values and sliderNCsvValues changes frequently

        //alt
        results = eddGrid.axisVariables[1].sliderCsvValues();
        expected = "-15";
        Test.ensureEqual(results, expected, "results=" + results);
        Test.ensureEqual(eddGrid.axisVariables[1].sliderNCsvValues(), 1, "");

        //lat
        results = eddGrid.axisVariables[2].sliderCsvValues();
        expected = 
"69.5, 68.5, 67.5, 66.5, 65.5, 64.5, 63.5, 62.5, 61.5, 60.5, 59.5, 58.5, 57.5, 56.5, " +
"55.5, 54.5, 53.5, 52.5, 51.5, 50.5, 49.5, 48.5, 47.5, 46.5, 45.5, 44.5, 43.5, 42.5, " +
"41.5, 40.5, 39.5, 38.5, 37.5, 36.5, 35.5, 34.5, 33.5, 32.5, 31.5, 30.5, 29.5, 28.5, " +
"27.5, 26.5, 25.5, 24.5, 23.5, 22.5, 21.5, 20.5, 19.5, 18.5, 17.5, 16.5, 15.5, 14.5, " +
"13.5, 12.5, 11.5, 10.5, 9.5, 8.5, 7.5, 6.5, 5.5, 4.5, 3.5, 2.5, 1.5, 0.5, -0.5, -1.5, " +
"-2.5, -3.5, -4.5, -5.5, -6.5, -7.5, -8.5, -9.5, -10.5, -11.5, -12.5, -13.5, -14.5, -15.5, " +
"-16.5, -17.5, -18.5, -19.5, -20.5, -21.5, -22.5, -23.5, -24.5, -25.5, -26.5, -27.5, -28.5, " +
"-29.5, -30.5, -31.5, -32.5, -33.5, -34.5, -35.5, -36.5, -37.5, -38.5, -39.5, -40.5, -41.5, " +
"-42.5, -43.5, -44.5, -45.5, -46.5, -47.5, -48.5, -49.5, -50.5, -51.5, -52.5, -53.5, -54.5, " +
"-55.5, -56.5, -57.5, -58.5, -59.5, -60.5, -61.5, -62.5, -63.5, -64.5, -65.5, -66.5, -67.5, -68.5, -69.5"; 
        Test.ensureEqual(results, expected, "results=" + results);
        Test.ensureEqual(eddGrid.axisVariables[2].sliderNCsvValues(), 140, "");

        //lon
        results = eddGrid.axisVariables[3].sliderCsvValues();
        expected = "20.5, 22.5, 24.5, 26.5,";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);
        expected = "414.5, 416.5, 418.5, 419.5";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "results=" + results);
        Test.ensureEqual(eddGrid.axisVariables[3].sliderNCsvValues(), 201, "");


        //.das     das isn't affected by userDapQuery
        tName = eddGrid.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String lastTime = "" +  eddGrid.axisVariables[eddGrid.timeIndex].lastDestinationValue();
        lastTime = String2.replaceAll(lastTime, "E", "e+");
        expected = 
"Attributes {\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 7.196256e+8, " + lastTime + ";\n" + //stop time changes periodically
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    Float32 actual_range -15.0, -15.0;\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float32 actual_range 69.5, -69.5;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float32 actual_range 20.5, 419.5;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  u {\n" +
"    Float64 colorBarMaximum 0.4;\n" +
"    Float64 colorBarMinimum -0.4;\n" +
"    String generic_name \"u\";\n" +
"    String ioos_category \"Currents\";\n" +
"    String long_name \"Ocean Surface Zonal Currents\";\n" +
"    Float32 missing_value NaN;\n" +
"    String name \"u\";\n" +
"    String standard_name \"eastward_sea_water_velocity\";\n" +
"    String units \"m s-1\";\n" +
"  }\n";
        tResults = results.substring(0, expected.length());
        Test.ensureEqual(tResults, expected, "\nresults=\n" + results);
        expected = 
"    String DATASUBTYPE \"unfiltered\";\n" +
"    String DATATYPE \"5-Day Interval\";\n" +
"    String date \"27-Apr-2007\";\n" +
"    String description \"Sea Surface Velocity\";\n" +
"    String history \"" + today + " http://dapper.pmel.noaa.gov/dapper/oscar/world-unfilter.nc\n" +
today + " " + EDStatic.erddapUrl + //in tests, always non-https url
            "/griddap/pmelOscar.das\";\n" +
"    String infoUrl \"http://www.oscar.noaa.gov/\";\n" +
"    String institution \"NOAA PMEL\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    String reference1 \"Bonjean F. and G.S.E. Lagerloef, 2002, \\\"Diagnostic model and analysis of the surface currents in the tropical Pacific ocean\\\", J. Phys. Oceanogr., 32, 2,938-2,954\";\n" +
"    String source \"Gary Lagerloef, ESR (lager@esr.org) and Fabrice Bonjean (bonjean@esr.org)\";\n" +
"    String sourceUrl \"http://dapper.pmel.noaa.gov/dapper/oscar/world-unfilter.nc\";\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"This project is developing a processing system and data center to provide operational ocean surface velocity fields from satellite altimeter and vector wind data. The method to derive surface currents with satellite altimeter and scatterometer data is the outcome of several years NASA sponsored research. The project will transition that capability to operational oceanographic applications. The end product is velocity maps updated daily, with a goal for eventual 2-day maximum delay from time of satellite measurement. Grid resolution is 100 km for the basin scale, and finer resolution in the vicinity of the Pacific Islands.\";\n" +
"    String title \"OSCAR - Ocean Surface Current Analyses, Real-Time\";\n" +
"    String VARIABLE \"Ocean Surface Currents\";\n" +
"    Float32 version 2006.0;\n" +
"  }\n" +
"}\n";
        int tpo = results.indexOf(expected.substring(0, 17));
        if (tpo < 0) String2.log("results=\n" + results);
        Test.ensureEqual(results.substring(tpo), expected, "results=\n" + results);


        //.dds     dds isn't affected by userDapQuery
        tName = eddGrid.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        int tnTime = eddGrid.axisVariables[eddGrid.timeIndex].sourceValues().size();
        expected = 
"Dataset {\n" +
"  Float64 time[time = " + tnTime + "];\n" +    //changes periodically
"  Float32 altitude[altitude = 1];\n" +
"  Float32 latitude[latitude = 140];\n" +
"  Float32 longitude[longitude = 400];\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 u[time = " + tnTime + "][altitude = 1][latitude = 140][longitude = 400];\n" +
"    MAPS:\n" +
"      Float64 time[time = " + tnTime + "];\n" +
"      Float32 altitude[altitude = 1];\n" +
"      Float32 latitude[latitude = 140];\n" +
"      Float32 longitude[longitude = 400];\n" +
"  } u;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 v[time = " + tnTime + "][altitude = 1][latitude = 140][longitude = 400];\n" +
"    MAPS:\n" +
"      Float64 time[time = " + tnTime + "];\n" +
"      Float32 altitude[altitude = 1];\n" +
"      Float32 latitude[latitude = 140];\n" +
"      Float32 longitude[longitude = 400];\n" +
"  } v;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 u_anom[time = " + tnTime + "][altitude = 1][latitude = 140][longitude = 400];\n" +
"    MAPS:\n" +
"      Float64 time[time = " + tnTime + "];\n" +
"      Float32 altitude[altitude = 1];\n" +
"      Float32 latitude[latitude = 140];\n" +
"      Float32 longitude[longitude = 400];\n" +
"  } u_anom;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 v_anom[time = " + tnTime + "][altitude = 1][latitude = 140][longitude = 400];\n" +
"    MAPS:\n" +
"      Float64 time[time = " + tnTime + "];\n" +
"      Float32 altitude[altitude = 1];\n" +
"      Float32 latitude[latitude = 140];\n" +
"      Float32 longitude[longitude = 400];\n" +
"  } v_anom;\n" +
"} pmelOscar;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //.nc lat values subset (subset correct? actual_range correct?)
        userDapQuery = "latitude[(69.5):10:(-69.5)]"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "Lat", ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory + tName, true);
        String2.log(results);
        expected = //note actual range is low to high
"netcdf EDDGridFromDapLat.nc {\n" +
" dimensions:\n" +
"   latitude = 14;\n" +   // (has coord.var)\n" +  //changed when switched to netcdf-java 4.0, 2009-02-23
" variables:\n" +
"   float latitude(latitude=14);\n" +
"     :_CoordinateAxisType = \"Lat\";\n" +
"     :actual_range = -60.5f, 69.5f; // float\n" +
"     :axis = \"Y\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Latitude\";\n" +
"     :standard_name = \"latitude\";\n" +
"     :units = \"degrees_north\";\n" +
"\n" +
" :ANOM_MEAN_PERIOD = \"1993-01-01 to 2006-12-31\";\n" +
" :cdm_data_type = \"Grid\";\n" +
" :company = \"Earth & Space Research, Seattle, WA\";\n" +
" :contact = \"Fabrice Bonjean (bonjean@esr.org) or John T. Gunn (gunn@esr.org)\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n";

        tResults = results.substring(0, expected.length());
        Test.ensureEqual(tResults, expected, "results=" + results);
        expected = //not geospatial_lat_min max;  note that internal " are not slashed, but that is ncDump's problem
" :DATASUBTYPE = \"unfiltered\";\n" +
" :DATATYPE = \"5-Day Interval\";\n" +
" :date = \"27-Apr-2007\";\n" +
" :description = \"Sea Surface Velocity\";\n" +
" :geospatial_lat_max = 69.5f; // float\n" +
" :geospatial_lat_min = -60.5f; // float\n" +
" :geospatial_lat_units = \"degrees_north\";\n" +
" :history = \"" + today + " http://dapper.pmel.noaa.gov/dapper/oscar/world-unfilter.nc\n" +
today + " " + EDStatic.erddapUrl + //in tests, always non-https url
    "/griddap/pmelOscar.nc?latitude[(69.5):10:(-69.5)]\";\n" +
" :infoUrl = \"http://www.oscar.noaa.gov/\";\n" +
" :institution = \"NOAA PMEL\";\n" +
" :license = \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
" :Northernmost_Northing = 69.5f; // float\n" +
" :reference1 = \"Bonjean F. and G.S.E. Lagerloef, 2002, \\\"Diagnostic model and analysis of the surface currents in the tropical Pacific ocean\\\", J. Phys. Oceanogr., 32, 2,938-2,954\";\n" +
" :source = \"Gary Lagerloef, ESR (lager@esr.org) and Fabrice Bonjean (bonjean@esr.org)\";\n" +
" :sourceUrl = \"http://dapper.pmel.noaa.gov/dapper/oscar/world-unfilter.nc\";\n" +
" :Southernmost_Northing = -60.5f; // float\n" +
" :standard_name_vocabulary = \"CF-11\";\n" +
" :summary = \"This project is developing a processing system and data center to provide operational ocean surface velocity fields from satellite altimeter and vector wind data. The method to derive surface currents with satellite altimeter and scatterometer data is the outcome of several years NASA sponsored research. The project will transition that capability to operational oceanographic applications. The end product is velocity maps updated daily, with a goal for eventual 2-day maximum delay from time of satellite measurement. Grid resolution is 100 km for the basin scale, and finer resolution in the vicinity of the Pacific Islands.\";\n" +
" :title = \"OSCAR - Ocean Surface Current Analyses, Real-Time\";\n" +
" :VARIABLE = \"Ocean Surface Currents\";\n" +
" :version = 2006.0f; // float\n" +
" data:\n" +
"latitude =\n" +
"  {69.5, 59.5, 49.5, 39.5, 29.5, 19.5, 9.5, -0.5, -10.5, -20.5, -30.5, -40.5, -50.5, -60.5}\n" +
"}\n";
        tpo = results.indexOf(expected.substring(0, 17));
        if (tpo < 0) String2.log("results=\n" + results);
        Test.ensureEqual(results.substring(tpo), expected, "results=\n" + results);

        //.csv data 
        userDapQuery = "u[0][0][(69.5):10:(-69.5)][0]"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);
        expected = 
"time, altitude, latitude, longitude, u\n" +
"UTC, m, degrees_north, degrees_east, m s-1\n" +
"1992-10-21T00:00:00Z, -15.0, 69.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, 59.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, 49.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, 39.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, 29.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, 19.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, 9.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, -0.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, -10.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, -20.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, -30.5, 20.5, NaN\n" +
"1992-10-21T00:00:00Z, -15.0, -40.5, 20.5, 0.08152205\n" +
"1992-10-21T00:00:00Z, -15.0, -50.5, 20.5, 0.17953366\n" +
"1992-10-21T00:00:00Z, -15.0, -60.5, 20.5, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        String mapDapQuery = 
            "u[(2008-08-06T00:00:00Z)][][][]" +
            "&.draw=surface&.vars=longitude|latitude|u&.colorBar=|C|Linear|||&.land=under";
        
        tName = eddGrid.makeNewFileForDapQuery(null, null, mapDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "_Map", ".png"); 
        if (doGraphicsTests) SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        String transparentQuery = 
            "u[(2008-08-06T00:00:00Z)][][][]" +
            "&.draw=surface&.vars=longitude|latitude|u&.colorBar=|C|Linear|||&.land=under";
        tName = eddGrid.makeNewFileForDapQuery(null, null, transparentQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "_Transparent", ".transparentPng"); 
        if (doGraphicsTests) SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        String query180 = 
            "u[(2008-08-06T00:00:00Z)][][][0:(179)]" +
            "&.draw=surface&.vars=longitude|latitude|u&.colorBar=|C|Linear|||&.land=under";
        tName = eddGrid.makeNewFileForDapQuery(null, null, query180, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "_Map", ".kml"); 
        if (doGraphicsTests) SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

//currently doesn't work, either ERDDAP needs to rearrange lat and lon values        
//  or GeotiffWritter needs to accept descending axis values.
//        tName = eddGrid.makeNewFileForDapQuery(null, null, query180, EDStatic.fullTestCacheDirectory, 
//            eddGrid.className() + "_Map", ".geotif"); 
//        SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

        tName = eddGrid.makeNewFileForDapQuery(null, null, "u[(2008-08-06T00:00:00Z)][][0:50:(last)][0:50:(179)]", 
            EDStatic.fullTestCacheDirectory, eddGrid.className() + "_Map", ".esriAscii"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);
//It seems to toggle frequently between these similar results ...
//changed slightly 2008-09-07! and 2008-09-24  2008-10-09 2008-10-13 2008-11-11

        expected= 
"ncols 4\n" +
"nrows 3\n" +
"xllcenter 20.5\n" +
"yllcenter -30.5\n" +
"cellsize 50.0\n" +
"nodata_value -9999999\n" +
"-9999999 -9999999 -9999999 -9999999\n";
//"-9999999 0.085097924 0.032853972 0.16006929\n" +
//"-9999999 -0.009836693 -9999999 -0.047798216\n";

//and infinite variants of last 2 lines:
//"-9999999 0.08509792 0.032853972 0.16006929\n" +
//"-9999999 -0.009836693 -9999999 -0.047798216\n";

//"-9999999 0.085097924 0.03285397 0.16006929\n" +
//"-9999999 -0.009836693 -9999999 -0.047798216\n";

//"-9999999 0.085097924 0.032853976 0.1600693\n" +
//"-9999999 -0.009836693 -9999999 -0.047798216\n";

//"-9999999 0.08509793 0.032853976 0.1600693\n" +
//"-9999999 -0.009836693 -9999999 -0.047798216\n";

//"-9999999 0.08509792 0.032853976 0.1600693\n" +
//"-9999999 -0.009836693 -9999999 -0.047798216\n";

//but on 2009-02-04 it changed to 
//-9999999 0.07428373 -9999999 0.18457247
//-9999999 0.0044357306 -9999999 -0.05290417

        //I gave up testing: it changes too often.
        //Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
        //Test.ensureTrue(results.indexOf(" 0.085097") > 0, "\nresults=\n" + results);
        //Test.ensureTrue(results.indexOf(" 0.032853") > 0, "\nresults=\n" + results);
        //Test.ensureTrue(results.indexOf(" 0.160069") > 0, "\nresults=\n" + results);
        //Test.ensureTrue(results.indexOf("-0.00983669") > 0, "\nresults=\n" + results);
        //Test.ensureTrue(results.indexOf("-0.0477982") > 0, "\nresults=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error accessing " + EDStatic.erddapUrl + //in tests, always non-https url
                "\nPress ^C to stop or Enter to continue..."); 
        }

    }

    /**
     * This does some tests for Ellyn (not usually run).
     *
     * @throws Throwable if trouble
     */
    public static void testForEllyn() throws Throwable {
        String2.log("\n*** test for Ellyn");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        EDDGrid eddGrid = (EDDGrid)oneFromDatasetXml("mb-7201adc"); 

        //.das     das isn't affected by userDapQuery
        tName = eddGrid.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".das"); 
        results = String2.readFromFile(EDStatic.fullTestCacheDirectory + tName)[1];
        expected = 
"Attributes {\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.0673667e+9, 1.0747107e+9;\n" +
"    String axis \"T\";\n" +
"    Int32 epic_code 624;\n" +
"    String FORTRAN_format \"F10.2\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String type \"EVEN\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    Float32 actual_range -9.452215, -0.9522152;\n" +
"    String axis \"Z\";\n" +
"    Float64 bin_size 0.5;\n" +
"    Float64 blanking_distance 0.4399999976158142;\n" +
"    Int32 epic_code 3;\n" +
"    String FORTRAN_format \"F10.2\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String NOTE \"Depth values were calculated using Surface.exe output\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String type \"EVEN\";\n" +
"    String units \"m\";\n" +
"    Float64 xducer_offset_from_bottom 1.2599999904632568;\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float32 actual_range 33.6496, 33.6496;\n" +
"    String axis \"Y\";\n" +
"    Int32 epic_code 500;\n" +
"    String FORTRAN_format \"F10.2\";\n" +
"    String generic_name \"lat\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String name \"LAT\";\n" +
"    String standard_name \"latitude\";\n" +
"    String type \"EVEN\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float32 actual_range -78.7894, -78.7894;\n" +
"    String axis \"X\";\n" +
"    Int32 epic_code 502;\n" +
"    String FORTRAN_format \"f10.4\";\n" +
"    String generic_name \"lon\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String name \"LON\";\n" +
"    String standard_name \"longitude\";\n" +
"    String type \"EVEN\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  u_1205 {\n" +
"    Float32 _FillValue 1.0E35;\n" +
"    Float64 bins_questioned 17.0, 18.0, 19.0;\n" +
"    Float64 bins_std_dev_threshold_for_fill 2.0, 0.5, 0.0;\n" +
"    Int32 epic_code 1205;\n" +
"    String generic_name \"u\";\n" +
"    String ioos_category \"Currents\";\n" +
"    String long_name \"Eastward Velocity (cm/s)\";\n" +
"    Float32 maximum 15.260121, 15.359408, 17.115356, 20.168367, 22.383686, 25.54994, 28.047798, 29.958483, 31.26962, 32.359756, 33.601105, 34.743168, 37.12874, 37.92423, 38.83359, 40.78402, 38.88334, 35.099434;\n" +
"    Float32 minimum -20.984797, -21.533491, -22.015848, -21.91846, -24.096342, -23.670574, -23.147821, -23.790667, -24.070864, -23.311993, -23.055223, -23.374365, -22.824059, -22.077686, -22.545044, -23.29182, -23.85412, -22.895117;\n" +
"    String name \"u\";\n" +
"    String NOTE \"Questionable data in bins likely contaminated by side-lobe surface reflection set to FillValue_\";\n" +
"    Float64 sensor_depth 10.452199935913086;\n" +
"    String sensor_type \"RD Instruments ADCP\";\n" +
"    Float64 serial_number 159.0;\n" +
"    String standard_name \"eastward_current_velocity\";\n" +
"    String units \"cm/s\";\n" +
"    Float64 valid_range -1000.0, 1000.0;\n" +
"  }\n";
        tResults = results.substring(0, expected.length());
        Test.ensureEqual(tResults, expected, "\nresults=\n" + results);
        expected = 
"  PGd_1203 {\n" +
"    Float32 _FillValue 1.0E35;\n" +
"    Int32 epic_code 1203;\n" +
"    String generic_name \"PGd\";\n" +
"    String ioos_category \"Unknown\";\n" +
"    String long_name \"Percent Good Pings\";\n" +
"    Float32 maximum 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0, 100.0;\n" +
"    Float32 minimum 94.25, 50.0, 50.0, 50.0, 50.0, 50.0, 39.5, 34.5, 32.75, 32.5, 32.25, 32.5, 32.0, 31.75, 31.5, 31.75, 27.0, 41.25;\n" +
"    String name \"PGd\";\n" +
"    Float64 sensor_depth 10.452199935913086;\n" +
"    String sensor_type \"RD Instruments ADCP\";\n" +
"    Float64 serial_number 159.0;\n" +
"    String standard_name \"none\";\n" +
"    String units \"counts\";\n" +
"    Float64 valid_range 0.0, 100.0;\n" +
"  }\n" +
"  NC_GLOBAL {\n" +
"    Int32 ADCP_serial_number 159;\n" +
"    Int32 beam_angle 20;\n" +
"    String beam_pattern \"convex\";\n" +
"    String cdm_data_type \"Grid\";\n" +
"    Int32 COMPOSITE 0;\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    String COORD_SYSTEM \"GEOGRAPHIC\";\n" +
"    String CREATION_DATE \"30-Aug-2006 11:56:52\";\n" +
"    String DATA_CMNT \"additional information\";\n" +
"    String DATA_ORIGIN \"USGS WHFS Sed Trans Group\";\n" +
"    String DATA_SUBTYPE \"MOORED\";\n" +
"    String DATA_TYPE \"ADCP\";\n" +
"    String DELTA_T \"900\";\n" +
"    String Deployment_date \"28-Oct-2003\";\n" +
"    Int32 DEPTH_CONST 0;\n" +
"    String DESCRIPT \"Site 1 ADCP\";\n" +
"    Int32 DRIFTER 0;\n" +
"    Int32 ending_water_layer 5;\n" +
"    Int32 error_velocity_threshold 2000;\n" +
"    String EXPERIMENT \"Myrtle Beach\";\n" +
"    Int32 false_target_reject_values 255, 255;\n" +
"    Float64 FILL_FLAG 1.0;\n" +
"    Float32 firmware_version 16.21;\n" +
"    Int32 frequency 1200;\n" +
"    Float64 geospatial_vertical_max 9.452215;\n" +
"    Float64 geospatial_vertical_min 0.9522152;\n" +
"    String geospatial_vertical_positive \"down\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"histories combined, fill_vals removed from coordinate vars.:WATER_DEPTH related attributes corrected.:Depth subsampled by dolly.m.:Min and max attributes on u, v, and z recalculated using recalculate_adcp_minmax_vals.m on 24-Mar-2006 10:27:10 by C. Sullivan, WHSC; Questionable velocity data in bins likely contaminated by side-lobe surface reflection set to FillValue_ by clean_adcp_bins_std.m V 1.1 on 02-Dec-2005 10:12:05 by C. Sullivan, USGS WHSC; Extra variables deleted by dolly.m.:Written to an EPIC standard data file by adcp2ep.m (version 1.1):Transformed to earth coordinates by runbm2g.m:Bins were trimmed by trimBins.m using 94% of the RDI surface output.:Transformed to earth coordinates by runbm2g.m:Bins were trimmed by trimBins.m using 94% of the RDI surface output.:Ensembles recorded pre and post deployment were trimmed by goodends.m.:The data were filtered using rdi quality control factors in runmask.m.:Converted to netCDF via MATLAB by rdi2cdf.m 3.0 10-Jan-2003\n" +
today + " http://coast-enviro.er.usgs.gov/thredds/dodsC/DATAFILES/MYRTLEBEACH/7201adc-a.nc\n" +
today + " " + EDStatic.erddapUrl + //in tests, always non-https url
            "/griddap/mb-7201adc.das\";\n" +
"    String infoUrl \"http://stellwagen.er.usgs.gov/myrtlebeach.html\";\n" +
"    Float32 inst_depth 10.4522;\n" +
"    String inst_depth_note \"inst_depth = (water_depth - inst_height); nominal depth below the surface\";\n" +
"    Float32 inst_height 1.26;\n" +
"    String inst_height_note \"height in meters above bottom: accurate for tripod mounted intstruments\";\n" +
"    String INST_TYPE \"RD Instruments ADCP\";\n" +
"    String institution \"USGS/CMGP\";\n" +
"    String janus \"4 Beam\";\n" +
"    Float64 latitude 33.6496;\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 longitude -78.78939819335938;\n" +
"    Float64 magnetic_variation -8.22;\n" +
"    Int32 minmax_percent_good 0, 100;\n" +
"    String MOORING \"7201\";\n" +
"    String orientation \"UP\";\n" +
"    Int32 pings_per_ensemble 300;\n" +
"    Int32 POS_CONST 0;\n" +
"    Float32 pred_accuracy 0.4;\n" +
"    String PROJECT \"WHFC\";\n" +
"    String Recovery_date \"21-Jan-2004\";\n" +
"    String sourceUrl \"http://coast-enviro.er.usgs.gov/thredds/dodsC/DATAFILES/MYRTLEBEACH/7201adc-a.nc\";\n" +
"    String standard_name_vocabulary \"EPIC, CF-1.0\";\n" +
"    String start_time \"28-Oct-2003 18:45:00\";\n" +
"    Int32 starting_water_layer 1;\n" +
"    String stop_time \"21-Jan-2004 18:45:00\";\n" +
"    String summary \"velocity data from the ADCP on mooring 720\";\n" +
"    Float32 time_between_ping_groups 1.0;\n" +
"    String title \"South Carolina Coastal Erosion Study -adcp7201\";\n" +
"    String transform \"EARTH\";\n" +
"    Int32 transmit_pulse_length_cm 49;\n" +
"    Int32 valid_correlation_range 64, 255;\n" +
"    String VAR_DESC \"u:v:w:Werr:AGC:PGd:hght:Tx\";\n" +
"    Float64 VAR_FILL 1.0000000409184788e+35;\n" +
"    Float32 WATER_DEPTH 11.7122;\n" +
"    String WATER_DEPTH_NOTE \"from ADCP: (m) \";\n" +
"    String WATER_MASS \"?\";\n" +
"  }\n" +
"}\n";
        int tpo = results.indexOf(expected.substring(0, 17));
        if (tpo < 0) String2.log("results=\n" + results);
        Test.ensureEqual(results.substring(tpo), expected, "results=\n" + results);


        //.dds     dds isn't affected by userDapQuery
        tName = eddGrid.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"Dataset {\n" +
"  Float64 time[time = 8161];\n" +
"  Float32 altitude[altitude = 18];\n" +
"  Float32 latitude[latitude = 1];\n" +
"  Float32 longitude[longitude = 1];\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 u_1205[time = 8161][altitude = 18][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 8161];\n" +
"      Float32 altitude[altitude = 18];\n" +
"      Float32 latitude[latitude = 1];\n" +
"      Float32 longitude[longitude = 1];\n" +
"  } u_1205;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 v_1206[time = 8161][altitude = 18][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 8161];\n" +
"      Float32 altitude[altitude = 18];\n" +
"      Float32 latitude[latitude = 1];\n" +
"      Float32 longitude[longitude = 1];\n" +
"  } v_1206;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 w_1204[time = 8161][altitude = 18][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 8161];\n" +
"      Float32 altitude[altitude = 18];\n" +
"      Float32 latitude[latitude = 1];\n" +
"      Float32 longitude[longitude = 1];\n" +
"  } w_1204;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 Werr_1201[time = 8161][altitude = 18][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 8161];\n" +
"      Float32 altitude[altitude = 18];\n" +
"      Float32 latitude[latitude = 1];\n" +
"      Float32 longitude[longitude = 1];\n" +
"  } Werr_1201;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 AGC_1202[time = 8161][altitude = 18][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 8161];\n" +
"      Float32 altitude[altitude = 18];\n" +
"      Float32 latitude[latitude = 1];\n" +
"      Float32 longitude[longitude = 1];\n" +
"  } AGC_1202;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 PGd_1203[time = 8161][altitude = 18][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 8161];\n" +
"      Float32 altitude[altitude = 18];\n" +
"      Float32 latitude[latitude = 1];\n" +
"      Float32 longitude[longitude = 1];\n" +
"  } PGd_1203;\n" +
"} mb-7201adc;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //.csv   altitude values
        userDapQuery = "altitude"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "Alt", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);
        expected = 
"altitude\n" +
"m\n" +
"-9.452215\n" +
"-8.952215\n" +
"-8.452215\n" +
"-7.952215\n" +
"-7.452215\n" +
"-6.952215\n" +
"-6.452215\n" +
"-5.952215\n" +
"-5.452215\n" +
"-4.952215\n" +
"-4.452215\n" +
"-3.9522152\n" +
"-3.4522152\n" +
"-2.9522152\n" +
"-2.4522152\n" +
"-1.9522152\n" +
"-1.4522152\n" +
"-0.9522152\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //.csv data values
        userDapQuery = "u_1205[0][0:17][0][0]"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);
        expected = 
"time, altitude, latitude, longitude, u_1205\n" +
"UTC, m, degrees_north, degrees_east, cm/s\n" +
"2003-10-28T18:45:00Z, -9.452215, 33.6496, -78.7894, 4.3488593\n" +
"2003-10-28T18:45:00Z, -8.952215, 33.6496, -78.7894, 5.9566345\n" +
"2003-10-28T18:45:00Z, -8.452215, 33.6496, -78.7894, 6.794666\n" +
"2003-10-28T18:45:00Z, -7.952215, 33.6496, -78.7894, 8.219185\n" +
"2003-10-28T18:45:00Z, -7.452215, 33.6496, -78.7894, 8.064997\n" +
"2003-10-28T18:45:00Z, -6.952215, 33.6496, -78.7894, 8.445462\n" +
"2003-10-28T18:45:00Z, -6.452215, 33.6496, -78.7894, 9.1305\n" +
"2003-10-28T18:45:00Z, -5.952215, 33.6496, -78.7894, 8.146257\n" +
"2003-10-28T18:45:00Z, -5.452215, 33.6496, -78.7894, 10.3772335\n" +
"2003-10-28T18:45:00Z, -4.952215, 33.6496, -78.7894, 9.835191\n" +
"2003-10-28T18:45:00Z, -4.452215, 33.6496, -78.7894, 8.704758\n" +
"2003-10-28T18:45:00Z, -3.9522152, 33.6496, -78.7894, 6.0494637\n" +
"2003-10-28T18:45:00Z, -3.4522152, 33.6496, -78.7894, 4.7303987\n" +
"2003-10-28T18:45:00Z, -2.9522152, 33.6496, -78.7894, 3.260833\n" +
"2003-10-28T18:45:00Z, -2.4522152, 33.6496, -78.7894, 1.9206865\n" +
"2003-10-28T18:45:00Z, -1.9522152, 33.6496, -78.7894, 1.6506728\n" +
"2003-10-28T18:45:00Z, -1.4522152, 33.6496, -78.7894, 1.1185195\n" +
"2003-10-28T18:45:00Z, -0.9522152, 33.6496, -78.7894, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


    }

    /** 
     * This tests that the query parser is properly dealing with altitude 
     * metersPerSourceUnit = -1 (which makes ascending values into descending).
     * Originally, Dave found a specific problem related to .mat files that had already been fixed but not released. 
     */
    public static void testMetersPerSourceUnit() throws Throwable {
        testVerboseOn();
        EDDGrid gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdSoda202d"); 
        String query = 
            //Dave had (-500):(-5.01) which succeeded for .htmlTable but failed for .mat 
            //but I had already fixed/cleaned up erddap's handling of descending axis vars 
            //(including altitude axes with negative metersPerSourceUnit
            //so reformed request with (-500):(-5.01) passes .htmlTable and .mat
            "temp[(2001-12-15T00:00:00)][(-5.01):(-500)][(23.1)][(185.2)]," + 
            "salt[(2001-12-15T00:00:00)][(-5.01):(-500)][(23.1)][(185.2)]," + 
               "u[(2001-12-15T00:00:00)][(-5.01):(-500)][(23.1)][(185.2)]," + 
               "v[(2001-12-15T00:00:00)][(-5.01):(-500)][(23.1)][(185.2)]," + 
               "w[(2001-12-15T00:00:00)][(-5.01):(-500)][(23.1)][(185.2)]," + 
           "CFC11[(2001-12-15T00:00:00)][(-5.01):(-500)][(23.1)][(185.2)]";

        String tName = gridDataset.makeNewFileForDapQuery(null, null, query,
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_erdSoda202d", ".htmlTable"); //was ok
        String results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String expected = 
"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \n" +
"  \"http://www.w3.org/TR/html4/loose.dtd\"> \n" +
"<html>\n" +
"<head>\n" +
"  <title>EDDGridFromDap_erdSoda202d</title>\n" +
"\n" +
"<style type=\"text/CSS\"> <!--\n" +
"  table.erd {border-collapse:collapse; border:1px solid gray; }\n" +
"  table.erd th, table.erd td {padding:2px; border:1px solid gray; }\n" +
"--> </style>\n" +
"</head>\n" +
"<body style=\"color:black; background:white; font-family:Arial,Helvetica,sans-serif; font-size:85%;\">\n" +
"<table class=\"erd\" bgcolor=\"#FFFFCC\" cellspacing=\"0\">\n" +
"<tr>\n" +
"<th>time</th>\n" +
"<th>altitude</th>\n" +
"<th>latitude</th>\n" +
"<th>longitude</th>\n" +
"<th>temp</th>\n" +
"<th>salt</th>\n" +
"<th>u</th>\n" +
"<th>v</th>\n" +
"<th>w</th>\n" +
"<th>CFC11</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<th>UTC</th>\n" +
"<th>m</th>\n" +
"<th>degrees_north</th>\n" +
"<th>degrees_east</th>\n" +
"<th>degree_C</th>\n" +
"<th>g kg-1</th>\n" +
"<th>m s-1</th>\n" +
"<th>m s-1</th>\n" +
"<th>m s-1</th>\n" +
"<th>mmole m-3</th>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>2001-12-15T00:00:00Z</td>\n" +
"<td nowrap>-5.01</td>\n" +
"<td>23.25</td>\n" +
"<td>185.25</td>\n" +
"<td>26.01415</td>\n" +
"<td>35.505432</td>\n" +
"<td nowrap>-0.08560439</td>\n" +
"<td>0.11106719</td>\n" +
"<td nowrap>2.4947973E-8</td>\n" +
"<td nowrap>1.7414581E-9</td>\n" +
"</tr>\n" +
"<tr>\n" +
"<td nowrap>2001-12-15T00:00:00Z</td>\n" +
"<td nowrap>-15.07</td>\n" +
"<td>23.25</td>\n" +
"<td>185.25</td>\n" +
"<td>26.007967</td>\n" +
"<td>35.505795</td>\n" +
"<td nowrap>-0.06710795</td>\n" +
"<td>0.10303306</td>\n" +
"<td nowrap>3.0305154E-7</td>\n" +
"<td nowrap>1.7414655E-9</td>\n" +
"</tr>\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);
     
        tName = gridDataset.makeNewFileForDapQuery(null, null, query,
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_erdSoda202d", ".mat");  //threw an exception
        results = File2.hexDump(EDStatic.fullTestCacheDirectory + tName, 1000000);
        expected = 
"4d 41 54 4c 41 42 20 35   2e 30 20 4d 41 54 2d 66   MATLAB 5.0 MAT-f |\n" +
"69 6c 65 2c 20 43 72 65   61 74 65 64 20 62 79 3a   ile, Created by: |\n" +
"20 67 6f 76 2e 6e 6f 61   61 2e 70 66 65 6c 2e 63    gov.noaa.pfel.c |\n" +
"6f 61 73 74 77 61 74 63   68 2e 4d 61 74 6c 61 62   oastwatch.Matlab |\n" +
//"2c 20 43 72 65 61 74 65   64 20 6f 6e 3a 20 4d 6f   , Created on: Mo |\n" +
//"6e 20 4f 63 74 20 31 33   20 31 35 3a 31 36 3a 32   n Oct 13 15:16:2 |\n" +
//"34 20 32 30 30 38 20 20   20 20 20 20 20 20 20 20   4 2008           |\n" +
"20 20 20 20 00 00 00 00   00 00 00 00 01 00 4d 49                 MI |\n" +
"00 00 00 0e 00 00 06 78   00 00 00 06 00 00 00 08          x         |\n" +
"00 00 00 02 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 0b                    |\n" +
"65 72 64 53 6f 64 61 32   30 32 64 00 00 00 00 00   erdSoda202d      |\n" +
"00 04 00 05 00 00 00 20   00 00 00 01 00 00 01 40                  @ |\n" +
"74 69 6d 65 00 00 00 00   00 00 00 00 00 00 00 00   time             |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"61 6c 74 69 74 75 64 65   00 00 00 00 00 00 00 00   altitude         |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"6c 61 74 69 74 75 64 65   00 00 00 00 00 00 00 00   latitude         |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"6c 6f 6e 67 69 74 75 64   65 00 00 00 00 00 00 00   longitude        |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"74 65 6d 70 00 00 00 00   00 00 00 00 00 00 00 00   temp             |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"73 61 6c 74 00 00 00 00   00 00 00 00 00 00 00 00   salt             |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"75 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00   u                |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"76 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00   v                |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"77 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00   w                |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"43 46 43 31 31 00 00 00   00 00 00 00 00 00 00 00   CFC11            |\n" +
"00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
"00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 08   41 ce 0d 49 40 00 00 00           A  I@    |\n" +
"00 00 00 0e 00 00 00 c8   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 13 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 98   c0 14 0a 3d 70 a3 d7 0a              =p    |\n" +
"c0 2e 23 d7 0a 3d 70 a4   c0 39 47 ae 14 7a e1 48    .#  =p  9G  z H |\n" +
"c0 41 e1 47 ae 14 7a e1   c0 47 4e 14 7a e1 47 ae    A G  z  GN z G  |\n" +
"c0 4c fd 70 a3 d7 0a 3d   c0 51 81 47 ae 14 7a e1    L p   = Q G  z  |\n" +
"c0 54 ba e1 47 ae 14 7b   c0 58 3a e1 47 ae 14 7b    T  G  { X: G  { |\n" +
"c0 5c 14 7a e1 47 ae 14   c0 60 2f ae 14 7a e1 48    \\ z G   `/  z H |\n" +
"c0 62 9e b8 51 eb 85 1f   c0 65 6c cc cc cc cc cd    b  Q    el      |\n" +
"c0 68 b9 47 ae 14 7a e1   c0 6c af 5c 28 f5 c2 8f    h G  z  l \\(    |\n" +
"c0 70 c7 5c 28 f5 c2 8f   c0 73 da 66 66 66 66 66    p \\(    s fffff |\n" +
"c0 77 d6 3d 70 a3 d7 0a   c0 7d 1e 8f 5c 28 f5 c3    w =p    }  \\(   |\n" +
"00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 08   40 37 40 00 00 00 00 00           @7@      |\n" +
"00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
"00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
"00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
"00 00 00 09 00 00 00 08   40 67 28 00 00 00 00 00           @g(      |\n" +
"00 00 00 0e 00 00 00 88   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 07 00 00 00 00   00 00 00 05 00 00 00 10                    |\n" +
"00 00 00 01 00 00 00 13   00 00 00 01 00 00 00 01                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 07 00 00 00 4c                  L |\n" +
"41 d0 1c fb 41 d0 10 51   41 d0 0d e8 41 d0 26 9c   A   A  QA   A &  |\n" +
"41 d0 54 34 41 d0 5f 7a   41 d0 33 59 41 cb 85 15   A T4A _zA 3YA    |\n" +
"41 c1 4d 7c 41 ba 53 68   41 b4 bd 11 41 af 94 1a   A M|A ShA   A    |\n" +
"41 a8 6b 51 41 a1 9a 80   41 96 26 32 41 87 25 c5   A kQA   A &2A %  |\n" +
"41 67 3d c2 41 3e 6f 50   41 17 65 81 00 00 00 00   Ag= A>oPA e      |\n" +
"00 00 00 0e 00 00 00 88   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 07 00 00 00 00   00 00 00 05 00 00 00 10                    |\n" +
"00 00 00 01 00 00 00 13   00 00 00 01 00 00 00 01                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 07 00 00 00 4c                  L |\n" +
"42 0e 05 90 42 0e 05 ef   42 0e 07 7a 42 0e 17 9e   B   B   B  zB    |\n" +
"42 0e 27 21 42 0e 41 8e   42 0e 55 be 42 0e 3e 93   B '!B A B U B >  |\n" +
"42 0e 20 9e 42 0e 21 9d   42 0d f7 05 42 0d 97 ac   B   B ! B   B    |\n" +
"42 0d 43 af 42 0c fd 2e   42 0c 93 84 42 0b e8 3e   B C B  .B   B  > |\n" +
"42 0a ed a6 42 09 ea 23   42 08 f4 1e 00 00 00 00   B   B  #B        |\n" +
"00 00 00 0e 00 00 00 88   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 07 00 00 00 00   00 00 00 05 00 00 00 10                    |\n" +
"00 00 00 01 00 00 00 13   00 00 00 01 00 00 00 01                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 07 00 00 00 4c                  L |\n" +
"bd af 51 5b bd 89 6f e5   bd 75 6c a6 bd 7d 68 5e     Q[  o  ul  }h^ |\n" +
"bd 85 10 fa bd 92 54 05   bd 9f 38 48 bd aa 83 9a         T   8H     |\n" +
"bd b5 2a d8 bd c1 5d f9   bd cd 70 66 bd d7 4c 57     *   ]   pf  LW |\n" +
"bd db df 9a bd da f6 de   bd d7 a8 ee bd d4 09 39                  9 |\n" +
"bd ca aa 4c bd b8 65 4d   bd a6 0f 5a 00 00 00 00      L  eM   Z     |\n" +
"00 00 00 0e 00 00 00 88   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 07 00 00 00 00   00 00 00 05 00 00 00 10                    |\n" +
"00 00 00 01 00 00 00 13   00 00 00 01 00 00 00 01                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 07 00 00 00 4c                  L |\n" +
"3d e3 77 32 3d d3 02 ff   3d bd a7 cf 3d ab d5 52   = w2=   =   =  R |\n" +
"3d a2 10 80 3d 9b b5 a6   3d 9a d7 a6 3d a0 55 f3   =   =   =   = U  |\n" +
"3d 9c f0 6e 3d 91 a7 92   3d 7e f0 9b 3d 50 34 68   =  n=   =~  =P4h |\n" +
"3d 19 1e ab 3c bc 77 9e   3c 0d b4 31 bb 9c 79 d2   =   < w <  1  y  |\n" +
"bc 88 be 29 bc d4 ac 4b   bd 07 35 d8 00 00 00 00      )   K  5      |\n" +
"00 00 00 0e 00 00 00 88   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 07 00 00 00 00   00 00 00 05 00 00 00 10                    |\n" +
"00 00 00 01 00 00 00 13   00 00 00 01 00 00 00 01                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 07 00 00 00 4c                  L |\n" +
"32 d6 4d 2c 34 a2 b3 16   35 16 6e 32 35 15 a6 bd   2 M,4   5 n25    |\n" +
"34 b2 3e 20 35 2e 37 4d   35 c1 ad 8a 36 1d ce 7b   4 > 5.7M5   6  { |\n" +
"36 44 ba 81 36 6e 9b cd   36 8e 4c 84 36 a5 a7 a2   6D  6n  6 L 6    |\n" +
"36 bc 33 c9 36 d1 ab 84   36 e5 9f e6 36 f6 c3 af   6 3 6   6   6    |\n" +
"37 02 23 bd 37 07 95 6b   37 0e bf 29 00 00 00 00   7 # 7  k7  )     |\n" +
"00 00 00 0e 00 00 00 88   00 00 00 06 00 00 00 08                    |\n" +
"00 00 00 07 00 00 00 00   00 00 00 05 00 00 00 10                    |\n" +
"00 00 00 01 00 00 00 13   00 00 00 01 00 00 00 01                    |\n" +
"00 00 00 01 00 00 00 00   00 00 00 07 00 00 00 4c                  L |\n" +
"30 ef 58 1c 30 ef 58 5f   30 ef 58 c1 30 ef 4a 3b   0 X 0 X_0 X 0 J; |\n" +
"30 ef 45 84 30 ef 20 75   30 f0 33 30 30 fa e8 36   0 E 0  u0 300  6 |\n" +
"31 05 f1 ef 31 0a 61 30   31 0d ff 49 31 11 99 6b   1   1 a01  I1  k |\n" +
"31 14 6c a5 31 15 7d e6   31 14 2b 88 31 0f de 55   1 l 1 } 1 + 1  U |\n" +
"31 07 49 15 30 ef 50 ba   30 b9 43 35 00 00 00 00   1 I 0 P 0 C5     |\n";
        Test.ensureEqual(
            results.substring(0, 71 * 4) + results.substring(71 * 7), //remove the creation dateTime
            expected, "results=" + results);    

        tName = gridDataset.makeNewFileForDapQuery(null, null, query,
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_erdSoda202d", ".asc"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"Dataset {\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 temp[time = 1][altitude = 19][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 1];\n" +
"      Float64 altitude[altitude = 19];\n" +
"      Float64 latitude[latitude = 1];\n" +
"      Float64 longitude[longitude = 1];\n" +
"  } temp;\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 salt[time = 1][altitude = 19][latitude = 1][longitude = 1];\n" +
"    MAPS:\n" +
"      Float64 time[time = 1];\n" +
"      Float64 altitude[altitude = 19];\n" +
"      Float64 latitude[latitude = 1];\n" +
"      Float64 longitude[longitude = 1];\n" +
"  } salt;";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);    


        //***** tests that should fail
        query = //same as above, but altitude values are reversed
            "temp[(2001-12-15T00:00:00)][(-500):(-5.01)][(23.1)][(185.2)]," + 
            "salt[(2001-12-15T00:00:00)][(-500):(-5.01)][(23.1)][(185.2)]," + 
               "u[(2001-12-15T00:00:00)][(-500):(-5.01)][(23.1)][(185.2)]," + 
               "v[(2001-12-15T00:00:00)][(-500):(-5.01)][(23.1)][(185.2)]," + 
               "w[(2001-12-15T00:00:00)][(-500):(-5.01)][(23.1)][(185.2)]," + 
           "CFC11[(2001-12-15T00:00:00)][(-500):(-5.01)][(23.1)][(185.2)]";
        expected = "SimpleException: Query error: requestStartIndex=18 is less than requestStopIndex=0 " +
            "in constraint (for variable=temp, axis=1, constraint=[(-500):(-5.01)]).";

        for (int i = 0; i < dataFileTypeNames.length; i++) {
            String fileType = dataFileTypeNames[i];
            if (String2.indexOf(new String[]{".das", ".dds", ".graph", ".help", ".html"}, fileType) >= 0)
                continue;

            results = "";
            try {
                tName = gridDataset.makeNewFileForDapQuery(null, null, query,
                    EDStatic.fullTestCacheDirectory, gridDataset.className() + "_erdSoda202d", fileType); 
            } catch (Throwable t) {
                results = MustBe.throwableToString(t);
            }
            Test.ensureEqual(results.substring(0, Math.min(results.length(), expected.length())), expected, 
                "fileType=" + fileType + " results=" + results);
        }

    }

    /** This tests sliderCsvValues. */
    public static void testSliderCsv() throws Throwable {
        testVerboseOn();
        String name, tName, results, expected;
        EDDGridFromDap gridDataset;
        
        //test erdBAssta5day
        gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdBAssta5day"); 

        //time
        results = gridDataset.axisVariables[0].sliderCsvValues();
        expected = "\"2002-07-06T12:00:00Z\", \"2002-07-16T12:00:00Z\", \"2002-07-26T12:00:00Z\", \"2002-08-05T12:00:00Z\",";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);
        //end dates and sliderNCsvValues changes frequently

        //alt
        results = gridDataset.axisVariables[1].sliderCsvValues();
        expected = "0";
        Test.ensureEqual(results, expected, "results=" + results);
        Test.ensureEqual(gridDataset.axisVariables[1].sliderNCsvValues(), 1, "");

        //lat
        results = gridDataset.axisVariables[2].sliderCsvValues();
        expected = "-75, -74, -73, -72, -71, -70, -69, -68, -67, -66,";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);
        expected = "64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "results=" + results);
        Test.ensureEqual(gridDataset.axisVariables[2].sliderNCsvValues(), 151, "");

        //lon
        results = gridDataset.axisVariables[3].sliderCsvValues();
        expected = "0, 2, 4, 6, 8, 10, 12, 14, 16, 18,";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "results=" + results);
        expected = "350, 352, 354, 356, 358, 360";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "results=" + results);
        Test.ensureEqual(gridDataset.axisVariables[3].sliderNCsvValues(), 181, "");

        //*** SEE also the oscar tests
    }


    /** This tests saveAsKml. */
    public static void testKml() throws Throwable {
        testVerboseOn();
        try {

        EDDGridFromDap gridDataset = (EDDGridFromDap)oneFromDatasetXml("erdBAssta5day"); 
        String name, tName, results, expected;

        //overall kml
        tName = gridDataset.makeNewFileForDapQuery(null, null, "sst[(2008-11-01T12:00:00Z)][][][]",
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_testKml", ".kml"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n" +
"<Document>\n" +
"  <name>SST, Blended, Global, EXPERIMENTAL (5 Day Composite)</name>\n" +
"  <description><![CDATA[Time: 2008-11-01T12:00:00Z<br />\n" +
"Data courtesy of: NOAA CoastWatch, West Coast Node<br />\n" +
"<a href=\"" + EDStatic.erddapUrl + //in tests, always non-https url
            "/griddap/erdBAssta5day.html?sst\">Download data from this dataset.</a><br />\n" +
"    ]]></description>\n" +
"  <Region>\n" +
"    <Lod><minLodPixels>2</minLodPixels></Lod>\n" +
"    <LatLonAltBox>\n" +
"      <west>0.0</west>\n" +
"      <east>360.0</east>\n" +
"      <south>-75.0</south>\n" +
"      <north>75.0</north>\n" +
"    </LatLonAltBox>\n" +
"  </Region>\n" +
"  <NetworkLink>\n" +
"    <name>1_0_0_0</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>0.0</west>\n" +
"        <east>180.0</east>\n" +
"        <south>-75.0</south>\n" +
"        <north>4.163336E-15</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(-75.0):(4.163336E-15)][(0.0):(180.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <NetworkLink>\n" +
"    <name>1_0_0_1</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>0.0</west>\n" +
"        <east>180.0</east>\n" +
"        <south>4.163336E-15</south>\n" +
"        <north>75.0</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                    "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(4.163336E-15):(75.0)][(0.0):(180.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <NetworkLink>\n" +
"    <name>1_0_0_2</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>-180.0</west>\n" +
"        <east>0.0</east>\n" +
"        <south>-75.0</south>\n" +
"        <north>4.163336E-15</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                        "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(-75.0):(4.163336E-15)][(180.0):(360.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <NetworkLink>\n" +
"    <name>1_0_0_3</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>-180.0</west>\n" +
"        <east>0.0</east>\n" +
"        <south>4.163336E-15</south>\n" +
"        <north>75.0</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                            "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(4.163336E-15):(75.0)][(180.0):(360.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <GroundOverlay>\n" +
"    <drawOrder>1</drawOrder>\n" +
"    <Icon>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                                "/griddap/erdBAssta5day.transparentPng?sst[(2008-11-01T12:00:00Z)][0][(-75.0):4:(75.0)][(0.0):4:(360.0)]</href>\n" +
"    </Icon>\n" +
"    <LatLonBox>\n" +
"      <west>0.0</west>\n" +
"      <east>360.0</east>\n" +
"      <south>-75.0</south>\n" +
"      <north>75.0</north>\n" +
"    </LatLonBox>\n" +
"  </GroundOverlay>\n" +
"  <ScreenOverlay id=\"Logo\">\n" +
"    <description>" + EDStatic.erddapUrl + //in tests, always non-https url
                                    "</description>\n" +
"    <name>Logo</name>\n" +
"    <Icon><href>" + EDStatic.imageDirUrl + "nlogo.gif</href></Icon>\n" +
"    <overlayXY x=\"0.005\" y=\".04\" xunits=\"fraction\" yunits=\"fraction\"/>\n" +
"    <screenXY x=\"0.005\" y=\".04\" xunits=\"fraction\" yunits=\"fraction\"/>\n" +
"    <size x=\"0\" y=\"0\" xunits=\"pixels\" yunits=\"pixels\"/>\n" +
"  </ScreenOverlay>\n" +
"</Document>\n" +
"</kml>\n";
        Test.ensureEqual(results, expected, "results=" + results);    

        //a quadrant
        tName = gridDataset.makeNewFileForDapQuery(null, null, 
            "sst[(2008-11-01T12:00:00Z)][0][(-75.0):(4.163336E-15)][(180.0):(360.0)]",
            EDStatic.fullTestCacheDirectory, gridDataset.className() + "_testKml2", ".kml"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
"<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n" +
"<Document>\n" +
"  <name>2_1_0</name>\n" +
"  <Region>\n" +
"    <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"    <LatLonAltBox>\n" +
"      <west>-180.0</west>\n" +
"      <east>0.0</east>\n" +
"      <south>-75.0</south>\n" +
"      <north>4.163336E-15</north>\n" +
"    </LatLonAltBox>\n" +
"  </Region>\n" +
"  <NetworkLink>\n" +
"    <name>2_1_0_0</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>-180.0</west>\n" +
"        <east>-90.0</east>\n" +
"        <south>-75.0</south>\n" +
"        <north>-37.5</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
            "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(-75.0):(-37.5)][(180.0):(270.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <NetworkLink>\n" +
"    <name>2_1_0_1</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>-180.0</west>\n" +
"        <east>-90.0</east>\n" +
"        <south>-37.5</south>\n" +
"        <north>4.163336E-15</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(-37.5):(4.163336E-15)][(180.0):(270.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <NetworkLink>\n" +
"    <name>2_1_0_2</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>-90.0</west>\n" +
"        <east>0.0</east>\n" +
"        <south>-75.0</south>\n" +
"        <north>-37.5</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                    "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(-75.0):(-37.5)][(270.0):(360.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <NetworkLink>\n" +
"    <name>2_1_0_3</name>\n" +
"    <Region>\n" +
"      <Lod><minLodPixels>256</minLodPixels></Lod>\n" +
"      <LatLonAltBox>\n" +
"        <west>-90.0</west>\n" +
"        <east>0.0</east>\n" +
"        <south>-37.5</south>\n" +
"        <north>4.163336E-15</north>\n" +
"      </LatLonAltBox>\n" +
"    </Region>\n" +
"    <Link>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                        "/griddap/erdBAssta5day.kml?sst[(2008-11-01T12:00:00Z)][0][(-37.5):(4.163336E-15)][(270.0):(360.0)]</href>\n" +
"      <viewRefreshMode>onRegion</viewRefreshMode>\n" +
"    </Link>\n" +
"  </NetworkLink>\n" +
"  <GroundOverlay>\n" +
"    <drawOrder>2</drawOrder>\n" +
"    <Icon>\n" +
"      <href>" + EDStatic.erddapUrl + //in tests, always non-https url
                            "/griddap/erdBAssta5day.transparentPng?sst[(2008-11-01T12:00:00Z)][0][(-75.0):2:(4.163336E-15)][(180.0):2:(360.0)]</href>\n" +
"    </Icon>\n" +
"    <LatLonBox>\n" +
"      <west>-180.0</west>\n" +
"      <east>0.0</east>\n" +
"      <south>-75.0</south>\n" +
"      <north>4.163336E-15</north>\n" +
"    </LatLonBox>\n" +
"  </GroundOverlay>\n" +
"</Document>\n" +
"</kml>\n";
        Test.ensureEqual(results, expected, "results=" + results);    


        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing erdBAssta5day at " + EDStatic.erddapUrl + //in tests, always non-https url
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }


    /** This tests non-nc-"Grid" data variable (dimensions don't have axis/coordinate variable). 
     * This also tests generateDatasetsXml for this special case.
     */
    public static void testNoAxisVariable() throws Throwable {
        String2.log("\n*** testNoAxisVariable");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        try {
        EDDGrid eddGrid = (EDDGrid)oneFromDatasetXml("testNoAxisVariable"); 

        //.das     das isn't affected by userDapQuery
        tName = eddGrid.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);
        expected = 
"Attributes {\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.075464e+9, 1.075464e+9;\n" +
"    String axis \"T\";\n" +
"    String calendar \"standard\";\n" +
"    String field \"time, scalar, series\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"averaged time since initialization\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  eta_rho {\n" +
"    Int16 actual_range 0, 641;\n" +
"    String ioos_category \"Location\";\n" +
"    String units \"count\";\n" +
"  }\n" +
"  xi_rho {\n" +
"    Int16 actual_range 0, 225;\n" +
"    String ioos_category \"Location\";\n" +
"    String units \"count\";\n" +
"  }\n" +
"  zeta {\n" +
"    String coordinates \"lat_rho lon_rho\";\n" +
"    String field \"free-surface, scalar, series\";\n" +
"    String ioos_category \"Other\";\n" +
"    String long_name \"averaged free-surface\";\n" +
"    String time \"ocean_time\";\n" +
"    String units \"meter\";\n" +
"  }\n" +
"  NC_GLOBAL {\n" +
"    String avg_base \"nep4_avg\";\n" +
"    String bry_file \"/wrkdir/kate/NEP4_djd/NEP4_bry_CCSM_2000-2004.nc\";\n";
        tResults = results.substring(0, expected.length());
        Test.ensureEqual(tResults, expected, "\nresults=\n" + results);

        expected = 
"String tiling \"003x016\";\n" +
"    String title \"ROMS/TOMS 3.0 - Northeast Pacific 10km Grid (NEP4)\";\n" +
"    String type \"ROMS/TOMS averages file\";\n" +
"    String var_info \"External/varinfo.dat\";\n" +
"  }\n" +
"}\n";
        int tpo = results.indexOf("String tiling ");
        if (tpo < 0) String2.log("results=\n" + results);
        Test.ensureEqual(results.substring(tpo), expected, "results=\n" + results);


        //.dds     dds isn't affected by userDapQuery
        tName = eddGrid.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddGrid.className(), ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"Dataset {\n" +
"  Float64 time[time = 1];\n" +
"  Int16 eta_rho[eta_rho = 642];\n" +
"  Int16 xi_rho[xi_rho = 226];\n" +
"  GRID {\n" +
"    ARRAY:\n" +
"      Float32 zeta[time = 1][eta_rho = 642][xi_rho = 226];\n" +
"    MAPS:\n" +
"      Float64 time[time = 1];\n" +
"      Int16 eta_rho[eta_rho = 642];\n" +
"      Int16 xi_rho[xi_rho = 226];\n" +
"  } zeta;\n" +
"} testNoAxisVariable;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //.csv   an index values
        userDapQuery = "xi_rho[0:2:5]"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "XiRho", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"xi_rho\n" +
"count\n" +
"0\n" +
"2\n" +
"4\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //.csv data values
        userDapQuery = "zeta[0][0:2][0:2]"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "_NAV", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
//from source
//  http://edac-dap2.northerngulfinstitute.org/thredds/dodsC/roms/al_roms/nep4_004.nc.ascii?zeta[0:1:0][0:1:2][0:1:2]
//Dataset {
//    Float32 zeta[ocean_time = 1][eta_rho = 3][xi_rho = 3];
//} roms/al_roms/nep4_004.nc;
//---------------------------------------------
//zeta[1][3][3]
//[0][0], 0.6276099, 0.6276099, 0.6215845
//[0][1], 0.6276099, 0.6276089, 0.6215834
//[0][2], 0.6242121, 0.62421095, 0.6181293

"time, eta_rho, xi_rho, zeta\n" +
"UTC, count, count, meter\n" +
"2004-01-30T12:00:00Z, 0, 0, 0.6276099\n" +
"2004-01-30T12:00:00Z, 0, 1, 0.6276099\n" +
"2004-01-30T12:00:00Z, 0, 2, 0.6215845\n" +
"2004-01-30T12:00:00Z, 1, 0, 0.6276099\n" +
"2004-01-30T12:00:00Z, 1, 1, 0.6276089\n" +
"2004-01-30T12:00:00Z, 1, 2, 0.6215834\n" +
"2004-01-30T12:00:00Z, 2, 0, 0.6242121\n" +
"2004-01-30T12:00:00Z, 2, 1, 0.62421095\n" +
"2004-01-30T12:00:00Z, 2, 2, 0.6181293\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //analogous query
        userDapQuery = "zeta[0][(0):(2)][(0):(2)]"; 
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "_NAV2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //generateDatasetsXml
        results = generateDatasetsXml("http://edac-dap2.northerngulfinstitute.org/thredds/dodsC/roms/al_roms/nep4_004.nc");
        expected = 
"<!-- Directions:\n" +
" * Read about this type of dataset in\n" +
"   http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html .\n" +
" * Read http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html#addAttributes\n" +
"   so that you understand about sourceAttributes and addAttributes.\n" +
" * All of the content below that starts with \"???\" is a guess. It must be edited.\n" +
" * All of the other tags and their content are based on information from the source.\n" +
" * For the att tags, you should either:\n" +
"    * Delete the att tag (so ERDDAP will use the unchanged source attribute).\n" +
"    * Change the att tag's value (because it isn't quite right).\n" +
"    * Or, remove the att tag's value, but leave the att tag\n" +
"      (so the source attribute will be removed by ERDDAP).\n" +
" * You can reorder data variables, but don't reorder axis variables.\n" +
" * The current IOOS category options are:\n" +
"      Bathymetry, Biology, Bottom Character, Contaminants, Currents, Dissolved \n" +
"      Nutrients, Dissolved O2, Ecology, Fish Abundance, Fish Species, Heat \n" +
"      Flux, Ice Distribution, Identifier, Location, Meteorology, Ocean Color, \n" +
"      Optical Properties, Other, Pathogens, Phytoplankton Species, Pressure, \n" +
"      Productivity, Salinity, Sea Level, Surface Waves, Taxonomy, Temperature, \n" +
"      Time, Unknown, Wind, Zooplankton Species, Zooplankton Abundance\n" +
" * For longitude, latitude, altitude (or depth), and time variables:\n" +
"   * If the sourceName isn't \"longitude\", \"latitude\", \"altitude\", or \"time\",\n" +
"     you need to specify \"longitude\", \"latitude\", \"altitude\", or \"time\"\n" +
"     with a destinationName tag.\n" +
"   * For EDDTable datasets: if possible, add an actual_range attribute\n" +
"     if one isn't already there.\n" +
"   * (Usually) remove all other attributes. They usually aren't needed.\n" +
"     Attributes will be added automatically.\n" +
"-->\n" +
"<!-- If the multidimensional variables from a data source don't all use the same dimensions,\n" +
"you need to separate the data source into more than one dataset.\n" +
"The multidimensional variables for this data source are:\n" +
"tnu2 1[tracer]\n" +
"Akt_bak 1[tracer]\n" +
"Tnudg 1[tracer]\n" +
"FSobc_in 1[boundary]\n" +
"FSobc_out 1[boundary]\n" +
"M2obc_in 1[boundary]\n" +
"M2obc_out 1[boundary]\n" +
"Tobc_in 2[boundary, tracer]\n" +
"Tobc_out 2[boundary, tracer]\n" +
"M3obc_in 1[boundary]\n" +
"M3obc_out 1[boundary]\n" +
"Cs_r 1[s_rho]\n" +
"Cs_w 1[s_w]\n" +
"zeta 3[ocean_time, eta_rho, xi_rho]\n" +
"ubar 3[ocean_time, eta_u, xi_u]\n" +
"vbar 3[ocean_time, eta_v, xi_v]\n" +
"u 4[ocean_time, s_rho, eta_u, xi_u]\n" +
"v 4[ocean_time, s_rho, eta_v, xi_v]\n" +
"omega 4[ocean_time, s_w, eta_rho, xi_rho]\n" +
"temp 4[ocean_time, s_rho, eta_rho, xi_rho]\n" +
"salt 4[ocean_time, s_rho, eta_rho, xi_rho]\n" +
"Hsbl 3[ocean_time, eta_rho, xi_rho]\n" +
"AKv 4[ocean_time, s_w, eta_rho, xi_rho]\n" +
"AKt 4[ocean_time, s_w, eta_rho, xi_rho]\n" +
"AKs 4[ocean_time, s_w, eta_rho, xi_rho]\n" +
"shflux 3[ocean_time, eta_rho, xi_rho]\n" +
"ssflux 3[ocean_time, eta_rho, xi_rho]\n" +
"latent 3[ocean_time, eta_rho, xi_rho]\n" +
"sensible 3[ocean_time, eta_rho, xi_rho]\n" +
"lwrad 3[ocean_time, eta_rho, xi_rho]\n" +
"swrad 3[ocean_time, eta_rho, xi_rho]\n" +
"sustr 3[ocean_time, eta_u, xi_u]\n" +
"svstr 3[ocean_time, eta_v, xi_v]\n" +
"uice 3[ocean_time, eta_u, xi_u]\n" +
"vice 3[ocean_time, eta_v, xi_v]\n" +
"aice 3[ocean_time, eta_rho, xi_rho]\n" +
"hice 3[ocean_time, eta_rho, xi_rho]\n" +
"tisrf 3[ocean_time, eta_rho, xi_rho]\n" +
"snow_thick 3[ocean_time, eta_rho, xi_rho]\n" +
"sfwat 3[ocean_time, eta_rho, xi_rho]\n" +
"ti 3[ocean_time, eta_rho, xi_rho]\n" +
"-->\n" +
"    <dataset type=\"EDDGridFromDap\" datasetID=\"???\">\n" +
"        <sourceUrl>http://edac-dap2.northerngulfinstitute.org/thredds/dodsC/roms/al_roms/nep4_004.nc</sourceUrl>\n" +
"        <reloadEveryNMinutes>???60</reloadEveryNMinutes>\n" +
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n" +
"        <addAttributes>\n" +
"            <att name=\"avg_base\">nep4_avg</att>\n" +
"            <att name=\"bry_file\">/wrkdir/kate/NEP4_djd/NEP4_bry_CCSM_2000-2004.nc</att>\n" +
"            <att name=\"cdm_data_type\">Grid</att>\n" +
"            <att name=\"compiler_command\">mpxlf95_r</att>\n" +
"            <att name=\"compiler_flags\">-qsuffix=f=f90 -qmaxmem=-1 -qarch=pwr4 -qtune=pwr4 -q64 -O3 -qstrict</att>\n" +
"            <att name=\"compiler_system\">xlf</att>\n" +
"            <att name=\"Conventions\">CF-1.0??????COARDS, CF-1.0, Unidata Dataset Discovery v1.0</att>\n" +
"            <att name=\"CPP_options\"> ANA_BSFLUX, ANA_BTFLUX, ASSUMED_SHAPE, AVERAGES, AVERAGES_AKS, AVERAGES_AKT, AVERAGES_FLUXES, BULK_FLUXES, CURVGRID, DIURNAL_SRFLUX, DJ_GRADPS, DOUBLE_PRECISION, EASTERN_WALL, EMINUSP, ICE_ADVECT, ICE_BULK_FLUXES, ICE_EVP, ICE_MK, ICE_MODEL, ICE_MOMENTUM, ICE_SMOLAR, ICE_THERMO, LMD_CONVEC, LMD_MIXING, LMD_NONLOCAL, LMD_RIMIX, LMD_SKPP, LONGWAVE_OUT, MASKING, MIX_GEO_TS, MIX_S_UV, MPI, NEP4, NONLINEAR, NONLIN_EOS, NORTHERN_WALL, NO_WRITE_GRID, POWER_LAW, PROFILE, RADIATION_2D, RST_SINGLE, SALINITY, SOLAR_SOURCE, SOLVE3D, SOUTH_FSCHAPMAN, SOUTH_M2FLATHER, SOUTH_M3NUDGING, SOUTH_M3RADIATION, SOUTH_TNUDGING, SOUTH_TRADIATION, SPLINES, SPONGE, TS_U3HADVECTION, TS_SVADVECTION, TS_DIF2, TS_PSOURCE, UV_ADV, UV_COR, UV_U3HADVECTION, UV_SADVECTION, UV_QDRAG, UV_PSOURCE, UV_VIS2, VAR_RHO_2D, WEST_FSCHAPMAN, WEST_M2FLATHER, WEST_M3NUDGING, WEST_M3RADIATION, WEST_TNUDGING, WEST_TRADIATION,</att>\n" +
"            <att name=\"cpu\">00203FCA4C00</att>\n" +
"            <att name=\"frc_file_01\">/wrkdir/kate/NEP4_djd/Pair_47.nc</att>\n" +
"            <att name=\"frc_file_02\">/wrkdir/kate/NEP4_djd/LWrad_47.nc</att>\n" +
"            <att name=\"frc_file_03\">/wrkdir/kate/NEP4_djd/Qair_47.nc</att>\n" +
"            <att name=\"frc_file_04\">/wrkdir/kate/NEP4_djd/SWrad_47.nc</att>\n" +
"            <att name=\"frc_file_05\">/wrkdir/kate/NEP4_djd/Tair_47.nc</att>\n" +
"            <att name=\"frc_file_06\">/wrkdir/kate/NEP4_djd/Uwind_47.nc</att>\n" +
"            <att name=\"frc_file_07\">/wrkdir/kate/NEP4_djd/Vwind_47.nc</att>\n" +
"            <att name=\"frc_file_08\">/wrkdir/kate/NEP4_djd/rain_01.nc</att>\n" +
"            <att name=\"frc_file_09\">/wrkdir/kate/NEP4_djd/NEP4_rivers_57-04_djd.nc</att>\n" +
"            <att name=\"grd_file\">/wrkdir/kate/NEP4_djd/NEP4_grid.nc</att>\n" +
"            <att name=\"his_base\">nep4_his</att>\n" +
"            <att name=\"history\">Tue May 16 15:06:14 2006: /u1/uaf/kate/bin/ncra -F -d ocean_time,36,42 nep4_avg_1680.nc nep4_avg_1681.nc nep4_avg_1682.nc nep4_avg_1683.nc nep4_avg_1684.nc nep4_avg_1685.nc nep4_avg_1686.nc nep4_avg_1687.nc nep4_avg_1688.nc nep4_avg_1689.nc nep4_avg_1690.nc nep4_avg_1691.nc nep4_avg_1692.nc nep4_avg_1693.nc nep4_avg_1694.nc nep4_avg_1695.nc nep4_avg_1696.nc nep4_avg_1697.nc nep4_avg_1698.nc nep4_avg_1699.nc nep4_avg_1700.nc nep4_avg_1701.nc nep4_avg_1702.nc nep4_avg_1703.nc nep4_avg_1704.nc nep4_avg_1705.nc nep4_avg_1706.nc nep4_avg_1707.nc nep4_avg_1708.nc nep4_avg_1709.nc nep4_avg_1710.nc nep4_avg_1711.nc nep4_avg_1712.nc nep4_avg_1713.nc nep4_avg_1714.nc nep4_avg_1715.nc nep4_avg_1716.nc nep4_avg_1717.nc weeks_2004/nep4_004.nc\n" +
"ROMS/TOMS, Version 3.0, Monday - May 8, 2006 -  7:09:04 PM</att>\n" +
"            <att name=\"infoUrl\">???</att>\n" +
"            <att name=\"ini_file\">nep4_rst.nc</att>\n" +
"            <att name=\"institution\">???</att>\n" +
"            <att name=\"license\">???[standard]</att>\n" +
"            <att name=\"nco_openmp_thread_number\" type=\"int\">4</att>\n" +
"            <att name=\"os\">AIX</att>\n" +
"            <att name=\"rst_file\">nep4_rst.nc</att>\n" +
"            <att name=\"script_file\">External/ocean_nep.in</att>\n" +
"            <att name=\"standard_name_vocabulary\">???CF-11</att>\n" +
"            <att name=\"summary\">???</att>\n" +
"            <att name=\"tiling\">003x016</att>\n" +
"            <att name=\"title\">ROMS/TOMS 3.0 - Northeast Pacific 10km Grid (NEP4)</att>\n" +
"            <att name=\"type\">ROMS/TOMS averages file</att>\n" +
"            <att name=\"var_info\">External/varinfo.dat</att>\n" +
"        </addAttributes>\n" +
"        <axisVariable>\n" +
"            <sourceName>tracer</sourceName>\n" +
"            <destinationName>???tracer</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Tracer</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">???</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n" +
"        <axisVariable>\n" +
"            <sourceName>boundary</sourceName>\n" +
"            <destinationName>???boundary</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Boundary</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">???</att>\n" +
"            </addAttributes>\n" +
"        </axisVariable>\n";
        Test.ensureEqual(results.substring(0, Math.min(results.length(), expected.length())), 
            expected, "results=\n" + results);

        expected = 
"<sourceName>snow_thick</sourceName>\n" +
"            <destinationName>???snow_thick</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"field\">snow thickness, scalar, series</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">averaged thickness of snow cover</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"time\">ocean_time</att>\n" +
"                <att name=\"units\">meter</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>sfwat</sourceName>\n" +
"            <destinationName>???sfwat</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"field\">melt water thickness, scalar, series</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">averaged surface melt water thickness on ice</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"time\">ocean_time</att>\n" +
"                <att name=\"units\">meter</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>ti</sourceName>\n" +
"            <destinationName>???ti</destinationName>\n" +
"            <addAttributes>\n" +
"                <att name=\"colorBarMaximum\" type=\"double\">NaN</att>\n" +
"                <att name=\"colorBarMinimum\" type=\"double\">NaN</att>\n" +
"                <att name=\"field\">interior temperature, scalar, series</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">averaged interior ice temperature</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"time\">ocean_time</att>\n" +
"                <att name=\"units\">degrees Celcius</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"    </dataset>\n";
        int po = results.indexOf("<sourceName>snow_thick</sourceName>");
        Test.ensureEqual(results.substring(po), expected, "results=\n" + results);

        } catch (Exception t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing testNoAxisVariable in datasets.xml."  +
                "\nPress ^C to stop or Enter to continue..."); 
        }

    }
     

    /** This tests a climatology time problem.   */
    public static void testClimatologyTime() throws Throwable {
        String2.log("\n*** testClimatologyTime");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery;

        try {
        EDDGrid eddGrid = (EDDGrid)oneFromDatasetXml("ncdcOwClm9505"); 
        userDapQuery = "u[(0000-12-15)][][(22)][(225)]";
        tName = eddGrid.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddGrid.className() + "_clim", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"time, altitude, latitude, longitude, u\n" +
"UTC, m, degrees_north, degrees_east, m s-1\n" +
"0000-12-15T00:00:00Z, 10.0, 22.0, 225.0, -6.749089\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError accessing ncdcOwClm9505 at " + EDStatic.erddapUrl + //in tests, always non-https url
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /** This tests accessibleTo. */
    public static void testAccessibleTo() throws Throwable {
        testVerboseOn();
        EDDGrid eddGrid = (EDDGrid)oneFromDatasetXml("erdBAssta5day"); 
        String roleNull[] = null;
        String roleNone[] = new String[0];
        String roleBob[] = new String[]{"bob"};
        String roleBMT[] = new String[]{"bob", "mike", "tom"};

        //test accessible = null
        eddGrid.setAccessibleTo(null);
        Test.ensureTrue(eddGrid.isAccessibleTo(roleNull), "");
        Test.ensureTrue(eddGrid.isAccessibleTo(roleNone), "");
        Test.ensureTrue(eddGrid.isAccessibleTo(roleBob), "");
        Test.ensureTrue(eddGrid.isAccessibleTo(roleBMT), "");

        //test accessible = ""
        eddGrid.setAccessibleTo("");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleNull), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleNone), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleBob), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleBMT), "");

        //test accessible = "john, tom"
        eddGrid.setAccessibleTo("john, tom");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleNull), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleNone), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleBob), "");
        Test.ensureTrue(eddGrid.isAccessibleTo(roleBMT), "");

        //test accessible = "albert, stan"
        eddGrid.setAccessibleTo("albert, stan");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleNull), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleNone), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleBob), "");
        Test.ensureTrue(!eddGrid.isAccessibleTo(roleBMT), "");
    }


    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void test(boolean doGraphicsTests) throws Throwable {

        String2.log("\n****************** EDDGridFromDap.test() *****************\n");

        /* standard tests */
        testBasic();
        testAccessibleTo();
        if (doGraphicsTests) testGraphics();
        testOpendap();
        testScaleAddOffset();
        testPmelOscar(doGraphicsTests);
        testGenerateDatasetsXml();
        testMetersPerSourceUnit();
        testSliderCsv();
        testKml();
        testNoAxisVariable();
        testClimatologyTime();
        /* */

        //not regularly done
        //testForCarleton();
        //testForDave();
        //testForEllyn();
        //testOneTime();

        String2.log("\n*** EDDGridFromDap.test finished.");

    }



}
