/* 
 * EDDGridFromErddap Copyright 2008, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.FloatArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.ResourceBundle2;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

/** The Java DAP classes.  */
import dods.dap.*;

import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.sgt.SgtGraph;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.util.Subscriptions;
import gov.noaa.pfel.erddap.variable.*;

import java.io.FileWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Enumeration;

/**
 * Get netcdf-X.X.XX.jar from http://www.unidata.ucar.edu/software/netcdf-java/index.htm
 * and copy it to <context>/WEB-INF/lib renamed as netcdf-latest.jar.
 * Get slf4j-jdk14.jar from 
 * ftp://ftp.unidata.ucar.edu/pub/netcdf-java/slf4j-jdk14.jar
 * and copy it to <context>/WEB-INF/lib.
 * Put both of these .jar files in the classpath for the compiler and for Java.
 */
import ucar.nc2.*;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dods.*;
import ucar.nc2.util.*;
import ucar.ma2.*;  //only Array is needed; all other ucar is for testing netcdf-java

/** 
 * This class represents a grid dataset from an opendap DAP source.
 * 
 * @author Bob Simons (bob.simons@noaa.gov) 2007-06-04
 */
public class EDDGridFromErddap extends EDDGrid { 

    protected String sourceErddapUrl;

    /** Indicates if data can be transmitted in a compressed form.
     * It is unlikely anyone would want to change this. */
    public static boolean acceptDeflate = true;

    /**
     * This constructs an EDDGridFromErddap based on the information in an .xml file.
     * 
     * @param xmlReader with the &lt;erddapDatasets&gt;&lt;dataset type="EDDGridFromErddap"&gt; 
     *    having just been read.  
     * @return an EDDGridFromErddap.
     *    When this returns, xmlReader will have just read &lt;erddapDatasets&gt;&lt;/dataset&gt; .
     * @throws Throwable if trouble
     */
    public static EDDGridFromErddap fromXml(SimpleXMLReader xmlReader) throws Throwable {

        //data to be obtained (or not)
        if (verbose) String2.log("\n*** constructing EDDGridFromErddap(xmlReader)...");
        String tDatasetID = xmlReader.attributeValue("datasetID"); 
        int tReloadEveryNMinutes = DEFAULT_RELOAD_EVERY_N_MINUTES;
        String tAccessibleTo = null;
        StringArray tOnChange = new StringArray();
        String tSourceUrl = null;

        //process the tags
        String startOfTags = xmlReader.allTags();
        int startOfTagsN = xmlReader.stackSize();
        int startOfTagsLength = startOfTags.length();

        while (true) {
            xmlReader.nextTag();
            String tags = xmlReader.allTags();
            String content = xmlReader.content();
            //if (reallyVerbose) String2.log("  tags=" + tags + content);
            if (xmlReader.stackSize() == startOfTagsN) 
                break; //the </dataset> tag
            String localTags = tags.substring(startOfTagsLength);

            //try to make the tag names as consistent, descriptive and readable as possible
            if      (localTags.equals( "<reloadEveryNMinutes>")) {}
            else if (localTags.equals("</reloadEveryNMinutes>")) tReloadEveryNMinutes = String2.parseInt(content); 

            //Since this erddap can never be logged in to the remote ERDDAP, 
            //it can never get dataset info from the remote erddap dataset (which should have restricted access).
            //Plus there is no way to pass accessibleTo info between ERDDAP's (but not to users).
            //So there is currently no way to make this work. 
            //So it is disabled.
            //else if (localTags.equals( "<accessibleTo>")) {}
            //else if (localTags.equals("</accessibleTo>")) tAccessibleTo = content;

            else if (localTags.equals( "<sourceUrl>")) {}
            else if (localTags.equals("</sourceUrl>")) tSourceUrl = content; 

            //onChange
            else if (localTags.equals( "<onChange>")) {}
            else if (localTags.equals("</onChange>")) 
                tOnChange.add(content); 

            else xmlReader.unexpectedTagException();
        }
        return new EDDGridFromErddap(tDatasetID, tAccessibleTo, tOnChange, tReloadEveryNMinutes, tSourceUrl);
    }

    /**
     * The constructor.
     *
     * @param tAccessibleTo is a comma separated list of 0 or more
     *    roles which will have access to this dataset.
     *    <br>If null, everyone will have access to this dataset (even if not logged in).
     *    <br>If "", no one will have access to this dataset.
     * @param tOnChange 0 or more actions (starting with "http://" or "mailto:")
     *    to be done whenever the dataset changes significantly
     * @param tReloadEveryNMinutes indicates how often the source should
     *    be checked for new data (use Integer.MAX_VALUE for never).
     * @param tSourceUrl the url to which .das or .dds or ... can be added
     * @throws Throwable if trouble
     */
    public EDDGridFromErddap(String tDatasetID, String tAccessibleTo,
        StringArray tOnChange, int tReloadEveryNMinutes,
        String tSourceUrl) throws Throwable {

        if (verbose) String2.log(
            "\n*** constructing EDDGridFromErddap " + tDatasetID); 
        long constructionStartMillis = System.currentTimeMillis();
        String errorInMethod = "Error in EDDGridFromErddap(" + 
            tDatasetID + ") constructor:\n";
            
        //save some of the parameters
        className = "EDDGridFromErddap"; 
        datasetID = tDatasetID;
        //setAccessibleTo(tAccessibleTo);  disabled. see above.
        onChange = tOnChange;
        sourceErddapUrl = tSourceUrl;
        setReloadEveryNMinutes(tReloadEveryNMinutes);

        //open the connection to the opendap source
        DConnect dConnect = new DConnect(sourceErddapUrl, acceptDeflate, 1, 1);

        //setup via info.json
        //source http://coastwatch.pfeg.noaa.gov/erddap/griddap/erdMHchla5day
        //json   http://coastwatch.pfeg.noaa.gov/erddap/info/erdMHchla5day/index.json
        String jsonUrl = String2.replaceAll(tSourceUrl, "/griddap/", "/info/") + "/index.json";
        String sourceInfo = SSR.getUrlResponseString(jsonUrl);
        Table table = new Table();
        table.readJson(jsonUrl, sourceInfo);

        //go through the rows of table from bottom to top
        int nRows = table.nRows();
        Attributes tSourceAttributes = new Attributes();
        ArrayList tAxisVariables = new ArrayList();
        ArrayList tDataVariables = new ArrayList();
        for (int row = nRows-1; row >= 0; row--) {  

            //"columnNames": ["Row Type", "Variable Name", "Attribute Name", "Data Type", "Value"],
            //"columnTypes": ["String", "String", "String", "String", "String"],
            //"rows": [
            //     ["attribute", "NC_GLOBAL", "acknowledgement", "String", "NOAA NESDIS COASTWATCH, NOAA SWFSC ERD"],
            //     ["dimension", "longitude", "", "double", "nValues=8640, evenlySpaced=true, averageSpacing=0.04166667052552379"],
            //     atts...
            //     ["variable", "chlorophyll", "", "float", "time, altitude, latitude, longitude"],
            //     atts...
            String rowType  = table.getStringData(0, row);
            String varName  = table.getStringData(1, row);
            String attName  = table.getStringData(2, row);
            String dataType = table.getStringData(3, row);
            String value    = table.getStringData(4, row);

            if (rowType.equals("attribute")) {
                if (dataType.equals("String")) {
                    tSourceAttributes.add(attName, value);
                } else {
                    Class tClass = PrimitiveArray.elementStringToType(dataType);
                    PrimitiveArray pa = PrimitiveArray.csvFactory(tClass, value);
                    tSourceAttributes.add(attName, pa);
                }

            } else if (rowType.equals("dimension")) {
                PrimitiveArray tSourceValues = OpendapHelper.getPrimitiveArray(dConnect, "?" + varName);
                Attributes tAddAttributes = new Attributes();
                //is this the lon axis?
                if (EDV.LON_NAME.equals(varName)) {
                    tAxisVariables.add(new EDVLonGridAxis(varName, 
                        tSourceAttributes, tAddAttributes, tSourceValues));

                //is this the lat axis?
                } else if (EDV.LAT_NAME.equals(varName)) {
                    tAxisVariables.add(new EDVLatGridAxis(varName, 
                        tSourceAttributes, tAddAttributes, tSourceValues));

                //is this the alt axis?
                } else if (EDV.ALT_NAME.equals(varName)) {
                    tAxisVariables.add(new EDVAltGridAxis(varName, 
                        tSourceAttributes, tAddAttributes, tSourceValues,
                        1)); //tAltMetersPerSourceUnit always 1

                //is this the time axis?
                } else if (EDV.TIME_NAME.equals(varName)) {
                    tAxisVariables.add(new EDVTimeGridAxis(varName, 
                        tSourceAttributes, tAddAttributes, tSourceValues));

                //it is some other axis variable
                } else {
                    tAxisVariables.add(new EDVGridAxis(varName, varName,
                        tSourceAttributes, tAddAttributes, tSourceValues));
                    ((EDVGridAxis)tAxisVariables.get(tAxisVariables.size() - 1)).setActualRangeFromDestinationMinMax();
                }

                //make new tSourceAttributes
                tSourceAttributes = new Attributes();

            //a grid variable
            } else if (rowType.equals("variable")) {
                EDV edv = new EDV(
                    varName, varName, 
                    tSourceAttributes, new Attributes(),
                    dataType, 
                    Double.NaN, Double.NaN);  //hard to get min and max
                edv.extractAndSetActualRange();
                tDataVariables.add(edv);
   
                //make new tSourceAttributes
                tSourceAttributes = new Attributes();

            //unexpected type
            } else throw new RuntimeException("Unexpected rowType=" + rowType + ".");
        }
        sourceGlobalAttributes = tSourceAttributes; //at the top of table, so collected last
        addGlobalAttributes = new Attributes();
        combinedGlobalAttributes = new Attributes(addGlobalAttributes, sourceGlobalAttributes); //order is important
        combinedGlobalAttributes.removeValue("null");

        axisVariables = new EDVGridAxis[tAxisVariables.size()]; 
        for (int av = 0; av < tAxisVariables.size(); av++) {
            axisVariables[av] = 
                (EDVGridAxis)tAxisVariables.get(tAxisVariables.size() - av - 1); //reverse the order
            String tName = axisVariables[av].destinationName();
            if      (tName.equals(EDV.LON_NAME))  lonIndex  = av;
            else if (tName.equals(EDV.LAT_NAME))  latIndex  = av;
            else if (tName.equals(EDV.ALT_NAME))  altIndex  = av;
            else if (tName.equals(EDV.TIME_NAME)) timeIndex = av;
        }
        dataVariables = new EDV[tDataVariables.size()];
        for (int dv = 0; dv < tDataVariables.size(); dv++)
            dataVariables[dv] = (EDV)tDataVariables.get(dv);

        //ensure the setup is valid
        ensureValid();  //this ensures many things are set, e.g., sourceUrl

        //try to subscribe to the remote dataset
        //It's ok that this is done every time. 
        //  emailIfAlreadyValid=false so there won't be excess email confirmations 
        //  and if flagKeyKey changes, the new tFlagUrl will be sent.
        //There is analogous code in EDDTableFromErddap.
        try {
            int gpo = tSourceUrl.indexOf("/griddap/");
            String subscriptionUrl = tSourceUrl.substring(0, gpo + 1) + Subscriptions.ADD_HTML + "?" +
                "datasetID=" + File2.getNameNoExtension(tSourceUrl) + 
                "&email=" + EDStatic.emailEverythingTo +
                "&emailIfAlreadyValid=false" + 
                "&action=\"" + flagUrl(datasetID()) + "\""; //the quotes allow for ? and & within flagUrl
            //String2.log("subscriptionUrl=" + subscriptionUrl); //don't normally display; flags are ~confidential
            SSR.touchUrl(subscriptionUrl, 60000);  
        } catch (Throwable st) {
            String2.log(
                "\nWARNING: an exception occurred while trying to subscribe to the remote ERDDAP dataset.\n" + 
                "If the subscription hasn't been set up already, you may need to\n" + 
                "use a small reloadEveryNMinutes, or have the remote ERDDAP admin add onChange.\n\n" 
                //+ MustBe.throwableToString(st) //don't normally display; flags are ~confidential
                );
        }

        //finally
        if (verbose) String2.log(
            (reallyVerbose? "\n" + toString() : "") +
            "\n*** EDDGridFromErddap " + datasetID() + " constructor finished. TIME=" + 
            (System.currentTimeMillis() - constructionStartMillis) + "\n"); 
    }

    /**
     * This returns the source ERDDAP's url.
     */
    public String getNextSourceErddapUrl() {
//future: if several sourceErddapUrl, return the next one
        return sourceErddapUrl;
    }

    /**
     * This makes a sibling dataset, based on the new sourceUrl.
     *
     * @param tSourceUrl
     * @param ensureAxisValuesAreEqual If Integer.MAX_VALUE, no axis sourceValue tests are performed. 
     *    If 0, this tests if sourceValues for axis-variable #0+ are same.
     *    If 1, this tests if sourceValues for axis-variable #1+ are same.
     *    (This is useful if the, for example, lat and lon values vary slightly and you 
     *    are willing to accept the initial values as the correct values.)
     *    Actually, the tests are always done but this determines whether
     *    the error is just logged or whether it throws an exception.
     * @param shareInfo if true, this ensures that the sibling's 
     *    axis and data variables are basically the same as this datasets,
     *    and then makes the new dataset point to the this instance's data structures
     *    to save memory. (AxisVariable #0 isn't duplicated.)
     *    Saving memory is important if there are 1000's of siblings in ERDDAP.
     * @return EDDGrid
     * @throws Throwable if trouble  (e.g., try to shareInfo, but datasets not similar)
     */
    public EDDGrid sibling(String tSourceUrl, int ensureAxisValuesAreEqual, boolean shareInfo) throws Throwable {
        if (verbose) String2.log("EDDGridFromErddap.sibling " + tSourceUrl);

        int nAv = axisVariables.length;
        int nDv = dataVariables.length;

        EDDGridFromErddap newEDDGrid = new EDDGridFromErddap(
            datasetID, 
            String2.toSSVString(accessibleTo),
            shareInfo? onChange : (StringArray)onChange.clone(), 
            getReloadEveryNMinutes(), tSourceUrl);

        //if shareInfo, point to same internal data
        if (shareInfo) {

            //ensure similar
            boolean testAV0 = false;
            String results = similar(newEDDGrid, ensureAxisValuesAreEqual, false); 
            if (results.length() > 0)
                throw new RuntimeException("Error in EDDGrid.sibling: " + results);

            //shareInfo
            for (int av = 1; av < nAv; av++) //not av0
                newEDDGrid.axisVariables()[av] = axisVariables[av];
            newEDDGrid.dataVariables = dataVariables;

            //shareInfo  (the EDDGrid variables)
            newEDDGrid.axisVariableSourceNames      = axisVariableSourceNames(); //() makes the array
            newEDDGrid.axisVariableDestinationNames = axisVariableDestinationNames();

            //shareInfo  (the EDD variables)
            newEDDGrid.dataVariableSourceNames      = dataVariableSourceNames();
            newEDDGrid.dataVariableDestinationNames = dataVariableDestinationNames();
            newEDDGrid.title                        = title();
            newEDDGrid.summary                      = summary();
            newEDDGrid.institution                  = institution();
            newEDDGrid.infoUrl                      = infoUrl();
            newEDDGrid.cdmDataType                  = cdmDataType();
            newEDDGrid.searchString                 = searchString();
            //not sourceUrl, which will be different
            newEDDGrid.sourceGlobalAttributes       = sourceGlobalAttributes();
            newEDDGrid.addGlobalAttributes          = addGlobalAttributes();
            newEDDGrid.combinedGlobalAttributes     = combinedGlobalAttributes();

        }

        return newEDDGrid;
    }

    /** 
     * This gets data (not yet standardized) from the data 
     * source for this EDDGrid.     
     * 
     * @param tDataVariables
     * @param tConstraints
     * @return a PrimitiveArray[] where the first axisVariables.length elements
     *   are the axisValues and the next tDataVariables.length elements
     *   are the dataValues.
     *   Both the axisValues and dataValues are straight from the source,
     *   not modified.
     * @throws Throwable if trouble
     */
    public PrimitiveArray[] getSourceData(EDV tDataVariables[], IntArray tConstraints) 
        throws Throwable {

        //build String form of the constraint
        //String errorInMethod = "Error in EDDGridFromErddap.getSourceData for " + datasetID() + ": "; 
        String constraint = buildDapArrayQuery(tConstraints);

        DConnect dConnect = new DConnect(sourceErddapUrl, acceptDeflate, 1, 1);
        PrimitiveArray results[] = new PrimitiveArray[axisVariables.length + tDataVariables.length];
        for (int dv = 0; dv < tDataVariables.length; dv++) {
            //???why not get all the dataVariables at once?

            //get the data
            PrimitiveArray pa[] = null;
            try {
                pa = OpendapHelper.getPrimitiveArrays(dConnect, 
                    "?" + tDataVariables[dv].sourceName() + constraint);
            } catch (Throwable t) {
                requestReloadASAP(); 
                throw new WaitThenTryAgainException(EDStatic.waitThenTryAgain, t); 
            }
            if (pa.length != axisVariables.length + 1) {
                requestReloadASAP(); 
                throw new WaitThenTryAgainException(EDStatic.waitThenTryAgain + 
                    "\nDetails: An unexpected data structure was returned from the source.");
            }
            results[axisVariables.length + dv] = pa[0];
            if (dv == 0) {
                //I think GridDataAccessor compares observed and expected axis values
                for (int av = 0; av < axisVariables.length; av++) {
                    results[av] = pa[av + 1];
                }
            } else {
                for (int av = 0; av < axisVariables.length; av++) {
                    String tError = results[av].almostEqual(pa[av + 1]); 
                    if (tError.length() > 0) {
                        requestReloadASAP(); 
                        throw new WaitThenTryAgainException(
                            EDStatic.waitThenTryAgain +
                            "\nDetails: The axis values for dataVariable=0,axis=" + av +  
                            ")\ndon't equal the axis values for dataVariable=" + dv + ",axis=" + av + ".\n" +
                            tError);
                    }
                }
            }
        }
        return results;
    }

    /** 
     * This generates the datasets.xml entries for all EDDGrid from a remote ERDDAP.
     * The XML can then be edited by hand and added to the datasets.xml file.
     *
     * @param url the base url for the dataset, e.g., 
     *   "http://coastwatch.pfeg.noaa.gov/erddap"
     * @throws Throwable if trouble
     */
    public static String generateDatasetsXml(String url) 
        throws Throwable {

        String2.log("EDDGridFromErddap.generateDatasetsXml " + url);

        //make the stringBuffer to hold the results and add documentation
        StringBuffer sb = new StringBuffer();
        sb.append(  //there is very similar text in EDDGridFromErddap
"<!-- Directions:\n" +
" * The XML below includes information for all of the EDDGrid datasets at the remote ERDDAP\n" +
"   " + XML.encodeAsXML(url) + "\n" +
"   (except for etopo180 and etopo360, which are built into every ERDDAP).\n" +
" * If you want to add all of these datasets to your ERDDAP, just paste the XML\n" +
"   into your datasets.xml file.\n" +
" * !!!You need to ensure that none of the datasetID's below is the same as\n" +
"   one of your existing datasetIDs!!!\n" +
" * !!!reloadEveryNMinutes is left as the default 10080=oncePerWeek on the assumption\n" +
"   that the remote ERDDAP will accept your ERDDAP's request to subscribe to the dataset.\n" +
"   If you don't get emails from the remote ERDDAP asking you to validate your subscription\n" +
"   requests (perhaps because the remote ERDDAP has the subscription system turned off),\n" +
"   send an email to the admin asking that s/he add onChange tags to the datasets.\n" +
"   See the EDDGridFromErddap documentation.\n" + 
" * The XML needed for EDDGridFromErddap in datasets.xml has few options.  See\n" +
"   http://coastwatch.pfeg.noaa.gov/erddap/download/setupDatasetsXml.html#EDDGridFromErddap .\n" +
"   If you want to alter a dataset's metadata or make other changes to a dataset,\n" +
"   use EDDGridFromDap to access the dataset instead of EDDGridFromErddap.\n" +
"-->\n");

        //get the griddap datasets in a json table
        String jsonUrl = url + "/griddap/index.json";
        String sourceInfo = SSR.getUrlResponseString(jsonUrl);
        if (reallyVerbose) String2.log(sourceInfo.substring(0, Math.min(sourceInfo.length(), 2000)));
        if (sourceInfo.indexOf("\"table\"") > 0) {
            Table table = new Table();
            table.readJson(jsonUrl, sourceInfo);   //they are sorted by title

            PrimitiveArray urlCol = table.findColumn("griddap");
            PrimitiveArray titleCol = table.findColumn("Title");
            PrimitiveArray datasetIdCol = table.findColumn("Dataset ID");

            //go through the rows of the table
            int nRows = table.nRows();
            for (int row = 0; row < nRows; row++) {
                String id = datasetIdCol.getString(row);
                if (id.equals("etopo180") || id.equals("etopo360"))
                    continue;
                sb.append(
"    <dataset type=\"EDDGridFromErddap\" datasetID=\"" + id + "\" active=\"true\">\n" +
"        <!-- " + XML.encodeAsXML(titleCol.getString(row)) + " -->\n" +
"        <sourceUrl>" + XML.encodeAsXML(urlCol.getString(row)) + "</sourceUrl>\n" +
"    </dataset>\n");
            }
        }

        //get the EDDGridFromErddap datasets 
        jsonUrl = url + "/search/index.json?searchFor=EDDGridFromErddap";
        sourceInfo = "";
        try {
            sourceInfo = SSR.getUrlResponseString(jsonUrl);
        } catch (Throwable t) {
            //error if remote erddap has no EDDGridFromErddap's
        }
        if (reallyVerbose) String2.log(sourceInfo.substring(0, Math.min(sourceInfo.length(), 2000)));
        PrimitiveArray datasetIdCol;
        if (sourceInfo.indexOf("\"table\"") > 0) {
            Table table = new Table();
            table.readJson(jsonUrl, sourceInfo);   //they are sorted by title
            datasetIdCol = table.findColumn("Dataset ID");
        } else {
            datasetIdCol = new StringArray();
        }

        sb.append(
            "\n<!-- Of the datasets above, the following datasets are EDDGridFromErddap's at the remote ERDDAP.\n" +
            "It would be best if you contacted the remote ERDDAP's administrator and requested the dataset XML\n" +
            "that is being using for these datasets so your ERDDAP can access the original ERDDAP source.\n" +
            "The remote EDDGridFromErddap datasets are:\n");
        if (datasetIdCol.size() == 0)
            sb.append("(none)");
        else sb.append(String2.noLongLines(datasetIdCol.toString(), 70, ""));
        sb.append("\n-->\n");

        return sb.toString();
    }

    /**
     * testGenerateDatasetsXml
     */
    public static void testGenerateDatasetsXml() throws Throwable {
        testVerboseOn();

        //test local generateDatasetsXml
        try {
            String results = generateDatasetsXml(EDStatic.erddapUrl); //in tests, always non-https url
            String2.log("results=\n" + results);

String expected = 
"<!-- Directions:\n" +
" * The XML below includes information for all of the EDDGrid datasets at the remote ERDDAP\n";

            Test.ensureEqual(results.substring(0, Math.min(results.length(), expected.length())), 
                expected, "results=\n" + results);

expected = 
"erdMHchla8day\" active=\"true\">\n" +
"        <!-- Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite) -->\n" +
"        <sourceUrl>http://127.0.0.1:8080/cwexperimental/griddap/erdMHchla8day</sourceUrl>\n" +
"    </dataset>\n" +
"    <dataset type=\"EDDGridFromErddap\" datasetID=\"";

            int po = results.indexOf(expected.substring(0, 13));
            Test.ensureEqual(results.substring(po, po + expected.length()), expected, "results=\n" + results);

expected = 
"erdBAssta5day\" active=\"true\">\n" +
"        <!-- SST, Blended, Global, EXPERIMENTAL (5 Day Composite) -->\n" +
"        <sourceUrl>http://127.0.0.1:8080/cwexperimental/griddap/erdBAssta5day</sourceUrl>\n" +
"    </dataset>\n" +
"    <dataset type=\"EDDGridFromErddap\" datasetID=\"";

            po = results.indexOf(expected.substring(0, 13));
            Test.ensureEqual(results.substring(po, po + expected.length()), expected, "results=\n" + results);

expected = 
"<!-- Of the datasets above, the following datasets are EDDGridFromErddap's at the remote ERDDAP.\n";
            po = results.indexOf(expected.substring(0, 20));
            Test.ensureEqual(results.substring(po, po + expected.length()), expected, "results=\n" + results);
            Test.ensureTrue(results.indexOf("rMHchla8day", po) > 0, "results=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nError using generateDatasetsXml on " + EDStatic.erddapUrl + //in tests, always non-https url
                "\nPress ^C to stop or Enter to continue..."); 
        }

    }

    /** This does some basic tests. */
    public static void testBasic(boolean testLocalErddapToo) throws Throwable {
        testVerboseOn();
        EDDGridFromErddap gridDataset;
        String name, tName, axisDapQuery, query, results, expected, expected2, error;
        int tPo;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
        String userDapQuery  = "chlorophyll[(2007-02-06)][][(29):10:(50)][(225):10:(247)]";
        String graphDapQuery = "chlorophyll[0:10:200][][(29)][(225)]"; 
        String mapDapQuery   = "chlorophyll[200][][(29):(50)][(225):(247)]"; //stride irrelevant 
        StringArray destinationNames = new StringArray();
        IntArray constraints = new IntArray();
        String localUrl = EDStatic.erddapUrl + "/griddap/rMHchla8day"; //in tests, always non-https url

        try {

            gridDataset = (EDDGridFromErddap)oneFromDatasetXml("rMHchla8day"); 

            //*** test getting das for entire dataset
            String2.log("\n****************** EDDGridFromErddap test entire dataset\n");
            tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Entire", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
"Attributes {\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.0260864e+9, 1.2373344e+9;\n" + //last value changes periodically
"    String axis \"T\";\n" +
"    Int32 fraction_digits 0;\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Centered Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    Float64 actual_range 0.0, 0.0;\n" +
"    String axis \"Z\";\n" +
"    Int32 fraction_digits 0;\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -89.97916, 89.97916;\n" +
"    String axis \"Y\";\n" +
"    String coordsys \"geographic\";\n" +
"    Int32 fraction_digits 4;\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String point_spacing \"even\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range 0.02083333, 359.9792;\n" +
"    String axis \"X\";\n" +
"    String coordsys \"geographic\";\n" +
"    Int32 fraction_digits 4;\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String point_spacing \"even\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  chlorophyll {\n" +
"    Float32 _FillValue -9999999.0;\n" +
"    Float64 colorBarMaximum 30.0;\n" +
"    Float64 colorBarMinimum 0.03;\n" +
"    String colorBarScale \"Log\";\n" +
"    String coordsys \"geographic\";\n" +
"    Int32 fraction_digits 2;\n" +
"    String ioos_category \"Ocean Color\";\n" +
"    String long_name \"Concentration Of Chlorophyll In Sea Water\";\n" +
"    Float32 missing_value -9999999.0;\n" +
"    String standard_name \"concentration_of_chlorophyll_in_sea_water\";\n" +
"    String units \"mg m-3\";\n" +
"  }\n" +
"  NC_GLOBAL {\n" +
"    String acknowledgement \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
"    String cdm_data_type \"Grid\";\n" +
"    String composite \"true\";\n" +
"    String contributor_name \"NASA GSFC (G. Feldman)\";\n" +
"    String contributor_role \"Source of level 2 data.\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    String creator_email \"dave.foley@noaa.gov\";\n" +
"    String creator_name \"NOAA CoastWatch, West Coast Node\";\n" +
"    String creator_url \"http://coastwatch.pfel.noaa.gov\";\n" +
"    String date_created \"2009-04-21Z\";\n" +
"    String date_issued \"2009-04-21Z\";\n" +
"    Float64 Easternmost_Easting 359.9792;\n" +
"    Float64 geospatial_lat_max 89.97916;\n" +
"    Float64 geospatial_lat_min -89.97916;\n" +
"    Float64 geospatial_lat_resolution 0.0416666635795323;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 359.9792;\n" +
"    Float64 geospatial_lon_min 0.02083333;\n" +
"    Float64 geospatial_lon_resolution 0.04166667052552379;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    Float64 geospatial_vertical_max 0.0;\n" +
"    Float64 geospatial_vertical_min 0.0;\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"NASA GSFC (G. Feldman)\n" +
"2009-04-21T03:44:34Z NOAA CoastWatch (West Coast Node) and NOAA SFSC ERD\n" + //changes sometimes
today + " http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day\n";
//today + " http://127.0.0.1:8080/cwexperimental/griddap/rMHchla8day.das\";\n" +   //This is it working locally.
//or           coastwatch ...      //what I expected/wanted.  This really appears as if remote dataset.

expected2 =
"    String infoUrl \"http://coastwatch.pfeg.noaa.gov/infog/MH_chla_las.html\";\n" +
"    String institution \"NOAA CoastWatch, West Coast Node\";\n" +
"    String keywords \"EARTH SCIENCE > Oceans > Ocean Chemistry > Chlorophyll\";\n" +
"    String keywords_vocabulary \"GCMD Science Keywords\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended for legal use, since it may contain inaccuracies. Neither the data Contributor, CoastWatch, NOAA, nor the United States Government, nor any of their employees or contractors, makes any warranty, express or implied, including warranties of merchantability and fitness for a particular purpose, or assumes any legal liability for the accuracy, completeness, or usefulness, of this information.\";\n" +
"    String naming_authority \"gov.noaa.pfel.coastwatch\";\n" +
"    Float64 Northernmost_Northing 89.97916;\n" +
"    String origin \"NASA GSFC (G. Feldman)\";\n" +
"    String processing_level \"3\";\n" +
"    String project \"CoastWatch (http://coastwatch.noaa.gov/)\";\n" +
"    String projection \"geographic\";\n" +
"    String projection_type \"mapped\";\n" +
"    String references \"Aqua/MODIS information: http://oceancolor.gsfc.nasa.gov/ . MODIS information: http://coastwatch.noaa.gov/modis_ocolor_overview.html .\";\n" +
"    String satellite \"Aqua\";\n" +
"    String sensor \"MODIS\";\n" +
"    String source \"satellite observation: Aqua, MODIS\";\n" +
"    String sourceUrl \"http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day\";\n" +
"    Float64 Southernmost_Northing -89.97916;\n" +
"    String standard_name_vocabulary \"CF-1.0\";\n" +
"    String summary \"NOAA CoastWatch distributes chlorophyll-a concentration data from NASA's Aqua Spacecraft.  Measurements are gathered by the Moderate Resolution Imaging Spectroradiometer (MODIS) carried aboard the spacecraft.   This is Science Quality data.\";\n" +
"    String title \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
"    Float64 Westernmost_Easting 0.02083333;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
            tPo = results.indexOf("    String infoUrl ");
            Test.ensureEqual(results.substring(tPo), expected2, "\nresults=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".das");
                Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
                tPo = results.indexOf("    String infoUrl ");
                Test.ensureEqual(results.substring(tPo), expected2, "\nresults=\n" + results);
            }

            //*** test getting dds for entire dataset
            tName = gridDataset.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Entire", ".dds"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
    "Dataset {\n" +
    "  Float64 time[time = 302];\n" +   //302 will change sometimes   (and a few places below)
    "  Float64 altitude[altitude = 1];\n" +
    "  Float64 latitude[latitude = 4320];\n" +
    "  Float64 longitude[longitude = 8640];\n" +
    "  GRID {\n" +
    "    ARRAY:\n" +
    "      Float32 chlorophyll[time = 302][altitude = 1][latitude = 4320][longitude = 8640];\n" +
    "    MAPS:\n" +
    "      Float64 time[time = 302];\n" +
    "      Float64 altitude[altitude = 1];\n" +
    "      Float64 latitude[latitude = 4320];\n" +
    "      Float64 longitude[longitude = 8640];\n" +
    "  } chlorophyll;\n" +
    "} "; //rMHchla8day;\n";
            Test.ensureEqual(results, expected + "rMHchla8day;\n", "\nresults=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".dds");
                Test.ensureEqual(results, expected + "erdMHchla8day;\n", "\nresults=\n" + results);
            }

            //********************************************** test getting axis data

            //.asc
            String2.log("\n*** EDDGridFromErddap test get .ASC axis data\n");
            query = "time[0:100:200],longitude[last]";
            tName = gridDataset.makeNewFileForDapQuery(null, null, query, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Axis", ".asc"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
    "Dataset {\n" +
    "  Float64 time[time = 3];\n" +
    "  Float64 longitude[longitude = 1];\n" +
    "} "; //r
            expected2 = "MHchla8day;\n" +
    "---------------------------------------------\n" +
    "Data:\n" +
    "time[3]\n" +
    "1.0260864E9, 1.0960704E9, 1.1661408E9\n" +
    "longitude[1]\n" +
    "359.9792\n";
            Test.ensureEqual(results, expected + "r" + expected2, "RESULTS=\n" + results);
     
            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".asc?" + query);
                Test.ensureEqual(results, expected + "erd" + expected2, "\nresults=\n" + results);
            }

            //.csv
            String2.log("\n*** EDDGridFromErddap test get .CSV axis data\n");
            query = "time[0:100:200],longitude[last]";
            tName = gridDataset.makeNewFileForDapQuery(null, null, query, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Axis", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
            expected = 
    "time, longitude\n" +
    "UTC, degrees_east\n" +
    "2002-07-08T00:00:00Z, 359.9792\n" +
    "2004-09-25T00:00:00Z, NaN\n" +
    "2006-12-15T00:00:00Z, NaN\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".csv?" + query);
                Test.ensureEqual(results, expected, "\nresults=\n" + results);
            }

            //.csv  test of gridName.axisName notation
            String2.log("\n*** EDDGridFromErddap test get .CSV axis data\n");
            query = "chlorophyll.time[0:100:200],chlorophyll.longitude[last]";
            tName = gridDataset.makeNewFileForDapQuery(null, null, query, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_AxisG.A", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            //SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
            expected = 
    "time, longitude\n" +
    "UTC, degrees_east\n" +
    "2002-07-08T00:00:00Z, 359.9792\n" +
    "2004-09-25T00:00:00Z, NaN\n" +
    "2006-12-15T00:00:00Z, NaN\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".csv?" + query);
                Test.ensureEqual(results, expected, "\nresults=\n" + results);
            }

            //.dods
            String2.log("\n*** EDDGridFromErddap test get .DODS axis data\n");
            query = "time[0:100:200],longitude[last]";
            tName = gridDataset.makeNewFileForDapQuery(null, null, query, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Axis", ".dods"); 
            results = String2.annotatedString(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));
            //String2.log(results);
            expected = 
    "Dataset {[10]\n" +
    "  Float64 time[time = 3];[10]\n" +
    "  Float64 longitude[longitude = 1];[10]\n" +
    "} ";  //r
            expected2 = "MHchla8day;[10]\n" +
    "[10]\n" +
    "Data:[10]\n" +
    "[0][0][0][3][0][0][0][3]A[206][8221]k[0][0][0][0]A[208]U-@[0][0][0]A[209]`y`[0][0][0][0][0][0][1][0][0][0][1]@v[127][170][205][382][402][228][end]";
            Test.ensureEqual(results, expected + "r" + expected2, "RESULTS=\n" + results);

            if (testLocalErddapToo) {
                results = String2.annotatedString(SSR.getUrlResponseString(localUrl + ".dods?" + query));
                Test.ensureEqual(results, expected + "erd" + expected2, "\nresults=\n" + results);
            }

            //.mat
            //octave> load('c:/temp/griddap/EDDGridFromErddap_Axis.mat');
            //octave> erdMHchla8day
            String matlabAxisQuery = "time[0:100:200],longitude[last]"; 
            String2.log("\n*** EDDGridFromErddap test get .MAT axis data\n");
            tName = gridDataset.makeNewFileForDapQuery(null, null, matlabAxisQuery, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Axis", ".mat"); 
            String2.log(".mat test file is " + EDStatic.fullTestCacheDirectory + tName);
            results = File2.hexDump(EDStatic.fullTestCacheDirectory + tName, 1000000);
            String2.log(results);
            expected = 
    "4d 41 54 4c 41 42 20 35   2e 30 20 4d 41 54 2d 66   MATLAB 5.0 MAT-f |\n" +
    "69 6c 65 2c 20 43 72 65   61 74 65 64 20 62 79 3a   ile, Created by: |\n" +
    "20 67 6f 76 2e 6e 6f 61   61 2e 70 66 65 6c 2e 63    gov.noaa.pfel.c |\n" +
    "6f 61 73 74 77 61 74 63   68 2e 4d 61 74 6c 61 62   oastwatch.Matlab |\n" +
    //"2c 20 43 72 65 61 74 65   64 20 6f 6e 3a 20 54 75   , Created on: Tu |\n" +
    //"65 20 4f 63 74 20 31 34   20 30 38 3a 35 36 3a 35   e Oct 14 08:56:5 |\n" +
    //"34 20 32 30 30 38 20 20   20 20 20 20 20 20 20 20   4 2008           |\n" +
    "20 20 20 20 00 00 00 00   00 00 00 00 01 00 4d 49                 MI |\n" +
    "00 00 00 0e 00 00 01 18   00 00 00 06 00 00 00 08                    |\n" +
    "00 00 00 02 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
    "00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 0b                    |\n" +
    "72 4d 48 63 68 6c 61 38   64 61 79 00 00 00 00 00   rMHchla8day      |\n" +
    "00 04 00 05 00 00 00 20   00 00 00 01 00 00 00 40                  @ |\n" +
    "74 69 6d 65 00 00 00 00   00 00 00 00 00 00 00 00   time             |\n" +
    "00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
    "6c 6f 6e 67 69 74 75 64   65 00 00 00 00 00 00 00   longitude        |\n" +
    "00 00 00 00 00 00 00 00   00 00 00 00 00 00 00 00                    |\n" +
    "00 00 00 0e 00 00 00 48   00 00 00 06 00 00 00 08          H         |\n" +
    "00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
    "00 00 00 03 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
    "00 00 00 09 00 00 00 18   41 ce 94 6b 00 00 00 00           A  k     |\n" +
    "41 d0 55 2d 40 00 00 00   41 d1 60 79 60 00 00 00   A U-@   A `y`    |\n" +
    "00 00 00 0e 00 00 00 38   00 00 00 06 00 00 00 08          8         |\n" +
    "00 00 00 06 00 00 00 00   00 00 00 05 00 00 00 08                    |\n" +
    "00 00 00 01 00 00 00 01   00 00 00 01 00 00 00 00                    |\n" +
    "00 00 00 09 00 00 00 08   40 76 7f aa cd 9e 83 e4           @v       |\n";
            Test.ensureEqual(
                results.substring(0, 71 * 4) + results.substring(71 * 7), //remove the creation dateTime
                expected, "RESULTS(" + EDStatic.fullTestCacheDirectory + tName + ")=\n" + results);

            //.ncHeader
            String2.log("\n*** EDDGridFromErddap test get .NCHEADER axis data\n");
            query = "time[0:100:200],longitude[last]";
            tName = gridDataset.makeNewFileForDapQuery(null, null, query, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Axis", ".ncHeader"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            //String2.log(results);
            expected = 
//    "netcdf EDDGridFromErddap_Axis.nc {\n" +
//    " dimensions:\n" +
//    "   time = 3;\n" +   // (has coord.var)\n" +   //changed when switched to netcdf-java 4.0, 2009-02-23   
//    "   longitude = 1;\n" +   // (has coord.var)\n" +   //but won't change for testLocalErddapToo until next release
    " variables:\n" +
    "   double time(time=3);\n" +
    "     :_CoordinateAxisType = \"Time\";\n" +
    "     :actual_range = 1.0260864E9, 1.1661408E9; // double\n" +  //up-to-date
    "     :axis = \"T\";\n" +
    "     :fraction_digits = 0; // int\n" +
    "     :ioos_category = \"Time\";\n" +
    "     :long_name = \"Centered Time\";\n" +
    "     :standard_name = \"time\";\n" +
    "     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
    "     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
    "   double longitude(longitude=1);\n" +
    "     :_CoordinateAxisType = \"Lon\";\n" +
    "     :actual_range = 359.9792, 359.9792; // double\n" +
    "     :axis = \"X\";\n" +
    "     :coordsys = \"geographic\";\n" +
    "     :fraction_digits = 4; // int\n" +
    "     :ioos_category = \"Location\";\n" +
    "     :long_name = \"Longitude\";\n" +
    "     :point_spacing = \"even\";\n" +
    "     :standard_name = \"longitude\";\n" +
    "     :units = \"degrees_east\";\n" +
    "\n" +
    " :acknowledgement = \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n";
            tPo = results.indexOf(" variables:\n");
            Test.ensureEqual(results.substring(tPo, tPo + expected.length()), expected, "RESULTS=\n" + results);
            expected2 = 
    " :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
    " :Westernmost_Easting = 359.9792; // double\n" +
    " data:\n" +
    "}\n";
            Test.ensureTrue(results.indexOf(expected2) > 0, "RESULTS=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".ncHeader?" + query);
                tPo = results.indexOf(" variables:\n");
                Test.ensureEqual(results.substring(tPo, tPo + expected.length()), expected, "\nresults=\n" + results);
                Test.ensureTrue(results.indexOf(expected2) > 0, "RESULTS=\n" + results);
            }



            //********************************************** test getting grid data
            //.csv
            String2.log("\n*** EDDGridFromErddap test get .CSV data\n");
            tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Data", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected =  //missing values are "NaN"
    "time, altitude, latitude, longitude, chlorophyll\n" +
    "UTC, m, degrees_north, degrees_east, mg m-3\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 224.97918749730292, NaN\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.39585420255816, NaN\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.8125209078134, 0.096\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.22918761306863, 0.119\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.64585431832387, NaN\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 227.0625210235791, 0.102\n";
            Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
            expected2 = "2007-02-06T00:00:00Z, 0.0, 49.437496337115064, 232.06252148664197, 0.367\n";
            Test.ensureTrue(results.indexOf(expected2) > 0, "RESULTS=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".csv?" + userDapQuery);
                Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
                Test.ensureTrue(results.indexOf(expected2) > 0, "RESULTS=\n" + results);
            }

            //.csv   test gridName.gridName notation
            String2.log("\n*** EDDGridFromErddap test get .CSV data\n");
            tName = gridDataset.makeNewFileForDapQuery(null, null, "chlorophyll." + userDapQuery, 
                EDStatic.fullTestCacheDirectory, gridDataset.className() + "_DotNotation", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
    "time, altitude, latitude, longitude, chlorophyll\n" +
    "UTC, m, degrees_north, degrees_east, mg m-3\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 224.97918749730292, NaN\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.39585420255816, NaN\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 225.8125209078134, 0.096\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.22918761306863, 0.119\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 226.64585431832387, NaN\n" +
    "2007-02-06T00:00:00Z, 0.0, 29.020831183144253, 227.0625210235791, 0.102\n";
            Test.ensureTrue(results.indexOf(expected) == 0, "RESULTS=\n" + results);
            expected2 = "2007-02-06T00:00:00Z, 0.0, 49.437496337115064, 232.06252148664197, 0.367\n";
            Test.ensureTrue(results.indexOf(expected2) > 0, "RESULTS=\n" + results);

            if (testLocalErddapToo) {
                results = SSR.getUrlResponseString(localUrl + ".csv" + "?chlorophyll." + userDapQuery);
                Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
                Test.ensureTrue(results.indexOf(expected2) > 0, "RESULTS=\n" + results);
            }

            //.nc
            String2.log("\n*** EDDGridFromErddap test get .NC data\n");
            tName = gridDataset.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
                gridDataset.className() + "_Data", ".nc"); 
            results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory  + tName, true);
            expected = 
    "netcdf EDDGridFromErddap_Data.nc {\n" +
    " dimensions:\n" +
    "   time = 1;\n" +   // (has coord.var)\n" +   //changed when switched to netcdf-java 4.0, 2009-02-23
    "   altitude = 1;\n" +   // (has coord.var)\n" +
    "   latitude = 51;\n" +   // (has coord.var)\n" +
    "   longitude = 53;\n" +   // (has coord.var)\n" +
    " variables:\n" +
    "   double time(time=1);\n" +
    "     :_CoordinateAxisType = \"Time\";\n" +
    "     :actual_range = 1.17072E9, 1.17072E9; // double\n" +
    "     :axis = \"T\";\n" +
    "     :fraction_digits = 0; // int\n" +
    "     :ioos_category = \"Time\";\n" +
    "     :long_name = \"Centered Time\";\n" +
    "     :standard_name = \"time\";\n" +
    "     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
    "     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
    "   double altitude(altitude=1);\n" +
    "     :_CoordinateAxisType = \"Height\";\n" +
    "     :_CoordinateZisPositive = \"up\";\n" +
    "     :actual_range = 0.0, 0.0; // double\n" +
    "     :axis = \"Z\";\n" +
    "     :fraction_digits = 0; // int\n" +
    "     :ioos_category = \"Location\";\n" +
    "     :long_name = \"Altitude\";\n" +
    "     :positive = \"up\";\n" +
    "     :standard_name = \"altitude\";\n" +
    "     :units = \"m\";\n" +
    "   double latitude(latitude=51);\n" +
    "     :_CoordinateAxisType = \"Lat\";\n" +
    "     :actual_range = 29.020831183144253, 49.8541629729104; // double\n" +
    "     :axis = \"Y\";\n" +
    "     :coordsys = \"geographic\";\n" +
    "     :fraction_digits = 4; // int\n" +
    "     :ioos_category = \"Location\";\n" +
    "     :long_name = \"Latitude\";\n" +
    "     :point_spacing = \"even\";\n" +
    "     :standard_name = \"latitude\";\n" +
    "     :units = \"degrees_north\";\n" +
    "   double longitude(longitude=53);\n" +
    "     :_CoordinateAxisType = \"Lon\";\n" +
    "     :actual_range = 224.97918749730292, 246.64585617057529; // double\n" +
    "     :axis = \"X\";\n" +
    "     :coordsys = \"geographic\";\n" +
    "     :fraction_digits = 4; // int\n" +
    "     :ioos_category = \"Location\";\n" +
    "     :long_name = \"Longitude\";\n" +
    "     :point_spacing = \"even\";\n" +
    "     :standard_name = \"longitude\";\n" +
    "     :units = \"degrees_east\";\n" +
    "   float chlorophyll(time=1, altitude=1, latitude=51, longitude=53);\n" +
    "     :_FillValue = -9999999.0f; // float\n" +
    "     :colorBarMaximum = 30.0; // double\n" +
    "     :colorBarMinimum = 0.03; // double\n" +
    "     :colorBarScale = \"Log\";\n" +
    "     :coordsys = \"geographic\";\n" +
    "     :fraction_digits = 2; // int\n" +
    "     :ioos_category = \"Ocean Color\";\n" +
    "     :long_name = \"Concentration Of Chlorophyll In Sea Water\";\n" +
    "     :missing_value = -9999999.0f; // float\n" +
    "     :standard_name = \"concentration_of_chlorophyll_in_sea_water\";\n" +
    "     :units = \"mg m-3\";\n" +
    "\n" +
    " :acknowledgement = \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD\";\n" +
    " :cdm_data_type = \"Grid\";\n" +
    " :composite = \"true\";\n" +
    " :contributor_name = \"NASA GSFC (G. Feldman)\";\n" +
    " :contributor_role = \"Source of level 2 data.\";\n" +
    " :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
    " :creator_email = \"dave.foley@noaa.gov\";\n" +
    " :creator_name = \"NOAA CoastWatch, West Coast Node\";\n" +
    " :creator_url = \"http://coastwatch.pfel.noaa.gov\";\n" +
    " :date_created = \"2009-04-21Z\";\n" + //changes periodically
    " :date_issued = \"2009-04-21Z\";\n" +
    " :Easternmost_Easting = 246.64585617057529; // double\n" +
    " :geospatial_lat_max = 49.8541629729104; // double\n" +
    " :geospatial_lat_min = 29.020831183144253; // double\n" +
    " :geospatial_lat_resolution = 0.0416666635795323; // double\n" +
    " :geospatial_lat_units = \"degrees_north\";\n" +
    " :geospatial_lon_max = 246.64585617057529; // double\n" +
    " :geospatial_lon_min = 224.97918749730292; // double\n" +
    " :geospatial_lon_resolution = 0.04166667052552379; // double\n" +
    " :geospatial_lon_units = \"degrees_east\";\n" +
    " :geospatial_vertical_max = 0.0; // double\n" +
    " :geospatial_vertical_min = 0.0; // double\n" +
    " :geospatial_vertical_positive = \"up\";\n" +
    " :geospatial_vertical_units = \"m\";\n" +
    " :history = \"NASA GSFC (G. Feldman)";
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
            expected = //note original missing values
    " :infoUrl = \"http://coastwatch.pfeg.noaa.gov/infog/MH_chla_las.html\";\n" +
    " :institution = \"NOAA CoastWatch, West Coast Node\";\n" +
    " :keywords = \"EARTH SCIENCE > Oceans > Ocean Chemistry > Chlorophyll\";\n" +
    " :keywords_vocabulary = \"GCMD Science Keywords\";\n" +
    " :license = \"The data may be used and redistributed for free but is not intended for legal use, since it may contain inaccuracies. Neither the data Contributor, CoastWatch, NOAA, nor the United States Government, nor any of their employees or contractors, makes any warranty, express or implied, including warranties of merchantability and fitness for a particular purpose, or assumes any legal liability for the accuracy, completeness, or usefulness, of this information.\";\n" +
    " :naming_authority = \"gov.noaa.pfel.coastwatch\";\n" +
    " :Northernmost_Northing = 49.8541629729104; // double\n" +
    " :origin = \"NASA GSFC (G. Feldman)\";\n" +
    " :processing_level = \"3\";\n" +
    " :project = \"CoastWatch (http://coastwatch.noaa.gov/)\";\n" +
    " :projection = \"geographic\";\n" +
    " :projection_type = \"mapped\";\n" +
    " :references = \"Aqua/MODIS information: http://oceancolor.gsfc.nasa.gov/ . MODIS information: http://coastwatch.noaa.gov/modis_ocolor_overview.html .\";\n" +
    " :satellite = \"Aqua\";\n" +
    " :sensor = \"MODIS\";\n" +
    " :source = \"satellite observation: Aqua, MODIS\";\n" +
    " :sourceUrl = \"http://192.168.31.18/thredds/dodsC/satellite/MH/chla/8day\";\n" +
    " :Southernmost_Northing = 29.020831183144253; // double\n" +
    " :standard_name_vocabulary = \"CF-1.0\";\n" +
    " :summary = \"NOAA CoastWatch distributes chlorophyll-a concentration data from NASA's Aqua Spacecraft.  Measurements are gathered by the Moderate Resolution Imaging Spectroradiometer (MODIS) carried aboard the spacecraft.   This is Science Quality data.\";\n" +
    " :time_coverage_end = \"2007-02-06T00:00:00Z\";\n" +
    " :time_coverage_start = \"2007-02-06T00:00:00Z\";\n" +
    " :title = \"Chlorophyll-a, Aqua MODIS, NPP, Global, Science Quality (8 Day Composite)\";\n" +
    " :Westernmost_Easting = 224.97918749730292; // double\n" +
    " data:\n" +
    "time =\n" +
    "  {1.17072E9}\n" +
    "altitude =\n" +
    "  {0.0}\n" +
    "latitude =\n" +
    "  {29.020831183144253, 29.437497818939576, 29.8541644547349, 30.27083109053021, 30.68749772632553, 31.104164362120855, 31.52083099791618, 31.9374976337115, 32.354164269506825, 32.77083090530215, 33.18749754109747, 33.604164176892795, 34.02083081268812, 34.43749744848344, 34.854164084278764, 35.27083072007409, 35.68749735586941, 36.104163991664734, 36.52083062746006, 36.93749726325538, 37.354163899050704, 37.77083053484603, 38.18749717064135, 38.60416380643669, 39.020830442232, 39.437497078027334, 39.85416371382264, 40.27083034961795, 40.68749698541329, 41.1041636212086, 41.52083025700394, 41.937496892799246, 42.35416352859458, 42.77083016438989, 43.18749680018523, 43.60416343598054, 44.020830071775876, 44.437496707571185, 44.85416334336652, 45.27082997916183, 45.68749661495717, 46.10416325075248, 46.520829886547816, 46.937496522343125, 47.35416315813846, 47.77082979393377, 48.18749642972911, 48.60416306552442, 49.020829701319755, 49.437496337115064, 49.8541629729104}\n" +
    "longitude =\n" +
    "  {224.97918749730292, 225.39585420255816, 225.8125209078134, 226.22918761306863, 226.64585431832387, 227.0625210235791, 227.47918772883435, 227.89585443408959, 228.31252113934482, 228.72918784460006, 229.1458545498553, 229.56252125511054, 229.97918796036578, 230.39585466562102, 230.81252137087625, 231.2291880761315, 231.64585478138673, 232.06252148664197, 232.4791881918972, 232.89585489715245, 233.31252160240768, 233.72918830766292, 234.14585501291813, 234.56252171817337, 234.9791884234286, 235.39585512868385, 235.8125218339391, 236.22918853919433, 236.64585524444956, 237.0625219497048, 237.47918865496004, 237.89585536021528, 238.31252206547052, 238.72918877072576, 239.145855475981, 239.56252218123623, 239.97918888649147, 240.3958555917467, 240.81252229700195, 241.2291890022572, 241.64585570751242, 242.06252241276766, 242.4791891180229, 242.89585582327814, 243.31252252853338, 243.72918923378862, 244.14585593904386, 244.5625226442991, 244.97918934955433, 245.39585605480957, 245.8125227600648, 246.22918946532005, 246.64585617057529}\n" +
    "chlorophyll =\n" +
    "  {\n" +
    "    {\n" +
    "      {\n" +
    "        {-9999999.0, -9999999.0, 0.096, 0.119, -9999999.0, 0.102, 0.09, 0.09, 0.084, 0.103, -9999999.0, 0.1, 0.071, 0.067, -9999999.0, 0.071, 0.074, -9999999.0, -9999999.0, -9999999.0, 0.074, -9999999.0, 0.09, -9999999.0, -9999999.0, -9999999.0, 0.094, -9999999.0, 0.078, 0.077, 0.093, -9999999.0, -9999999.0, 0.127, 0.189, 0.152, 0.188, 0.151, 0.136, 0.163, -9999999.0, 0.253, 0.245, 0.267, 0.275, 0.301, 0.396, 0.384, 0.545, -9999999.0, -9999999.0, -9999999.0, 1.205},\n";
            tPo = results.indexOf(" :infoUrl");
            Test.ensureEqual(results.substring(tPo, tPo + expected.length()), expected, "RESULTS=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\n*** This EDDGridFromErddap test requires erdMHchla8day on coastwatch's erddap" +
                (testLocalErddapToo? "\n    AND rMHchla8day on localhost's erddap." : "") +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void test() throws Throwable {

        String2.log("\n****************** EDDGridFromErddap.test() *****************\n");

        /* standard tests */
        testBasic(false);
        testBasic(true);
        testGenerateDatasetsXml();

        //not regularly done

        String2.log("\n*** EDDGridFromErddap.test finished.");

    }



}
