/* 
 * EDDGridFromEtopo Copyright 2008, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.ShortArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.FileNameUtility;
import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.*;

import java.io.RandomAccessFile;

/** 
 * This class represents a grid dataset with Etopo2v2g bathymetry data.
 * 
 * @author Bob Simons (bob.simons@noaa.gov) 2008-02-20
 */
public class EDDGridFromEtopo extends EDDGrid { 


    /** Properties of the datafile */
    protected static String fileName = EDStatic.contextDirectory + "WEB-INF/ref/ETOPO2v2g_MSB.raw";
    protected final static double fileMinLon = -180, fileMaxLon = 180; 
    protected final static double fileMinLat = -90,  fileMaxLat = 90;
    protected final static int fileNLons = 10801, fileNLats = 5401;
    protected final static int bytesPerValue = 2;
    protected static double fileLonSpacing = (fileMaxLon - fileMinLon) / (fileNLons - 1);
    protected static double fileLatSpacing = (fileMaxLat - fileMinLat) / (fileNLats - 1);
    protected static double fileLons[] = DataHelper.getRegularArray(fileNLons, fileMinLon, fileLonSpacing); 
    protected static double fileLats[] = DataHelper.getRegularArray(fileNLats, fileMinLat, fileLatSpacing); 

    /** Set by the constructor */
    protected boolean is180;

    /**
     * This constructs an EDDGridFromBinaryFile based on the information in an .xml file.
     * 
     * @param xmlReader with the &lt;erddapDatasets&gt;&lt;dataset type="EDDGridFromBinaryFile"&gt; 
     *    having just been read.  
     * @return an EDDGridFromBinaryFile.
     *    When this returns, xmlReader will have just read &lt;erddapDatasets&gt;&lt;/dataset&gt; .
     * @throws Throwable if trouble
     */
    public static EDDGridFromEtopo fromXml(SimpleXMLReader xmlReader) throws Throwable {
        //data to be obtained (or not)
        if (verbose) String2.log("\n*** constructing EDDGridFromBinaryFile(xmlReader)...");
        String tDatasetID = xmlReader.attributeValue("datasetID"); 

        //process the tags
        int startOfTagsN = xmlReader.stackSize();
        while (true) {
            xmlReader.nextTag();
            if (xmlReader.stackSize() == startOfTagsN) 
                break; //the </dataset> tag

            //try to make the tag names as consistent, descriptive and readable as possible

            //no support for active, since always active
            //no support for accessibleTo, since accessible to all
            //no support for onChange since dataset never changes

            xmlReader.unexpectedTagException();
        }

        return new EDDGridFromEtopo(tDatasetID);
    }

    /**
     * The constructor.
     *
     * @throws Throwable if trouble
     */
    public EDDGridFromEtopo(String tDatasetID) throws Throwable {

        if (verbose) String2.log(
            "\n*** constructing EDDGridFromEtopo " + tDatasetID); 
        long constructionStartMillis = System.currentTimeMillis();
        String errorInMethod = "Error in EDDGridFromEtopo(" + 
            tDatasetID + ") constructor:\n";

        className = "EDDGridFromEtopo"; 
        datasetID = tDatasetID;
        is180 = datasetID.equals("etopo180");
        Test.ensureTrue(is180 || datasetID.equals("etopo360"),
            errorInMethod + "datasetID must be \"etopo180\" or \"etopo360\".");

        sourceGlobalAttributes = new Attributes();
        sourceGlobalAttributes.add("acknowledgement", "NOAA NGDC");
        sourceGlobalAttributes.add("cdm_data_type", "Grid");
        sourceGlobalAttributes.add("contributor_name", 
            "Smith & Sandwell, GLOBE, IBCAO, NGDC Coastal Relief Model (CRM), " +
            "NGDC Great Lakes Bathymetric Data, and Caspian Sea Bathymetry");
        sourceGlobalAttributes.add("contributor_role", "source data");
        sourceGlobalAttributes.add("Conventions", "COARDS, CF-1.0, Unidata Dataset Discovery v1.0");
        sourceGlobalAttributes.add("creator_email", "david.c.schoolcraft@noaa.gov");
        sourceGlobalAttributes.add("creator_name", "NOAA NGDC");
        sourceGlobalAttributes.add("creator_url", "http://www.ngdc.noaa.gov/mgg/fliers/01mgg04.html");
        sourceGlobalAttributes.add("data_source", "NOAA NGDC ETOPO2v2g_MSB.raw");
        sourceGlobalAttributes.add("history", "Sampled from http://www.ngdc.noaa.gov/mgg/global/relief/ETOPO2/ETOPO2v2-2006/ETOPO2v2g/raw_binary/ETOPO2v2g_i2_MSB.zip");
        sourceGlobalAttributes.add("id", "SampledFromETOPO2v2g_MSB.raw");
        sourceGlobalAttributes.add("infoUrl", "http://www.ngdc.noaa.gov/mgg/fliers/01mgg04.html");
        sourceGlobalAttributes.add("institution", "NOAA NGDC");
        sourceGlobalAttributes.add("keywords", "EARTH SCIENCE > Oceans > Bathymetry/Seafloor Topography > Bathymetry");
        sourceGlobalAttributes.add("keywords_vocabulary", "GCMD Science Keywords");
        sourceGlobalAttributes.add("license", EDStatic.standardLicense);
        sourceGlobalAttributes.add("naming_authority", "gov.noaa.pfel.coastwatch");
        sourceGlobalAttributes.add("project", "NOAA NGDC ETOPO");
        sourceGlobalAttributes.add("projection", "geographic");
        sourceGlobalAttributes.add("projection_type", "mapped");
        sourceGlobalAttributes.add("references", 
            "U.S. Department of Commerce, National Oceanic and Atmospheric Administration, " +
            "National Geophysical Data Center, 2006. 2-minute Gridded Global Relief Data (ETOPO2v2)");
        sourceGlobalAttributes.add("sourceUrl", "http://www.ngdc.noaa.gov/mgg/global/relief/ETOPO2/ETOPO2v2-2006/ETOPO2v2g/raw_binary/ETOPO2v2g_i2_MSB.zip");
        sourceGlobalAttributes.add("standard_name_vocabulary", FileNameUtility.getStandardNameVocabulary());
        sourceGlobalAttributes.add("summary", 
            "Two-minute gridded global relief for both ocean and land areas are available " +
            "in the ETOPO2v2 (2006) database. The horizontal datum is WGS-84, the vertical " +
            "datum is Mean Sea Level. ETOPO2v2 is a compilation of several datasets: Smith & " +
            "Sandwell, GLOBE, IBCAO, NGDC Coastal Relief Model (CRM), NGDC Great Lakes " +
            "Bathymetric Data, and Caspian Sea Bathymetry. Keywords: Bathymetry, Digital Elevation. " +
            "This is the grid-centered version, with cell boundaries defined by lines of odd minutes " +
            "of latitude and longitude, meaning that cells were centered on the integer multiples of " +
            "2 minutes [even minutes] of latitude and longitude.");
        sourceGlobalAttributes.add("title", 
            "Topography, ETOPO2v2g, 0.033333 degrees, Global (longitude " + 
            (is180? "-180 to 180)" : "0 to 360)"));        

        addGlobalAttributes = new Attributes();
        combinedGlobalAttributes = new Attributes(addGlobalAttributes, sourceGlobalAttributes); //order is important
        combinedGlobalAttributes.removeValue("null");

        //make the axisVariables
        axisVariables = new EDVGridAxis[2];
        latIndex = 0;
        axisVariables[latIndex] = new EDVLatGridAxis(EDV.LAT_NAME, 
            new Attributes(), new Attributes(), 
            new DoubleArray(DataHelper.getRegularArray(5401, -90, 1/30.0)));
        lonIndex = 1;
        axisVariables[lonIndex] = new EDVLonGridAxis(EDV.LON_NAME, 
            new Attributes(), new Attributes(), 
            new DoubleArray(DataHelper.getRegularArray(10801, is180? -180 : 0, 1/30.0)));

        //make the dataVariable
        //???I'm not sure if some attributes should only be used for axisVariables.
        Attributes dAtt = new Attributes();
        //dAtt.set("_CoordinateAxisType", "Height"); //don't treat as axisVariable
        //dAtt.set("_CoordinateZisPositive", "up");
        dAtt.set("_FillValue", -9999999);
        dAtt.set("actual_range", new IntArray(new int[]{-5527, 4064}));
        //dAtt.set("axis", "Z");
        dAtt.set("coordsys", "geographic");
        dAtt.set("ioos_category", "Location");
        dAtt.set("long_name", "Altitude");
        dAtt.set("missing_value", -9999999);
        dAtt.set("positive", "up");
        dAtt.set("standard_name", "altitude");
        dAtt.set("colorBarMinimum", -8000.0); //.0 makes it a double
        dAtt.set("colorBarMaximum", 8000.0);
        dAtt.set("colorBarPalette", "Topography");
        dAtt.set("units", "m");
        dataVariables = new EDV[1];
        dataVariables[0] = new EDV("altitude", "", dAtt, new Attributes(), "short");  
        dataVariables[0].setActualRangeFromDestinationMinMax();

        //ensure the setup is valid
        ensureValid();

        //finally
        if (verbose) String2.log(
            (reallyVerbose? "\n" + toString() : "") +
            "\n*** EDDGridFromEtopo " + datasetID() + " constructor finished. TIME=" + 
            (System.currentTimeMillis() - constructionStartMillis) + "\n"); 
    }

    /**
     * This makes a sibling dataset, based on the new sourceUrl.
     *
     * @throws Throwable always (since this class doesn't support sibling())
     */
    public EDDGrid sibling(String tSourceUrl, int ensureAxisValuesAreEqual, 
        boolean shareInfo) throws Throwable {
        throw new SimpleException("Error: " + 
            "EDDGridFromEtopo doesn't support method=\"sibling\".");

    }

    /** 
     * This gets data (not yet standardized) from the data 
     * source for this EDDGrid.     
     * 
     * @param tDataVariables
     * @param tConstraints
     * @return a PrimitiveArray[] where the first axisVariables.length elements
     *   are the axisValues and the next tDataVariables.length elements
     *   are the dataValues.
     *   Both the axisValues and dataValues are straight from the source,
     *   not modified.
     * @throws Throwable if trouble
     */
    public PrimitiveArray[] getSourceData(EDV tDataVariables[], IntArray tConstraints) 
        throws Throwable {

        //Currently ETOPO2v2g_MSB.raw (version 2).
        //ETOPO2v2g_MSB.raw (grid centered, MSB 16 bit signed integers)
        //5401 rows by 10801 columns.
        //Data is stored row by row, starting at 90, going down to -90,
        //with lon -180 to 180 on each row (the first and last points on each row are duplicates).
        //The data is grid centered, so the data associated with a given lon,lat represents
        //a cell which extends 1 minute N, S, E, and W of the lon, lat.
        //I verified this interpretation with Lynn.

        DoubleArray lats = (DoubleArray)axisVariables[0].sourceValues().subset(
            tConstraints.get(0), tConstraints.get(1), tConstraints.get(2));
        DoubleArray lons = (DoubleArray)axisVariables[1].sourceValues().subset(
            tConstraints.get(3), tConstraints.get(4), tConstraints.get(5));
        int nLats = lats.size();
        int nLons = lons.size();
        short data[] = new short[nLats * nLons];

        PrimitiveArray results[] = new PrimitiveArray[3];
        results[0] = lats;
        results[1] = lons;
        results[2] = new ShortArray(data);

        //find the offsets for the start of the rows for the resulting lon values
        int lonOffsets[] = new int[nLons];
        for (int i = 0; i < nLons; i++) {
            double tLon = lons.get(i);
            while (tLon < fileMinLon) tLon += 360;
            while (tLon > fileMaxLon) tLon -= 360;
            //findClosest since may differ by roundoff error
            int closestLon = Math2.binaryFindClosest(fileLons, tLon); //never any ties, so no need to findFirst or findLast
            //never any ties, so no need to findFirst or findLast
            lonOffsets[i] = bytesPerValue * closestLon;  
            //String2.log("tLon=" + tLon + " closestLon=" + closestLon + " offset=" + offsetLon[i]);
        }

        //find the offsets for the start of the columns closest to the desiredLon values
        int latOffsets[] = new int[nLats];
        for (int i = 0; i < nLats; i++) {
            double tLat = lats.get(i);
            while (tLat < fileMinLat) tLat += 90;
            while (tLat > fileMaxLat) tLat -= 90;
            int closestLat = Math2.binaryFindClosest(fileLats, tLat); //never any ties, so no need to findFirst or findLast
            //adjust lat, since fileLat is ascending, but file stores data top row at start of file
            closestLat = fileNLats - 1 - closestLat;
            latOffsets[i] = bytesPerValue * closestLat * fileNLons; 
            //String2.log("tLat=" + tLat + " closestLat=" + closestLat + " offset=" + offsetLat[i]);
        }

        //open the file  (reading is thread safe)
        RandomAccessFile raf = new RandomAccessFile(fileName, "r");

        //fill data array
        //lat is outer loop because file is lat major
        //and loop is backwards since stored top to bottom
        //(goal is to read basically from start to end of file)
        for (int lati = nLats - 1; lati >= 0; lati--) { 
            int po = lati * nLons;
            for (int loni = 0; loni < nLons; loni++) { 
               raf.seek(latOffsets[lati] + lonOffsets[loni]);
               data[po++] = raf.readShort();
            }
        }

        //close the file 
        raf.close();

        return results;
    }


    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void test(boolean doGraphicsTests) throws Throwable {

        String2.log("\n****************** EDDGridFromEtopo.test() *****************\n");
        verbose = true;
        reallyVerbose = true;
        GridDataAccessor.verbose = true;
        GridDataAccessor.reallyVerbose = true;
        String name, tName, axisDapQuery, userDapQuery, results, expected, error;
        int tPo;
        EDDGridFromEtopo data180 = new EDDGridFromEtopo("etopo180");
        EDDGridFromEtopo data360 = new EDDGridFromEtopo("etopo360");

        //*** test getting .nc for entire dataset
        String2.log("\n****************** EDDGridFromEtopo test entire dataset\n");
        tName = data180.makeNewFileForDapQuery(null, null, "altitude[(-90):500:(90)][(-180):500:(180)]", 
            EDStatic.fullTestCacheDirectory, data180.className() + "_Entire", ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory  + tName, true);
        expected = 
"netcdf EDDGridFromEtopo_Entire.nc {\n" +
" dimensions:\n" +
"   latitude = 11;\n" +   // (has coord.var)\n" +  //changed when switched to netcdf-java 4.0, 2009-02-23
"   longitude = 22;\n" +   // (has coord.var)\n" +
" variables:\n" +
"   double latitude(latitude=11);\n" +
"     :_CoordinateAxisType = \"Lat\";\n" +
"     :actual_range = -90.0, 76.66666666666666; // double\n" +
"     :axis = \"Y\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Latitude\";\n" +
"     :standard_name = \"latitude\";\n" +
"     :units = \"degrees_north\";\n" +
"   double longitude(longitude=22);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = -180.0, 170.0; // double\n" +
"     :axis = \"X\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"   short altitude(latitude=11, longitude=22);\n" +
//"     :_CoordinateAxisType = \"Height\";\n" +
//"     :_CoordinateZisPositive = \"up\";\n" +
"     :_FillValue = 32767s; // short\n" +
//"     :axis = \"Z\";\n" +
"     :colorBarMaximum = 8000.0; // double\n" +
"     :colorBarMinimum = -8000.0; // double\n" +
"     :colorBarPalette = \"Topography\";\n" +
"     :coordsys = \"geographic\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Altitude\";\n" +
"     :missing_value = 32767s; // short\n" +
"     :positive = \"up\";\n" +
"     :standard_name = \"altitude\";\n" +
"     :units = \"m\";\n" +
"\n" +
" :acknowledgement = \"NOAA NGDC\";\n" +
" :cdm_data_type = \"Grid\";\n" +
" :contributor_name = \"Smith & Sandwell, GLOBE, IBCAO, NGDC Coastal Relief Model (CRM), NGDC Great Lakes Bathymetric Data, and Caspian Sea Bathymetry\";\n" +
" :contributor_role = \"source data\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
" :creator_email = \"david.c.schoolcraft@noaa.gov\";\n" +
" :creator_name = \"NOAA NGDC\";\n" +
" :creator_url = \"http://www.ngdc.noaa.gov/mgg/fliers/01mgg04.html\";\n" +
" :data_source = \"NOAA NGDC ETOPO2v2g_MSB.raw\";\n" +
" :Easternmost_Easting = 170.0; // double\n" +
" :geospatial_lat_max = 76.66666666666666; // double\n" +
" :geospatial_lat_min = -90.0; // double\n" +
" :geospatial_lon_max = 170.0; // double\n" +
" :geospatial_lon_min = -180.0; // double\n" +
" :history = \"Sampled from http://www.ngdc.noaa.gov/mgg/global/relief/ETOPO2/ETOPO2v2-2006/ETOPO2v2g/raw_binary/ETOPO2v2g_i2_MSB.zip\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        expected = 
" :id = \"SampledFromETOPO2v2g_MSB.raw\";\n" +
" :infoUrl = \"http://www.ngdc.noaa.gov/mgg/fliers/01mgg04.html\";\n" +
" :institution = \"NOAA NGDC\";\n" +
" :keywords = \"EARTH SCIENCE > Oceans > Bathymetry/Seafloor Topography > Bathymetry\";\n" +
" :keywords_vocabulary = \"GCMD Science Keywords\";\n" +
" :license = \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
" :naming_authority = \"gov.noaa.pfel.coastwatch\";\n" +
" :Northernmost_Northing = 76.66666666666666; // double\n" +
" :project = \"NOAA NGDC ETOPO\";\n" +
" :projection = \"geographic\";\n" +
" :projection_type = \"mapped\";\n" +
" :references = \"U.S. Department of Commerce, National Oceanic and Atmospheric Administration, National Geophysical Data Center, 2006. 2-minute Gridded Global Relief Data (ETOPO2v2)\";\n" +
" :sourceUrl = \"http://www.ngdc.noaa.gov/mgg/global/relief/ETOPO2/ETOPO2v2-2006/ETOPO2v2g/raw_binary/ETOPO2v2g_i2_MSB.zip\";\n" +
" :Southernmost_Northing = -90.0; // double\n" +
" :standard_name_vocabulary = \"CF-11\";\n" +
" :summary = \"Two-minute gridded global relief for both ocean and land areas are available in the ETOPO2v2 (2006) database. The horizontal datum is WGS-84, the vertical datum is Mean Sea Level. ETOPO2v2 is a compilation of several datasets: Smith & Sandwell, GLOBE, IBCAO, NGDC Coastal Relief Model (CRM), NGDC Great Lakes Bathymetric Data, and Caspian Sea Bathymetry. Keywords: Bathymetry, Digital Elevation. This is the grid-centered version, with cell boundaries defined by lines of odd minutes of latitude and longitude, meaning that cells were centered on the integer multiples of 2 minutes [even minutes] of latitude and longitude.\";\n" +
" :title = \"Topography, ETOPO2v2g, 0.033333 degrees, Global (longitude -180 to 180)\";\n" +
" :Westernmost_Easting = -180.0; // double\n" +
" data:\n" +
"latitude =\n" +
"  {-90.0, -73.33333333333333, -56.666666666666664, -40.0, -23.33333333333333, -6.666666666666671, 10.0, 26.66666666666667, 43.33333333333334, 60.0, 76.66666666666666}\n" +
"longitude =\n" +
"  {-180.0, -163.33333333333334, -146.66666666666666, -130.0, -113.33333333333333, -96.66666666666667, -80.0, -63.33333333333333, -46.66666666666666, -30.0, -13.333333333333343, 3.333333333333343, 20.0, 36.66666666666666, 53.33333333333334, 70.0, 86.66666666666669, 103.33333333333331, 120.0, 136.66666666666669, 153.33333333333331, 170.0}\n" +
"altitude =\n" +
"  {\n" +
"    {2774, 2776, 2774, 2774, 2776, 2774, 2776, 2776, 2776, 2776, 2776, 2778, 2778, 2776, 2778, 2778, 2776, 2776, 2776, 2776, 2776, 2776},\n" +
"    {-415, -3861, -4063, -1151, -521, 208, 148, 1244, -379, -3409, 394, 2782, 3046, 2956, 3042, 1508, 3304, 3218, 3100, 2800, 2332, -79},\n" +
"    {-5013, -4137, -2731, -3639, -3801, -4543, -4573, -4051, -3927, -3351, -4191, -3793, -4671, -5303, -5127, -2435, -4445, -4207, -4641, -3817, -3523, -5229},\n" +
"    {-3147, -5089, -5373, -4855, -3099, -3767, -4229, 38, -5195, -4445, -2881, -4933, -4923, -4263, -3805, -4181, -3871, -4237, -4931, -5151, -4609, -835},\n" +
"    {-2705, -5191, -5029, -3365, -3455, -3935, -4607, 224, 916, -5303, -2833, -4883, 1264, -2355, -4465, -3537, -4067, -5655, 628, 246, -413, -3371},\n" +
"    {-5555, -5657, -4903, -4611, -4091, -3795, 68, 74, 394, -5283, -3143, -5365, 720, 774, -3679, -3691, -5145, -2063, -1507, -35, -4473, -3959},\n" +
"    {-6091, -4363, -5167, -4791, -3667, -3927, -2169, 222, -4817, -5351, 142, 360, 386, 1836, -4359, -4541, -3493, -5, -1421, -4781, -5623, -3639},\n" +
"    {-5269, -5025, -5049, -4599, -29, -79, -95, -5639, -3735, -5185, 118, 304, 222, 306, -77, 78, 234, 2612, 168, -5367, -5935, -5681},\n" +
"    {-5887, -5665, -4659, -3283, 1584, 418, 250, -183, -4101, -2715, -5179, 24, 1118, -2165, 190, 880, 3332, 1608, 396, -3677, -5249, -2449},\n" +
"    {-2643, 4, -99, 1388, 278, 170, -115, -169, -383, -1411, -847, -247, -11, 152, 184, 64, 182, 380, 334, 744, 360, -39},\n" +
"    {-1137, -819, -3725, -2579, -231, -17, 1032, 1342, 2694, 2158, -211, -3119, -135, -201, -235, -215, -27, 226, -59, -25, -43, -269}\n" +
"  }\n" +
"}";
        tPo = results.indexOf(" :id =");
        Test.ensureEqual(results.substring(tPo, Math.min(results.length(), tPo + expected.length())), expected, 
            "RESULTS=\n" + results);

        tName = data360.makeNewFileForDapQuery(null, null, "altitude[(-90):1000:(90)][(0):1000:(360)]", 
            EDStatic.fullTestCacheDirectory, data360.className() + "_Entire", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"latitude, longitude, altitude\n" +
"degrees_north, degrees_east, m\n" +
"-90.0, 0.0, 2778\n" +
"-90.0, 33.333333333333336, 2778\n" +
"-90.0, 66.66666666666667, 2776\n" +
"-90.0, 100.0, 2776\n" +
"-90.0, 133.33333333333334, 2776\n" +
"-90.0, 166.66666666666666, 2776\n" +
"-90.0, 200.0, 2774\n" +
"-90.0, 233.33333333333334, 2774\n" +
"-90.0, 266.6666666666667, 2776\n" +
"-90.0, 300.0, 2776\n" +
"-90.0, 333.3333333333333, 2776\n" +
"-56.666666666666664, 0.0, -4009\n" +
"-56.666666666666664, 33.333333333333336, -5347\n" +
"-56.666666666666664, 66.66666666666667, -2033\n" +
"-56.666666666666664, 100.0, -3959\n" +
"-56.666666666666664, 133.33333333333334, -4433\n" +
"-56.666666666666664, 166.66666666666666, -5197\n" +
"-56.666666666666664, 200.0, -4475\n" +
"-56.666666666666664, 233.33333333333334, -3919\n" +
"-56.666666666666664, 266.6666666666667, -5367\n" +
"-56.666666666666664, 300.0, -3925\n" +
"-56.666666666666664, 333.3333333333333, -1845\n" +
"-23.33333333333333, 0.0, -5201\n" +
"-23.33333333333333, 33.333333333333336, 54\n" +
"-23.33333333333333, 66.66666666666667, -4047\n" +
"-23.33333333333333, 100.0, -5889\n" +
"-23.33333333333333, 133.33333333333334, 636\n" +
"-23.33333333333333, 166.66666666666666, -1823\n" +
"-23.33333333333333, 200.0, -4575\n" +
"-23.33333333333333, 233.33333333333334, -3541\n" +
"-23.33333333333333, 266.6666666666667, -3831\n" +
"-23.33333333333333, 300.0, 122\n" +
"-23.33333333333333, 333.3333333333333, -5683\n" +
"10.0, 0.0, 154\n" +
"10.0, 33.333333333333336, 420\n" +
"10.0, 66.66666666666667, -4395\n" +
"10.0, 100.0, -23\n" +
"10.0, 133.33333333333334, -5435\n" +
"10.0, 166.66666666666666, -4487\n" +
"10.0, 200.0, -5269\n" +
"10.0, 233.33333333333334, -4671\n" +
"10.0, 266.6666666666667, -3649\n" +
"10.0, 300.0, -105\n" +
"10.0, 333.3333333333333, -5179\n" +
"43.33333333333334, 0.0, 274\n" +
"43.33333333333334, 33.333333333333336, -2157\n" +
"43.33333333333334, 66.66666666666667, 170\n" +
"43.33333333333334, 100.0, 1498\n" +
"43.33333333333334, 133.33333333333334, 228\n" +
"43.33333333333334, 166.66666666666666, -4887\n" +
"43.33333333333334, 200.0, -5429\n" +
"43.33333333333334, 233.33333333333334, -2953\n" +
"43.33333333333334, 266.6666666666667, 386\n" +
"43.33333333333334, 300.0, -1533\n" +
"43.33333333333334, 333.3333333333333, -2965\n" +
"76.66666666666666, 0.0, -3253\n" +
"76.66666666666666, 33.333333333333336, -153\n" +
"76.66666666666666, 66.66666666666667, 254\n" +
"76.66666666666666, 100.0, -29\n" +
"76.66666666666666, 133.33333333333334, -35\n" +
"76.66666666666666, 166.66666666666666, -197\n" +
"76.66666666666666, 200.0, -2059\n" +
"76.66666666666666, 233.33333333333334, -927\n" +
"76.66666666666666, 266.6666666666667, 148\n" +
"76.66666666666666, 300.0, 1478\n" +
"76.66666666666666, 333.3333333333333, 1886\n";
        Test.ensureEqual(results, expected, "RESULTS=\n" + results);


        if (doGraphicsTests) {
            tName = data180.makeNewFileForDapQuery(null, null, 
                "altitude[(-90):(90)][(-180):(180)]" +
                "&.vars=longitude|latitude|altitude&.colorBar=Ocean|C|Linear|-8000|0", 
                EDStatic.fullTestCacheDirectory, data180.className() + "_Map180", ".png"); 
            SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
    
            tName = data360.makeNewFileForDapQuery(null, null, "altitude[(-90):(90)][(0):(360)]", 
                EDStatic.fullTestCacheDirectory, data360.className() + "_Map360", ".png"); 
            SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);

            tName = data360.makeNewFileForDapQuery(null, null, 
                "altitude[][]" +
                "&.vars=longitude|latitude|altitude&.colorBar=Topography|C|Linear|-8000|8000&.land=under", 
                EDStatic.fullTestCacheDirectory, data360.className() + "_TopoUnder", ".png"); 
            SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
    
            tName = data360.makeNewFileForDapQuery(null, null, 
                "altitude[][]" +
                "&.vars=longitude|latitude|altitude&.colorBar=Topography|C|Linear|-8000|8000&.land=over", 
                EDStatic.fullTestCacheDirectory, data360.className() + "_TopoOver", ".png"); 
            SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
    
        }


        String2.log("\n*** EDDGridFromEtopo.test finished.");

    }


}
