/* 
 * EDDTable Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.DoubleArray;
import com.cohort.array.NDimensionalIndex;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.Test;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.griddata.DataHelper;
import gov.noaa.pfel.coastwatch.griddata.Grid;
import gov.noaa.pfel.coastwatch.griddata.Matlab;
import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.SSR;
import gov.noaa.pfel.coastwatch.sgt.CompoundColorMap;
import gov.noaa.pfel.coastwatch.sgt.GraphDataLayer;
import gov.noaa.pfel.coastwatch.sgt.SgtGraph;
import gov.noaa.pfel.coastwatch.sgt.SgtMap;
import gov.noaa.pfel.coastwatch.sgt.SgtUtil;
import gov.noaa.pfel.erddap.util.*;
import gov.noaa.pfel.erddap.variable.*;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import javax.servlet.http.HttpServletRequest;

/**
 * Get netcdf-X.X.XX.jar from http://www.unidata.ucar.edu/software/netcdf-java/index.htm
 * and copy it to <context>/WEB-INF/lib renamed as netcdf-latest.jar.
 * Get slf4j-jdk14.jar from 
 * ftp://ftp.unidata.ucar.edu/pub/netcdf-java/slf4j-jdk14.jar
 * and copy it to <context>/WEB-INF/lib.
 * Put both of these .jar files in the classpath for the compiler and for Java.
 */
import ucar.nc2.*;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dods.*;
import ucar.nc2.util.*;
import ucar.ma2.*;

/** 
 * This class represents a dataset where the results can be presented as a table.
 * 
 * @author Bob Simons (bob.simons@noaa.gov) 2007-06-08
 */
public abstract class EDDTable extends EDD { 

    public final static String dapProtocol = "tabledap"; 
    public final static String SEQUENCE_NAME = "s"; //same for all datasets

    /** The regular expression operator. The DAP spec says =~, so that is the ERDDAP standard.
     * But some implementations use ~=, so see sourceRegexOp. */
    public final static String REGEX_OP = "=~";

    /** 
     * This is a list of all operator symbols (for my convenience in parseUserDapQuery: 
     * 2 letter ops are first). 
     * Note that all variables (numeric or String, axisVariable or dataVariable)
     * can be constrained via a ERDDAP constraint using any of these
     * operators.
     * If the source can't support the operator (e.g., 
     *    no source supports numeric =~ testing,
     *    and no source supports string &lt; &lt;= &gt; &gt;= testing),
     * ERDDAP will just get all the relevant data and do the test itself. 
     */
    public final static String OPERATORS[] = { 
        //EDDTableFromFiles.isOK relies on this order
        "!=", REGEX_OP, "<=", ">=",   
        "=", "<", ">"}; 
    private final static int opDefaults[] = {
        String2.indexOf(OPERATORS, ">="),
        String2.indexOf(OPERATORS, "<=")};

    /** A list of the greater-than less-than operators, which some sources can't handle for strings. */
    public final static String GTLT_OPERATORS[] = {
        "<=", ">=", "<", ">"}; 
    /** The index of REGEX_OP in OPERATORS. */
    public static int REGEX_OP_INDEX = String2.indexOf(OPERATORS, REGEX_OP);

    /** This is used in many file types as the row identifier. */
    public final static String ROW_NAME = "row";

    /** These are needed for EDD-required methods of the same name. */
    public final static String[] dataFileTypeNames = {  
        ".asc", ".csv", ".das", ".dds", ".dods", ".geoJson", ".graph", ".help", ".html", ".htmlTable", 
        ".json", ".mat", ".nc", ".ncHeader", ".tsv", ".xhtml"};
    public final static String[] dataFileTypeExtensions = {
        ".asc", ".csv", ".das", ".dds", ".dods", ".json",    ".html", ".html", ".html", ".html",
        ".json", ".mat", ".nc", ".txt", ".tsv", ".xhtml"};
    //These all used to have " (It may take a minute. Please be patient.)" at the end.
    public static String[] dataFileTypeDescriptions = {
        "View the data as OPeNDAP-style comma-separated ASCII text.",
        "Download the data as comma-separated ASCII text table (missing value = 'NaN'; times are ISO 8601 strings).",
        "View the data's metadata via an OPeNDAP Dataset Attribute Structure (DAS).",
        "View the data's structure via an OPeNDAP Dataset Descriptor Structure (DDS).",
        "OPeNDAP clients use this to download the data in the DODS binary format.",
        "Download longitude,latitude,otherColumns data as a GeoJSON file.",
        "View a Make A Graph web page.",
        "View a web page with a description of tabledap.",
        "View an OPeNDAP-style HTML Data Access Form.",
        "View an HTML file with the data in a table (times are ISO 8601 strings).",
        "Download the data as a JSON table (missing value = 'null'; times are ISO 8601 strings).",
        "Download a MATLAB binary file.",
        "Download a NetCDF binary file with COARDS/CF/THREDDS metadata.",
        "View the header (the metadata) for the NetCDF file.",
        "Download the data as tab-separated ASCII text table (missing value = 'NaN'; times are ISO 8601 strings).",
        "View an XHTML file with the data in a table (times are ISO 8601 strings)."};
    public static String[] dataFileTypeInfo = {
        "http://www.opendap.org/user/guide-html/guide_20.html#id4", //dap ascii
        "http://www.creativyst.com/Doc/Articles/CSV/CSV01.htm", //csv
        "http://www.opendap.org/user/guide-html/guide_66.html", //das
        "http://www.opendap.org/user/guide-html/guide_65.html", //dds
        "http://www.opendap.org", //dods
        "http://wiki.geojson.org/Main_Page", //geoJSON
        "http://coastwatch.pfeg.noaa.gov/erddap/griddap/index.html#GraphicsCommands", //GraphicsCommands
        "http://www.opendap.org/user/guide-html/guide_20.html#id8", //help
        "http://www.opendap.org/user/guide-html/guide_21.html", //html
        "http://www.w3schools.com/html/html_tables.asp", //htmlTable
        "http://www.json.org/", //json
        "http://www.mathworks.com/", //mat
        "http://www.unidata.ucar.edu/software/netcdf/", //nc
        "http://www.unidata.ucar.edu/software/netcdf/guide_12.html#SEC95", //ncHeader
        "http://www.cs.tut.fi/~jkorpela/TSV.html",  //tsv
        "http://www.w3schools.com/html/html_tables.asp"}; //xhtml
        //"http://www.tizag.com/htmlT/tables.php" //xhtml

    public final static String[] imageFileTypeNames = {
        ".kml", ".smallPdf", ".pdf", ".largePdf", ".smallPng", ".png", ".largePng"};  
    public final static String[] imageFileTypeExtensions = {
        ".kml", ".pdf", ".pdf", ".pdf", ".png", ".png", ".png"};
    public static String[] imageFileTypeDescriptions = {
        "View a .kml file, suitable for Google Earth, with a reference to the data you selected.",
        "View a small .pdf image file with a graph/map of the data you selected.",
        "View a standard, medium-sized .pdf image file with a graph/map of the data you selected.",
        "View a large .pdf image file with a graph/map of the data you selected.",
        "View a small .png image file with a graph/map of the data you selected.",
        "View a standard, medium-sized .png image file with a graph/map of the data you selected.",
        "View a large .png image file with a graph/map of the data you selected."};
    public static String[] imageFileTypeInfo = {
        "http://earth.google.com/", //kml
        "http://www.adobe.com/products/acrobat/readstep2.html", //pdf
        "http://www.adobe.com/products/acrobat/readstep2.html", //pdf
        "http://www.adobe.com/products/acrobat/readstep2.html", //pdf
        "http://www.libpng.org/pub/png/", //png
        "http://www.libpng.org/pub/png/", //png
        "http://www.libpng.org/pub/png/" //png
    };

    private static String[] allFileTypeOptions, allFileTypeNames;
    private static int defaultFileTypeOption;


    //static constructor
    static {
        int nDFTN = dataFileTypeNames.length;
        int nIFTN = imageFileTypeNames.length;
        Test.ensureEqual(nDFTN, dataFileTypeDescriptions.length,
            "'dataFileTypeNames.length' not equal to 'dataFileTypeDescriptions.length'.");                                     
        Test.ensureEqual(nDFTN, dataFileTypeExtensions.length,
            "'dataFileTypeNames.length' not equal to 'dataFileTypeExtensions.length'.");                                     
        Test.ensureEqual(nDFTN, dataFileTypeInfo.length,
            "'dataFileTypeNames.length' not equal to 'dataFileTypeInfo.length'.");                                     
        Test.ensureEqual(nIFTN, imageFileTypeDescriptions.length,
            "'imageFileTypeNames.length' not equal to 'imageFileTypeDescriptions.length'.");                                     
        Test.ensureEqual(nIFTN, imageFileTypeExtensions.length,
            "'imageFileTypeNames.length' not equal to 'imageFileTypeExtensions.length'.");                                     
        Test.ensureEqual(nIFTN, imageFileTypeInfo.length,
            "'imageFileTypeNames.length' not equal to 'imageFileTypeInfo.length'.");                                     
        defaultFileTypeOption = String2.indexOf(dataFileTypeNames, ".htmlTable");

        //construct allFileTypeOptions
        allFileTypeOptions = new String[nDFTN + nIFTN];
        allFileTypeNames = new String[nDFTN + nIFTN];
        for (int i = 0; i < nDFTN; i++) {
            allFileTypeOptions[i] = dataFileTypeNames[i] + " - " + dataFileTypeDescriptions[i];
            allFileTypeNames[i] = dataFileTypeNames[i];
        }
        for (int i = 0; i < nIFTN; i++) {
            allFileTypeOptions[nDFTN + i] = imageFileTypeNames[i] + " - " + imageFileTypeDescriptions[i];
            allFileTypeNames[nDFTN + i] = imageFileTypeNames[i];
        }
    }


    //*********** end of static declarations ***************************

    /**
     * These specifies which variables the source can constrain
     * (either via constraints in the query or in the getSourceData method).
     * <br>For example, DAPPER lets you constrain the lon,lat,depth,time
     *   but not dataVariables
     *   (http://www.epic.noaa.gov/epic/software/dapper/dapperdocs/conventions/
     *   and see the Constraints heading).
     * <br>It is presumed that no source can handle regex's for numeric variables.
     * <p>CONSTRAIN_PARTIAL indicates the constraint will be partly handled by 
     *   getSourceData and fully handled by standarizeResultsTable.
     * The constructor must set these.
     */
    public final static int CONSTRAIN_NO = 0, CONSTRAIN_PARTIAL = 1, CONSTRAIN_YES = 2;
    protected int sourceCanConstrainNumericData = -1;  //not set
    protected int sourceCanConstrainStringData  = -1;  //not set
    /** Some sources can constrain String regexes (use =~ or ~=); some can't (use ""). See REGEX_OP.*/
    protected String sourceCanConstrainStringRegex = ""; 

    /** If present, these dataVariables[xxxIndex] values are set by the constructor.  
     * idIndex is the number of the variable with the 
     * offering/station/trajectory/profile id/name (e.g., for SOS).
     */
    protected int lonIndex = -1, latIndex = -1, altIndex = -1, timeIndex = -1, idIndex = -1; 

    /**
     * If an instance supports SOS offerings, the constructor should set these
     * parallel data structures with the information for each offering (e.g., station/trajectory/profile)
     * or overwrite the sosXxx methods below. 
     */
    protected PrimitiveArray sosOfferingName, //but this one almost always a StringArray
        sosMinLongitude, sosMaxLongitude,
        sosMinLatitude,  sosMaxLatitude,
        sosMinTime,      sosMaxTime;

    /** 
     * Constructors in subclasses call gatherSosObservedProperties (below) to set this
     * list of SOS observedProperties available from the offerings/stations.
     * The properties are assumed to be the same for all offerings/stations
     * (which may not be strictly true, but which is reasonable).
     * This returns null if not applicable.
     */
    protected String[] sosObservedProperties;

    /**
     * This returns the dapProtocol
     *
     * @return the dapProtocol
     */
    public String dapProtocol() {return dapProtocol; }

    /**
     * This returns the dapDescription 
     *
     * @return the dapDescription
     */
    public String dapDescription() {return EDStatic.EDDTableDapDescription; }

    /** It is useful to have a static field and a non-static method to access this. */
    public static String longDapDescriptionHtml =
        dapProtocol + " lets you request tabular data (for example, buoy data) and graphs of tabular data, via specially formed URLs.\n" +
        "<br>The URLs specify everything: the dataset, the desired file type for the response, and the subset of data that you want to receive.\n" +
        "<br>You can form these URLs by hand or with a computer program,\n" +
        "<br>or (much easier) you can use the dataset's \"Data Access Form\" (oriented to requests for data)\n" +
        "<br>or the dataset's \"Make A Graph\" form (oriented to requests for graphs and maps),\n" +
        "<br>both of which generate the URL when the form is submitted.\n" +
        "<br>" + dapProtocol + " is a superset of the \n" +
        "<a href=\"http://www.opendap.org\">OPeNDAP</a>\n" +
        "<a href=\"http://www.opendap.org/pdf/ESE-RFC-004v1.1.pdf\">DAP</a>\n" +
        "<a href=\"http://www.opendap.org/user/guide-html/guide_33.html\">constraint protocol</a>.\n";

    /**
     * This returns the standard long HTML description of this DAP protocol.
     */
    public String longDapDescriptionHtml() {
        return longDapDescriptionHtml;
    }

    /**
     * This MUST be used by all subclass constructors to ensure that 
     * all of the items common to all EDDTables are properly set.
     *
     * @throws Throwable if any required item isn't properly set
     */
    public void ensureValid() throws Throwable {
        super.ensureValid();

        String errorInMethod = "datasets.xml/EDDTable.ensureValid error for " + 
            datasetID + ":\n ";

        Test.ensureTrue(lonIndex < 0 || dataVariables[lonIndex] instanceof EDVLon, 
            errorInMethod + "dataVariable[lonIndex=" + lonIndex + "] isn't an EDVLon.");
        Test.ensureTrue(latIndex < 0 || dataVariables[latIndex] instanceof EDVLat, 
            errorInMethod + "dataVariable[latIndex=" + latIndex + "] isn't an EDVLat.");
        Test.ensureTrue(altIndex < 0 || dataVariables[altIndex] instanceof EDVAlt, 
            errorInMethod + "dataVariable[altIndex=" + altIndex + "] isn't an EDVAlt.");
        Test.ensureTrue(timeIndex < 0 || dataVariables[timeIndex] instanceof EDVTime, 
            errorInMethod + "dataVariable[timeIndex=" + timeIndex + "] isn't an EDVTime.");

        Test.ensureTrue(sourceCanConstrainNumericData >= 0 && sourceCanConstrainNumericData <= 2, 
            errorInMethod + "sourceCanConstrainNumericData=" + sourceCanConstrainNumericData + " must be 0, 1, or 2.");
        Test.ensureTrue(sourceCanConstrainStringData >= 0 && sourceCanConstrainStringData <= 2, 
            errorInMethod + "sourceCanConstrainStringData=" + sourceCanConstrainStringData + " must be 0, 1, or 2.");
        Test.ensureTrue(sourceCanConstrainStringRegex.equals("=~") ||
            sourceCanConstrainStringRegex.equals("~=") ||
            sourceCanConstrainStringRegex.equals(""),
            errorInMethod + "sourceCanConstrainStringRegex=\"" + 
                sourceCanConstrainStringData + "\" must be \"=~\", \"~=\", or \"\".");

        //add standard metadata to combinedGlobalAttributes
        //(This should always be done, so shouldn't be in an optional method...)
        String destNames[] = dataVariableDestinationNames();
        //lon
        int dv = String2.indexOf(destNames, EDV.LON_NAME);
        if (dv >= 0) {
            combinedGlobalAttributes.add("geospatial_lon_units", EDV.LON_UNITS);
            PrimitiveArray pa = dataVariables[dv].combinedAttributes().get("actual_range");
            if (pa != null) {
                combinedGlobalAttributes.add("geospatial_lon_min", pa.getNiceDouble(0));
                combinedGlobalAttributes.add("geospatial_lon_max", pa.getNiceDouble(1));
                combinedGlobalAttributes.add("Westernmost_Easting", pa.getNiceDouble(0));
                combinedGlobalAttributes.add("Easternmost_Easting", pa.getNiceDouble(1));
            }
        }
        //lat
        dv = String2.indexOf(destNames, EDV.LAT_NAME); 
        if (dv >= 0) {
            combinedGlobalAttributes.add("geospatial_lat_units", EDV.LAT_UNITS);
            PrimitiveArray pa = dataVariables[dv].combinedAttributes().get("actual_range");
            if (pa != null) {
                combinedGlobalAttributes.add("geospatial_lat_min", pa.getNiceDouble(0));
                combinedGlobalAttributes.add("geospatial_lat_max", pa.getNiceDouble(1));
                combinedGlobalAttributes.add("Southernmost_Northing", pa.getNiceDouble(0));
                combinedGlobalAttributes.add("Northernmost_Northing", pa.getNiceDouble(1));
            }
        }
        //alt
        dv = String2.indexOf(destNames, EDV.ALT_NAME);
        if (dv >= 0) {
            combinedGlobalAttributes.add("geospatial_vertical_positive", "up");
            combinedGlobalAttributes.add("geospatial_vertical_units", EDV.ALT_UNITS);
            PrimitiveArray pa = dataVariables[dv].combinedAttributes().get("actual_range");
            if (pa != null) {
                combinedGlobalAttributes.add("geospatial_vertical_min", pa.getNiceDouble(0));
                combinedGlobalAttributes.add("geospatial_vertical_max", pa.getNiceDouble(1));
            }
        }
        //time
        dv = String2.indexOf(destNames, EDV.TIME_NAME);
        if (dv >= 0) {
            PrimitiveArray pa = dataVariables[dv].combinedAttributes().get("actual_range");
            if (pa != null) {
                double d = pa.getDouble(0);
                if (!Double.isNaN(d))
                    combinedGlobalAttributes.set("time_coverage_start", Calendar2.epochSecondsToIsoStringT(d) + "Z");
                d = pa.getDouble(1);  //will be NaN for 'present'.   Deal with this better???
                if (!Double.isNaN(d))
                    combinedGlobalAttributes.set("time_coverage_end",   Calendar2.epochSecondsToIsoStringT(d) + "Z");
            }
        }

    }


    /** 
     * The index of the longitude variable (-1 if not present).
     * @return the index of the longitude variable (-1 if not present)
     */
    public int lonIndex() {return lonIndex; }

    /** 
     * The index of the latitude variable (-1 if not present).
     * @return the index of the latitude variable (-1 if not present)
     */
    public int latIndex() {return latIndex; }

    /** 
     * The index of the altitude variable (-1 if not present).
     * @return the index of the altitude variable (-1 if not present)
     */
    public int altIndex() {return altIndex; }

    /** 
     * The index of the time variable (-1 if not present).
     * @return the index of the time variable (-1 if not present)
     */
    public int timeIndex() {return timeIndex; }

    /**
     * The string representation of this tableDataSet (for diagnostic purposes).
     *
     * @return the string representation of this tableDataSet.
     */
    public String toString() {  
        //make this JSON format?
        StringBuffer sb = new StringBuffer();
        sb.append("//** EDDTable " + super.toString() +
            "\nsourceCanConstrainNumericData=" + sourceCanConstrainNumericData +
            "\nsourceCanConstrainStringData="  + sourceCanConstrainStringData +
            "\nsourceCanConstrainStringRegex=\"" + sourceCanConstrainStringRegex + "\"" +
            "\n\\**\n\n");
        return sb.toString();
    }

    /** 
     * This returns 0=CONSTRAIN_NO (not at all), 1=CONSTRAIN_PARTIAL, or
     * 2=CONSTRAIN_YES (completely), indicating if the source can handle 
     * all non-regex constraints on numericDataVariables
     * (either via constraints in the query or in the getSourceData method).
     *
     * <p>CONSTRAIN_NO indicates that sourceQueryFromDapQuery should remove
     * these constraints, because the source doesn't even want to see them.
     * <br>CONSTRAIN_YES indicates that standardizeResultsTable needn't
     * do the tests for these constraints.
     *
     * <p>Note that CONSTRAIN_PARTIAL is very useful. It says that
     * the source would like to see all of the constraints (e.g., 
     * from sourceQueryFromDapQuery) so that it can then further prune
     * the constraints as needed and use them or not. 
     * Then, standardizeResultsTable will test all of the constraints.
     * It is a flexible and reliable approach.
     *
     * <p>It is assumed that no source can handle regex for numeric data, so this says nothing about regex.
     *
     * @return 0=CONSTRAIN_NO (not at all), 1=CONSTRAIN_PARTIAL, or
     * 2=CONSTRAIN_YES (completely) indicating if the source can handle 
     * all non-regex constraints on numericDataVariables 
     * (ignoring regex, which it is assumed none can handle).
     */
    public int sourceCanConstrainNumericData() {
        return sourceCanConstrainNumericData;
    }

    /** 
     * This returns 0=CONSTRAIN_NO (not at all), 1=CONSTRAIN_PARTIAL, or
     * 2=CONSTRAIN_YES (completely), indicating if the source can handle 
     * all non-regex constraints on stringDataVariables
     * (either via constraints in the query or in the getSourceData method).
     *
     * <p>CONSTRAIN_NO indicates that sourceQueryFromDapQuery should remove
     * these constraints, because the source doesn't even want to see them.
     * <br>CONSTRAIN_YES indicates that standardizeResultsTable needn't
     * do the tests for these constraints.
     *
     * <p>Note that CONSTRAIN_PARTIAL is very useful. It says that
     * the source would like to see all of the constraints (e.g., 
     * from sourceQueryFromDapQuery) so that it can then further prune
     * the constraints as needed and use them or not. 
     * Then, standardizeResultsTable will test all of the constraints.
     * It is a flexible and reliable approach.
     *
     * <p>PARTIAL and YES say nothing about regex; see sourceCanConstrainStringRegex.
     * So this=YES + sourceCanConstrainStringReges="" is a valid setup.
     *
     * @return 0=CONSTRAIN_NO (not at all), 1=CONSTRAIN_PARTIAL, or
     * 2=CONSTRAIN_YES (completely) indicating if the source can handle 
     * all non-regex constraints on stringDataVariables.
     */
    public int sourceCanConstrainStringData() {
        return sourceCanConstrainStringData;
    }

    /** 
     * This returns "=~" or "~=" (indicating the operator that the source uses)
     * if the source can handle some source regex constraints, or "" if it can't.
     * <br>sourceCanConstrainStringData must be PARTIAL or YES if this isn't "".
     * <br>If this isn't "", the implication is always that the source can
     *    only handle the regex PARTIAL-ly (i.e., it is always also checked in
     *    standardizeResultsTable).
     *
     * @return "=~" or "~=" if the source can handle some source regex constraints,
     * or "" if it can't.
     */
    public String sourceCanConstrainStringRegex() {
        return sourceCanConstrainStringRegex;
    }


    /** 
     * This is a convenience for the subclass' getDataForDapQuery methods.
     * <br>This is the first step in converting a userDapQuery into a sourceDapQuery.
     * <br>The userDapQuery refers to destinationNames 
     *   and allows any constraint op on any variable in the dataset.
     * <br>The sourceDapQuery refers to sourceNames and utilizes the 
     *   sourceCanConstraintXxx settings to move some constraintVariables
     *   into resultsVariables so they can be tested by
     *   standardizeResultsTable (and by the source, or not).
     * <br>This removes constraints which can't be handled by the source 
     *   (e.g., regex for numberic source variables).
     * <br>But the subclass will almost always need to fine-tune the
     *   results from this method to PRUNE out additional constraints
     *   that the source can't handle (e.g., constraints on inner variables
     *   in a DAP sequence).
     *
     * <p>You can do a numeric test for "=NaN" or "!=NaN", but
     *   the NaN must always be represented by "NaN" (not just an invalid number).
     *
     * @param userDapQuery  the query from the user after the '?', still percentEncoded (shouldn't be null)
     *  (e.g., var1,var2&var3%3C=40) 
     *   referring to destinationNames (e.g., LAT), not sourceNames.
     * @param resultsVariables will receive the list of variables that 
     *   it should request.
     *   <br>If a constraintVariable isn't in the original resultsVariables list
     *     that the user submitted, it will be added here.
     * @param constraintVariables will receive the constraint variables
     *     that the source has indicated (via sourceCanConstrainNumericData and
     *     sourceCanConstrainStringData) that it would like to see.
     * @param constraintOps will receive the operators corresponding to the 
     *     constraint variables.
     *   <br>REGEX_OP remains REGEX_OP (it isn't converted to sourceCanConstrainRegex string).
     * @param constraintValues will receive the values corresponding to the constraint variables.
     *   <br>Note that EDVTimeStamp values for non-regex ops will be returned as epoch seconds, 
     *      not in the source format.
     *   <br>For non-regex ops, edv.scale_factor and edv.add_offset will have been applied in reverse  
     *      so that the constraintValues are appropriate for the source.
     * @throws Throwable if trouble (e.g., improper format or unrecognized variable)
     */
    public void getSourceQueryFromDapQuery(String userDapQuery, 
        StringArray resultsVariables,
        StringArray constraintVariables,
        StringArray constraintOps,
        StringArray constraintValues) throws Throwable {
        if (reallyVerbose) String2.log("\nEDDTable.getSourceQueryFromDapQuery...");

        //pick the query apart
        resultsVariables.clear();
        constraintVariables.clear();
        constraintOps.clear();
        constraintValues.clear();
        parseUserDapQuery(userDapQuery, resultsVariables,
            constraintVariables, constraintOps, constraintValues, false);

        //remove any fixedValue variables (source can't provide their data values)
        //work backwards since I may remove some variables
        for (int rv = resultsVariables.size() - 1; rv >= 0; rv--) {
            EDV edv = findDataVariableByDestinationName(resultsVariables.get(rv)); 
            if (edv.isSourceFixedValue())
                resultsVariables.remove(rv);
        }

        //make source query (which the source can handle, and which gets all 
        //  the data needed to do all the constraints and make the final table)

        //Deal with each of the constraintVariables.
        //Work backwards because I often delete the current one.
        //!!!!! VERY IMPORTANT CODE. THINK IT THROUGH CAREFULLY.
        //!!!!! THE IF/ELSE STRUCTURE HERE IS EXACTLY THE SAME AS IN standardizeResultsTable.
        //!!!!! SO IF YOU MAKE A CHANGE HERE, MAKE A CHANGE THERE.
        for (int cv = constraintVariables.size() - 1; cv >= 0; cv--) { 
            String constraintVariable = constraintVariables.get(cv);
            int dv = String2.indexOf(dataVariableDestinationNames(), constraintVariable);
            EDV edv = dataVariables[dv];
            Class sourceClass = edv.sourceDataTypeClass();
            Class destClass = edv.destinationDataTypeClass(); 

            String constraintOp = constraintOps.get(cv);
            String constraintValue = constraintValues.get(cv);
            double constraintValueD = String2.parseDouble(constraintValue);
            //Only valid numeric constraintValue NaN is "NaN".
            //Test this because it helps discover other constraint syntax errors.
            if (destClass != String.class &&
                !constraintOp.equals(REGEX_OP) &&
                Double.isNaN(constraintValueD) && 
                !constraintValue.equals("NaN"))
                throw new SimpleException("Query error: " +
                    "Numeric constraint value='" + constraintValue + 
                    "' evaluating to NaN must be 'NaN'.");
            if (reallyVerbose) String2.log("  Looking at constraint#" + cv + ": " + constraintVariable + 
                " " + constraintOp + " " + constraintValue);

            //is constraintVariable a sourceFixedValue? 
            if (edv.isSourceFixedValue()) {
                //it's a fixedValue; always numeric; do constraint now
                //pass=that's nice    fail=NO DATA
                if (!testValueOpValue(edv.destinationMin(), //min = max, in destination units
                    constraintOp, constraintValueD))
                    throw new SimpleException(EDStatic.THERE_IS_NO_DATA + 
                        " (fixed value variable=" + 
                        edv.destinationName() + " failed " + 
                        edv.destinationMin() + constraintOp + constraintValueD + ")");

                //It passed, so remove the constraint from list passed to source.
                constraintVariables.remove(cv);
                constraintOps.remove(cv);
                constraintValues.remove(cv);
                if (reallyVerbose) 
                    String2.log("    The sourceFixedValue constraint passes.");

            //constraint source is String                  
            } else if (sourceClass == String.class) {
                if (sourceCanConstrainStringData == CONSTRAIN_NO ||
                    (constraintOp.equals(REGEX_OP) && sourceCanConstrainStringRegex.length() == 0)) {
                    //remove from constraints and constrain after the fact in standardizeResultsTable
                    constraintVariables.remove(cv);
                    constraintOps.remove(cv);
                    constraintValues.remove(cv);                            
                } else {
                    //source will (partially) handle it
                    if (reallyVerbose) 
                        String2.log("    The string constraint will be (partially) handled by source.");
                }

                //if source handling is NO or PARTIAL, data is needed to do the test in standardizeResultsTable
                if (sourceCanConstrainStringData == CONSTRAIN_NO ||
                    sourceCanConstrainStringData == CONSTRAIN_PARTIAL ||
                    constraintOp.equals(REGEX_OP)) {    //regex always (also) done in standardizeResultsTable
                    if (resultsVariables.indexOf(constraintVariable) < 0)
                        resultsVariables.add(constraintVariable);
                }

            //constraint source is numeric
            } else {  
                if (sourceCanConstrainNumericData == CONSTRAIN_NO ||
                    constraintOp.equals(REGEX_OP)) {
                    //remove from constraints and constrain after the fact in standardizeResultsTable
                    constraintVariables.remove(cv);
                    constraintOps.remove(cv);
                    constraintValues.remove(cv);                            
                } else {
                    //source will (partially) handle it
                    if (reallyVerbose) 
                        String2.log("    The numeric constraint will be (partially) handled by source.");
                }

                //if source handling is NO or PARTIAL, data is needed to do the test below
                if (sourceCanConstrainNumericData == CONSTRAIN_NO ||
                    sourceCanConstrainNumericData == CONSTRAIN_PARTIAL ||
                    constraintOp.equals(REGEX_OP)) {
                    if (resultsVariables.indexOf(constraintVariable) < 0)
                        resultsVariables.add(constraintVariable);
                }
            }
        }


        //Convert resultsVariables and constraintVariables to sourceNames.
        //Do last because sourceNames may not be unique (e.g., for derived axes, e.g., "=0")
        //And cleaner and safer to do all at once.
        for (int rv = 0; rv < resultsVariables.size(); rv++) {
            EDV edv = findDataVariableByDestinationName(resultsVariables.get(rv));
            resultsVariables.set(rv, edv.sourceName());
        }
        for (int cv = 0; cv < constraintVariables.size(); cv++) {
            EDV edv = findDataVariableByDestinationName(constraintVariables.get(cv));
            constraintVariables.set(cv, edv.sourceName());

            //and if constaintVariable isn't in resultsVariables, add it
            if (resultsVariables.indexOf(edv.sourceName()) < 0)
                resultsVariables.add(edv.sourceName());

            //and apply scale_factor and add_offset to non-regex constraintValues
            // sourceValue = (destintationValue - addOffset) / scaleFactor;
            if (edv.scaleAddOffset() && !constraintOps.get(cv).equals(REGEX_OP)) {
                constraintValues.setDouble(cv, 
                    (constraintValues.getDouble(cv) - edv.addOffset()) / edv.scaleFactor());
            }

        }

        if (reallyVerbose) String2.log("getSourceQueryFromDapQuery done");

    }

    /** 
     * This is a convenience for the subclass' getData methods.
     * This converts the table of data from the source dap query into
     * the finished, standardized, results table.  If need be, it tests constraints
     * that the source couldn't test.
     *
     * <p>This always does all regex tests.
     *
     * <p>Coming into this method, the column names are sourceNames.
     *  Coming out, they are destinationNames.
     *
     * <p>Coming into this method, missing values in the table
     *  will be the original edv.sourceMissingValues or sourceFillValues.
     *  Coming out, they will be edv.destinationMissingValues or destinationFillValues.
     *
     * <p>This removes any global or column attributes from the table.
     *  This then adds a copy of the combinedGlobalAttributes,
     *  and adds a copy of each variable's combinedAttributes.
     *
     * <p>This doesn't call setActualRangeAndBoundingBox
     *   (since you need the entire table to get correct values).
     *   In fact, all the file-specific metadata is removed (e.g., actual_range) here.
     *   Currently, the only file types that need file-specific metadata are .nc and .ncHeader.
     *   (.das always shows original metadata, not subset metadata).
     *   Currently the only place the file-specific metadata is added
     *   is in TableWriterAll (which has access to all the data).
     *
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery  the OPeNDAP DAP-style query from the user, 
     *   after the '?', still percentEncoded (shouldn't be null).
     *   (e.g., var1,var2&var3&lt;40) 
     *   referring to variable's destinationNames (e.g., LAT), instead of the
     *   sourceNames.
     *   Presumably, getSourceDapQuery has ensured that this is a valid query
     *   and has tested the constraints on sourceFixedValue axes.
     * @param table the table from the source query which will be modified
     *   to become the finished standardized table.
     *   It must have all the requested source resultsVariables (using sourceNames),
     *     and variables needed for constraints that need to be handled here.
     *   It may have additional variables.
     *   It doesn't need to already have globalAttributes or variable attributes.
     * @throws Throwable if trouble (e.g., improper format or
     *    unrecognized variable).
     *    Because this may be called separately for each station, 
     *    this doesn't throw exception if no data at end; 
     *    tableWriter.finish will catch such problems.
     *    But it does throw exception if no data at beginning.
     *    
     */
    public void standardizeResultsTable(String requestUrl, String userDapQuery, 
            Table table) throws Throwable {
        if (reallyVerbose) String2.log("\nstandardizeResultsTable...");
        //String2.log("table=\n" + table.toString());
        if (table.nRows() == 0) 
            throw new SimpleException(EDStatic.THERE_IS_NO_DATA);

        //pick the query apart
        StringArray resultsVariables    = new StringArray();
        StringArray constraintVariables = new StringArray();
        StringArray constraintOps       = new StringArray();
        StringArray constraintValues    = new StringArray();
        parseUserDapQuery(userDapQuery, resultsVariables,
            constraintVariables, constraintOps, constraintValues, false);

        //set the globalAttributes (this takes care of title, summary, ...)
        table.globalAttributes().clear(); //remove any existing atts
        table.globalAttributes().set(combinedGlobalAttributes); //make a copy

        //fix up global attributes  (always to a local COPY of global attributes)
        EDD.addToHistory(table.globalAttributes(), sourceUrl());
        EDD.addToHistory(table.globalAttributes(), 
            EDStatic.baseUrl + requestUrl + 
            (userDapQuery == null? "" : "?" + userDapQuery));

        //Change all table columnNames from sourceName to destinationName.
        //Do first because sourceNames in Erddap may not be unique 
        //(e.g., for derived vars, e.g., "=0").
        //But all table columnNames are unique because they are direct from source.
        //And cleaner and safer to do all at once.
        int nRows = table.nRows();
        for (int col = table.nColumns() - 1; col >= 0; col--) { //work backward since may remove some cols

            //These are direct from source, so no danger of ambiguous fixedValue source names.
            String columnSourceName = table.getColumnName(col);
            int dv = String2.indexOf(dataVariableSourceNames(), columnSourceName);
            if (dv < 0) {
                //remove unexpected column
                if (reallyVerbose) String2.log("  removing unexpected source column=" + columnSourceName);
                table.removeColumn(col);
                continue;
            }

            //convert source values to destination values
            //missing values stay as source missing values
            //String2.log("  col=" + col + " dv=" + dv + " nRows=" + nRows);
            EDV edv = dataVariables[dv];
            table.setColumnName(col, edv.destinationName());
            table.columnAttributes(col).clear(); //remove any existing atts
            table.columnAttributes(col).set(edv.combinedAttributes()); //make a copy
            table.setColumn(col, edv.toDestination(table.getColumn(col)));
        }

        //move the requested resultsVariables columns into place
        for (int rv = 0; rv < resultsVariables.size(); rv++) {

            int dv = String2.indexOf(dataVariableDestinationNames(), resultsVariables.get(rv));
            EDV edv = dataVariables[dv];
            if (edv.isSourceFixedValue()) {
                //edv is a sourceFixedValue
                table.addColumn(rv, resultsVariables.get(rv), 
                    PrimitiveArray.factory(edv.destinationDataTypeClass(), nRows, "" + edv.destinationMin()),
                    new Attributes(edv.combinedAttributes())); //make a copy
            } else {
                //edv is not sourceFixedValue, so it must be in table. 
                int col = table.findColumnNumber(edv.destinationName());
                if (col < 0)
                    throw new SimpleException("Error: " +
                        "variable=" + edv.destinationName() + 
                        " not found in results table.\n" +
                        "colNames=" + String2.toCSVString(table.getColumnNames()));
                if (col < rv)
                    throw new SimpleException("Error: " +
                        "variable=" + edv.destinationName() + " is already at column=" + 
                        col + " in the results table.\n" +
                        "colNames=" + String2.toCSVString(table.getColumnNames()));

                //move the column into place
                table.moveColumn(col, rv);
                table.columnAttributes(rv).set(edv.combinedAttributes()); //make a copy
            }
        }
        //String2.log("table after rearrange cols=\n" + table.toString());

        //deal with the constraints one-by-one
        //!!!!! VERY IMPORTANT CODE. THINK IT THROUGH CAREFULLY.
        //!!!!! THE IF/ELSE STRUCTURE HERE IS ALMOST EXACTLY THE SAME AS IN getSourceDapQuery.
        //Difference: above, only CONSTRAIN_NO items are removed from what goes to source
        //            below, CONSTRAIN_NO and CONSTRAIN_PARTIAL items are tested here
        //!!!!! SO IF YOU MAKE A CHANGE HERE, MAKE A CHANGE THERE.
        BitSet keep = new BitSet();
        keep.set(0, nRows, true); 
        if (reallyVerbose) String2.log("  nRows=" + nRows);
        for (int cv = 0; cv < constraintVariables.size(); cv++) { 
            String constraintVariable = constraintVariables.get(cv);
            String constraintOp       = constraintOps.get(cv);
            int dv = String2.indexOf(dataVariableDestinationNames(), constraintVariable);
            EDV edv = dataVariables[dv];
            Class sourceClass = edv.sourceDataTypeClass();
            Class destClass = edv.destinationDataTypeClass();

            //is constraintVariable a sourceFixedValue? 
            if (edv.isSourceFixedValue()) {
                //it's a fixedValue axis; always numeric; test was done by getSourceDapQuery
                continue;

            //constraint source is String
            } else if (sourceClass == String.class) {
                if (sourceCanConstrainStringData == CONSTRAIN_NO ||
                    sourceCanConstrainStringData == CONSTRAIN_PARTIAL ||
                    constraintOp.equals(REGEX_OP)) {  //always do all regex tests here (perhaps in addition to source)
                    //fall through to test below
                } else {
                    //source did the test
                    continue;
                }

            //constraint source is numeric
            } else {
                if (sourceCanConstrainNumericData == CONSTRAIN_NO ||
                    sourceCanConstrainNumericData == CONSTRAIN_PARTIAL ||
                    constraintOp.equals(REGEX_OP)) { //always do all regex tests here
                    //fall through to test below
                } else {
                    //source did the test
                    continue;
                }
            }

            //The constraint needs to be tested here. Test it now.
            //Note that Time and Alt values have been converted to standardized units above.
            String constraintValue = constraintValues.get(cv);
            double constraintValueD = String2.parseDouble(constraintValue);
            if (reallyVerbose) String2.log("  Handling constraint #" + cv + " here: " + 
                constraintVariable + " " + constraintOp + " " + constraintValue);
            boolean doStringTest = destClass == String.class ||
                constraintOp.equals(REGEX_OP); //do numeric regex test via string testValueOpValue
            //String2.log("    colNames=" + String2.toCSVString(table.getColumnNames()));
            PrimitiveArray dataPa = table.findColumn(constraintVariable); //throws Throwable if not found
            int nStillGood = 0;
            //just test the rows where keep==true
            for(int row = keep.nextSetBit(0); row >= 0; row = keep.nextSetBit(row + 1)) {
                boolean b = doStringTest?
                    testValueOpValue(dataPa.getString(row), constraintOp, constraintValue) :
                    testValueOpValue(dataPa.getNiceDouble(row), constraintOp, //"nice" is important
                        //do constrantValue=NaN tests with safeDestinationMissingValue
                        Double.isNaN(constraintValueD)? edv.safeDestinationMissingValue() : constraintValueD);
                if (b) {
                    nStillGood++;
                } else {
                    //if (reallyVerbose) String2.log(
                    //    "    constraint test failed: doStringTest=" + doStringTest + 
                    //    ": data=" + dataPa.getString(row) + " " + constraintOp + " constr=" + constraintValue);
                    keep.clear(row);
                }               
            }
            if (reallyVerbose) String2.log("    nStillGood=" + nStillGood);
        }

        //remove the non-keep rows
        table.subset(keep);

        //discard excess columns (presumably constraintVariables not handled by source)
        for (int v = table.nColumns() - 1; v >= resultsVariables.size(); v--)
            table.removeColumn(v);

        //unsetActualRangeAndBoundingBox  (see comments in method javadocs above)
        //(see TableWriterAllWithMetadata which adds them back)
        table.unsetActualRangeAndBoundingBox(
            table.findColumnNumber(EDV.LON_NAME), 
            table.findColumnNumber(EDV.LAT_NAME), 
            -1, //depth
            table.findColumnNumber(EDV.ALT_NAME), 
            table.findColumnNumber(EDV.TIME_NAME));

        if (reallyVerbose) String2.log("standardizeResultsTable done.");

    }


    /** 
     * This gets the data (chunk by chunk) from this EDDTable for the 
     * OPeNDAP DAP-style query and writes it to the TableWriter. 
     *
     * <p>This methods allows any constraint on any variable.
     * Since many data sources just support constraints on certain variables
     * (e.g., Dapper only supports axis variable constraints), the implementations
     * here must separate constraints that the server can handle and constraints 
     * that must be handled separately afterwards.
     * See getSourceQueryFromDapQuery.
     *
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery  the OPeNDAP DAP-style  
     *   query from the user after the '?', still percentEncoded (shouldn't be null).
     *   (e.g., var1,var2&var3%3C=40) 
     *   perhaps referring to standard variables (e.g., LAT instead of the
     *   latVariable.sourceName) or fixed value axis variables.
     *   See parseUserDapQuery.
     * @param tableWriter
     * @throws Throwable if trouble
     */
    public abstract void getDataForDapQuery(String requestUrl, String userDapQuery, 
        TableWriter tableWriter) throws Throwable;


    /** 
     * This is used by subclasses getDataForDapQuery to write a chunk of table 
     * data to a tableWriter if nCells &gt; partialRequestMaxCells
     * or finish==true.
     * This converts columns to expected sourceTypes (if they weren't already).
     * 
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery  after the '?', still percentEncoded (shouldn't be null).
     * @param table with a chunk of the total source data.
     * @param tableWriter to which the table's data will be written
     * @param finish If true, the will be written to the tableWriter.
     *    If false, the data will only be written to tableWriter if there are 
     *    a sufficient number of cells of data.
     * @return true Returns true if it wrote the table data to tableWriter (table was changed, so should be thrown away).
     *    Returns false if the data wasn't written -- add some more rows of data and call this again.
     */
    protected boolean writeChunkToTableWriter(String requestUrl, String userDapQuery, 
        Table table, TableWriter tableWriter, boolean finish) throws Throwable {

        //write to tableWriter?
        //String2.log("writeChunkToTableWriter table.nRows=" + table.nRows());
        if (table.nRows() > 200 || //that's a small number, but user is probably eager to see some results
            (table.nRows() * (long)table.nColumns() > EDStatic.partialRequestMaxCells) || 
            finish) {

            if (table.nRows() == 0 && finish) {
                tableWriter.finish();
                return true;
            }

            //String2.log("\ntable at end of getSourceData=\n" + table.toString("row", 10));
            //convert columns to stated type
            for (int col = 0; col < table.nColumns(); col++) {
                String colName = table.getColumnName(col);
                EDV edv = findDataVariableBySourceName(colName); 
                Class shouldBeType = edv.sourceDataTypeClass();
                Class isType = table.getColumn(col).getElementType();
                if (shouldBeType != isType) {
                    //if (reallyVerbose) String2.log("  converting col=" + col + 
                    //    " from=" + PrimitiveArray.elementTypeToString(isType) + 
                    //    " to="   + PrimitiveArray.elementTypeToString(shouldBeType));
                    PrimitiveArray newPa = PrimitiveArray.factory(shouldBeType, 1, false);
                    newPa.append(table.getColumn(col));
                    table.setColumn(col, newPa);
                }            
            }

            //standardize the results table
            if (table.nRows() > 0) {
                standardizeResultsTable(requestUrl, userDapQuery, table); //changes sourceNames to destinationNames
                tableWriter.writeSome(table);
            }

            //done?
            if (finish)
                tableWriter.finish();

            return true;
        }

        return false;
    }

    /**
     * This returns the types of data files that this dataset can be returned as.
     * These are short descriptive names that are put in the 
     * request url after the dataset name and before the "?", e.g., ".nc". 
     *
     * @return the types of data files that this dataset can be returned as.
     */
    public String[] dataFileTypeNames() {return dataFileTypeNames; }

    /**
     * This returns the file extensions corresponding to the dataFileTypes.
     * E.g., dataFileTypeName=".htmlTable" returns dataFileTypeExtension=".html".
     *
     * @return the file extensions corresponding to the dataFileTypes.
     */
    public String[] dataFileTypeExtensions() {return dataFileTypeExtensions; }

    /**
     * This returns descriptions (up to 80 characters long, suitable for a tooltip)
     * corresponding to the dataFileTypes. 
     *
     * @return descriptions corresponding to the dataFileTypes.
     */
    public String[] dataFileTypeDescriptions() {return dataFileTypeDescriptions; }

    /**
     * This returns an info URL corresponding to the dataFileTypes. 
     *
     * @return an info URL corresponding to the dataFileTypes.
     */
    public String[] dataFileTypeInfo() {return dataFileTypeInfo; }

    /**
     * This returns the types of image files that this dataset can be returned 
     * as. These are short descriptive names that are put in the 
     * request url after the dataset name and before the "?", e.g., ".largePng". 
     *
     * @return the types of image files that this dataset can be returned as.
     */
    public String[] imageFileTypeNames() {return imageFileTypeNames; }

    /**
     * This returns the file extensions corresponding to the imageFileTypes,
     * e.g., imageFileTypeNames=".largePng" returns imageFileTypeExtensions=".png".
     *
     * @return the file extensions corresponding to the imageFileTypes.
     */
    public String[] imageFileTypeExtensions() {return imageFileTypeExtensions; }

    /**
     * This returns descriptions corresponding to the imageFileTypes 
     * (each is suitable for a tooltip).
     *
     * @return descriptions corresponding to the imageFileTypes.
     */
    public String[] imageFileTypeDescriptions() {return imageFileTypeDescriptions; }

    /**
     * This returns an info URL corresponding to the imageFileTypes. 
     *
     * @return an info URL corresponding to the imageFileTypes.
     */
    public String[] imageFileTypeInfo() {return imageFileTypeInfo; }
    
    /**
     * This returns the "[name] - [description]" for all dataFileTypes and imageFileTypes.
     *
     * @return the "[name] - [description]" for all dataFileTypes and imageFileTypes.
     */
    public String[] allFileTypeOptions() {return allFileTypeOptions; }
     
    /**
     * This parses a PERCENT ENCODED OPeNDAP DAP-style query.
     * This checks the validity of the resultsVariable and constraintVariabe names.
     *
     * <p>Unofficially (e.g., for testing) the query can be already percent decoded
     * if there are no %dd in the query.
     *
     * <p>There can be a constraints on variables that
     * aren't in the user-specified results variables.
     * This procedure adds those variables to the returned resultsVariables.
     *
     * <p>If the constraintVariable is time, the value
     * can be numeric ("seconds since 1970-01-01") 
     * or String (ISO format, e.g., "1996-01-31T12:40:00", at least YYYY-MM).
     * The ISO 8601 String format is converted to the numeric format.
     *
     * @param userDapQuery is the opendap DAP-style query, 
     *      after the '?', still percentEncoded (shouldn't be null), e.g.,
     *      <tt>var1,var2,var3&amp;var4=value4&amp;var5%3E=value5</tt> .
     *    <br>Values for String variable constraints should be in double quotes.
     *    <br>A more specific example is 
     *      <tt>genus,species&amp;genus="Macrocystis"&amp;LAT%3C=53&amp;LAT%3C=54</tt> .
     *    <br>If no results variables are specified, all will be returned.
     *      See DAP specification, section 6.1.1.
     *    <br>Note that each constraint's left hand side must be a variable
     *      and its right hand side must be a value.
     *    <br>The valid operators (for numeric and String variables) are "=", "!=", "&lt;", "&lt;=",  
     *      "&gt;", "&gt;=", and "=~" (REGEX_OP, which looks for data values
     *      matching the regular expression on the right hand side),
     *      but in percent encoded form. 
     *      (see http://www.opendap.org/user/guide-html/guide_35.html).
     *    <br>If an &amp;-separated part is "distinct()", "orderBy("...")", 
     *      "orderByMax("...")", 
     *      it is ignored.
     *    <br>If an &amp;-separated part starts with ".", it is ignored.
     *      It can't be a variable name.
     *      &amp;.[param]=value is used to pass special values (e.g., &amp;.colorBar=...).
     * @param resultsVariables to be appended with the results variables' 
     *    destinationNames, e.g., {var1,var2,var3}.
     *    If a variable is needed for constraint testing in standardizeResultsTable
     *    but is not among the user-requested resultsVariables, it will be added here.
     * @param constraintVariables to be appended with the constraint variables' 
     *    destinationNames, e.g., {var4,var5}.
     *    This method makes sure they are valid.
     *    These will not be percent encoded.
     * @param constraintOps to be appended with the constraint operators, e.g., {"=", "&gt;="}.
     *    These will not be percent encoded.
     * @param constraintValues to be appended with the constraint values, e.g., {value4,value5}.
     *    ISO times are returned as epochSeconds.
     *    These will not be percent encoded.
     * @param repair if true, this method tries to do its best repair problems (guess at intent), 
     *     not to throw exceptions 
     * @throws Throwable if invalid query
     *     (0 resultsVariables is a valid query)
     */
    public void parseUserDapQuery(String userDapQuery, StringArray resultsVariables,
        StringArray constraintVariables, StringArray constraintOps, StringArray constraintValues,
        boolean repair) throws Throwable {

        //parse userDapQuery into parts
        String parts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
        resultsVariables.clear();
        constraintVariables.clear(); 
        constraintOps.clear(); 
        constraintValues.clear();

        //expand no resultsVariables (or entire sequence) into all results variables
        //look at part0 with comma-separated vars 
        if (parts[0].length() == 0 || parts[0].equals(SEQUENCE_NAME)) {
            if (reallyVerbose) String2.log("  userDapQuery parts[0]=\"" + parts[0] + 
                "\" is expanded to request all variables.");
            for (int v = 0; v < dataVariables.length; v++) {
                resultsVariables.add(dataVariables[v].destinationName());
            }
        } else {
            String cParts[] = String2.split(parts[0], ',');
            for (int cp = 0; cp < cParts.length; cp++) {

                //request uses sequence.dataVarName notation?
                String tVar = cParts[cp].trim();
                int period = tVar.indexOf('.');
                if (period > 0 && tVar.substring(0, period).equals(SEQUENCE_NAME)) 
                    tVar = tVar.substring(period + 1);

                //is it a valid destinationName?
                int po = String2.indexOf(dataVariableDestinationNames(), tVar);
                if (po < 0) {
                    if (!repair) {
                        if (tVar.equals(SEQUENCE_NAME))
                            throw new SimpleException("Query error: " +
                                 "If " + SEQUENCE_NAME + " is requested, it must be the only requested variable.");
                        throw new SimpleException("Query error: " +
                             "Unrecognized variable=" + tVar);
                    }
                } else {
                    //it's valid; is it a duplicate?
                    if (resultsVariables.indexOf(tVar) >= 0) {
                        if (!repair) 
                            throw new SimpleException("Query error: " +
                                "variable=" + tVar + " is listed twice in the results variables list.");
                    } else {
                        resultsVariables.add(tVar);
                    }
                }
            }
        }
        //String2.log("resultsVariables=" + resultsVariables);

        //get the constraints 
        for (int p = 1; p < parts.length; p++) {
            //deal with one constraint at a time
            String constraint = parts[p]; 
            int constraintLength = constraint.length();
            //String2.log("constraint=" + constraint);            
            
            //special case: ignore constraint starting with "." 
            //(can't be a variable name)
            //use for e.g., .colorBar=...
            if (constraint.startsWith("."))
                continue;

            //special case: server-side functions
            if (constraint.equals("distinct()") ||
                (constraint.startsWith("orderBy(\"") && constraint.endsWith("\")")) ||
                (constraint.startsWith("orderByMax(\"") && constraint.endsWith("\")")))
                continue;

            //look for == (common mistake, but not allowed)
            if (constraint.indexOf("==") >= 0) {
                if (repair) constraint = String2.replaceAll(constraint, "==", "=");
                else throw new SimpleException("Query error: " +
                    "Use '=' instead of '==' in constraints.");
            }

            //look for ~= (common mistake, but not allowed)
            if (constraint.indexOf("~=") >= 0) {
                if (repair) constraint = String2.replaceAll(constraint, "~=", "=~");
                else throw new SimpleException("Query error: " +
                    "Use '=~' instead of '~=' in constraints.");
            }

            //find the valid op            
            int op = 0;
            int opPo = -1;
            while (op < OPERATORS.length && 
                (opPo = constraint.indexOf(OPERATORS[op])) < 0)
                op++;
            if (opPo < 0) {
                if (repair) continue; //was IllegalArgumentException
                else throw new SimpleException("Query error: " +
                    "No operator found in constraint=\"" + constraint + "\".");
            }

            //request uses sequenceName.dataVarName notation?
            String tName = constraint.substring(0, opPo);
            int period = tName.indexOf('.');
            if (period > 0 && tName.substring(0, period).equals(SEQUENCE_NAME)) 
                tName = tName.substring(period + 1);

            //is it a valid destinationName?
            int dvi = String2.indexOf(dataVariableDestinationNames(), tName);
            if (dvi < 0) {
                if (repair) continue;
                else throw new SimpleException("Query error: " +
                    "Unrecognized constraint variable=" + tName);
            }

            EDV conEdv = dataVariables[dvi];
            constraintVariables.add(tName);
            constraintOps.add(OPERATORS[op]);
            String tValue = constraint.substring(opPo + OPERATORS[op].length());
            constraintValues.add(tValue);

            //convert <time><op><isoString> to <time><op><epochSeconds>   
            if (conEdv instanceof EDVTimeStamp) {
//this isn't precise!!!   it should either be required or not
                if (tValue.startsWith("\"") && tValue.endsWith("\"")) { 
                    tValue = String2.fromJson(tValue);
                    constraintValues.set(constraintValues.size() - 1, tValue);
                }                

                //if not for regex, convert isoString to epochSeconds 
                if (OPERATORS[op] != REGEX_OP) {
                    if (Calendar2.isIsoDate(tValue)) {
                        double valueD = repair? Calendar2.safeIsoStringToEpochSeconds(tValue) :
                            Calendar2.isoStringToEpochSeconds(tValue);
                        constraintValues.set(constraintValues.size() - 1,
                            "" + valueD);
                        if (reallyVerbose) String2.log("  TIME CONSTRAINT converted in parseUserDapQuery: " +
                            tValue + " -> " + valueD);
                    }
                }
            } else if (conEdv.destinationDataTypeClass() == String.class) {

                //String variables must have " around constraintValues
                if (tValue.startsWith("\"") && tValue.endsWith("\"")) {
                    //remove the quotes
                    tValue = String2.fromJson(tValue);
                    constraintValues.set(constraintValues.size() - 1, tValue);
                } else {
                    throw new SimpleException("Query error: " +
                        "For constraints of String variables, " +
                        "the right-hand-side value must be surrounded by double quotes.\n" +
                        "Bad constraint: " + constraint);
                }

            } else {
                //numeric variables

                //if op=regex, value must have "'s around it
                if (OPERATORS[op] == REGEX_OP) {
                    if (!tValue.startsWith("\"") || !tValue.endsWith("\"")) 
                        throw new SimpleException("Query error: " +
                            "For =~ constraints of numeric variables, " +
                            "the right-hand-side value must be surrounded by double quotes.\n" +
                            "Bad constraint: " + constraint);
                    tValue = String2.fromJson(tValue);
                    constraintValues.set(constraintValues.size() - 1, tValue);

                } else {
                    //if op!=regex, numeric values must NOT have "'s around them
                    if (tValue.startsWith("\"") || tValue.endsWith("\"")) 
                        throw new SimpleException("Query error: " +
                            "For non =~ constraints of numeric variables, " +
                            "the right-hand-side value must not be surrounded by double quotes.\n" +
                            "Bad constraint: " + constraint);

                    //test of value=NaN must use "NaN", not somthing just a badly formatted number
                    double td = String2.parseDouble(tValue);
                    if (Double.isNaN(td) && !tValue.equals("NaN")) 
                        throw new SimpleException("Query error: " +
                            "Numeric tests of NaN must use \"NaN\", not value=\"" + tValue + "\".");
                }
            }
        }

        if (reallyVerbose) {
            String2.log("  Output from parseUserDapQuery:" +
                "\n    resultsVariables=" + resultsVariables +
                "\n    constraintVariables=" + constraintVariables +
                "\n    constraintOps=" + constraintOps +
                "\n    constraintValues=" + constraintValues);
        }
    }

    /**
     * As a service to getDataForDapQuery implementations,  
     * given the userDapQuery (the DESTINATION (not source) request),
     * this fills in the requestedMin and requestedMax arrays 
     * [0=lon, 1=lat, 2=alt, 3=time] (in destination units, may be NaN).
     *
     * <p>If a given variable (e.g., altIndex) isn't defined, the requestedMin/Max
     * will be NaN.
     *
     * @param userDapQuery  a userDapQuery (usually including longitude,latitude,
     *    altitude,time variables), after the '?', still percentEncoded (shouldn't be null).
     * @param useVariablesDestinationMinMax if true, the results are further constrained
     *    by the LLAT variable's destinationMin and destinationMax.
     * @param requestedMin will catch the minimum LLAT constraints;
     *     NaN if a given var doesn't exist or no info or constraints. 
     * @param requestedMax will catch the maximum LLAT constraints;
     *     NaN if a given var doesn't exist or no info or constraints. 
     * @throws Throwable if trouble (e.g., a requestedMin &gt; a requestedMax)
     */
    public void getRequestedDestinationMinMax(String userDapQuery, 
        boolean useVariablesDestinationMinMax,
        double requestedMin[], double requestedMax[]) throws Throwable {

        Arrays.fill(requestedMin, Double.NaN);
        Arrays.fill(requestedMax, Double.NaN);

        //parseUserDapQuery
        StringArray resultsVariables    = new StringArray();
        StringArray constraintVariables = new StringArray();
        StringArray constraintOps       = new StringArray();
        StringArray constraintValues    = new StringArray();
        parseUserDapQuery(userDapQuery, resultsVariables,
            constraintVariables, constraintOps, constraintValues, false);

        //try to get the LLAT variables
        EDV edv[] = new EDV[]{
            lonIndex  >= 0? dataVariables[lonIndex] : null,
            latIndex  >= 0? dataVariables[latIndex] : null,
            altIndex  >= 0? dataVariables[altIndex] : null,
            timeIndex >= 0? dataVariables[timeIndex] : null};

        //go through the constraints
        int nConstraints = constraintVariables.size();
        for (int constraint = 0; constraint < nConstraints; constraint++) {

            String destName = constraintVariables.get(constraint);
            int conDVI = String2.indexOf(dataVariableDestinationNames(), destName);
            if (conDVI < 0)
                throw new SimpleException("Query error: " +
                    "constraint variable=" + destName + " wasn't found.");
            String op = constraintOps.get(constraint);
            String conValue = constraintValues.get(constraint);
            double conValueD = String2.parseDouble(conValue); //ok for times: they are epochSeconds

            //constraint affects which of LLAT
            int index4 = -1;
            if      (conDVI == lonIndex)  index4 = 0;
            else if (conDVI == latIndex)  index4 = 1;
            else if (conDVI == altIndex)  index4 = 2;
            else if (conDVI == timeIndex) index4 = 3;
            //String2.log("index4:" + index4 + " " + op + " " + conValueD);

            if (index4 >= 0) {
                if (op.equals("=") || op.equals(">=") || op.equals(">")) 
                    requestedMin[index4] = Double.isNaN(requestedMin[index4])? conValueD :
                        Math.max(requestedMin[index4], conValueD); 
                if (op.equals("=") || op.equals("<=") || op.equals("<")) 
                    requestedMax[index4] = Double.isNaN(requestedMax[index4])? conValueD :
                        Math.min(requestedMax[index4], conValueD); 
            }
        }
        
        //invalid user constraints?
        for (int i = 0; i < 4; i++) {
            if (edv[i] == null) 
                continue;
            String tName = edv[i].destinationName();
            if (reallyVerbose) String2.log("  " + tName + " requested min=" + requestedMin[i] + " max=" + requestedMax[i]);
            if (!Double.isNaN(requestedMin[i]) && !Double.isNaN(requestedMax[i])) {
                if (requestedMin[i] > requestedMax[i]) {
                    String minS = i == 3? Calendar2.epochSecondsToIsoStringT(requestedMin[3]) : "" + requestedMin[i];
                    String maxS = i == 3? Calendar2.epochSecondsToIsoStringT(requestedMax[3]) : "" + requestedMax[i];
                    throw new SimpleException("Query error: " +
                        "The requested " + tName + " min=" + minS +
                        " is greater than the requested max=" + maxS + ".");
                }
            }
        }

       
        //use the variable's destinationMin, max?
        if (!useVariablesDestinationMinMax)
            return;

        for (int i = 0; i < 4; i++) {
            if (edv[i] != null) {
                double tMin = edv[i].destinationMin();
                if (Double.isNaN(tMin)) {
                } else {
                    if (Double.isNaN(requestedMin[i]))
                        requestedMin[i] = tMin;
                    else requestedMin[i] = Math.max(tMin, requestedMin[i]);
                }
                double tMax = edv[i].destinationMax();
                if (Double.isNaN(tMax)) {
                } else {
                    if (Double.isNaN(requestedMax[i]))
                        requestedMax[i] = tMax;
                    else requestedMax[i] = Math.min(tMax, requestedMax[i]);
                }
            }
        }
        if (Double.isNaN(requestedMax[3]))   
            requestedMax[3] = Calendar2.gcToEpochSeconds(Calendar2.newGCalendarZulu()) + 
                Calendar2.SECONDS_PER_HOUR; //now + 1 hr     
            //???is this trouble for models which predict future?  (are any models EDDTables?)

        //recheck. If invalid now, it's No Data due to variable's destinationMin/Max
        for (int i = 0; i < 4; i++) {
            if (edv[i] == null) 
                continue;
            String tName = edv[i].destinationName();
            if (reallyVerbose) String2.log("  " + tName + " requested min=" + requestedMin[i] + " max=" + requestedMax[i]);
            if (!Double.isNaN(requestedMin[i]) && !Double.isNaN(requestedMax[i])) {
                if (requestedMin[i] > requestedMax[i]) {
                    String minS = i == 3? Calendar2.epochSecondsToIsoStringT(requestedMin[3]) : "" + requestedMin[i];
                    String maxS = i == 3? Calendar2.epochSecondsToIsoStringT(requestedMax[3]) : "" + requestedMax[i];
                    throw new SimpleException(EDStatic.THERE_IS_NO_DATA +
                        "\n(" + tName + " min=" + minS +
                        " is greater than max=" + maxS + ")");
                }
            }
        }

    }

    /**
     * This formats the resultsVariables and constraints as an OPeNDAP DAP-style query.
     * This does no checking of the validity of the resultsVariable or 
     * constraintVariabe names, so, e.g., axisVariables may be referred to by
     * the soureNames or the destinationNames, so this method can be used in various ways.
     *
     * @param resultsVariables  
     * @param constraintVariables 
     * @param constraintOps   are usually the standard OPERATORS
     *     (or ~=, the non-standard regex op)
     * @param constraintValues 
     * @return the OPeNDAP DAP-style query,
     *   <tt>var1,var2,var3&amp;var4=value4&amp;var5&amp;gt;=value5</tt>.
     *   See parseUserDapQuery.
     */
    public static String formatAsDapQuery(String resultsVariables[],
        String constraintVariables[], String constraintOps[], String constraintValues[]) {

        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < resultsVariables.length; i++) {
            if (i > 0) 
                sb.append(',');
            sb.append(resultsVariables[i]);
        }

        //do I need to put quotes around constraintValues???
        for (int i = 0; i < constraintVariables.length; i++) 
            sb.append("&" + constraintVariables[i] + constraintOps[i] + constraintValues[i]);
        
        return sb.toString();
    }

    /** 
     * This is similar to formatAsDapQuery, but properly SSR.percentEncodes the parts.
     * 
     */
    public static String formatAsPercentEncodedDapQuery(String resultsVariables[],
        String constraintVariables[], String constraintOps[], String constraintValues[]) 
        throws Throwable {

        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < resultsVariables.length; i++) {
            if (i > 0) 
                sb.append(',');
            sb.append(SSR.minimalPercentEncode(resultsVariables[i]));
        }

        for (int i = 0; i < constraintVariables.length; i++) 
            sb.append("&" + 
                SSR.minimalPercentEncode(constraintVariables[i]) + 
                constraintOps[i] + 
                SSR.minimalPercentEncode(constraintValues[i]));
        
        return sb.toString();
    }

    /**
     * This tests if 'value1 op value2' is true.
     * The &lt;=, &gt;=, and = tests are (partly) done with Math2.almostEqual9
     *   so there is a little fudge factor.
     * The =~ regex test must be tested with String testValueOpValue, not here,
     *   because value2 is a regex (not a double).
     * 
     * @param value1
     * @param op one of OPERATORS
     * @param value2
     * @return true if 'value1 op value2' is true.
     *    <br>Tests of "NaN = NaN" will evaluate to true.
     *    <br>Tests of "nonNaN != NaN" will evaluate to true.
     *    <br>All other tests where value1 is NaN or value2 is NaN will evaluate to false.
     * @throws Throwable if trouble (e.g., invalid op)
     */
     public static boolean testValueOpValue(double value1, String op, double value2) 
         throws Throwable {
         //String2.log("testValueOpValue: " + value1 + op + value2);
         if (op.equals("<=")) return value1 <= value2 || Math2.almostEqual(9, value1, value2);
         if (op.equals(">=")) return value1 >= value2 || Math2.almostEqual(9, value1, value2);
         if (op.equals("="))  return (Double.isNaN(value1) && Double.isNaN(value2)) ||
                                     Math2.almostEqual(9, value1, value2);
         if (op.equals("<"))  return value1 < value2;
         if (op.equals(">"))  return value1 > value2;
         if (op.equals("!=")) return Double.isNaN(value1) && Double.isNaN(value2)? false :
                                         value1 != value2;
         //Regex test has to be handled via String testValueOpValue 
         //  if (op.equals(REGEX_OP))  
         throw new SimpleException("Query error: " +
             "Unknown operator=\"" + op + "\".");
     }

    /**
     * This tests if 'value1 op value2' is true.
     * The ops containing with &lt; and &gt; compare value1.toLowerCase()
     * and value2.toLowerCase().
     * 
     * @param value1   (shouldn't be null)
     * @param op one of OPERATORS
     * @param value2   (shouldn't be null)
     * @return true if 'value1 op value2' is true.
     * @throws Throwable if trouble (e.g., invalid op)
     */
     public static boolean testValueOpValue(String value1, String op, String value2) 
         throws Throwable {
         //String2.log("testValueOpValue: " + value1 + op + value2);
         if (op.equals("="))  return value1.equals(value2);
         if (op.equals("!=")) return !value1.equals(value2);
         if (op.equals(REGEX_OP)) return value1.matches(value2);  //regex test

         int t = value1.toLowerCase().compareTo(value2.toLowerCase());
         if (op.equals("<=")) return t <= 0;  
         if (op.equals(">=")) return t >= 0;
         if (op.equals("<"))  return t < 0;
         if (op.equals(">"))  return t > 0;
         throw new SimpleException("Query error: " +
             "Unknown operator=\"" + op + "\".");
     }

    /**
     * This responds to a DAP-style query.
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery the part of the user's request after the '?', still percentEncoded (shouldn't be null).
     * @param outputStreamSource  the source of an outputStream that receives the 
     *     results, usually already buffered.
     * @param dir the directory (on this computer's hard drive) to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param fileTypeName the fileTypeName for the new file  (e.g., .largePng)
     * @throws Throwable if trouble
     */
    public void respondToDapQuery(HttpServletRequest request, String loggedInAs,
        String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource,
        String dir, String fileName, String fileTypeName) throws Throwable {

        if (reallyVerbose) String2.log("\n//** EDDTable.respondToDapQuery..." +
            "\n  datasetID=" + datasetID() +
            "\n  userDapQuery=" + userDapQuery +
            "\n  dir=" + dir +
            "\n  fileName=" + fileName +
            "\n  fileTypeName=" + fileTypeName);
        long makeTime = System.currentTimeMillis();
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);

        //.das, .dds, and .html requests can be handled without getting data 
        if (fileTypeName.equals(".das")) {
            //.das is always the same regardless of the userDapQuery.
            //(That's what THREDDS does -- DAP 2.0 7.2.1 is vague.
            //THREDDs doesn't even object if userDapQuery is invalid.)
            Table table = makeEmptyDestinationTable(requestUrl, "", true); //as if userDapQuery was for everything

            //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit
            table.saveAsDAS(outputStreamSource.outputStream("ISO-8859-1"), 
                SEQUENCE_NAME);
            return;
        }
        if (fileTypeName.equals(".dds")) {
            Table table = makeEmptyDestinationTable(requestUrl, userDapQuery, false);
            //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible common 8bit
            table.saveAsDDS(outputStreamSource.outputStream("ISO-8859-1"),
                SEQUENCE_NAME);
            return;
        }

        if (fileTypeName.equals(".graph")) {
            respondToGraphQuery(request, loggedInAs, requestUrl, userDapQuery, outputStreamSource,
                dir, fileName, fileTypeName);
            return;
        }
        
        if (fileTypeName.equals(".html")) {
            //first
            Table table = makeEmptyDestinationTable(requestUrl, userDapQuery, true);
            //it is important that this use outputStreamSource so stream is compressed (if possible)
            //DAP 2.0 section 3.2.3 says US-ASCII (7bit), so might as well go for compatible unicode
            OutputStream out = outputStreamSource.outputStream("UTF-8");
            Writer writer = new OutputStreamWriter(out, "UTF-8"); 
            writer.write(EDStatic.startHeadHtml(tErddapUrl, "Data Access Form for " + 
                XML.encodeAsXML(title() + " from " + institution())));
            writer.write(EDStatic.standardHead);
            writer.write("\n" + rssHeadLink());
            writer.write("\n</head>\n");
            writer.write(EDStatic.startBodyHtml(loggedInAs));
            writer.write("\n");
            writer.write(HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl));   //this is a link to a script
            writer.write(HtmlWidgets.dragDropScript(EDStatic.imageDirUrl));      //this is a link to a script
            writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
            try {
                writer.write(EDStatic.youAreHereWithHelp(tErddapUrl, dapProtocol, "Data Access Form", 
                    EDStatic.EDDTableDataAccessFormHtml + 
                    "<p>" + EDStatic.EDDTableDownloadDataHtml +
                    "</ol>\n" +
                    "This web page just simplifies the creation of tabledap URLs. " +
                    "<br>If you want, you can create these URLs by hand or have a computer program do it." +
                    "<br>See the 'Just generate the URL' button below, or see the tabledap documentation."));
                writeHtmlDatasetInfo(loggedInAs, writer, false, true, userDapQuery, "");
                writeDapHtmlForm(loggedInAs, userDapQuery, writer);
                writer.write("<hr noshade>\n");
                writer.write("<h2>The Dataset Attribute Structure (.das) for this Dataset</h2>\n" +
                    "<pre>\n");
                table.writeDAS(writer, SEQUENCE_NAME, true); //useful so search engines find all relevant words
                writer.write("</pre>\n");
                writer.write("<br>&nbsp;\n");
                writer.write("<hr noshade>\n");
                writeGeneralDapHtmlInstructions(tErddapUrl, writer, false); 
                writer.write("<br>&nbsp;\n" +
                    "<br><small>" + EDStatic.ProgramName + " Version " + EDStatic.erddapVersion + "</small>\n");
                if (EDStatic.displayDiagnosticInfo) 
                    EDStatic.writeDiagnosticInfoHtml(writer);
            } catch (Throwable t) {
                writer.write(EDStatic.htmlForException(t));
            }
            writer.write(EDStatic.endBodyHtml(tErddapUrl));
            writer.write("\n</html>\n");
            writer.flush(); //essential
            out.close();
            return;
        }
        

        //*** get the data and write to a tableWriter
        TableWriter tableWriter = null;
        TableWriterAllWithMetadata twawm = null;
        if (fileTypeName.equals(".asc")) 
            tableWriter = new TableWriterDodsAscii(outputStreamSource, SEQUENCE_NAME);
        else if (fileTypeName.equals(".csv")) 
            tableWriter = new TableWriterSeparatedValue(outputStreamSource, ", ", true, true);
        else if (fileTypeName.equals(".dods")) 
            tableWriter = new TableWriterDods(outputStreamSource, SEQUENCE_NAME);
        else if (fileTypeName.equals(".geoJson") || 
                 fileTypeName.equals(".json")) {
            //did query include &.jsonp= ?
            String parts[] = getUserQueryParts(userDapQuery);
            String jsonp = String2.stringStartsWith(parts, ".jsonp="); //may be null
            if (jsonp != null) 
                jsonp = SSR.percentDecode(jsonp.substring(7));
            if (fileTypeName.equals(".geoJson"))
                tableWriter = new TableWriterGeoJson(outputStreamSource, jsonp);
            if (fileTypeName.equals(".json"))
                tableWriter = new TableWriterJson(outputStreamSource, jsonp, true);
        } else if (fileTypeName.equals(".htmlTable")) 
            tableWriter = new TableWriterHtmlTable(outputStreamSource, fileName, false, "", "", true, true);
        else if (fileTypeName.equals(".mat")) { 
            twawm = new TableWriterAllWithMetadata(dir, fileName);  //used after getDataForDapQuery below...
            tableWriter = twawm;
        } else if (fileTypeName.equals(".tsv")) 
            tableWriter = new TableWriterSeparatedValue(outputStreamSource, "\t", false, true);
        else if (fileTypeName.equals(".xhtml")) 
            tableWriter = new TableWriterHtmlTable(outputStreamSource, fileName, true, "", "", true, true);

        if (tableWriter != null) {
            tableWriter = encloseTableWriter(dir, fileName, tableWriter, userDapQuery);
            getDataForDapQuery(requestUrl, userDapQuery, tableWriter);

            //special case
            if (fileTypeName.equals(".mat")) 
                saveAsMatlab(outputStreamSource, twawm, datasetID());
            return;
        }

        //*** make a file (then copy it to outputStream)
        //nc files are handled this way because .ncHeader needs to call
        //  NcHelper.dumpString(aRealFile, false). 
        String fileTypeExtension = fileTypeExtension(fileTypeName);
        String fullName = dir + fileName + fileTypeExtension;

        //does the file already exist?
        //if .ncHeader, make sure the .nc file exists (and it is the better file to cache)
        String cacheFullName = fileTypeName.equals(".ncHeader")?  //the only exception there will ever be
            dir + fileName + ".nc" : fullName;
        int random = Math2.random(Integer.MAX_VALUE);

        if (File2.isFile(cacheFullName)) { //don't 'touch()'; files for latest data will change
            if (verbose) String2.log("  reusing cached " + cacheFullName);

        } else if (fileTypeName.equals(".nc") || fileTypeName.equals(".ncHeader")) {
            //if .ncHeader, make sure the .nc file exists (and it is the better file to cache)
            twawm = new TableWriterAllWithMetadata(dir, fileName);
            tableWriter = encloseTableWriter(dir, fileName, twawm, userDapQuery);
            getDataForDapQuery(requestUrl, userDapQuery, tableWriter);  
            saveAsFlatNc(cacheFullName, twawm); //internally, it writes to temp file, then rename to cacheFullName

        } else {
            //all other types
            //create random file; and if error, only random file will be created
            FileOutputStream fos = new FileOutputStream(cacheFullName + random);
            OutputStreamSourceSimple osss = new OutputStreamSourceSimple(fos);
            boolean ok; 
            if (fileTypeName.equals(".kml")) {
                ok = saveAsKml(loggedInAs, requestUrl, userDapQuery, dir, fileName, osss);

            } else if (String2.indexOf(imageFileTypeNames, fileTypeName) >= 0) {
                //pdf and png  //do last so .kml caught above
                ok = saveAsImage(requestUrl, userDapQuery, dir, fileName, osss, fileTypeName);

            } else { 
                fos.close();
                File2.delete(cacheFullName + random);
                throw new SimpleException("Error: " +
                    "fileType=" + fileTypeName + " isn't supported by this dataset.");
            }

            fos.close();
            File2.renameIfNewDoesntExist(cacheFullName + random, cacheFullName); //make available in an instant
            if (!ok) //make eligible to be removed from cache in 5 minutes
                File2.touch(cacheFullName, 
                    Math.max(0, EDStatic.cacheMillis - 5 * Calendar2.MILLIS_PER_MINUTE));
        } 

        //then handle .ncHeader
        if (fileTypeName.equals(".ncHeader")) {
            String error = String2.writeToFile(fullName, 
                NcHelper.dumpString(cacheFullName, false)); //!!!this doesn't do anything to internal " in a String attribute value.
            if (error.length() != 0)
                throw new RuntimeException(error);
        }

        //copy file to outputStream
        //(I delayed getting actual outputStream as long as possible.)
        if (!File2.copy(fullName, outputStreamSource.outputStream(
            fileTypeName.equals(".ncHeader")? "UTF-8" : 
            fileTypeName.equals(".kml")? "UTF-8" : 
            ""))) { 
            //outputStream contentType already set,
            //so I can't go back to html and display error message
            //note than the message is thrown if user cancels the transmission; so don't email to me
            String2.log("Error while transmitting " + fileName + fileTypeExtension);
        }

        //done
        if (reallyVerbose) String2.log(
            "\n\\\\** EDDTable.respondToDapQuery finished successfully. TIME=" +
            (System.currentTimeMillis() - makeTime));

    }

    /**
     * If a part of the userDapQuery is distinct(), orderBy("..."),
     * this wraps tableWriter in (e.g.) TableWriterDistinct
     * in the order they occur in userDapQuery.
     *
     * @param dir a private cache directory for storing the intermediate files
     * @param fileNameNoExt is the fileName without dir or extension (used as basis for temp files).
     * @param tableWriter the final tableWriter 
     * @param userDapQuery
     * @return the (possibly) wrapped tableWriter
     */
    protected TableWriter encloseTableWriter(String dir, String fileName, 
        TableWriter tableWriter, String userDapQuery) throws Throwable {
        
        //work backwards through parts so tableWriters are added in correct order
        String[] parts = getUserQueryParts(userDapQuery);
        for (int part = parts.length - 1; part >= 0; part--) {
            String p = parts[part];
            if (p.equals("distinct()")) {
                tableWriter = new TableWriterDistinct(dir, fileName, tableWriter);
            } else if (p.startsWith("orderBy(\"") && p.endsWith("\")")) {
                TableWriterOrderBy twob = new TableWriterOrderBy(dir, fileName, tableWriter, 
                    p.substring(9, p.length() - 2));
                tableWriter = twob;
                //minimal test: ensure orderBy columns are valid column names
                for (int ob = 0; ob < twob.orderBy.length; ob++) {
                    if (String2.indexOf(dataVariableDestinationNames(), twob.orderBy[ob]) < 0)
                        throw new SimpleException("Query error: " +
                            "'orderBy' variable=" + twob.orderBy[ob] + " isn't in the dataset.");
                }
            } else if (p.startsWith("orderByMax(\"") && p.endsWith("\")")) {
                TableWriterOrderByMax twobm = new TableWriterOrderByMax(dir, fileName, tableWriter, 
                    p.substring(12, p.length() - 2));
                tableWriter = twobm;
                //minimal test: ensure orderBy columns are valid column names
                for (int ob = 0; ob < twobm.orderBy.length; ob++) {
                    if (String2.indexOf(dataVariableDestinationNames(), twobm.orderBy[ob]) < 0)
                        throw new SimpleException("Query error: " +
                            "'orderByMax' variable=" + twobm.orderBy[ob] + " isn't in the dataset.");
                }
            }
        }
        return tableWriter;
    }

    /**
     * This makes an empty table (with destination columns, but without rows) 
     * corresponding to the request.
     *
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery the part after the '?', still percentEncoded (shouldn't be null).
     * @param withAttributes
     * @return an empty table (with columns, but without rows) 
     *    corresponding to the request
     * @throws Throwable if trouble
     */
    protected Table makeEmptyDestinationTable(String requestUrl, String userDapQuery, 
        boolean withAttributes) throws Throwable {
        if (reallyVerbose) String2.log("  makeEmptyDestinationTable...");

        //pick the query apart
        StringArray resultsVariables    = new StringArray();
        StringArray constraintVariables = new StringArray();
        StringArray constraintOps       = new StringArray();
        StringArray constraintValues    = new StringArray();
        parseUserDapQuery(userDapQuery, resultsVariables,
            constraintVariables, constraintOps, constraintValues, false);

        //make the table
        //attributes are added in standardizeResultsTable
        Table table = new Table();        
        if (withAttributes) {
            table.globalAttributes().set(combinedGlobalAttributes); //make a copy
            //fix up global attributes  (always to a local COPY of global attributes)
            EDD.addToHistory(table.globalAttributes(), sourceUrl());
            EDD.addToHistory(table.globalAttributes(), 
                EDStatic.baseUrl + requestUrl); //userDapQuery is irrelevant
        }

        //add the columns
        for (int col = 0; col < resultsVariables.size(); col++) {
            int dv = String2.indexOf(dataVariableDestinationNames(), resultsVariables.get(col));
            EDV edv = dataVariables[dv]; 
            Attributes atts = new Attributes();
            if (withAttributes) 
                atts.set(edv.combinedAttributes()); //make a copy
            table.addColumn(col, edv.destinationName(), 
                PrimitiveArray.factory(edv.destinationDataTypeClass(), 1, false),
                atts); 
        }

        //can't setActualRangeAndBoundingBox because no data
        //so remove actual_range and bounding box attributes??? 
        //  no, they are dealt with in standardizeResultsTable

        if (reallyVerbose) String2.log("  makeEmptyDestinationTable done.");
        return table;
    }


    /**
     * This makes a .kml file.
     * The userDapQuery must include the EDV.LON_NAME and EDV.LAT_NAME columns 
     * (and preferably also EDV.ALT_NAME and EDV.TIME_NAME column) in the results variables.
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl 
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery the part after the '?', still percentEncoded (shouldn't be null).
     * @param dir the directory (on this computer's hard drive) to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param outputStreamSource
     * @return true of written ok; false if exception occurred (and written on image)
     * @throws Throwable if trouble
     */
    protected boolean saveAsKml(String loggedInAs,
        String requestUrl, String userDapQuery,
        String dir, String fileName,
        OutputStreamSource outputStreamSource) throws Throwable {

        //before any work is done, 
        //  ensure LON_NAME and LAT_NAME are among resultsVariables
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        StringArray resultsVariables    = new StringArray();
        StringArray constraintVariables = new StringArray();
        StringArray constraintOps       = new StringArray();
        StringArray constraintValues    = new StringArray();
        parseUserDapQuery(userDapQuery, resultsVariables,
            constraintVariables, constraintOps, constraintValues, false);
        Test.ensureTrue(resultsVariables.indexOf(EDV.LON_NAME) >= 0, ".kml requests must include the " + EDV.LON_NAME + " variable.");
        Test.ensureTrue(resultsVariables.indexOf(EDV.LAT_NAME) >= 0, ".kml requests must include the " + EDV.LAT_NAME + " variable.");

        //get the table with all the data
        TableWriterAllWithMetadata twawm = new TableWriterAllWithMetadata(dir, fileName);
        TableWriter tableWriter = encloseTableWriter(dir, fileName, twawm, userDapQuery);
        getDataForDapQuery(requestUrl, userDapQuery, tableWriter);
        Table table = twawm.cumulativeTable();
        twawm.releaseResources();
        table.convertToStandardMissingValues(); //so stored as NaNs

        //double check that lon and lat were found
        int lonCol  = table.findColumnNumber(EDV.LON_NAME);
        int latCol  = table.findColumnNumber(EDV.LAT_NAME);
        int altCol  = table.findColumnNumber(EDV.ALT_NAME); 
        int timeCol = table.findColumnNumber(EDV.TIME_NAME);
        Test.ensureTrue(lonCol >= 0, ".kml requests must include the " + EDV.LON_NAME + " variable.");
        Test.ensureTrue(latCol >= 0, ".kml requests must include the " + EDV.LAT_NAME + " variable.");

        //remember this may be many stations one time, or one station many times, or many/many
        //sort table by lat, lon, depth, then time (if possible)
        if (altCol >= 0 && timeCol >= 0)
            table.sort(new int[]{lonCol, latCol, altCol, timeCol}, new boolean[]{true, true, true, true});
        else if (timeCol >= 0)
            table.sort(new int[]{lonCol, latCol, timeCol}, new boolean[]{true, true, true});
        else if (altCol >= 0)
            table.sort(new int[]{lonCol, latCol, altCol},  new boolean[]{true, true, true});
        else 
            table.sort(new int[]{lonCol, latCol}, new boolean[]{true, true});
        //String2.log(table.toString("row", 10));

        //get lat and lon range (needed to create icon size and ensure there is data to be plotted)
        double minLon = Double.NaN, maxLon = Double.NaN, minLat = Double.NaN, maxLat = Double.NaN;
        if (lonCol >= 0) {
            double stats[] = table.getColumn(lonCol).calculateStats();
            minLon = stats[PrimitiveArray.STATS_MIN]; 
            maxLon = stats[PrimitiveArray.STATS_MAX];
        }
        if (latCol >= 0) {
            double stats[] = table.getColumn(latCol).calculateStats();
            minLat = stats[PrimitiveArray.STATS_MIN]; 
            maxLat = stats[PrimitiveArray.STATS_MAX];
        }
        if (Double.isNaN(minLon) || Double.isNaN(minLat))
            throw new SimpleException(EDStatic.THERE_IS_NO_DATA);
        double lonRange = maxLon - minLon;
        double latRange = maxLat - minLat;
        double maxRange = Math.max(lonRange, latRange);

        //get time range and prep moreTime constraints

        String moreTime = "";
        double minTime = Double.NaN, maxTime = Double.NaN;
        if (timeCol >= 0) {
            //time is in the response
            double stats[] = table.getColumn(timeCol).calculateStats();
            minTime = stats[PrimitiveArray.STATS_MIN]; 
            maxTime = stats[PrimitiveArray.STATS_MAX];
            if (!Double.isNaN(minTime)) { //there are time values
                //at least a week
                double tMinTime = Math.min(minTime, maxTime - 7 * Calendar2.SECONDS_PER_DAY);
                moreTime = 
                    "&" + SSR.minimalPercentEncode(EDV.TIME_NAME + ">=" + Calendar2.epochSecondsToIsoStringT(tMinTime)) +  
                    "&" + SSR.minimalPercentEncode(EDV.TIME_NAME + "<=" + Calendar2.epochSecondsToIsoStringT(maxTime));
            }
        } else {
            //look for time in constraints
            for (int c = 0; c < constraintVariables.size(); c++) {
                if (EDV.TIME_NAME.equals(constraintVariables.get(c))) {
                    double tTime = String2.parseDouble(constraintValues.get(c));
                    char ch1 = constraintOps.get(c).charAt(0);
                    if (ch1 == '>' || ch1 == '=') 
                        minTime = Double.isNaN(minTime)? tTime : Math.min(minTime, tTime);
                    if (ch1 == '<' || ch1 == '=')
                        maxTime = Double.isNaN(maxTime)? tTime : Math.max(maxTime, tTime);
                }
            }
            if (!Double.isNaN(minTime)) 
                moreTime = "&" + 
                    SSR.minimalPercentEncode(EDV.TIME_NAME + ">=" + 
                        Calendar2.epochSecondsToIsoStringT(minTime - 7 * Calendar2.SECONDS_PER_DAY));
            if (!Double.isNaN(maxTime)) 
                moreTime = "&" + 
                    SSR.minimalPercentEncode(EDV.TIME_NAME + "<=" + 
                        Calendar2.epochSecondsToIsoStringT(maxTime + 7 * Calendar2.SECONDS_PER_DAY));
        }

        //Google Earth .kml
        //(getting the outputStream was delayed until actually needed)
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
            outputStreamSource.outputStream("UTF-8"), "UTF-8"));

        //collect the units
        String columnUnits[] = new String[table.nColumns()];
        boolean columnIsString[] = new boolean[table.nColumns()];
        boolean columnIsTimeStamp[] = new boolean[table.nColumns()];
        for (int col = 0; col < table.nColumns(); col++) {
            String units = table.columnAttributes(col).getString("units");
            columnUnits[col] = (units == null || units.equals(EDV.UNITLESS))? "" :
                " " + units;
            //hasTimeUnits? actually, if timeStamp, units will be TIME_UNITS
            columnIsTimeStamp[col] = EDVTimeStamp.hasTimeUnits(units); 
            columnIsString[col] = table.getColumn(col) instanceof StringArray;
        }

        //based on kmz example from http://www.coriolis.eu.org/cdc/google_earth.htm
        //see copy in bob's c:/programs/kml/SE-LATEST-MONTH-STA.kml
        //kml docs: http://earth.google.com/kml/kml_tags.html
        //CDATA is necessary for url's with queries
        //kml/description docs recommend \n<br />
        String courtesy = institution().length() == 0? "" : 
            "Data courtesy of: " + institution();
        double iconSize = maxRange > 90? 1.2 : maxRange > 45? 1.0 : maxRange > 20? .8 : 
            maxRange > 10? .6 : maxRange > 5 ? .5 : .4;
        writer.write(
            "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
            "<kml xmlns=\"http://www.opengis.net/kml/2.2\">\n" +
            "<Document>\n" +
            //human-friendly, but descriptive, <name>
            //name is used as link title -- leads to <description> 
            "  <name>" + XML.encodeAsXML(title()) + "</name>\n" +
            //<description> appears in balloon
            "  <description><![CDATA[" +
            XML.encodeAsXML(courtesy) + "\n<br />" +
            String2.replaceAll(XML.encodeAsXML(summary()), "\n", "\n<br />") +
            //link to download this dataset
            "\n<br />" +
            "<a href=\"" + tErddapUrl + 
                "/tabledap/" + //don't use \n for the following lines
            datasetID() + ".html?" + userDapQuery + //already percentEncoded; XML.encodeAsXML isn't ok
            "\">View/download more data from this dataset.</a>\n" +
            "    ]]></description>\n" +
            "  <open>1</open>\n" +
            "  <Style id=\"BUOY ON\">\n" +
            "    <IconStyle>\n" +
            "      <color>ff0099ff</color>\n" + //abgr   orange
            "      <scale>" + iconSize + "</scale>\n" +
            "      <Icon>\n" +
            "        <href>root://icons/palette-2.png</href>\n" +
            "        <x>64</x>\n" +
            "        <y>128</y>\n" +
            "        <w>32</w>\n" +
            "        <h>32</h>\n" +
            "      </Icon>\n" +
            "    </IconStyle>\n" +
            "  </Style>\n" +
            "  <Style id=\"BUOY OUT\">\n" +
            "    <IconStyle>\n" +
            "      <color>ff0099ff</color>\n" +
            "        <scale>" + iconSize + "</scale>\n" +
            "        <Icon>\n" +
            "          <href>root://icons/palette-2.png</href>\n" +
            "          <x>64</x>\n" +
            "          <y>128</y>\n" +
            "          <w>32</w>\n" +
            "          <h>32</h>\n" +
            "        </Icon>\n" +
            "      </IconStyle>\n" +
            "    <LabelStyle><scale>0</scale></LabelStyle>\n" +
            "  </Style>\n" +
            "  <StyleMap id=\"BUOY\">\n" +
            "    <Pair><key>normal</key><styleUrl>#BUOY OUT</styleUrl></Pair>\n" +
            "    <Pair><key>highlight</key><styleUrl>#BUOY ON</styleUrl></Pair>\n" +
            "  </StyleMap>\n");
        
        //just one link for each station (same lat,lon/depth):
        //LON   LAT DEPTH   TIME    ID  WTMP
        //-130.36   42.58   0.0 1.1652336E9 NDBC 46002 met  13.458333174387613
        int nRows = table.nRows();  //there must be at least 1 row
        int startRow = 0;
        double startLon = table.getNiceDoubleData(lonCol, startRow);
        double startLat = table.getNiceDoubleData(latCol, startRow);
        for (int row = 1; row <= nRows; row++) { //yes, 1...n, since looking at previous row
            //look for a change in lastLon/Lat
            if (row == nRows || 
                startLon != table.getNiceDoubleData(lonCol, row) ||
                startLat != table.getNiceDoubleData(latCol, row)) {

                if (!Double.isNaN(startLon) && !Double.isNaN(startLat)) {
                                
                    //make a placemark for this station
                    double startLon180 = Math2.anglePM180(startLon); 
                    writer.write(
                        "  <Placemark>\n" +
                        "    <name>" + 
                            "Lat=" + String2.genEFormat10(startLat) +
                            ", Lon=" + String2.genEFormat10(startLon180) +
                            "</name>\n" +
                        "    <description><![CDATA[" + 
                        //kml/description docs recommend \n<br />
                        XML.encodeAsXML(title) +
                        "\n<br />" + XML.encodeAsXML(courtesy));

                    //if timeCol exists, find last row with valid time
                    //This solves problem with dapper data (last row for each station has just NaNs)
                    int displayRow = row - 1;
                    if (timeCol >= 0) {
                        while (displayRow - 1 >= startRow && 
                               Double.isNaN(table.getDoubleData(timeCol, displayRow))) //if displayRow is NaN...
                            displayRow--;
                    }

                    //display the last row of data  (better than nothing)
                    for (int col = 0; col < table.nColumns(); col++) {
                        double td = table.getNiceDoubleData(col, displayRow);
                        String ts = table.getStringData(col, displayRow);
                        writer.write("\n<br />" + 
                            XML.encodeAsXML(table.getColumnName(col) + " = " +
                                (columnIsTimeStamp[col] ? 
                                    (Double.isNaN(td)? "" : Calendar2.epochSecondsToIsoStringT(td) + "Z") :
                                 columnIsString[col]? ts :
                                 (Double.isNaN(td)? "NaN" : ts) + columnUnits[col])));
                    }
                    writer.write(
                        "\n<br /><a href=\"" + tErddapUrl + 
                            "/tabledap/" + //don't use \n for the following lines
                            datasetID() + ".htmlTable?" + 
                                //was SSR.minimalPercentEncode   XML.encodeAsXML isn't ok
                                //ignore userDapQuery    
                                //get just this station, all variables, at least 7 days                            
                                moreTime +  //already percentEncoded
                                //some data sources like dapper don't respond to lon=startLon lat=startLat
                                "&" + SSR.minimalPercentEncode(EDV.LON_NAME + ">" + (startLon - .01)) +  //not startLon180
                                "&" + SSR.minimalPercentEncode(EDV.LON_NAME + "<" + (startLon + .01)) +  
                                "&" + SSR.minimalPercentEncode(EDV.LAT_NAME + ">" + (startLat - .01)) +
                                "&" + SSR.minimalPercentEncode(EDV.LAT_NAME + "<" + (startLat + .01)) +
                        "\">View tabular data for this location.</a>\n" +
                        "\n<br /><a href=\"" + tErddapUrl + "/tabledap/" + //don't use \n for the following lines
                            datasetID() + ".html?" + userDapQuery + //already percentEncoded.  XML.encodeAsXML isn't ok 
                        "\">View/download more data from this dataset.</a>\n" +
                        "]]></description>\n" +
                        "    <styleUrl>#BUOY</styleUrl>\n" +
                        "    <Point>\n" +
                        "      <coordinates>" + 
                                   startLon180 + "," +
                                   startLat + 
                                "</coordinates>\n" +
                        "    </Point>\n" +
                        "  </Placemark>\n");
                }

                //reset startRow...
                startRow = row;
                if (startRow == nRows) {
                    //assist with LookAt
                    //it has trouble with 1 point: zoom in forever
                    //  and if lon range crossing dateline
                    double tMaxRange = Math.min(90, maxRange); //90 is most you can comfortably see, and useful fudge
                    tMaxRange = Math.max(2, tMaxRange);  //now it's 2 .. 90 
                    //fudge for smaller range
                    if (tMaxRange < 45) tMaxRange *= 1.5;
                    double eyeAt = 14.0e6 * tMaxRange / 90.0; //meters, 14e6 shows whole earth (~90 deg comfortably)  
                    double lookAtX = Math2.anglePM180((minLon + maxLon) / 2);
                    double lookAtY = (minLat + maxLat) / 2;
                    if (reallyVerbose) String2.log("KML minLon=" + minLon + " maxLon=" + maxLon + 
                        " minLat=" + minLat + " maxLat=" + maxLat + 
                        "\n  maxRange=" + maxRange + " tMaxRange=" + tMaxRange +
                        "\n  lookAtX=" + lookAtX + 
                        "  lookAtY=" + lookAtY + "  eyeAt=" + eyeAt); 
                    writer.write(
                        "  <LookAt>\n" +
                        "    <longitude>" + lookAtX + "</longitude>\n" +
                        "    <latitude>" + lookAtY + "</latitude>\n" +
                        "    <range>" + eyeAt + "</range>\n" + //meters  
                        "  </LookAt>\n");
                } else {
                    startLon = table.getNiceDoubleData(lonCol, startRow);
                    startLat = table.getNiceDoubleData(latCol, startRow);
                    minLon = Double.isNaN(minLon)? startLon : 
                             Double.isNaN(startLon)? minLon : Math.min(minLon, startLon);
                    maxLon = Double.isNaN(maxLon)? startLon : 
                             Double.isNaN(startLon)? maxLon : Math.max(maxLon, startLon);
                    minLat = Double.isNaN(minLat)? startLat : 
                             Double.isNaN(startLat)? minLat : Math.min(minLat, startLat);
                    maxLat = Double.isNaN(maxLat)? startLat : 
                             Double.isNaN(startLat)? maxLat : Math.max(maxLat, startLat);
                }

            } //end processing change in lon, lat, or depth
        } //end row loop

        //end of kml file
        writer.write(
            getKmlIconScreenOverlay() +
            "  </Document>\n" +
            "</kml>\n");
        writer.flush(); //essential
        return true;

    }

    /**
     * This saves the data in the table to the outputStream as an image.
     * If table.getColumnName(0)=LON_NAME and table.getColumnName(0)=LAT_NAME,
     * this plots the data on a map.
     * Otherwise, this makes a graph with x=col(0) and y=col(1).
     *
     * @param requestUrl
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery the part after the '?', still percentEncoded (shouldn't be null).
     * @param dir the directory (on this computer's hard drive) to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param outputStreamSource
     * @param fileTypeName
     * @return true of written ok; false if exception occurred (and written on image)
     * @throws Throwable if trouble
     */
    public boolean saveAsImage(String requestUrl, String userDapQuery,
        String dir, String fileName, 
        OutputStreamSource outputStreamSource, String fileTypeName) throws Throwable {

        if (reallyVerbose) String2.log("  EDDTable.saveAsImage query=" + userDapQuery);
        long time = System.currentTimeMillis();
        if (debugMode) String2.log("saveAsImage 1");
        //imageFileTypes
        //determine the size
        int sizeIndex = 
            fileTypeName.startsWith(".small")? 0 :
            fileTypeName.startsWith(".medium")? 1 :
            fileTypeName.startsWith(".large")? 2 : 1;
        boolean pdf = fileTypeName.toLowerCase().endsWith("pdf");
        boolean png = fileTypeName.toLowerCase().endsWith("png");
        if (!pdf && !png) 
            throw new SimpleException("Error: " +
                "Unexpected image type=" + fileTypeName);
        Object pdfInfo[] = null;
        BufferedImage bufferedImage = null;
        Graphics2D g2 = null;
        int width  = pdf? EDStatic.pdfWidths[ sizeIndex] : EDStatic.imageWidths[sizeIndex];
        int height = pdf? EDStatic.pdfHeights[sizeIndex] : EDStatic.imageHeights[sizeIndex];
        boolean ok = true;
       
        try {
            //get the user-specified resultsVariables
            StringArray resultsVariables    = new StringArray();
            StringArray constraintVariables = new StringArray();
            StringArray constraintOps       = new StringArray();
            StringArray constraintValues    = new StringArray();
            parseUserDapQuery(userDapQuery, resultsVariables,
                constraintVariables, constraintOps, constraintValues, false);
            if (debugMode) String2.log("saveAsImage 2");

            EDV xVar, yVar, zVar = null, tVar = null; 
            if (resultsVariables.size() < 2)
                throw new SimpleException("Query error: " +
                    "The query must include at least 2 results variables (for x and y on the graph).");
            //if lon and lat requested, it's a map
            boolean isMap =
                resultsVariables.indexOf(EDV.LON_NAME) >= 0 &&
                resultsVariables.indexOf(EDV.LON_NAME) < 2  &&
                resultsVariables.indexOf(EDV.LAT_NAME) >= 0 &&
                resultsVariables.indexOf(EDV.LAT_NAME) < 2;
            if (isMap) {
                xVar = dataVariables[lonIndex]; 
                yVar = dataVariables[latIndex]; 
            } else {
                //xVar,yVar are 1st and 2nd request variables
                xVar = findVariableByDestinationName(resultsVariables.get(0)); 
                yVar = findVariableByDestinationName(resultsVariables.get(1)); 
                if (yVar instanceof EDVTimeStamp) { //prefer time on x axis
                    EDV edv = xVar; xVar = yVar; yVar = edv;
                }
            }
            if (resultsVariables.size() >= 3)
                zVar = findVariableByDestinationName(resultsVariables.get(2));
            if (resultsVariables.size() >= 4)
                tVar = findVariableByDestinationName(resultsVariables.get(3));

            //get the table with all the data
            //errors here will be caught below
            TableWriterAllWithMetadata twawm = new TableWriterAllWithMetadata(dir, fileName);
            TableWriter tableWriter = encloseTableWriter(dir, fileName, twawm, userDapQuery);
            getDataForDapQuery(requestUrl, userDapQuery, tableWriter);
            Table table = twawm.cumulativeTable();
            twawm.releaseResources();
            table.convertToStandardMissingValues();
            if (debugMode) String2.log("saveAsImage 3");

            //units
            int xColN = table.findColumnNumber(xVar.destinationName());
            int yColN = table.findColumnNumber(yVar.destinationName());
            int zColN = zVar == null? -1 : table.findColumnNumber(zVar.destinationName());
            int tColN = tVar == null? -1 : table.findColumnNumber(tVar.destinationName());
            String xUnits = xVar instanceof EDVTimeStamp? "UTC" : xVar.units();
            String yUnits = yVar instanceof EDVTimeStamp? "UTC" : yVar.units();
            String zUnits = zVar == null? null : zVar instanceof EDVTimeStamp? "UTC" : zVar.units();
            String tUnits = tVar == null? null : tVar instanceof EDVTimeStamp? "UTC" : tVar.units();
            xUnits = xUnits == null? "" : " (" + xUnits + ")";
            yUnits = yUnits == null? "" : " (" + yUnits + ")";
            zUnits = zUnits == null? "" : " (" + zUnits + ")";
            tUnits = tUnits == null? "" : " (" + tUnits + ")";

            //extract optional .graphicsSettings from userDapQuery
            //  xRange, yRange, color and colorbar information
            //  title2 -- a prettified constraint string 
            boolean drawLines = false;
            boolean drawLinesAndMarkers = false;
            boolean drawMarkers = true;
            boolean drawSticks  = false;
            boolean drawVectors = false;
            int markerType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE;
            int markerSize = GraphDataLayer.MARKER_SIZE_SMALL;
            Color color = Color.black;

            //set colorBar defaults via zVar attributes
            String ts;
            ts = zVar == null? null : zVar.combinedAttributes().getString("colorBarPalette");
            String palette = ts == null? "" : ts;
            ts = zVar == null? null : zVar.combinedAttributes().getString("colorBarScale");
            String scale = ts == null? "Linear" : ts;
            double paletteMin = zVar == null? Double.NaN : zVar.combinedAttributes().getDouble("colorBarMinimum");
            double paletteMax = zVar == null? Double.NaN : zVar.combinedAttributes().getDouble("colorBarMaximum");
            int nSections = -1;
            ts = zVar == null? null : zVar.combinedAttributes().getString("colorBarContinuous");
            boolean continuous = String2.parseBoolean(ts); //defaults to true

            CompoundColorMap colorMap = null;
            double xMin = Double.NaN, xMax = Double.NaN, yMin = Double.NaN, yMax = Double.NaN;
            double fontScale = 1, vectorStandard = Double.NaN;
            boolean drawLandAsMask = true;
            StringBuffer title2 = new StringBuffer();
            String ampParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
            for (int ap = 0; ap < ampParts.length; ap++) {
                String ampPart = ampParts[ap];
                if (debugMode) String2.log("saveAsImage 4 " + ap);

                //.colorBar defaults: palette=""|continuous=C|scale=Linear|min=NaN|max=NaN|nSections=-1
                if (ampPart.startsWith(".colorBar=")) {
                    String pParts[] = String2.split(ampPart.substring(10), '|');
                    if (pParts == null) pParts = new String[0];
                    if (pParts.length > 0 && pParts[0].length() > 0) palette    = pParts[0];
                    if (pParts.length > 1 && pParts[1].length() > 0) continuous = !pParts[1].toLowerCase().startsWith("d");
                    if (pParts.length > 2 && pParts[2].length() > 0) scale      = pParts[2];
                    if (pParts.length > 3 && pParts[3].length() > 0) paletteMin = String2.parseDouble(pParts[3]);
                    if (pParts.length > 4 && pParts[4].length() > 0) paletteMax = String2.parseDouble(pParts[4]);
                    if (pParts.length > 5 && pParts[5].length() > 0) nSections  = String2.parseInt(pParts[5]);
                    if (String2.indexOf(EDStatic.palettes, palette) < 0) palette   = "";
                    if (String2.indexOf(EDV.VALID_SCALES, scale) < 0)    scale     = "Linear";
                    if (nSections < 0 || nSections >= 100)               nSections = -1;
                    if (reallyVerbose)
                        String2.log(".colorBar palette=" + palette +
                            " continuous=" + continuous +
                            " scale=" + scale +
                            " min=" + paletteMin + " max=" + paletteMax +
                            " nSections=" + nSections);                                

                //.color
                } else if (ampPart.startsWith(".color=")) {
                    int iColor = String2.parseInt(ampPart.substring(7));
                    if (iColor < Integer.MAX_VALUE) {
                        color = new Color(iColor);
                        if (reallyVerbose)
                            String2.log(".color=0x" + Integer.toHexString(iColor));
                    }

                //.draw 
                } else if (ampPart.startsWith(".draw=")) {
                    String tDraw = ampPart.substring(6);
                    //make all false
                    drawLines = false;
                    drawLinesAndMarkers = false;
                    drawMarkers = false;
                    drawSticks  = false;
                    drawVectors = false;
                    //set one option to true
                    if (tDraw.equals("sticks") && zVar != null) {drawSticks = true; isMap = false;}
                    else if (isMap && tDraw.equals("vectors") && zVar != null && tVar != null) drawVectors = true;
                    else if (tDraw.equals("lines")) drawLines = true;
                    else if (tDraw.equals("linesAndMarkers")) drawLinesAndMarkers = true;
                    else drawMarkers = true; //default

                //.font
                } else if (ampPart.startsWith(".font=")) {
                    String pParts[] = String2.split(ampPart.substring(6), '|'); //subparts may be ""; won't be null
                    if (pParts == null) pParts = new String[0];
                    if (pParts.length > 0) fontScale = String2.parseDouble(pParts[0]);
                    fontScale = Double.isNaN(fontScale)? 1 : fontScale < 0.1? 0.1 : fontScale > 10? 10 : fontScale;
                    if (reallyVerbose)
                        String2.log(".font= scale=" + fontScale);

                //.land 
                } else if (ampPart.startsWith(".land=")) {
                    String gt = ampPart.substring(6);
                    drawLandAsMask = !gt.equals("under");

                //.marker
                } else if (ampPart.startsWith(".marker=")) {
                    String pParts[] = String2.split(ampPart.substring(8), '|'); //subparts may be ""; won't be null
                    if (pParts == null) pParts = new String[0];
                    if (pParts.length > 0) markerType = String2.parseInt(pParts[0]);
                    if (pParts.length > 1) markerSize = String2.parseInt(pParts[1]);
                    if (markerType < 0 || markerType >= GraphDataLayer.MARKER_TYPES.length) 
                        markerType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE;
                    if (markerSize < 1 || markerSize > 50) markerSize = GraphDataLayer.MARKER_SIZE_SMALL;
                    if (reallyVerbose)
                        String2.log(".marker= type=" + markerType + " size=" + markerSize);


                //.vec
                } else if (ampPart.startsWith(".vec=")) {
                    vectorStandard = String2.parseDouble(ampPart.substring(5));
                    if (reallyVerbose)
                        String2.log(".vec " + vectorStandard);

                //.xRange   (supported, but currently not created by the Make A Graph form)
                //  prefer set via xVar constratints
                } else if (ampPart.startsWith(".xRange=")) {
                    String pParts[] = String2.split(ampPart.substring(8), '|');
                    if (pParts.length > 0) xMin = String2.parseDouble(pParts[0]);
                    if (pParts.length > 1) xMax = String2.parseDouble(pParts[1]);
                    if (reallyVerbose)
                        String2.log(".xRange min=" + xMin + " max=" + xMax);

                //.yRange   (supported, but currently not created by the Make A Graph form)
                //  prefer set via yVar constratints
                } else if (ampPart.startsWith(".yRange=")) {
                    String pParts[] = String2.split(ampPart.substring(8), '|');
                    if (pParts.length > 0) yMin = String2.parseDouble(pParts[0]);
                    if (pParts.length > 1) yMax = String2.parseDouble(pParts[1]);
                    if (reallyVerbose)
                        String2.log(".yRange min=" + yMin + " max=" + yMax);
    
                //ignore any unrecognized .something 
                } else if (ampPart.startsWith(".")) {

                //don't do anything with list of vars
                } else if (ap == 0) {

                //x and yVar constraints 
                } else if (ampPart.startsWith(xVar.destinationName()) || //x,y axis ranges indicate x,y var constraints
                           ampPart.startsWith(yVar.destinationName())) {
                    //don't include in title2  (better to leave in, but space often limited)

                //constraints on other vars
                } else {
                    //add to title2
                    if (title2.length() > 0) 
                        title2.append(", ");
                    title2.append(ampPart);
                }
            }
            if (title2.length() > 0) {
                title2.insert(0, "(");
                title2.append(")");
            }
            if (debugMode) String2.log("saveAsImage 5");

            //make colorMap (or set to null)
            if (drawLines || drawSticks || drawVectors) 
                colorMap = null;
            if ((drawLinesAndMarkers || drawMarkers) && zVar == null) 
                colorMap = null;
            if (drawVectors && zVar != null && Double.isNaN(vectorStandard)) {
                double zStats[] = table.getColumn(zColN).calculateStats();
                if (zStats[PrimitiveArray.STATS_N] == 0) {
                    vectorStandard = 1;
                } else {
                    double minMax[] = Math2.suggestLowHigh(0, 
                        Math.max(Math.abs(zStats[PrimitiveArray.STATS_MIN]),
                                 Math.abs(zStats[PrimitiveArray.STATS_MAX])));
                    vectorStandard = minMax[1];
                }
            }
            if ((drawLinesAndMarkers || drawMarkers) && zVar != null && colorMap == null) {
                if ((palette.length() == 0 || Double.isNaN(paletteMin) || Double.isNaN(paletteMax)) && 
                    zColN >= 0) {
                    //set missing items based on z data
                    double zStats[] = table.getColumn(zColN).calculateStats();
                    if (zStats[PrimitiveArray.STATS_N] > 0) {
                        double minMax[];
                        if (zVar instanceof EDVTimeStamp) {
                            //???I think this is too crude. Smarter code elsewhere? Or handled by compoundColorMap?
                            double r20 = 
                                (zStats[PrimitiveArray.STATS_MAX] -
                                 zStats[PrimitiveArray.STATS_MIN]) / 20;
                            minMax = new double[]{
                                zStats[PrimitiveArray.STATS_MIN] - r20,
                                zStats[PrimitiveArray.STATS_MAX] + r20};
                        } else {
                            minMax = Math2.suggestLowHigh(
                                zStats[PrimitiveArray.STATS_MIN],
                                zStats[PrimitiveArray.STATS_MAX]);
                        }

                        if (palette.length() == 0) {
                            if (minMax[1] >= minMax[0] / -2 && 
                                minMax[1] <= minMax[0] * -2) {
                                double td = Math.max(minMax[1], -minMax[0]);
                                minMax[0] = -td;
                                minMax[1] = td;
                                palette = "BlueWhiteRed";
                            } else {
                                palette = "Rainbow";
                            }
                        }
                        if (Double.isNaN(paletteMin)) 
                            paletteMin = minMax[0];
                        if (Double.isNaN(paletteMax)) 
                            paletteMax = minMax[1];
                    }                                             
                }
                if (palette.length() == 0 || Double.isNaN(paletteMin) || Double.isNaN(paletteMax)) {
                    String2.log("Warning in EDDTable.saveAsImage: NaNs not allowed (zVar has no numeric data):" +
                        " palette=" + palette +
                        " paletteMin=" + paletteMin +
                        " paletteMax=" + paletteMax);
                } else {
                    if (reallyVerbose)
                        String2.log("create colorBar palette=" + palette +
                            " continuous=" + continuous +
                            " scale=" + scale +
                            " min=" + paletteMin + " max=" + paletteMax +
                            " nSections=" + nSections);                                
                    if (zVar instanceof EDVTimeStamp)
                        colorMap = new CompoundColorMap(
                        EDStatic.fullPaletteDirectory, palette, false, //false= data is seconds
                        paletteMin, paletteMax, nSections, 
                        continuous, EDStatic.fullCptCacheDirectory);
                    else colorMap = new CompoundColorMap(
                        EDStatic.fullPaletteDirectory, palette, scale, 
                        paletteMin, paletteMax, nSections, 
                        continuous, EDStatic.fullCptCacheDirectory);
                }
            }

            //x|yVar > >= < <= constraints are relevant to x|yMin|Max (if not already set)
            for (int con = 0; con < constraintVariables.size(); con++) {
                boolean isX = constraintVariables.get(con).equals(xVar.destinationName());
                boolean isY = constraintVariables.get(con).equals(yVar.destinationName());
                if (isX || isY) {
                    String op = constraintOps.get(con);
                    boolean isG = op.startsWith(">");
                    boolean isL = op.startsWith("<");
                    if (isG || isL) {
                        boolean isTime = 
                            (isX && xVar instanceof EDVTimeStamp) || 
                            (isY && yVar instanceof EDVTimeStamp);
                        String vals = constraintValues.get(con);
                        double vald = isTime && Calendar2.isIsoDate(vals)?
                            Calendar2.safeIsoStringToEpochSeconds(vals) :
                            String2.parseDouble(vals);
                        if      (isX && isG && Double.isNaN(xMin)) xMin = vald;
                        else if (isX && isL && Double.isNaN(xMax)) xMax = vald;
                        else if (isY && isG && Double.isNaN(yMin)) yMin = vald;
                        else if (isY && isL && Double.isNaN(yMax)) yMax = vald;
                    }
                }
            }
            if (debugMode) String2.log("saveAsImage 6");

            String varTitle = "";
            if (drawLines) {
                varTitle = "";
            } else if (drawLinesAndMarkers || drawMarkers) {
                varTitle = zVar == null || colorMap == null? "" : zVar.longName() + zUnits;
            } else if (drawSticks) {
                varTitle = "x=" + yVar.destinationName() + 
                    (yUnits.equals(zUnits)? "" : yUnits) + 
                    ", y=" + zVar.destinationName() + zUnits;
            } else if (drawVectors) { 
                varTitle = "x=" + zVar.destinationName() + 
                    (zUnits.equals(tUnits)? "" : zUnits) + 
                    ", y=" + tVar.destinationName() + tUnits +
                    ", standard=" + (float)vectorStandard;
            }
            String yLabel = drawSticks? varTitle :
                yVar.longName() + yUnits;

            //make a graphDataLayer 
            GraphDataLayer graphDataLayer = new GraphDataLayer(
                -1, //which pointScreen
                xColN, yColN, zColN, tColN, zColN, //x,y,z1,z2,z3 column numbers
                drawSticks? GraphDataLayer.DRAW_STICKS :
                    drawVectors?  GraphDataLayer.DRAW_POINT_VECTORS :
                    drawLines? GraphDataLayer.DRAW_LINES :
                    drawLinesAndMarkers? GraphDataLayer.DRAW_MARKERS_AND_LINES :
                    GraphDataLayer.DRAW_MARKERS, //default
                true, false,
                xVar.longName() + xUnits, //x,yAxisTitle  for now, always std units 
                yVar.longName() + yUnits, 
                varTitle.length() > 0? varTitle : title,             
                varTitle.length() > 0? title : "",
                title2.toString(), 
                "Data courtesy of " + institution(), 
                table, null, null,
                colorMap, color,
                markerType, markerSize,
                vectorStandard,
                GraphDataLayer.REGRESS_NONE);
            ArrayList graphDataLayers = new ArrayList();
            graphDataLayers.add(graphDataLayer);
//???should transparentPng only be for certain graph types???

            //setup graphics2D
            String logoImageFile;
            if (pdf) {
                logoImageFile = EDStatic.highResLogoImageFile;
                fontScale *= 1.25; //SgtMap.PDF_FONTSCALE=1.5 is too big
                //getting the outputStream was delayed as long as possible to allow errors
                //to be detected and handled before committing to sending results to client
                pdfInfo = SgtUtil.createPdf(SgtUtil.PDF_PORTRAIT, 
                    width, height, outputStreamSource.outputStream("UTF-8"));
                g2 = (Graphics2D)pdfInfo[0];
            } else {
                logoImageFile = sizeIndex <= 1? EDStatic.lowResLogoImageFile : EDStatic.highResLogoImageFile;
                fontScale *= sizeIndex <= 1? 1: 1.25;
                bufferedImage = SgtUtil.getBufferedImage(width, height);
                g2 = (Graphics2D)bufferedImage.getGraphics();
            }
            g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                RenderingHints.VALUE_ANTIALIAS_ON);
            if (reallyVerbose) String2.log("  sizeIndex=" + sizeIndex + " pdf=" + pdf + 
                " width=" + width + " height=" + height);
            if (debugMode) String2.log("saveAsImage 7");

            if (isMap) {
                //create a map

                if (Double.isNaN(xMin) || Double.isNaN(xMax) ||
                    Double.isNaN(yMin) || Double.isNaN(yMax)) {

                    //calculate the xy axis ranges (this should be in make map!)
                    double xStats[] = table.getColumn(xColN).calculateStats();
                    double yStats[] = table.getColumn(yColN).calculateStats();

                    //old way  (too tied to big round numbers like 100, 200, 300) 
                    //often had big gap on one side
                    //double xLH[] = Math2.suggestLowHigh(
                    //    xStats[PrimitiveArray.STATS_MIN], 
                    //    xStats[PrimitiveArray.STATS_MAX]);
                    //double yLH[] = Math2.suggestLowHigh(
                    //    yStats[PrimitiveArray.STATS_MIN], 
                    //    yStats[PrimitiveArray.STATS_MAX]);

                    //new way
                    double xLH[] = {
                        xStats[PrimitiveArray.STATS_MIN], 
                        xStats[PrimitiveArray.STATS_MAX]};
                    double[] sd = Math2.suggestDivisions(xLH[1] - xLH[0]);
                    xLH[0] -= sd[1]; //tight range
                    xLH[1] += sd[1];
                    if (xStats[PrimitiveArray.STATS_N] == 0) {
                        xLH[0] = xVar.destinationMin(); 
                        xLH[1] = xVar.destinationMax();
                    }

                    double yLH[] = {
                        yStats[PrimitiveArray.STATS_MIN], 
                        yStats[PrimitiveArray.STATS_MAX]};
                    sd = Math2.suggestDivisions(yLH[1] - yLH[0]);
                    yLH[0] -= sd[1]; //tight range
                    yLH[1] += sd[1];
                    if (yStats[PrimitiveArray.STATS_N] == 0) {
                        yLH[0] = yVar.destinationMin(); 
                        yLH[1] = yVar.destinationMax();
                    }

                    //ensure reasonable for a map
                    if (xLH[0] < -180) xLH[0] = -180;
                    //deal with odd cases like pmelArgoAll: x<0 and >180
                    if (-xLH[0] > xLH[1]-180) { //i.e., if minX is farther below 0, than maxX is >180
                        if (xLH[1] > 180) xLH[1] = 180;
                    } else {
                        if (xLH[0] < 0)   xLH[0] = 0;
                        if (xLH[1] > 360) xLH[1] = 360;
                    }
                    if (yLH[0] < -90) yLH[0] = -90;
                    if (yLH[1] >  90) yLH[1] =  90;

                    //make square
                    if (true) {
                        double xRange = xLH[1] - xLH[0];
                        double yRange = yLH[1] - yLH[0];
                        if (xRange > yRange) {
                            double diff2 = (xRange - yRange) / 2;
                            yLH[0] = Math.max(-90, yLH[0] - diff2);
                            yLH[1] = Math.min( 90, yLH[1] + diff2);
                        } else {
                            double diff2 = (yRange - xRange) / 2;
                            //deal with odd cases like pmelArgoAll: x<0 and >180
                            if (-xLH[0] > xLH[1]-180) { //i.e., if minX is farther below 0, than maxX is >180
                                xLH[0] = Math.max(-180, xLH[0] - diff2);
                                xLH[1] = Math.min( 180, xLH[1] + diff2);
                            } else {
                                xLH[0] = Math.max(   0, xLH[0] - diff2);
                                xLH[1] = Math.min( 360, xLH[1] + diff2);
                            }
                        }
                    }

                    //set xyMin/Max
                    if (Double.isNaN(xMin) || Double.isNaN(xMax)) {
                        xMin = xLH[0]; xMax = xLH[1];
                    }
                    if (Double.isNaN(yMin) || Double.isNaN(yMax)) {
                        yMin = yLH[0]; yMax = yLH[1];
                    }
                }


                int predicted[] = SgtMap.predictGraphSize(1, width, height, 
                    xMin, xMax, yMin, yMax);
                Grid bath = table.nRows() == 0? null :
                    SgtMap.createBathymetryGrid(
                        EDStatic.fullSgtMapBathymetryCacheDirectory, 
                        xMin, xMax, yMin, yMax, 
                        predicted[0], predicted[1]);
                SgtMap.makeMap(SgtUtil.LEGEND_BELOW,
                    EDStatic.legendTitle1, EDStatic.legendTitle2,
                    EDStatic.imageDir, logoImageFile,
                    xMin, xMax, yMin, yMax, //predefined min/maxX/Y
                    drawLandAsMask,
                    bath != null, //plotGridData (bathymetry)
                    bath,
                    1, 1, 0, //double gridScaleFactor, gridAltScaleFactor, gridAltOffset,
                    SgtMap.bathymetryCptFullName,
                    null, //SgtMap.BATHYMETRY_BOLD_TITLE + " (" + SgtMap.BATHYMETRY_UNITS + ")",
                    "",
                    "",  
                    "", //"Data courtesy of " + SgtMap.BATHYMETRY_COURTESY,
                    false, null, 1, 1, 1, "", null, "", "", "", "", "", //plot contour 
                    graphDataLayers,
                    g2, 0, 0, width, height,
                    0, //no boundaryResAdjust,
                    fontScale);

            } else {
                //create a graph
                EDStatic.sgtGraph.makeGraph(
                    xVar.longName() + xUnits, //x,yAxisTitle  for now, always std units 
                    yLabel, 
                    SgtUtil.LEGEND_BELOW, EDStatic.legendTitle1, EDStatic.legendTitle2,
                    EDStatic.imageDir, logoImageFile,
                    xMin, xMax, yMin, yMax, 
                    xVar instanceof EDVTimeStamp, //x/yIsTimeAxis,
                    yVar instanceof EDVTimeStamp, 
                    graphDataLayers,
                    g2, 0, 0, width, height,  1, //graph width/height
                    fontScale); 
            }
            if (debugMode) String2.log("saveAsImage 8");

        } catch (Throwable t) {
            ok = false;
            try {
                //write exception info on image
                String msg = MustBe.getShortErrorMessage(t);
                String2.log(MustBe.throwableToString(t)); //log full message with stack trace
                double tFontScale = pdf? 1.25 : 1;
                int tHeight = Math2.roundToInt(tFontScale * 12);

                if (pdf) {
                    if (pdfInfo == null)
                        pdfInfo = SgtUtil.createPdf(SgtUtil.PDF_PORTRAIT, 
                            width, height, outputStreamSource.outputStream("UTF-8"));
                    if (g2 == null)
                        g2 = (Graphics2D)pdfInfo[0];
                } else {
                    //make a new image (I don't think pdf can work this way -- sent as created)
                    bufferedImage = SgtUtil.getBufferedImage(width, height);
                    g2 = (Graphics2D)bufferedImage.getGraphics();
                }
                g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, 
                    RenderingHints.VALUE_ANTIALIAS_ON);
                g2.setClip(0, 0, width, height); //unset in case set by sgtGraph
                msg = String2.noLongLines(msg, (width * 10 / 6) / tHeight, "    ");
                String lines[] = msg.split("\\n"); //not String2.split which trims
                g2.setColor(Color.black);
                g2.setFont(new Font(EDStatic.fontFamily, Font.PLAIN, tHeight));
                int ty = tHeight * 2;
                for (int i = 0; i < lines.length; i++) {
                    g2.drawString(lines[i], tHeight, ty);
                    ty += tHeight + 2;
                }
            } catch (Throwable t2) {
                String2.log("ERROR2 while creating image:\n" + MustBe.throwableToString(t2)); 
                if (pdf) {
                    if (pdfInfo == null) throw t;
                } else {
                    if (bufferedImage == null) throw t;
                }
                //else fall through to close/save image below
            }
        }
        if (debugMode) String2.log("saveAsImage 9");

        //save image
        if (pdf) {
            SgtUtil.closePdf(pdfInfo);
        } else {
            //getting the outputStream was delayed as long as possible to allow errors
            //to be detected and handled before committing to sending results to client
            SgtUtil.saveAsPng(bufferedImage, outputStreamSource.outputStream("")); 
        }

        outputStreamSource.outputStream("").flush(); //safety

        if (reallyVerbose) String2.log("  EDDTable.saveAsImage done. TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
        return ok;
    }

    /**
     * Save the TableWriterAllWithMetadata data as a Matlab .mat file.
     * This doesn't write attributes because .mat files don't store attributes.
     * This maintains the data types (Strings become char[][]).
     * 
     * @param outputStreamSource
     * @param twawm  all the results data, with missingValues stored as destinationMissingValues
     *    or destinationFillValues  (they are converted to NaNs)
     * @param structureName the name to use for the variable which holds all of the data, 
     *    usually the dataset's internal name (datasetID).
     * @throws Throwable 
     */
    public void saveAsMatlab(OutputStreamSource outputStreamSource, 
        TableWriterAllWithMetadata twawm, String structureName) throws Throwable {
        if (reallyVerbose) String2.log("EDDTable.saveAsMatlab"); 
        long time = System.currentTimeMillis();

        //make sure there is data
        long tnRows = twawm.nRows();
        if (tnRows == 0)
            throw new SimpleException(EDStatic.THERE_IS_NO_DATA);
        int nCols = twawm.nColumns();
        if (tnRows * nCols * 8 >= Integer.MAX_VALUE - 1000)
            throw new SimpleException(EDStatic.thereIsTooMuchData);
        int nRows = (int)tnRows;

        //open a dataOutputStream 
        DataOutputStream stream = new DataOutputStream(outputStreamSource.outputStream(""));

        //*** write Matlab Structure  see 1-32
        //*** THIS CODE MIMICS Table.saveAsMatlab. If make changes here, make them there, too.
        //    The code in EDDGrid.saveAsMatlab is similar, too.
        //write the header
        Matlab.writeMatlabHeader(stream);

        //calculate cumulative size of the structure
        byte structureNameInfo[] = Matlab.nameInfo(structureName); 
        NDimensionalIndex ndIndex[] = new NDimensionalIndex[nCols];
        int cumSize = //see 1-32
            16 + //for array flags
            16 + //my structure is always 2 dimensions 
            structureNameInfo.length +
            8 + //field name length (for all fields)
            8 + nCols * 32; //field names
        for (int col = 0; col < nCols; col++) {
            Class type = twawm.columnType(col);
            if (type == String.class)
                ndIndex[col] = Matlab.make2DNDIndex(nRows, twawm.columnMaxStringLength(col)); 
            else
                ndIndex[col] = Matlab.make2DNDIndex(nRows); 
            //add size of each cell
            cumSize += 8 + //type and size
                Matlab.sizeOfNDimensionalArray("", //without column names (they're stored separately)
                    type, ndIndex[col]);
        }

        //write the miMatrix dataType and nBytes
        stream.writeInt(Matlab.miMATRIX);        //dataType
        stream.writeInt(cumSize); 

        //write array flags 
        stream.writeInt(Matlab.miUINT32); //dataType
        stream.writeInt(8);  //fixed nBytes of data
        stream.writeInt(Matlab.mxSTRUCT_CLASS); //array flags  
        stream.writeInt(0); //reserved; ends on 8 byte boundary

        //write structure's dimension array 
        stream.writeInt(Matlab.miINT32); //dataType
        stream.writeInt(2 * 4);  //nBytes
        //matlab docs have 2,1, octave has 1,1. 
        //Think of structure as one row of a table, where elements are entire arrays:  e.g., sst.lon sst.lat sst.sst.
        //Having multidimensions (e.g., 2 here) lets you have additional rows, e.g., sst(2).lon sst(2).lat sst(2).sst.
        //So 1,1 makes sense.
        stream.writeInt(1);  
        stream.writeInt(1);
         
        //write structure name 
        stream.write(structureNameInfo, 0, structureNameInfo.length);

        //write length for all field names (always 32)  (short form)
        stream.writeShort(4);                //nBytes
        stream.writeShort(Matlab.miINT32);    //dataType
        stream.writeInt(32);                 //nBytes per field name

        //write the field names (each 32 bytes)
        stream.writeInt(Matlab.miINT8);    //dataType
        stream.writeInt(nCols * 32);      //nBytes per field name
        String nulls = String2.makeString('\u0000', 32);
        for (int col = 0; col < nCols; col++) 
            stream.write(String2.toByteArray(
                String2.noLongerThan(twawm.columnName(col), 31) + nulls), 0, 32);

        //write the structure's elements (one for each col)
        //This is pretty good at conserving memory (just one column in memory at a time).
        //It would be hard to make more conservative because Strings have
        //to be written out: all first chars, all second chars, all third chars...
        for (int col = 0; col < nCols; col++) {
            PrimitiveArray pa = twawm.column(col);
            if (!(pa instanceof StringArray)) {
                //convert missing values to NaNs
                pa.convertToStandardMissingValues( 
                    twawm.columnAttributes(col).getDouble("_FillValue"),
                    twawm.columnAttributes(col).getDouble("missing_value"));
            }
            Matlab.writeNDimensionalArray(stream, "", //without column names (they're stored separately)
                pa, ndIndex[col]);
        }

        //this doesn't write attributes because .mat files don't store attributes

        stream.flush(); //essential

        if (reallyVerbose) String2.log("  EDDTable.saveAsMatlab done. TIME=" + 
            (System.currentTimeMillis() - time) + "\n");
    }


    /**
     * Save this table of data as a flat netCDF .nc file (a column for each 
     * variable, all referencing one dimension) using the currently
     * available attributes.
     * <br>The data are written as separate variables, sharing a common dimension
     *   "observation", not as a Structure.
     * <br>The data values are written as their current data type 
     *   (e.g., float or int).
     * <br>This writes the lon values as they are currently in this table
     *   (e.g., +-180 or 0..360).
     * <br>This overwrites any existing file of the specified name.
     * <br>This makes an effort not to create a partial file if there is an error.
     * <br>If no exception is thrown, the file was successfully created.
     * <br>!!!The file must have at least one row, or an Exception will be thrown
     *   (nc dimensions can't be 0 length).
     * <br>!!!Missing values should be destinationMissingValues or 
     *   destinationFillValues and aren't changed.
     * 
     * @param fullName the full file name (dir+name+ext)
     * @param twawm provides access to all of the data
     * @throws Throwable if trouble (e.g., TOO_MUCH_DATA)
     */
    public void saveAsFlatNc(String fullName, TableWriterAllWithMetadata twawm) 
        throws Throwable {
        if (reallyVerbose) String2.log("  EDDTable.saveAsFlatNc " + fullName); 
        long time = System.currentTimeMillis();

        //delete any existing file
        File2.delete(fullName);

        //test for too much data
        if (twawm.nRows() >= Integer.MAX_VALUE)
            throw new SimpleException(EDStatic.thereIsTooMuchData);
        int nRows = (int)twawm.nRows(); 
        int nColumns = twawm.nColumns();
        if (nRows == 0) 
            throw new SimpleException(EDStatic.THERE_IS_NO_DATA);

        //POLICY: because this procedure may be used in more than one thread,
        //do work on unique temp files names using randomInt, then rename to proper file name.
        //If procedure fails half way through, there won't be a half-finished file.
        int randomInt = Math2.random(Integer.MAX_VALUE);

        //open the file (before 'try'); if it fails, no temp file to delete
        NetcdfFileWriteable nc = NetcdfFileWriteable.createNew(fullName + randomInt, false);
        try {
            //items determined by looking at a .nc file; items written in that order 

            //define the dimensions
            Dimension dimension  = nc.addDimension(ROW_NAME, nRows);
//javadoc says: if there is an unlimited dimension, all variables that use it are in a structure
//Dimension rowDimension  = nc.addDimension("row", nRows, true, true, false); //isShared, isUnlimited, isUnknown
//String2.log("unlimitied dimension exists: " + (nc.getUnlimitedDimension() != null));

            //add the variables
            for (int col = 0; col < nColumns; col++) {
                Class type = twawm.columnType(col);
                String tColName = twawm.columnName(col);
                if (type == String.class) {
                    int max = Math.max(1, twawm.columnMaxStringLength(col)); //nc libs want at least 1; 0 happens if no data
                    Dimension lengthDimension  = nc.addDimension("StringLengthForVariable" + col, max);
                    nc.addVariable(tColName, DataType.CHAR, 
                        new Dimension[]{dimension, lengthDimension}); 
                } else {
                    nc.addVariable(tColName, DataType.getType(type), new Dimension[]{dimension}); 
                }
//nc.addMemberVariable(recordStructure, nc.findVariable(tColName));
            }

//boolean bool = nc.addRecordStructure(); //creates a structure variable called "record"         
//String2.log("addRecordStructure: " + bool);
//Structure recordStructure = (Structure)nc.findVariable("record");

            //set id attribute = file name
            //currently, only the .nc saveAs types use attributes!
            Attributes globalAttributes = twawm.globalAttributes();
            if (globalAttributes.get("id") == null)
                globalAttributes.set("id", File2.getNameNoExtension(fullName));

            globalAttributes.set("observationDimension", ROW_NAME);
            String names[] = globalAttributes.getNames();
            for (int ni = 0; ni < names.length; ni++) { 
                PrimitiveArray tValue = globalAttributes.get(names[ni]);
                if (tValue == null || tValue.size() == 0 || tValue.toString().length() == 0)
                    String2.log("WARNING! Global Attribute: " + names[ni] + " is \"" + tValue + "\".");
                else nc.addGlobalAttribute(  
                    NcHelper.getAttribute(names[ni], tValue));
            }
            for (int col = 0; col < nColumns; col++) {
                String colName = twawm.columnName(col);

                names = twawm.columnAttributes(col).getNames();
                for (int ni = 0; ni < names.length; ni++) {
                    PrimitiveArray tValue = twawm.columnAttributes(col).get(names[ni]);
                    if (tValue == null || tValue.size() == 0 || tValue.toString().length() == 0)
                        String2.log("WARNING! column=" + col + " attribute: " + names[ni] + " is \"" + tValue + "\".");
                    else nc.addVariableAttribute(colName, 
                        NcHelper.getAttribute(names[ni], tValue));
                }
            }

            //leave "define" mode
            nc.create();

            //write the data
            for (int col = 0; col < nColumns; col++) {

                //missing values are already destinationMissingValues or destinationFillValues
                PrimitiveArray pa = twawm.column(col);

                nc.write(twawm.columnName(col), 
                    NcHelper.get1DArray(pa.toObjectArray()));

            }

            //if close throws Throwable, it is trouble
            nc.close(); //it calls flush() and doesn't like flush called separately

            //rename the file to the specified name
            File2.rename(fullName + randomInt, fullName);

            //diagnostic
            if (reallyVerbose) String2.log("  EDDTable.saveAsFlatNc done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
            //ncDump("End of Table.saveAsFlatNc", directory + name + ext, false);

        } catch (Throwable t) {
            //try to close the file
            try {
                nc.close(); //it calls flush() and doesn't like flush called separately
            } catch (Throwable t2) {
                //don't care
            }

            //delete the partial file
            File2.delete(fullName + randomInt);

            throw t;
        }

    }

    /**
     * This is used by administrators to get the empirical min and max for all 
     *   non-String variables by doing a request for a time range (sometimes one time point).
     * This could be used by constructors (at the end),
     * but the results vary with different isoDateTimes,
     * and would be wasteful to do this every time a dataset is constructed 
     * if results don't change.
     *
     * @param minTime the ISO 8601 min time to check (use null or "" if no min limit)
     * @param maxTime the ISO 8601 max time to check (use null or "" if no max limit)
     * @param makeChanges if true, the discovered values are used to set 
     *    the variable's min and max values if they aren't already set
     * @param stringsToo if true, this determines if the String variables have any values
     * @throws Throwable if trouble
     */
    public void getEmpiricalMinMax(String minTime, String maxTime, 
            boolean makeChanges, boolean stringsToo) throws Throwable {
        if (verbose) String2.log("\nEDDTable.getEmpiricalMinMax for " + datasetID() + ", " + 
            minTime + " to " + maxTime + " total nVars=" + dataVariables.length);
        long downloadTime = System.currentTimeMillis();

        StringBuffer query = new StringBuffer();
        int first = 0;     //change temporarily for bmde to get e.g., 0 - 100, 100 - 200, ...
        int last = dataVariables.length;  //change temporarily for bmde
        for (int dv = first; dv < last; dv++) {
            //get vars
            if (stringsToo || !dataVariables[dv].destinationDataType().equals("String"))
                query.append("," + dataVariables[dv].destinationName());
        }
        if (query.length() == 0) {
            String2.log("All variables are String variables.");
            return;
        }
        query.deleteCharAt(0); //first comma
        if (minTime != null && minTime.length() > 0)
            query.append("&time>=" + minTime);
        if (maxTime != null && maxTime.length() > 0)
            query.append("&time<=" + maxTime);

        //query
        String dir = EDStatic.fullTestCacheDirectory;
        String fileName = datasetID() + Math2.random(Integer.MAX_VALUE);
        TableWriterAllWithMetadata twawm = new TableWriterAllWithMetadata(dir, fileName);
        TableWriter tableWriter = encloseTableWriter(dir, fileName, twawm, query.toString());
        getDataForDapQuery("", query.toString(), tableWriter);
        Table table = twawm.cumulativeTable();
        twawm.releaseResources();
        String2.log("  downloadTime=" + (System.currentTimeMillis() - downloadTime));
        String2.log("  found nRows=" + table.nRows());
        for (int col = 0; col < table.nColumns(); col++) {
            String destName = table.getColumnName(col);
            EDV edv = findDataVariableByDestinationName(destName); //error if not found
            if (edv.destinationDataType().equals("String")) {
                String2.log("  " + destName + ": is String variable; maxStringLength found = " +
                    twawm.columnMaxStringLength(col) + "\n");
            } else {
                String tMin, tMax; 
                if (PrimitiveArray.isIntegerType(edv.destinationDataTypeClass())) {
                    tMin = "" + Math2.roundToLong(twawm.columnMinValue[col]); 
                    tMax = "" + Math2.roundToLong(twawm.columnMaxValue[col]); 
                } else {
                    tMin = "" + twawm.columnMinValue[col]; 
                    tMax = "" + twawm.columnMaxValue[col]; 
                }
                if (verbose) {
                    double loHi[] = Math2.suggestLowHigh(twawm.columnMinValue[col], twawm.columnMaxValue[col]);
                    String2.log("  " + destName + ":\n" +
                    "                <att name=\"actual_range\" type=\"" + edv.destinationDataType().toLowerCase() + 
                        "List\">" + tMin + " " + tMax + "</att>\n" +
                    "                <att name=\"colorBarMinimum\" type=\"double\">" + loHi[0] + "</att>\n" +
                    "                <att name=\"colorBarMaximum\" type=\"double\">" + loHi[1] + "</att>\n");
                }
                if (makeChanges && 
                    Double.isNaN(edv.destinationMin()) &&
                    Double.isNaN(edv.destinationMax())) {

                    edv.setDestinationMin(twawm.columnMinValue[col] * edv.scaleFactor() + edv.addOffset());
                    edv.setDestinationMax(twawm.columnMaxValue[col] * edv.scaleFactor() + edv.addOffset());
                    edv.setActualRangeFromDestinationMinMax();
                }
            }
        }
        if (verbose) String2.log("\ntotal nVars=" + dataVariables.length);
    }

    /**
     * This is used by administrator to get min,max time
     *   by doing a request for lon,lat.
     * This could be used by constructors (at the end).
     * The problem is that the results vary with different isoDateTimes,
     * and wasteful to do this every time constructed (if results don't change).
     *
     * @param lon the lon to check
     * @param lat the lat to check
     * @param dir the directory (on this computer's hard drive) to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @throws Throwable if trouble
     */
    public void getMinMaxTime(double lon, double lat, String dir, String fileName) throws Throwable {
        if (verbose) String2.log("\nEDDTable.getMinMaxTime for lon=" + lon + " lat=" + lat);

        StringBuffer query = new StringBuffer();
        for (int dv = 0; dv < dataVariables.length; dv++)
            //get all vars since different vars at different altitudes
            query.append("," + dataVariables[dv].destinationName());
        query.deleteCharAt(0); //first comma
        query.append("&" + EDV.LON_NAME + "=" + lon + "&" + EDV.LAT_NAME + "=" + lat);

        //auto get source min/max time   for that one lat,lon location
        TableWriterAllWithMetadata twawm = new TableWriterAllWithMetadata(dir, fileName);
        TableWriter tableWriter = encloseTableWriter(dir, fileName, twawm, query.toString());
        getDataForDapQuery("", query.toString(), tableWriter);
        Table table = twawm.makeEmptyTable(); //no need for twawm.cumulativeTable();
        twawm.releaseResources();
        int timeCol = table.findColumnNumber(EDV.TIME_NAME);
        double tMin = twawm.columnMinValue(timeCol); 
        double tMax = twawm.columnMaxValue(timeCol);
        if (verbose) String2.log("  found time min=" + tMin + "=" +
            (Double.isNaN(tMin)? "" : Calendar2.epochSecondsToIsoStringT(tMin)) + 
            (Double.isNaN(tMax)? "" : Calendar2.epochSecondsToIsoStringT(tMax)));
        if (!Double.isNaN(tMin)) dataVariables[timeIndex].setDestinationMin(tMin); //scaleFactor,addOffset not supported
        if (!Double.isNaN(tMax)) dataVariables[timeIndex].setDestinationMax(tMax);
        dataVariables[timeIndex].setActualRangeFromDestinationMinMax();
    }


    /**
     * This writes an HTML form requesting info from this dataset (like the DAP Data Access Forms).
     *
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param userDapQuery the part after the '?', still percentEncoded (shouldn't be null).
     * @param writer
     * @throws Throwable if trouble
     */
    public void writeDapHtmlForm(String loggedInAs,
        String userDapQuery, Writer writer) throws Throwable {
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl);

        //parse userDapQuery 
        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        if (userDapQuery == null)
            userDapQuery = "";
        userDapQuery = userDapQuery.trim();
        StringArray resultsVariables = new StringArray();
        StringArray constraintVariables = new StringArray();
        StringArray constraintOps = new StringArray();
        StringArray constraintValues = new StringArray();
        if (userDapQuery == null)
            userDapQuery = "";
        try {
            parseUserDapQuery(userDapQuery, resultsVariables,
                constraintVariables, constraintOps, constraintValues, true);
        } catch (Throwable t) {
            String2.log(MustBe.throwableToString(t));
            userDapQuery = ""; //as if no userDapQuery
        }

        //beginning of form
        String liClickSubmit = "\n" +
            "  <li> " + EDStatic.EDDClickOnSubmitHtml + "\n" +
            "  </ol>\n";
        writer.write("<p>");
        String formName = "form1";
        writer.write(widgets.beginForm(formName, "GET", "", ""));
        writer.write(HtmlWidgets.PERCENT_ENCODE_JS);

        //begin table
        writer.write(widgets.beginTable(0, 0, "")); 

        //write the table's column names   
        String gap = "&nbsp;&nbsp;&nbsp;";
        writer.write(
            "<tr>\n" +
            "  <th nowrap align=\"left\">" + EDStatic.EDDTableVariable + " " +
            EDStatic.htmlTooltipImage(EDStatic.EDDTableTabularDatasetHtml));
        
        StringBuffer checkAll = new StringBuffer();
        StringBuffer uncheckAll = new StringBuffer();
        int nDv = dataVariables.length;
        for (int dv = 0; dv < nDv; dv++) {
            checkAll.append(  formName + ".varch" + dv + ".checked=true;");
            uncheckAll.append(formName + ".varch" + dv + ".checked=false;");
        }
        writer.write(widgets.button("button", "CheckAll", EDStatic.EDDTableCheckAllTooltip,
            EDStatic.EDDTableCheckAll,   "onclick=\"" + checkAll.toString() + "\""));
        writer.write(widgets.button("button", "UncheckAll", EDStatic.EDDTableUncheckAllTooltip,
            EDStatic.EDDTableUncheckAll, "onclick=\"" + uncheckAll.toString() + "\""));

        writer.write(
            "  &nbsp;</th>\n" +
            "  <th colspan=\"2\">" + EDStatic.EDDTableOptConstraint1Html + " " + 
                EDStatic.htmlTooltipImage(EDStatic.EDDTableConstraintHtml) + "</th>\n" +
            "  <th colspan=\"2\">" + EDStatic.EDDTableOptConstraint2Html + " " + 
                EDStatic.htmlTooltipImage(EDStatic.EDDTableConstraintHtml) + "</th>\n" +
            "  <th nowrap align=\"left\">&nbsp;" + EDStatic.EDDMinimum + " " + 
                EDStatic.htmlTooltipImage(EDStatic.EDDTableMinimumTooltip) + "</th>\n" +
            "  <th nowrap align=\"left\">" + gap + EDStatic.EDDMaximum + " " + 
                EDStatic.htmlTooltipImage(EDStatic.EDDTableMaximumTooltip) + "</th>\n" +
            "</tr>\n");

        //a row for each dataVariable
        String sliderFromNames[] = new String[nDv];
        String sliderToNames[] = new String[nDv];
        int sliderNThumbs[] = new int[nDv];
        String sliderUserValuesCsvs[] = new String[nDv];
        int sliderInitFromPositions[] = new int[nDv];
        int sliderInitToPositions[] = new int[nDv];
        for (int dv = 0; dv < nDv; dv++) {
            EDV edv = dataVariables[dv];
            double tMin = edv.destinationMin();
            double tMax = edv.destinationMax();
            boolean isTime = dv == timeIndex;
            boolean isString = edv.destinationDataTypeClass() == String.class;
            writer.write("<tr>\n");
            
            //get the extra info   
            String extra = edv.units();
            if (dv == timeIndex)
                extra = "UTC"; //no longer true: "seconds since 1970-01-01..."
            if (extra == null) 
                extra = "";
            if (showLongName(edv.destinationName(), edv.longName())) 
                extra = edv.longName() + (extra.length() == 0? "" : ", " + extra);
            if (extra.length() > 0) 
                extra = " (" + extra + ")";

            //variables: checkbox varname (longName, units)
            writer.write("  <td nowrap>");
            writer.write(widgets.checkbox("varch" + dv, EDStatic.EDDTableCheckTheVariables, 
                userDapQuery.length() == 0? true : (resultsVariables.indexOf(edv.destinationName()) >= 0), 
                edv.destinationName(), 
                edv.destinationName() + extra + " " +
                    EDStatic.htmlTooltipImage(edv.destinationDataTypeClass(), edv.destinationName(),
                        edv.combinedAttributes()), 
                ""));
            writer.write("  &nbsp;</td>\n");

            //get default constraints
            String[] tOp = {">=", "<="};
            double[] tValD = {Double.NaN, Double.NaN};
            String[] tValS = {"", ""};
            if (userDapQuery.length() > 0) {
                //get constraints from userDapQuery?
                boolean done0 = false, done1 = false;
                //find first 2 constraints on this var
                for (int con = 0; con < 2; con++) { 
                    int tConi = constraintVariables.indexOf(edv.destinationName());
                    if (tConi >= 0) {
                        String cOp = constraintOps.get(tConi); 
                        int putIn = cOp.startsWith("<") || done0? 1 : 0; //prefer 0 first
                        if (putIn == 0) done0 = true; 
                        else done1 = true;
                        tOp[putIn] = cOp;
                        tValS[putIn] = constraintValues.get(tConi);
                        tValD[putIn] = String2.parseDouble(tValS[putIn]);
                        if (dv == timeIndex) {
                            //time constraint will be stored as double (then as a string)
                            //convert to iso format
                            if (!Double.isNaN(tValD[putIn]))
                                tValS[putIn] = Calendar2.epochSecondsToIsoStringT(tValD[putIn]) + "Z";
                        }
                        constraintVariables.remove(tConi);
                        constraintOps.remove(tConi);
                        constraintValues.remove(tConi);
                    }
                }
            } else {
                if (dv == timeIndex) {
                    double ttMax = tMax;
                    GregorianCalendar gc;
                    if (Double.isNaN(ttMax)) {
                        gc = Calendar2.newGCalendarZulu();
                    } else {
                        //only set max request if tMax is known
                        tValD[1] = ttMax;
                        tValS[1] = Calendar2.epochSecondsToIsoStringT(ttMax) + "Z";
                        gc = Calendar2.epochSecondsToGc(ttMax);
                    }
                    //round to previous midnight, then go back 1 week
                    Calendar2.clearSmallerFields(gc, Calendar2.DATE);
                    ttMax = Calendar2.gcToEpochSeconds(gc);
                    tValD[0] = ttMax - Calendar2.SECONDS_PER_DAY * 7;
                    tValS[0] = Calendar2.epochSecondsToIsoStringT(tValD[0]) + "Z";
                }
            }

            //write constraints html 
            String valueWidgetName = "val" + dv + "_";
            String tTooltip = 
                isTime?   EDStatic.EDDTableMakeATimeConstraintHtml :
                isString? EDStatic.EDDTableMakeAStringConstraintHtml :
                          EDStatic.EDDTableMakeANumericConstraintHtml;
            if (edv.destinationMinString().length() > 0)
                tTooltip += "<br>&nbsp;<br>" + edv.destinationName() + " ranges from " +
                    edv.destinationMinString() + " to " +
                    (edv.destinationMaxString().length() > 0? edv.destinationMaxString() : "(?)") + 
                    ".";
            for (int con = 0; con < 2; con++) {
                writer.write("  <td nowrap>" + (con == 0? "" : gap));
                writer.write(widgets.select("op" + dv + "_" + con, EDStatic.EDDTableSelectAnOperator, 1,
                    OPERATORS, String2.indexOf(OPERATORS, tOp[con]), ""));
                writer.write("  </td>\n");
                writer.write("  <td nowrap>");
                String tVal = tValS[con];
                if (tOp[con].equals(REGEX_OP) || isString) {
                    if (tVal.length() > 0)
                        tVal = String2.toJson(tVal); //enclose in "
                }
                writer.write(widgets.textField(valueWidgetName + con, tTooltip,
                    20, 255, tVal, ""));
                writer.write("  </td>\n");
            }

            //min max
            writer.write(
                "  <td nowrap>&nbsp;" + edv.destinationMinString() + "</td>\n" +
                "  <td nowrap>" + gap + edv.destinationMaxString() + "</td>\n");

            //end of row
            writer.write("</tr>\n");


            // *** and a slider for this dataVariable    (Data Access Form)
            if ((dv != lonIndex && dv != latIndex && dv != altIndex && dv != timeIndex) ||
                !Math2.isFinite(edv.destinationMin()) ||
                (!Math2.isFinite(edv.destinationMax()) && dv != timeIndex)) {

                //no slider
                sliderNThumbs[dv] = 0;
                sliderFromNames[dv] = "";
                sliderToNames[  dv] = "";
                sliderUserValuesCsvs[dv] = "";
                sliderInitFromPositions[dv] = 0;
                sliderInitToPositions[dv]   = 0;

            } else {
                //slider
                sliderNThumbs[dv] = 2;
                sliderFromNames[dv] = formName + "." + valueWidgetName + "0";
                sliderToNames[  dv] = formName + "." + valueWidgetName + "1";
                sliderUserValuesCsvs[dv] = edv.sliderCsvValues();
                sliderInitFromPositions[dv] = edv.closestSliderPosition(tValD[0]);
                sliderInitToPositions[dv]   = edv.closestSliderPosition(tValD[1]);
                if (sliderInitFromPositions[dv] == -1) sliderInitFromPositions[dv] = 0;
                if (sliderInitToPositions[  dv] == -1) sliderInitToPositions[  dv] = EDV.SLIDER_PIXELS - 1;
                writer.write(
                    "<tr align=\"left\">\n" +
                    "  <td nowrap colspan=\"5\" align=\"left\">\n" +
                    widgets.spacer(10, 1, "align=\"left\"") +
                    widgets.dualSlider(dv, EDV.SLIDER_PIXELS - 1, "align=\"left\"") +
                    "  </td>\n" +
                    "</tr>\n");
            }

        }

        //end of table
        writer.write(widgets.endTable());

        //filters
        String queryParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")
        writer.write("<p><b>Filters</b>\n");

        //distinct filter
        writer.write("<br>" + widgets.checkbox("distinct", 
            "Check this if you want to use the distinct() filter.",
            String2.indexOf(queryParts, "distinct()") >= 0, 
            "true", "distinct()", ""));
        writer.write(EDStatic.htmlTooltipImage(
            "If just one variable is selected above, checking this tells ERDDAP to return\n" +
            "<br>the sorted list of all unique values for that variable." +

            "<p>If more than one variable is selected above, checking this tells ERDDAP to return\n" +
            "<br>the sorted list of all unique combinations of values for those variables." +

            "<p>In other words, checking this tells ERDDAP to remove all duplicate rows" +
            "<br>from the response table and to sort the table by all of the variables" +
            "<br>(with the top/first variable selected above being the most important" +
            "<br>and other selected variables only being used to break ties)." +

            "<p>WARNING: if this is used without constraints, ERDDAP may have to read" +
            "<br>all rows from the datasource, which can be VERY SLOW for some datasets." +
            "<br>So you should usually include constraints (e.g., time minimum and maximum)."));

        //orderBy filter
        StringArray dvNames0 = new StringArray(dataVariableDestinationNames);
        dvNames0.add(0, "");
        String dvList0[] = dvNames0.toArray();
        int nOrderBy = 5;
        String obPart = String2.stringStartsWith(queryParts, "orderBy(\"");
        String orderBy[] = (obPart != null && obPart.endsWith("\")"))?
            String2.split(obPart.substring(9, obPart.length() - 2), ',') : new String[0];
        writer.write("\n<br>orderBy(&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;\"");  //&nbsp; make it line up with orderByMax
        String important[]={"most", "second most", "third most", "fourth most", "least"};
        for (int ob = 0; ob < nOrderBy; ob++) {
            if (ob > 0) writer.write(",\n");
            writer.write(widgets.select("orderBy" + ob,                 
                "This variable is the " + important[ob] + " important sort variable.\n",
                1, dvList0, 
                Math.max(0, dvNames0.indexOf(ob < orderBy.length? orderBy[ob] : "")),
                ""));
        }
        writer.write("\")\n");       
        writer.write(EDStatic.htmlTooltipImage(
            "orderBy lets you specify how the results table will be sorted.\n" +
            "<br>To use orderBy, select one or more variables.\n" +
            "<br>If you don't want to use orderBy, set all of the orderBy variables to \"\".\n"));

        //orderByMax filter
        String obmPart = String2.stringStartsWith(queryParts, "orderByMax(\"");
        String orderByMax[] = (obmPart != null && obmPart.endsWith("\")"))?
            String2.split(obmPart.substring(12, obmPart.length() - 2), ',') : new String[0];
        writer.write("\n<br>orderByMax(\"");
        for (int obm = 0; obm < nOrderBy; obm++) {
            if (obm > 0) writer.write(",\n");
            writer.write(widgets.select("orderByMax" + obm,                
                "This variable is the " + important[obm] + " important sort variable.\n" +
                (obm == nOrderBy - 1? "<br>(Only the row with the maximum value will be kept.)\n" : ""),
                1, dvList0, 
                Math.max(0, dvNames0.indexOf(obm < orderBy.length? orderBy[obm] : "")),
                ""));
        }
        writer.write("\")\n");       
        writer.write(EDStatic.htmlTooltipImage(
            "orderByMax lets you specify how the results table will be sorted and removes some rows.\n" +
            "<br>Within each group, only the row with the maximum value for the last orderBy variable will be kept.\n" +
            "<br>For example, orderByMax(\"stationID, time\") will return the row with the last time value for each station.\n" + 
            "<br>To use orderByMax, select one or more variables.\n" +
            "<br>If you don't want to use orderByMax, set all of the orderBy variables to \"\".\n"));

        //fileType
        writer.write("<p><b>" + EDStatic.EDDFileType + "</b>\n");
        writer.write(widgets.select("fileType", EDStatic.EDDSelectFileType, 1,
            allFileTypeOptions, defaultFileTypeOption, ""));

        //generate the javaScript
        String javaScript = 
            "var result = \"\";\n" +
            "try {\n" +
            "  var ft = " + formName + ".fileType.options[" + formName + ".fileType.selectedIndex].text;\n" +
            "  var tResult = \"" + tErddapUrl + "/tabledap/" + datasetID() + 
              "\" + ft.substring(0, ft.indexOf(\" - \")) + \"?\";\n" +
            "  var constraint = \"\";\n" +
            "  var active = {};\n" +
            "  for (var dv = 0; dv < " + nDv + "; dv++) {\n" +
            "    var tVar = eval(\"" + formName + ".varch\" + dv);\n" +
            "    if (tVar.checked) {\n" +
            "      tResult += (tResult.charAt(tResult.length-1)==\"?\"? \"\" : \",\") + tVar.value;\n" +
            "      active[tVar.value] = 1;\n" +
            "    }\n" +
            "    var tOp  = eval(\"" + formName + ".op\"  + dv + \"_0\");\n" +
            "    var tVal = eval(\"" + formName + ".val\" + dv + \"_0\");\n" +
            "    if (tVal.value.length > 0) constraint += \"&\" + tVar.value + tOp.options[tOp.selectedIndex].text + percentEncode(tVal.value);\n" +
            "    tOp  = eval(\"" + formName + ".op\"  + dv + \"_1\");\n" +
            "    tVal = eval(\"" + formName + ".val\" + dv + \"_1\");\n" +
            "    if (tVal.value.length > 0) constraint += \"&\" + tVar.value + tOp.options[tOp.selectedIndex].text + percentEncode(tVal.value);\n" +
            "  }\n" +
            "  if (" + formName + ".distinct.checked) constraint += \"&distinct()\";\n" +

            "  var nActiveOb = 0;\n" +
            "  for (var ob = 0; ob < " + nOrderBy + "; ob++) {\n" +
            "    var tOb = eval(\"" + formName + ".orderBy\" + ob);\n" +
            "    var obVar = tOb.options[tOb.selectedIndex].text;\n" +
            "    if (obVar != \"\") {\n" +
            "      constraint += (nActiveOb++ == 0? \"&orderBy(\\\"\" : \",\") + obVar;\n" +
            "      if (active[obVar] === undefined) tResult += (tResult.charAt(tResult.length-1)==\"?\"? \"\" : \",\") + obVar;\n" +
            "    }\n" +
            "  }\n" +
            "  if (nActiveOb > 0) constraint += \"\\\")\";\n" +

            "  var nActiveObm = 0;\n" +
            "  for (var obm = 0; obm < " + nOrderBy + "; obm++) {\n" +
            "    var tObm = eval(\"" + formName + ".orderByMax\" + obm);\n" +
            "    var obmVar = tObm.options[tObm.selectedIndex].text;\n" +
            "    if (obmVar != \"\") {\n" +
            "      constraint += (nActiveObm++ == 0? \"&orderByMax(\\\"\" : \",\") + obmVar;\n" +
            "      if (active[obmVar] === undefined) tResult += (tResult.charAt(tResult.length-1)==\"?\"? \"\" : \",\") + obmVar;\n" +
            "    }\n" +
            "  }\n" +
            "  if (nActiveObm > 0) constraint += \"\\\")\";\n" +

            "  result = tResult + constraint;\n" + //if all is well, make result
            "} catch (e) {alert(e);}\n" +  //result is in 'result'  (or "" if error)
            "" + formName + ".tUrl.value = result;\n";

        //just generate URL
        writer.write("<br>"); 
        writer.write(widgets.button("button", "getUrl", 
            String2.replaceAll(EDStatic.justGenerateAndViewHtml, "&protocolName;", dapProtocol),
            EDStatic.justGenerateAndView, 
            "onclick='" + javaScript + "'"));
        writer.write(widgets.textField("tUrl", 
            EDStatic.justGenerateAndViewUrl,
            95, 1000, "", ""));

        //submit
        writer.write(HtmlWidgets.ifJavaScriptDisabled);
        writer.write("<br>"); 
        writer.write(widgets.button("button", "submit1", 
            EDStatic.submitTooltip, EDStatic.submit, 
            "onclick='" + javaScript +
            "if (result.length > 0) window.location=result;\n" + //or open a new window: window.open(result);\n" +
            "'"));

        //end of form
        writer.write(widgets.endForm());
        writer.write("<br>&nbsp;\n");

        //the javascript for the sliders
        writer.write(widgets.sliderScript(sliderFromNames, sliderToNames, 
            sliderNThumbs, sliderUserValuesCsvs, 
            sliderInitFromPositions, sliderInitToPositions, EDV.SLIDER_PIXELS - 1));

        //be nice
        writer.flush(); 

    }


    /**
     * This write HTML info on forming OPeNDAP DAP-style requests for this type of dataset.
     *
     * @param tErddapUrl  from EDStatic.erddapUrl(loggedInAs)  (erddapUrl, or erddapHttpsUrl if user is logged in)
     * @param writer to which will be written HTML info on forming OPeNDAP 
     *    DAP-style requests for this type of dataset.
     * @param complete if false, this just writes a paragraph and shows a link
     *    to [protocol]/documentation.html
     * @throws Throwable if trouble
     */
    public static void writeGeneralDapHtmlInstructions(String tErddapUrl,
        Writer writer, boolean complete) throws Throwable {

        String dapBase = tErddapUrl + "/" + dapProtocol + "/";
        String datasetBase = dapBase + EDStatic.EDDTableIdExample;
        //all of the fullXxx examples are pre-encoded
        String fullDdsExample      = XML.encodeAsXML(datasetBase + ".dds");
        String fullValueExample    = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDTableDataValueExample);
        String fullTimeExample     = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDTableDataTimeExample);
        String fullTimeMatExample  = XML.encodeAsXML(datasetBase + ".mat?"       + EDStatic.EDDTableDataTimeExample);
        String fullGraphExample    = XML.encodeAsXML(datasetBase + ".png?"       + EDStatic.EDDTableGraphExample);
        String fullGraphMAGExample = XML.encodeAsXML(datasetBase + ".graph?"     + EDStatic.EDDTableGraphExample);
        String fullGraphDataExample= XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDTableGraphExample);
        String fullMapExample      = XML.encodeAsXML(datasetBase + ".png?"       + EDStatic.EDDTableMapExample);
        String fullMapMAGExample   = XML.encodeAsXML(datasetBase + ".graph?"     + EDStatic.EDDTableMapExample);
        String fullMapDataExample  = XML.encodeAsXML(datasetBase + ".htmlTable?" + EDStatic.EDDTableMapExample);
        writer.write(  
            "<h2><a name=\"instructions\">Using</a> tabledap to Request Data and Graphs from Tabular Datasets</h2>\n" +
            longDapDescriptionHtml +
            "<p>tabledap request URLs must be in the form\n" +
            "<br>&nbsp;&nbsp;&nbsp;" + dapBase + " " +
                                       //these links don't work in firefox
                "<i>datasetID</i> " +  //"<i><a href=\"#datasetID\">datasetID</a></i> " + 
                "<i>fileType</i> {?" + //"<i><a href=\"#fileType\">fileType</a></i> {?" + 
                "<i>query</i>}\n" +    //"<i><a href=\"#query\">query</a></i>}\n" +
            "<br>For example,\n" +
            "<br>&nbsp;&nbsp;&nbsp;<a href=\"" + fullTimeExample + "\">" + 
                                                 fullTimeExample + "</a>\n" +
            "\n");

        if (!complete) {
            writer.write(
            "<p>For details, see the <a href=\"" + dapBase + 
                "documentation.html\">" + dapProtocol + " Documentation</a>.\n");
            return;
        }
         
        //details
        writer.write(
            "<p><b>Details:</b><ul>\n" +
            "<li>Requests must not have any internal spaces.\n"+
            "<li>Requests are case sensitive.\n" +
            "<li>{} is notation to denote an optional part of the request.\n" + 
            "  <br>&nbsp;\n" +

            //datasetID
            "<li><a name=\"datasetID\"><i><b>datasetID</b></i></a> identifies the name that ERDDAP\n" +
            "      assigned to the source web site and dataset (for example, \"" + EDStatic.EDDTableIdExample + "\"). \n" +
            "      <br>You can see a list of " +
            "<a href=\"" + tErddapUrl + "/" + dapProtocol + 
                "/index.html\">datasetID options available via tabledap</a>.\n" +
            "   <br>&nbsp;\n" +

            //fileType
            "<li><a name=\"fileType\"><i><b>fileType</b></i></a> specifies the type of table data file that you want to download\n" +
            "     (for example, \".htmlTable\").\n" +
            "   <br>The actual extension of the resulting file may be slightly different than the fileType\n" +
            "     (for example, \".smallPdf\" returns a small .pdf file).\n" +
            "   <br>The fileType options for downloading tabular data are:\n" +
            "   <br>&nbsp;\n" +
            "  <table class=\"erd\" cellspacing=\"0\">\n" + 
            //"  <table border=\"1\" cellpadding=\"2\" cellspacing=\"0\" >\n" +
            "    <tr><th>Data<br>fileTypes</th><th>Description</th><th>Info</th><th>Example</th></tr>\n");
        for (int i = 0; i < dataFileTypeNames.length; i++) 
            writer.write(
                "    <tr>\n" +
                "      <td>" + dataFileTypeNames[i] + "</td>\n" +
                "      <td>" + dataFileTypeDescriptions[i] + "</td>\n" +
                "      <td>" + 
                      (dataFileTypeInfo[i] == null || dataFileTypeInfo[i].equals("")? 
                          "&nbsp;" : "<a href=\"" +  dataFileTypeInfo[i] + "\">info</a>") + 
                      "</td>\n" +
                "      <td><a href=\"" +  datasetBase + dataFileTypeNames[i] + "?" + 
                    XML.encodeAsXML(EDStatic.EDDTableDataTimeExample) + "\">example</a></td>\n" +
                "    </tr>\n");
        writer.write(
            "   </table>\n" +
            "   <br>For example, here is a complete request to download data:\n" +
            "   <br>&nbsp;&nbsp;<a href=\"" + fullTimeExample + "\">" + 
                                              fullTimeExample + "</a>\n" +
            "   \n" +
            "   <p><a name=\"matlab\"><b>MATLAB</b></a> users can download data from within MATLAB.\n" +
            "     Here is a one line example:<pre>\n" +
                 "load(urlwrite('" + fullTimeMatExample + "', 'test.mat'));</pre>\n" +
            "     The data will be in a MATLAB structure. \n" +
                 "The structure's name will be the datasetID (for example, \"" + EDStatic.EDDTableIdExample + "\"). \n" +
            "     <br>The structure's internal variables will be column vectors with the same names as in ERDDAP \n" +
                 "(for example, use \"<tt>fieldnames(" + EDStatic.EDDTableIdExample + ")</tt>\" ). \n" +
            "     <br>You can then make a scatterplot of any two columns. For example:<pre>\n" +
                 EDStatic.EDDTableMatlabPlotExample + "</pre>\n" +
            "\n" +
            "   <p><a name=\"jsonp\"><b>JSONP</b></a> - Requests for .json and .geoJson files may now include an optional\n" +
            "     <a href=\"http://niryariv.wordpress.com/2009/05/05/jsonp-quickly/\">jsonp</a> request\n" +
            "     <br>by adding \"&amp;.jsonp=<i>functionName</i>\" to the end of the query.\n" +
            "     <br>Basically, this just tells ERDDAP to add \"<i>functionName</i>(\" to the beginning of the response \n" +
            "     <br>and \")\" to the end of the response.\n" +
            "     <br>If originally there was no query, leave off the \"&amp;\" in your query.\n"); 

        //imageFile Types, graphs and maps
        writer.write(
            "   <p><a name=\"imageFileTypes\"><b>Making an Image with a Graph or Map of Tabular Data</b></a>\n" +
            "   <br>If a tabledap request URL specifies a subset of data which is suitable for making a graph or a map,\n" +
            "   <br>and the fileType is an image fileType, tabledap will return an image with a graph or map. \n" +
            "   <br>tabledap request URLs can include optional <a href=\"#GraphicsCommands\">graphics commands</a> which let you customize the graph or map.\n" +
            "   <br>As with other tabledap request URLs, you can create these URLs by hand or have a computer program do it. \n" +
            "   <br>Or, you can use the Make A Graph web pages, which simplify creating these URLs (see the \"graph\" links in the table of <a href=\"" + 
                dapBase + "index.html\">tabledap datasets</a>). \n" +
            "\n" +
            "   <p>The fileType options for downloading images of graphs and maps of table data are:\n" +
            "  <table class=\"erd\" cellspacing=\"0\">\n" + 
            //"  <table border=\"1\" cellpadding=\"2\" cellspacing=\"0\" >\n" +
            "    <tr><th>Image<br>fileTypes</th><th>Description</th><th>Info</th><th>Example</th></tr>\n");
        for (int i = 0; i < imageFileTypeNames.length; i++) 
            writer.write(
                "    <tr>\n" +
                "      <td>" + imageFileTypeNames[i] + "</td>\n" +
                "      <td>" + imageFileTypeDescriptions[i] + "</td>\n" +
                "      <td>" + 
                      (imageFileTypeInfo[i] == null || imageFileTypeInfo[i].equals("")? 
                          "&nbsp;" : "<a href=\"" +  imageFileTypeInfo[i] + "\">info</a>") + 
                      "</td>\n" + //must be mapExample below because kml doesn't work with graphExample
                "      <td><a href=\"" +  datasetBase + imageFileTypeNames[i] + "?" + 
                    XML.encodeAsXML(EDStatic.EDDTableMapExample) + "\">example</a></td>\n" +
                "    </tr>\n");
        writer.write(
            "   </table>\n" +
            "   <br>&nbsp;\n"); //linebreak


        //query
        writer.write(
            "<li><a name=\"query\"><i><b>query</b></i></a> is the part of the request after the \"?\".\n" +
            "       It specifies the subset of data that you want to receive.\n" +
            "       <br>In tabledap, it is an \n" +
            "       <a href=\"http://www.opendap.org/\">OPeNDAP</a> " +
            "       <a href=\"http://www.opendap.org/pdf/ESE-RFC-004v1.1.pdf\">DAP</a>\n" +
            "       <a href=\"http://www.opendap.org/user/guide-html/guide_33.html\">constraint query</a> " +
            "       in the form: {<i>resultsVariables</i>}{<i>constraints</i>} .\n " +
            "      <br>For example, \"" + XML.encodeAsXML(
                EDStatic.EDDTableVariablesExample + EDStatic.EDDTableConstraintsExample) + "\".\n" +
            "   <ul>\n" +
            "   <li><i><b>resultsVariables</b></i> is an optional comma-separated list of variables\n" +
            "         (for example, \"" + XML.encodeAsXML(EDStatic.EDDTableVariablesExample) + "\").\n" +
            "     <br>For each variable in resultsVariables, there will be a column in the \n" +
            "         results table, in the same order.\n" +
            "     <br>If you don't specify any results variables, the results table will include\n" +
            "         columns for all of the variables in the dataset.\n" +
            "   <li><i><b>constraints</b></i> is an optional &amp;-separated list of constraints\n" +
            "       (for example, \"" + XML.encodeAsXML(EDStatic.EDDTableConstraintsExample) + "\").\n" +
            "       <ul>\n" +
            "       <li>The constraints determine which rows of data from the original\n" +
            "           table are included in the results table. \n" +
            "         <br>The constraints are applied to each row of the original table.\n" +
            "         <br>If all the constraints evaluate to <tt>true</tt> for a given row,\n" +
            "           that row is included in the results table.\n" +
            "         <br>Thus, \"&amp;\" can be roughly interpreted as \"and\".\n" +
            "       <li>If you don't specify any constraints, all rows from the original table\n" +
            "           will be included in the results table.\n" +
            "         <br>For the fileTypes that don't actually return data (notably, .das, .dds, and .html)\n" +
            "           it is fine not to specify constraints. For example,\n" +
            "         <br><a href=\"" + fullDdsExample + "\">" + 
                                        fullDdsExample + "</a>\n" +
            "         <br>For the fileTypes that do return data, this may result in a very large results table.\n" + 
            "       <li>tabledap constraints are consistent with \n" +
            "         <a href=\"http://www.opendap.org/\">OPeNDAP</a> " +
            "         <a href=\"http://www.opendap.org/pdf/ESE-RFC-004v1.1.pdf\">DAP</a>\n" +
            "         <a href=\"http://www.opendap.org/user/guide-html/guide_33.html\">constraint queries</a>,\n" +
            "         but with a few additional features.\n" +
            "       <li>Each constraint is in the form <tt>&lt;variable&gt;&lt;operator&gt;&lt;value&gt;</tt>\n" +
            "         (for example, \"latitude&gt;45\").\n" +
            "       <li>The valid operators are =, != (not equals), =~ (a regular expression test),\n" +
            "          &lt;, &lt;=, &gt;, and &gt;= .\n" +
            "       <li>tabledap extends the DAP standard to allow any operator to be used with any data type.\n" +
            "         <br>(Standard DAP constraints don't allow &lt;, &lt;=, &gt;, or &gt;= \n" +
            "            to be used with string variables\n" +
            "            and don't allow =~ to be used with numeric variables.)\n" +
            "         <br>For String data, the &lt;, &lt;=, &gt;, and &gt;= operators act in a case-insensitive manner.\n" +
            "         <br>=~ tests if the value from the variable on the left matches the\n" +
            "            <a href=\"http://java.sun.com/javase/6/docs/api/java/util/regex/Pattern.html\">regular expression</a>\n" +
            "            on the right.\n" +
            "           <ul>\n" +
            "           <li>The syntax of regular expressions varies somewhat in different implementations.\n" +
            "              <br>tabledap uses the standard \n" +
            "                <a href=\"http://java.sun.com/javase/6/docs/api/java/util/regex/Pattern.html\">regular expression</a>\n" +
            "                procedures in Java.\n" +
            "           <li>For numeric variables,\n" +
            "                the test is performed on the String representation of the variable's value.\n" +
            "           </ul>\n" +
            "       <li>tabledap extends the DAP standard to allow constraints to be applied to any variable\n" +
            "           in the dataset.\n" +
            "         <br>The constraint variables don't have to be included in the resultsVariables.\n" +
            "         <br>(Standard DAP implementations usually only allow constraints to be applied\n" +
            "           to certain variables. The limitations vary with different implementations.)\n" +           
            "       <li>Although tabledap extends the DAP constraint standard (as noted above),\n" +
            "           these extensions (notably \"=~\") sometimes aren't practical\n" +
            "           because tabledap may need to download lots of extra data from the source\n" +
            "           (which takes time) in order to test the constraint.\n" +
            "       <li>tabledap always stores date/time values as numeric values in\n" +
            "           seconds since 1970-01-01T00:00:00Z.\n" +
            "         <br>Here is an example of a query which includes date/time numeric values:\n" +
            "         <br><a href=\"" + fullValueExample + "\">" + 
                                        fullValueExample + "</a>\n" +
            "         <br>Some fileTypes (notably, .csv, .tsv, .htmlTable, and .xhtml)\n" +
            "            display date/time values as \n" +
            "            <a href=\"http://www.iso.org/iso/date_and_time_format\">ISO 8601:2004 \"extended\" date/time strings</a>\n" +
            "            (e.g., 2002-08-03T12:30:00Z).\n" +
            "       <li>tabledap extends the DAP standard to allow you to specify time values in the \n" +
            "           ISO 8601 date/time format (<i>YYYY-MM-DD</i>T<i>hh:mm:ssZ</i>, where Z is 'Z' or a &plusmn;hh:mm offset from UTC). \n" +
            "         <br>If you omit Z (or the &plusmn;hh:mm offset), :ssZ, :mm:ssZ, or Thh:mm:ssZ from the ISO date/time that you specify, \n" +
            "           the missing fields are assumed to be 0.\n" +
            "         <br>Here is an example of a query which includes ISO date/time values:\n" +
            "         <br><a href=\"" + fullTimeExample + "\">" + 
                                        fullTimeExample + "</a> .\n" +
            "       <li><a name=\"QuoteStrings\">For</a> <a name=\"backslashEncoded\">all</a> constraints of String variables and for regex constraints of numeric variables,\n" +
            "           <br>the right-hand-side value MUST be enclosed in double quotes (e.g., <tt>id=\"NDBC41201\"</tt>) and \n" +
            "           <br>any internal special characters must be backslash encoded: \\ into \\\\, \" into \\\", newline into \\n, and tab into \\t.\n" +
            "       <li><a name=\"PercentEncode\">For</a> all constraints, the <tt>&lt;variable&gt;&lt;operator&gt;&lt;value&gt;</tt>\n" +
            "             MUST be <a href=\"http://en.wikipedia.org/wiki/Percent-encoding\">percent encoded</a>.\n" +
            "           <br>If you submit the request via a browser, the browser will do the percent encoding for you.\n" +
            "           <br>If you submit the request via a computer program, then the program needs to do the percent encoding.\n" +
            "           <br>In practice, this can be very minimal percent encoding: all you usually have to do is convert\n" +
            "           <br>% into %25, &amp; into %26, \", into %22, + into %2B, space into %20 (or +), &lt; into %3C, &gt; into %3E, ~ into %7E,\n" +
            "           <br>and convert all characters above #126 to their %HH form (where HH is the 2-digit hex value).\n" +
            "           <br>Unicode characters above #255 must be UTF-8 encoded and then each byte must be converted to %HH form\n" +
            "           <br>(ask a programmer for help).\n" +
            "       </ul>\n" +
            "   <li><a name=\"filters\"><b>Filters</b></a> - tabledap extends the DAP standard to allow some additional constraints that filter the results.\n" +
            "       <br>Currently, each of the filters takes a table as input and returns a table with the same columns in the same order (but the rows may be altered).\n" +
            "       <br>If you use more than one filter, they will be applied in the order that they appear in the request.\n" +
            "       <br>Each of these options appears near the bottom of the Data Access Form for each dataset.\n" + 
            "       <ul>\n" +
            "       <li><a name=\"distinct\">&amp;distinct()</a> - If you add &amp;distinct() to the end of a query, ERDDAP will sort all of the rows in the results table\n" +
            "          <br>(starting with the first requested variable, then using the second requested variable if the first variable has a tie, ...),\n" +
            "          <br>then remove all non-unique rows of data.\n" +
            "          <br>For example, the query \"stationType,stationID&amp;distinct()\" will return a sorted list of stationIDs associated with each stationType.\n" +
            "          <br>In many situations, ERDDAP can return distinct() values quickly and efficiently.\n" +
            "          <br>But in some cases, ERDDAP must look through all rows of the source dataset. \n" +
            "          <br>If the data set isn't local, this may be VERY slow, may return only some of the results (without an error), or may throw an error.\n" +
            "       <li><a name=\"orderBy\">&amp;orderBy()</a> - If you add &amp;orderBy(\"<i>comma-separated list of variable names</i>\") to the end of a query,\n" +
            "          <br>ERDDAP will sort all of the rows in the results table\n" +
            "          <br>(starting with the first variable, then using the second variable if the first variable has a tie, ...).\n" +
            "          <br>Normally, the rows of data in the response table are in the order that they arrived from the data source.\n" +
            "          <br>orderBy allows you to request that the results table be sorted in a specific way.\n" +
            "          <br>For example, use the query \"stationID,time,temperature&amp;time&gt;2009-04-01&amp;orderBy(\"stationID,time\")\"\n" +
            "          <br>to get the results sorted by stationID, then time.\n" +
            "          <br>Or use the query \"stationID,time,temperature&amp;time&gt;2009-04-01&amp;orderBy(\"time,stationID\")\"\n" +
            "          <br>to get the results sorted by time first, then stationID.\n" +
            "          <br>The orderBy variables MUST be included in the list of requested variables in the query as well as in the orderBy constraint.\n" +
            "       <li><a name=\"orderByMax\">&amp;orderByMax()</a> - If you add &amp;orderByMax(\"<i>comma-separated list of variable names</i>\") to the end of a query,\n" +
            "          <br>ERDDAP will sort all of the rows in the results table\n" +
            "          <br>(starting with the first variable, then using the second variable if the first variable has a tie, ...)\n" +
            "          <br>and then just keeps the rows where the value of the last sort variable is highest (for each combination of other values).\n" +
            "          <br>For example, use the query \"stationID,time,temperature&amp;time&gt;2009-05-21&amp;orderBy(\"stationID,time\")\"\n" +
            "          <br>to get just the rows of data with each station's maximum time value (for stations with data from after 2009-05-21).\n" +
            "          <br>(Without the time constraint, ERDDAP would have to look through all rows of the dataset, which might be VERY slow.)\n" +
            "          <br>The orderByMax variables MUST be included in the list of requested variables in the query as well as in the orderByMax constraint.\n" +
            "          <br>This is the closest thing in tabledap to griddap's allowing requests for the [last] axis value.\n" +
            "       </ul>\n" +
                
            //Graphics Commands
            "   <li><a name=\"GraphicsCommands\"><b>Graphics Commands</b></a> - <a name=\"MakeAGraph\">tabledap</a> extends the DAP standard by allowing graphics commands in the query. \n" +
            "      <br>The Make A Graph web pages simplify the creation of URLs with these graphics commands (see the \"graph\" links in the table of <a href=\"" + 
                dapBase + "index.html\">tabledap datasets</a>). \n" +
            "      <br>So we recommend using the Make A Graph web pages to generate URLs, and then, when needed, using the information here to modify the URLs for special purposes.\n" +
            "      <br>These commands are optional. If present, they must occur after the data request part of the query. \n" +
            "      <br>These commands are used by tabledap if you request an <a href=\"#imageFileTypes\">image fileType</a> (e.g., .png) and are ignored if you request a data file (e.g., .asc). \n" +
            "      <br>If relevant commands are not included in your request, tabledap uses the defaults and tries its best to generate a reasonable graph or map. \n" +
            "      <br>All of the commands are in the form &amp;.<i>commandName</i>=<i>value</i> . \n" +
            "      <br>If the value has sub-values, they are separated by the '|' character. \n" +
            "      <br>In most cases, sub-values can be omitted, causing the default value to be used.\n" +
            "      <br>In most cases, if sub-values are omitted from the end of the list, you can just truncate the list \n" +
            "      (for example, \"a||c\" can be used instead of \"a||c||||\").\n" +
            "      <br>The commands are:\n" +
            "      <ul>\n" +
            "      <li><tt>&amp;.colorBar=</tt> specifies the settings for a color bar.  The sub-values are:\n" +
            "        <ul>\n" +
            "        <li>palette - see a Make A Graph web page for options (e.g., \"Rainbow\"). \n" +
            "          <br>The default (no value) varies based on min and max: if -1*min ~= max, the default is \"BlueWhiteRed\";\n" +
            "          <br>otherwise, the default is \"Rainbow\".\n" +
            "        <li>continuous - must be either no value (the default), 'C' (for Continuous), or 'D' (for Discrete). \n" +
            "          The default is different for different datasets\n" +
            "        <li>scale - must be either no value (the default), \"Linear\", or \"Log\".  \n" +
            "          The default is different for different datasets.\n" +
            "        <li>min - The minimum value for the color bar. The default is different for different datasets.\n" +
            "        <li>max - The maximum value for the color bar. The default is different for different datasets.\n" +
            "        <li>nSections - The preferred number of sections (for Log color bars, this is a minimum value). \n" +
            "          The default is different for different datasets.\n" +
            "        </ul>\n" +
            "      <li><tt>&amp;.color=</tt> specifies the color for lines, markers, vectors, etc. The value must be specified as \n" +
            "        an 0xRRGGBB value (e.g., 0xFF0000 is red, 0x00FF00 is green). The default is 0x000000 (black).\n" +
            "      <li><tt>&amp;.draw=</tt> specifies how the data will be drawn, as \"lines\", \"linesAndMarkers\", \"markers\" (default), \"sticks\", or \"vectors\". \n" +
            "      <li><tt>&amp;.font=</tt> specifies a scale factor for the font (e.g., 1.5 would make the font 1.5 times as big as normal).\n" +
            "      <li><tt>&amp;.marker=</tt> specifies markerType|markerSize . \n" +
            "        <br>markerType is an integer: 0=None, 1=Plus, 2=X, 3=Dot, 4=Square, 5=Filled Square (default), 6=Circle, \n" +
            "           7=Filled Circle, 8=Up Triangle, 9=Filled Up Triangle.\n" +
            "        <br>markerSize is an integer from 3 to 50 (default=5)\n" +
            "      <li>There is no <tt>&amp;.vars=</tt> command. Instead, the results variables from the main part of the query are used.\n" +
            "        <br>The meaning associated with each position varies with the <tt>&amp;.draw</tt> value:\n" +
            "        <ul>\n" +
            "        <li>for lines: xAxis,yAxis\n" +
            "        <li>for linesAndMarkers: xAxis,yAxis,Color\n" +
            "        <li>for markers: xAxis,yAxis,Color\n" +
            "        <li>for sticks: xAxis,uComponent,vComponent\n" +
            "        <li>for vectors: xAxis,yAxis,uComponent,vComponent\n" +
            "        </ul>\n" +
            "        If xAxis=longitude and yAxis=latitude, you get a map; otherwise, you get a graph.\n " +
            "      <li><tt>&amp;.vec=</tt> specifies the data vector length (in data units) to be scaled to the size of \n" + 
            "        the sample vector in the legend. The default varies based on the data.\n" +
            "      <li><tt>&amp;.xRange=</tt> specifies the min|max for the X axis. The default varies based on the data.\n" +
            "      <li><tt>&amp;.yRange=</tt> specifies the min|max for the Y axis. The default varies based on the data.\n" +
            "      </ul>\n" +
            "    <br>A sample graph URL is \n" +
            "    <br><a href=\"" + fullGraphExample + "\">" + 
                                   fullGraphExample + "</a>\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to .graph, you can see a Make A Graph web page with that request loaded:\n" +
            "    <br><a href=\"" + fullGraphMAGExample + "\">" + 
                                   fullGraphMAGExample + "</a>\n" +
            "    <br>That makes it easy for humans to modify an image request to make a similar graph or map.\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to a data fileType (e.g., .htmlTable), you can download the data that was graphed:\n" +
            "    <br><a href=\"" + fullGraphDataExample + "\">" + 
                                   fullGraphDataExample + "</a>\n" +
            "\n" +
            "    <p>A sample map URL is \n" +
            "    <br><a href=\"" + fullMapExample + "\">" + 
                                   fullMapExample + "</a>\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to .graph, you can see a Make A Graph web page with that request loaded:\n" +
            "    <br><a href=\"" + fullMapMAGExample + "\">" + 
                                   fullMapMAGExample + "</a>\n" +
            "\n" +
            "    <p>Or, if you change the fileType in the URL from .png to a data fileType (e.g., .htmlTable), you can download the data that was mapped:\n" +
            "    <br><a href=\"" + fullMapDataExample + "\">" + 
                                   fullMapDataExample + "</a>\n" +
            "  </ul>\n" +
            "</ul>\n" +

            //other info
            "<a name=\"otherInformation\"><b>Other Information</b></a>\n" +
            "<ul>\n" +
            "<li><a name=\"dataModel\"><b>Data Model</b></a> - Each tabledap dataset can be represented as a\n" +
            "a table with one or more rows and one or more columns of data.\n" + 
            "  <ul>\n" +
            "  <li>Each column is also known as a \"data variable\" (or just a \"variable\").\n" +
            "    <br>Each data variable has data of one specific type.\n" +
            "    <br>The supported types are (int8, uint16, int16, int32, int64, float32, float64, String of any length).\n" +
            "    <br>Any cell in the table may have a no value (i.e., a missing value).\n" +
            "    <br>Each data variable has a name composed of a letter (A-Z, a-z) and then 0 or more characters (A-Z, a-z, 0-9, _).\n" +
            "    <br>Each data variable has a set of metadata expressed as Key=Value pairs.\n" +
            "  <li>Each dataset has a set of global metadata expressed as Key=Value pairs.\n" +
            "    <br>Each metadata value is either one or more numeric values (of one type), or one or more Strings\n" +
            "    <br>of any length (using \\n as the separator).\n" +
            "  </ul>\n" +
            "<li><a name=\"specialVariables\"><b>Special Variables</b></a>\n" +
            "  <ul>\n" +
            "  <li>In tabledap, a longitude variable (if present) always has the name \"" + EDV.LON_NAME + 
                 "\" and the units \"" + EDV.LON_UNITS + "\".\n" +
            "  <li>In tabledap, a latitude variable (if present) always has the name \"" + EDV.LAT_NAME + 
                 "\" and the units \"" + EDV.LAT_UNITS + "\".\n" +
            "  <li>In tabledap, an altitude variable (if present) always has the name \"" + EDV.ALT_NAME + 
                 "\" and the units \"" + EDV.ALT_UNITS + "\" above sea level.\n" +
            "     Locations below sea level have negative altitude values.\n" +
            "  <li>In tabledap, a time variable (if present) always has the name \"" + EDV.TIME_NAME + 
                 "\" and the units \"" + EDV.TIME_UNITS + "\".\n" +
            "    <br>If you request data and specify a time constraint, you can specify the time as\n" +
            "      a numeric value (seconds since 1970-01-01T00:00:00Z)\n" +
            "      or as a String value (e.g., \"2002-12-25T07:00:00\" in the GMT/Zulu time zone).\n" +
            "  <li>In tabledap, other variables can be timeStamp variables, which act like the time variable " +
                 "but have a different name.\n" +
            "  <li>Because the longitude, latitude, altitude, and time variables are specifically recognized,\n" +
                "ERDDAP is aware of the geo/temporal features of each dataset.\n" +
                "This is useful when making images with maps or time-series,\n" +
                "and when saving data in geo-referenced file types (e.g., .geoJson and .kml).\n" +
            "  </ul>\n" +
            "<li><a name=\"incompatibilities\"><b>Incompatibilities</b></a>\n" +
            "  <ul>\n" +
            "  <li>Constraints - Different types of data sources support different types of constraints.\n" +
            "    <br>Most data sources don't support all of the types of constraints that tabledap advertises.\n" +
            "    <br>For example, most data sources can't test <tt><i>Variable=~\"RegularExpression\"</i></tt>.\n" +
            "    <br>When a data source doesn't support a given type of constraint, tabledap gets extra data from the source, \n" +
            "    <br>then does that constraint test itself.  Sometimes, getting the extra data takes a lot of time.\n" +
            "  <li>File Types - Some results file types have restrictions.\n" +
            "    <br>For example, .kml is only appropriate for results with longitude and latitude values.\n" +
            "    <br>If a given request is incompatible with the requested file type, tabledap throws an error.\n" + 
            "  </ul>\n" +
            "<li>" + OutputStreamFromHttpResponse.acceptEncodingHtml +
            "</ul>\n" +
            "<br>&nbsp;\n");

        writer.flush(); 

    }                   


    /**
     * This deals with requests for a Make A Graph (MakeAGraph) web page for this dataset. 
     *
     * @param request may be null. If null, no attempt will be made to include 
     *   the loginStatus in startHtmlBody.
     * @param loggedInAs  the name of the logged in user (or null if not logged in).
     *   Normally, this is not used to test if this edd is accessibleTo loggedInAs, 
     *   but it unusual cases (EDDTableFromPost?) it could be.
     *   Normally, this is just used to determine which erddapUrl to use (http vs https).
     * @param requestUrl the part of the user's request, after EDStatic.baseUrl, before '?'.
     *   I think it's currently just used to add to "history" metadata.
     * @param userDapQuery    after the '?', still percentEncoded (shouldn't be null).
     *    If the query has missing or invalid parameters, defaults will be used.
     *    If the query has irrelevant parameters, they will be ignored.
     * @param outputStreamSource  the source of an outputStream that receives the results,
     *    usually already buffered.
     * @param dir the directory to use for temporary/cache files
     * @param fileName the name for the 'file' (no dir, no extension),
     *    which is used to write the suggested name for the file to the response 
     *    header.
     * @param fileTypeName must be .graph
     * @throws Throwable if trouble
     */
    public void respondToGraphQuery(HttpServletRequest request, String loggedInAs,
        String requestUrl, String userDapQuery, 
        OutputStreamSource outputStreamSource,
        String dir, String fileName, String fileTypeName) throws Throwable {

        String tErddapUrl = EDStatic.erddapUrl(loggedInAs);
        if (userDapQuery == null)
            userDapQuery = "";
        if (reallyVerbose)
            String2.log("*** respondToGraphQuery");
        if (debugMode) String2.log("respondToGraphQuery 1");

        String formName = "f1"; //change JavaScript below if this changes

        //gather LLAT range
        StringBuffer llatRange = new StringBuffer();
        int llat[] = {lonIndex, latIndex, altIndex, timeIndex};
        for (int llati = 0; llati < 4; llati++) {
            int dv = llat[llati];
            if (dv < 0) 
                continue;
            EDV edv = dataVariables[dv];
            if (Double.isNaN(edv.destinationMin()) && Double.isNaN(edv.destinationMax())) {}
            else {
                String tMin = Double.isNaN(edv.destinationMin())? "(?)" : 
                    llati == 3? edv.destinationMinString() : ""+((float)edv.destinationMin());
                String tMax = Double.isNaN(edv.destinationMax())? "(?)" : 
                    llati == 3? edv.destinationMaxString() : ""+((float)edv.destinationMax());
                llatRange.append(edv.destinationName() + "&nbsp;=&nbsp;" + 
                    tMin + "&nbsp;to&nbsp;" + tMax + ", ");
            }
        }
        String otherRows = "";
        if (llatRange.length() > 0) {
            llatRange.setLength(llatRange.length() - 2); //remove last ", "
            otherRows = "  <tr><td nowrap>Range:&nbsp;</td> <td>" + llatRange + "</td></tr>\n";
        }

        //write the header
        OutputStream out = outputStreamSource.outputStream("UTF-8");
        Writer writer = new OutputStreamWriter(out, "UTF-8"); 
        HtmlWidgets widgets = new HtmlWidgets("", true, EDStatic.imageDirUrl);
        writer.write(EDStatic.startHeadHtml(tErddapUrl, "Make A Graph for " + 
            XML.encodeAsXML(title() + " from " + institution())));
        writer.write(EDStatic.standardHead);
        writer.write("\n" + rssHeadLink());
        writer.write("\n</head>\n");
        writer.write(EDStatic.startBodyHtml(loggedInAs));
        writer.write("\n");
        writer.write(HtmlWidgets.htmlTooltipScript(EDStatic.imageDirUrl)); //this is a link to a script
        writer.flush(); //Steve Souder says: the sooner you can send some html to user, the better
        try {
            writer.write(EDStatic.youAreHereWithHelp(tErddapUrl, "tabledap", 
                "Make a Graph", 
                "<b>To make a graph of data from this tabular dataset, repeatedly:</b><ol>" +
                "<li>Change the 'Graph Type' and the variables for the graph's axes." +
                "<li>Change the 'Constraints' to specify a subset of the data." +
                "<li>Change the 'Graph Settings' as desired." +
                "<li>Press 'Redraw the Graph'." +
                "</ol>" +
                "This Make A Graph web page just simplifies the creation of tabledap URLs with graphics commands. " +
                "<br>If you want, you can create these URLs by hand or have a computer program do it." +
                "<br>See the 'view the URL' textfield below, or see 'Make A Graph' in the tabledap documentation."));
            writeHtmlDatasetInfo(loggedInAs, writer, true, false, userDapQuery, otherRows);

            //make the big table
            writer.write("<p>\n");
            writer.write(widgets.beginTable(0, 0, "")); 
            writer.write("<tr><td align=\"left\" valign=\"top\">\n"); 

            //begin the form
            writer.write(widgets.beginForm(formName, "GET", "", ""));

            if (debugMode) String2.log("respondToGraphQuery 2");

            //parse the query to get the constraints
            StringArray resultsVariables    = new StringArray();
            StringArray constraintVariables = new StringArray();
            StringArray constraintOps       = new StringArray();
            StringArray constraintValues    = new StringArray(); //times are ""+epochSeconds
            int conPo;
            parseUserDapQuery(userDapQuery, resultsVariables,
                constraintVariables, constraintOps, constraintValues, true);
            //String2.log("conVars=" + constraintVariables);
            //String2.log("conOps=" + constraintOps);
            //String2.log("conVals=" + constraintValues);

            //parse the query so &-separated parts are handy
            String paramName, paramValue, partName, partValue, pParts[];
            String queryParts[] = getUserQueryParts(userDapQuery); //always at least 1 part (may be "")

            //find the numeric dataVariables
            StringArray sa = new StringArray();
            StringArray nonLLSA = new StringArray();
            StringArray axisSA = new StringArray();
            StringArray nonAxisSA = new StringArray();
            EDVTime timeVar = null;
            int dvTime = -1;
            for (int dv = 0; dv < dataVariables.length; dv++) {
                if (dataVariables[dv].destinationDataTypeClass() != String.class) {
                    String dn = dataVariables[dv].destinationName();
                    if        (dv == lonIndex)  {axisSA.add(dn); 
                    } else if (dv == latIndex)  {axisSA.add(dn); 
                    } else if (dv == altIndex)  {axisSA.add(dn); 
                    } else if (dv == timeIndex) {axisSA.add(dn);  dvTime = sa.size(); 
                        timeVar = (EDVTime)dataVariables[dv]; 
                    } else { 
                        nonAxisSA.add(dn);
                    }

                    if (dv != lonIndex && dv != latIndex)
                        nonLLSA.add(dn);

                    sa.add(dn);
                }
            }
            String[] nonAxis = nonAxisSA.toArray(); nonAxisSA = null;
            String[] axis = axisSA.toArray(); axisSA = null;
            String[] nonLL = nonLLSA.toArray();     nonLLSA = null;
            String[] numDvNames = sa.toArray();
            sa.add(0, "");
            String[] numDvNames0 = sa.toArray();
            sa = null;
            String[] dvNames0 = new String[dataVariableDestinationNames.length + 1]; //all, including string vars
            dvNames0[0] = "";
            System.arraycopy(dataVariableDestinationNames, 0, dvNames0, 1, dataVariableDestinationNames.length);

            StringBuffer graphQuery = new StringBuffer();
            String gap = "&nbsp;&nbsp;&nbsp;";

            //*** set the Graph Type    //eeek! what if only 1 var?
            StringArray drawsSA = new StringArray();
            //it is important for javascript below that first 3 options are the very similar (L L&M M)
            drawsSA.add("lines");  
            drawsSA.add("linesAndMarkers"); 
            drawsSA.add("markers");  int defaultDraw = 2;
            if (axis.length >= 1 && nonAxis.length >= 2) 
                drawsSA.add("sticks");
            if (lonIndex >= 0 && latIndex >= 0 && nonAxis.length >= 2) 
                drawsSA.add("vectors");
            String draws[] = drawsSA.toArray();
            int draw = defaultDraw;
            partValue = String2.stringStartsWith(queryParts, partName = ".draw=");
            if (partValue != null) {
                draw = String2.indexOf(draws, partValue.substring(partName.length()));
                if (draw < 0)
                    draw = defaultDraw;
            }
            boolean drawLines = draws[draw].equals("lines");
            boolean drawLinesAndMarkers = draws[draw].equals("linesAndMarkers");
            boolean drawMarkers = draws[draw].equals("markers");
            boolean drawSticks  = draws[draw].equals("sticks");
            boolean drawVectors = draws[draw].equals("vectors");
            writer.write(widgets.beginTable(0, 0, "")); 
            paramName = "draw";
            writer.write(
                "<tr>\n" +
                "  <td nowrap><b>Graph Type:&nbsp;</b>" + 
                "  </td>\n" +
                "  <td>\n");
            writer.write(widgets.select(paramName, "", 
                1, draws, draw, 
                "onChange='mySubmit(" +  //change->submit so form always reflects graph type
                (draw < 3? "f1.draw.selectedIndex<3" : //if old and new draw are <3, do send var names
                    "false") + //else don't send var names
                ");'")); 
            writer.write(
                EDStatic.htmlTooltipImage(
                    "<b>Graph Type</b>" +
                    "<br>Graph Type = 'lines' draws lines on a graph." +
                    "<br>If X=longitude and Y=latitude, you will get a map." +

                    "<p>If Graph Type = 'linesAndMarkers', the optional Color variable can color the markers." +
                    "<br>If X=longitude and Y=latitude, you will get a map." +

                    "<p>If Graph Type = 'markers', the optional Color variable can color the markers." +
                    "<br>If X=longitude and Y=latitude, you will get a map." +

                    "<p>Graph Type = 'sticks' is usually used to plot time on the X axis," +
                    "<br>and with the sticks being draw from the x component and y component" +
                    "<br>of currents or wind data." +

                    "<p>Graph Type = 'vectors' requires that X=longitude and Y=latitude." +
                    "<br>The other variables provide the vector's x component and y component." +
                    "<br>So it is often used for currents or wind data." +

                    "<p><b>Changing the Graph Type automatically submits this form.</b>" +
                    "<br>For the remainder of this form:" +
                    "<br>make changes, then press 'Redraw the Graph' below.") + 
                "  </td>\n" +
                "</tr>\n");
            if (debugMode) String2.log("respondToGraphQuery 3");

            //if no userDapQuery, set default variables
            boolean useDefaultVars = userDapQuery.length() == 0 || userDapQuery.startsWith("&");
            int sameUnits1 = 0; //default  nonAxis index
            if (useDefaultVars) {
                resultsVariables.clear(); //parse() set resultsVariables to all variables

                //heuristic: look for two adjacent nonAxis vars that have same units
                String units1 = null;
                String units2 = findDataVariableByDestinationName(nonAxis[0]).units(); 
                for (int sameUnits2 = 1; sameUnits2 < nonAxis.length; sameUnits2++) {
                    units1 = units2;
                    units2 = findDataVariableByDestinationName(nonAxis[sameUnits2]).units(); 
                    if (units1 != null && units2 != null && units1.equals(units2)) {
                        sameUnits1 = sameUnits2 - 1;
                        break;
                    }
                }
            }

            //set draw-related things
            int nVars = -1, nonAxisPo = 0;
            String varLabel[], varHelp[], varOptions[][];
            if (drawLines) {
                nVars = 2;
                varLabel = new String[]{"X Axis:", "Y Axis:"};
                varHelp  = new String[]{"graph's X Axis.", "graph's X Axis."};
                varOptions = new String[][]{numDvNames, numDvNames};
                if (useDefaultVars) {
                    if (lonIndex >= 0 && latIndex >= 0) {
                        resultsVariables.add(EDV.LON_NAME);
                        resultsVariables.add(EDV.LAT_NAME);
                    } else if (timeIndex >= 0 && nonAxis.length >= 1) {
                        resultsVariables.add(EDV.TIME_NAME);
                        resultsVariables.add(nonAxis[0]);
                    } else if (nonAxis.length >= 2) {
                        resultsVariables.add(nonAxis[0]);
                        resultsVariables.add(nonAxis[1]);
                    } else {
                        resultsVariables.add(numDvNames[0]);
                        if (numDvNames.length > 0) resultsVariables.add(numDvNames[1]);
                    }
                }
            } else if (drawLinesAndMarkers || drawMarkers) {
                nVars = 3;
                varLabel = new String[]{"X Axis:", "Y Axis:", "Color:"};
                varHelp  = new String[]{"graph's X Axis.", "graph's X Axis.", 
                    "marker's color via the Color Bar (or leave blank)."};
                varOptions = new String[][]{numDvNames, numDvNames, numDvNames0};
                if (useDefaultVars) {
                    if (lonIndex >= 0 && latIndex >= 0 && nonAxis.length >= 1) {
                        resultsVariables.add(EDV.LON_NAME);
                        resultsVariables.add(EDV.LAT_NAME);
                        resultsVariables.add(nonAxis[0]);
                    } else if (timeIndex >= 0 && nonAxis.length >= 1) {
                        resultsVariables.add(EDV.TIME_NAME);
                        resultsVariables.add(nonAxis[0]);
                    } else if (nonAxis.length >= 2) {
                        resultsVariables.add(nonAxis[0]);
                        resultsVariables.add(nonAxis[1]);
                    } else {
                        resultsVariables.add(numDvNames[0]);
                        if (numDvNames.length > 0) resultsVariables.add(numDvNames[1]);
                    }
                }
            } else if (drawSticks) {
                nVars = 3;
                varLabel = new String[]{"X Axis:", "Stick X:", "Stick Y:"};
                varHelp  = new String[]{"graph's X Axis.", "stick's x-component.", "stick's y-component."};
                varOptions = new String[][]{numDvNames, numDvNames, numDvNames};
                if (useDefaultVars) {
                    //x var is an axis (prefer: time)
                    resultsVariables.add(timeIndex >= 0? EDV.TIME_NAME : axis[0]);
                    resultsVariables.add(nonAxis[sameUnits1]);
                    resultsVariables.add(nonAxis[sameUnits1 + 1]);
                }
            } else if (drawVectors) {
                nVars = 4;
                varLabel = new String[]{"X Axis:", "Y Axis:", "Vector X:", "Vector Y:"};
                varHelp  = new String[]{"map's X Axis.", "map's Y Axis.", "vector's x-component.", "vector's y-component."};
                varOptions = new String[][]{new String[]{EDV.LON_NAME}, new String[]{EDV.LAT_NAME}, nonAxis, nonAxis};
                if (useDefaultVars) {
                    resultsVariables.add(EDV.LON_NAME);
                    resultsVariables.add(EDV.LAT_NAME);
                    resultsVariables.add(nonAxis[sameUnits1]);
                    resultsVariables.add(nonAxis[sameUnits1 + 1]);
                }
            } else throw new SimpleException("Query error: " +
                "No drawXxx value was set.");
            if (debugMode) String2.log("respondToGraphQuery 4");

            String varName[] = new String[nVars];
            for (int v = 0; v < nVars; v++)
                varName[v] = resultsVariables.size() > v? resultsVariables.get(v) : "";

            //pick variables
            for (int v = 0; v < nVars; v++) {
                String tDvNames[] = varOptions[v];
                int vi = Math.max(0, String2.indexOf(tDvNames, varName[v]));
                varName[v] = tDvNames[vi];
                paramName = "var" + v;
                writer.write(
                    "<tr>\n" +
                    "  <td nowrap>" + varLabel[v] + "&nbsp;" +
                    "  </td>\n" +
                    "  <td>\n");
                writer.write(widgets.select(paramName, 
                    "Select the variable for the " + varHelp[v], 
                    1, tDvNames, vi, ""));
                writer.write(
                    "  </td>\n" +
                    "</tr>\n");
            }

            //end of variables table
            writer.write(widgets.endTable());

            //add varNames to graphQuery
            for (int v = 0; v < nVars; v++) {
                if (varName[v].length() > 0) 
                    graphQuery.append((v == 0? "" : ",") + varName[v]);
            }
            if (debugMode) String2.log("respondToGraphQuery 5");

            //*** write the constraint table
            writer.write("<p>\n");
            writer.write(widgets.beginTable(0, 0, "")); 
            writer.write(
                "<tr>\n" +
                "  <th nowrap align=\"left\">" + 
                    "Constraints " +
                        EDStatic.htmlTooltipImage(
                            "These optional constraints determine which rows of data from the\n" +
                            "<br>original table are plotted on the graph.\n" +
                            "<br>Any or all constraints can be left blank.") +
                "    &nbsp;</th>\n" +
                "  <th nowrap align=\"center\" colspan=\"2\">" + 
                    "Constraint #1 " + 
                    EDStatic.htmlTooltipImage(EDStatic.EDDTableConstraintHtml) + "</th>\n" +
                "  <th nowrap align=\"center\" colspan=\"2\">" + 
                    "Constraint #2 " + 
                    EDStatic.htmlTooltipImage(EDStatic.EDDTableConstraintHtml) + "</th>\n" +
                "</tr>\n");

            //a row for each constrained variable
            int nConRows = 5;
            int conVar[] = new int[nConRows];
            int conOp[][] = new int[nConRows][2];
            String conVal[][] = new String[nConRows][2];
            for (int cv = 0; cv < nConRows; cv++) {
                writer.write("<tr>\n"); 

                //get constraints from query
                //!!! since each constraintVar/Op/Val is removed after processing, 
                //    always work on constraintVar/Op/Val #0
                if (constraintVariables.size() > 0) { 
                    //parse() ensured that all constraintVariables and Ops are valid 
                    conVar[cv]    = String2.indexOf(dvNames0, constraintVariables.get(0));  //yes, 0 (see above)
                    boolean use1  = constraintOps.get(0).startsWith("<");
                    conOp[ cv][use1? 1 : 0] = String2.indexOf(OPERATORS, constraintOps.get(0)); 
                    conVal[cv][use1? 1 : 0] = constraintValues.get(0);    
                    constraintVariables.remove(0); //yes, 0 (see above)
                    constraintOps.remove(0); 
                    constraintValues.remove(0);
                    //another constraint for same var?
                    conPo = constraintVariables.indexOf(dvNames0[conVar[cv]]);
                    if (conPo >= 0) {
                        conOp[ cv][use1? 0 : 1] = String2.indexOf(OPERATORS, constraintOps.get(conPo)); 
                        conVal[cv][use1? 0 : 1] = constraintValues.get(conPo);    
                        constraintVariables.remove(conPo);
                        constraintOps.remove(conPo); 
                        constraintValues.remove(conPo);
                    } else {
                        conOp[ cv][use1? 0 : 1] = opDefaults[use1? 0 : 1];
                        conVal[cv][use1? 0 : 1] = "";
                    }
                } else { //else blanks
                    conVar[cv] = 0;
                    conOp[cv][0] = opDefaults[0]; 
                    conOp[cv][1] = opDefaults[1];
                    conVal[cv][0] = "";  
                    conVal[cv][1] = "";
                }
                EDV conEdv = null;
                if (conVar[cv] > 0)  // >0 since option 0=""
                    conEdv = dataVariables()[conVar[cv] - 1];

                for (int con = 0; con < 2; con++) {
                    String ab = con == 0? "a" : "b";

                    //ensure conVal[cv][con] is valid
                    if (conVal[cv][con].length() > 0) {
                        if (conEdv != null && conEdv instanceof EDVTimeStamp) { 
                            //convert numeric time (from parseUserDapQuery) to iso
                            conVal[cv][con] = ((EDVTimeStamp)conEdv).destinationToString(
                                String2.parseDouble(conVal[cv][con]));
                        } else if (dataVariables[conVar[cv] - 1].destinationDataTypeClass() == String.class) { 
                            //-1 above since option 0=""
                            //leave String constraint values intact
                        } else if (conVal[cv][con].toLowerCase().equals("nan")) { //clean up "NaN"
                            conVal[cv][con] = "NaN"; 
                        } else if (Double.isNaN(String2.parseDouble(conVal[cv][con]))) { 
                            //it should be numeric, but NaN, set to ""
                            conVal[cv][con] = ""; 
                        //else leave as it
                        }
                    }

                    if (cv == 0 && userDapQuery.length() == 0 && timeIndex >= 0) {
                        //suggest a first constraint: last week's worth of data
                        conVar[cv] = timeIndex + 1; // +1 since option 0=""
                        conEdv = dataVariables()[conVar[cv] - 1];
                        double d = dataVariables[timeIndex].destinationMax();
                        if (con == 0) {
                            //suggest previousMidnight-7days
                            GregorianCalendar gc = Double.isNaN(d)?
                                Calendar2.newGCalendarZulu() : //now
                                Calendar2.epochSecondsToGc(d);
                            Calendar2.clearSmallerFields(gc, Calendar2.DATE);
                            d = Calendar2.gcToEpochSeconds(gc) - Calendar2.SECONDS_PER_DAY * 7;
                        }
                        if (!Double.isNaN(d))
                            conVal[cv][con] = Calendar2.epochSecondsToIsoStringT(d) + "Z";
                    }

                    //variable list
                    if (con == 0) {
                        writer.write("  <td nowrap>");
                        writer.write(widgets.select("cVar" + cv, 
                            "Optional: select a variable to be constrained.", 
                            1, dvNames0, conVar[cv], ""));
                        writer.write("  &nbsp;</td>\n");
                    }

                    //make the constraintTooltip
                    //ConstraintHtml below is worded in generic way so works with number and time, too
                    String constraintTooltip = EDStatic.EDDTableMakeAStringConstraintHtml; 
                    //String2.log("cv=" + cv + " conVar[cv]=" + conVar[cv]);
                    if (conEdv != null && conEdv.destinationMinString().length() > 0) {
                        constraintTooltip += "<br>&nbsp;<br>" + 
                            conEdv.destinationName() + " ranges from " +
                            conEdv.destinationMinString() + " to " +
                            (conEdv.destinationMaxString().length() > 0? conEdv.destinationMaxString() : "(?)") + 
                            ".";    
                    }

                    //display the op and value
                    int tOp = conOp[cv][con];
                    writer.write("  <td nowrap>" + gap);
                    writer.write(widgets.select("cOp" + cv + ab, EDStatic.EDDTableSelectAnOperator, 
                        1, OPERATORS, tOp, ""));
                    writer.write("  </td>\n");

                    String tVal = conVal[cv][con];
                    if (tOp == REGEX_OP_INDEX || 
                        (conEdv != null && conEdv.destinationDataTypeClass() == String.class)) {
                        if (tVal.length() > 0)
                            tVal = String2.toJson(tVal); //enclose in "
                    }
                    writer.write("  <td nowrap>");
                    writer.write(widgets.textField("cVal" + cv + ab, constraintTooltip,                    
                        16, 255, tVal, ""));
                    writer.write("  </td>\n");

                    //add to graphQuery
                    if (conVar[cv] > 0 && tVal.length() > 0) 
                        graphQuery.append("&" + dvNames0[conVar[cv]] + 
                            OPERATORS[conOp[cv][con]] + SSR.minimalPercentEncode(tVal));
                }

                //end of row
                writer.write("</tr>\n");
            }
            if (debugMode) String2.log("respondToGraphQuery 6");

            //end of constraint table
            writer.write(widgets.endTable());

            //*** Graph Settings
            //add draw to graphQuery
            graphQuery.append("&.draw=" + draws[draw]); 

            writer.write("<p>\n");
            writer.write(widgets.beginTable(0, 0, "")); 
            writer.write("  <tr><th align=\"left\" colspan=\"2\" nowrap>Graph Settings</th></tr>\n");

            //get Marker settings
            int mType = -1, mSize = -1;
            String mSizes[] = {"3", "4", "5", "6", "7", "8", "9", "10", "11", 
                "12", "13", "14", "15", "16", "17", "18"};
            if (drawLinesAndMarkers || drawMarkers) {
                partValue = String2.stringStartsWith(queryParts, partName = ".marker=");
                if (partValue != null) {
                    pParts = String2.split(partValue.substring(partName.length()), '|');
                    if (pParts.length > 0) mType = String2.parseInt(pParts[0]);
                    if (pParts.length > 1) mSize = String2.parseInt(pParts[1]); //the literal, not the index
                    if (reallyVerbose)
                        String2.log(".marker type=" + mType + " size=" + mSize);
                }

                //markerType
                paramName = "mType";
                if (mType < 0 || mType >= GraphDataLayer.MARKER_TYPES.length)
                    mType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE;
                //if (varName[2].length() > 0 && 
                //    GraphDataLayer.MARKER_TYPES[mType].toLowerCase().indexOf("filled") < 0) 
                //    mType = GraphDataLayer.MARKER_TYPE_FILLED_SQUARE; //needs "filled" marker type
                writer.write("  <tr>\n" +
                             "    <td nowrap>Marker Type:&nbsp;</td>\n" +
                             "    <td nowrap>");
                writer.write(widgets.select(paramName, "", 
                    1, GraphDataLayer.MARKER_TYPES, mType, ""));

                //markerSize
                paramName = "mSize";
                mSize = String2.indexOf(mSizes, "" + mSize); //convert from literal 3.. to index in mSizes[0..]
                if (mSize < 0)
                    mSize = String2.indexOf(mSizes, "" + GraphDataLayer.MARKER_SIZE_SMALL);
                writer.write(gap + "Size: ");
                writer.write(widgets.select(paramName, "", 1, mSizes, mSize, ""));
                writer.write("    </td>\n" +
                             "  </tr>\n");

                //add to graphQuery
                graphQuery.append("&.marker=" + mType + "|" + mSizes[mSize]);
            }
            if (debugMode) String2.log("respondToGraphQuery 7");

            //color
            paramName = "colr";
            String colors[] = HtmlWidgets.PALETTE17;
            partValue = String2.stringStartsWith(queryParts, partName = ".color=0x");
            int colori = String2.indexOf(colors, 
                partValue == null? "" : partValue.substring(partName.length()));
            if (colori < 0)
                colori = String2.indexOf(colors, "000000");
            writer.write("  <tr>\n" +
                         "    <td nowrap>Color:&nbsp;</td>\n" +
                         "    <td nowrap>");
            writer.write(widgets.color17("", paramName, "", colori, ""));
            writer.write("    </td>\n" +
                         "  </tr>\n");

            //add to graphQuery
            graphQuery.append("&.color=0x" + HtmlWidgets.PALETTE17[colori]);

            //color bar
            int palette = 0, continuous = 0, scale = 0, pSections = 0;
            String palMin = "", palMax = "";
            if (drawLinesAndMarkers || drawMarkers) {
                partValue = String2.stringStartsWith(queryParts, partName = ".colorBar=");
                pParts = partValue == null? new String[0] : String2.split(partValue.substring(partName.length()), '|');
                if (reallyVerbose)
                    String2.log(".colorBar=" + String2.toCSVString(pParts));

                //find dataVariable relevant to colorBar
                //(force change in values if this var changes?  but how know, since no state?)
                int tDataVariablePo = String2.indexOf(dataVariableDestinationNames(), 
                    varName[2]); //currently, only relevant representation uses varName[2] for "Color"
                EDV tDataVariable = tDataVariablePo >= 0? dataVariables[tDataVariablePo] : null;

                paramName = "p";
                String defaultPalette = ""; 
                palette = Math.max(0, 
                    String2.indexOf(EDStatic.palettes0, pParts.length > 0? pParts[0] : defaultPalette));
                writer.write("  <tr>\n" +
                             "    <td colspan=\"2\" nowrap>Color Bar: ");
                writer.write(widgets.select(paramName, 
                    "Select a palette for the color bar (or leave blank for the default).", 
                    1, EDStatic.palettes0, palette, ""));

                String conDisAr[] = new String[]{"", "Continuous", "Discrete"};
                paramName = "pc";
                continuous = pParts.length > 1? (pParts[1].equals("D")? 2 : pParts[1].equals("C")? 1 : 0) : 0;
                writer.write(gap + "Continuity: ");
                writer.write(widgets.select(paramName, 
                    "Specify whether the colors should be continuous or discrete<br>(or leave blank for the default).", 
                    1, conDisAr, continuous, ""));

                paramName = "ps";
                String defaultScale = "";
                scale = Math.max(0, String2.indexOf(EDV.VALID_SCALES0, pParts.length > 2? pParts[2] : defaultScale));
                writer.write(gap + "Scale: ");
                writer.write(widgets.select(paramName, 
                    "Select a scale for the color bar (or leave blank for the default).", 
                    1, EDV.VALID_SCALES0, scale, ""));
                writer.write(
                    "    </td>\n" +
                    "  </tr>\n");

                paramName = "pMin";
                String defaultMin = "";
                palMin = pParts.length > 3? pParts[3] : defaultMin;
                writer.write(
                    "  <tr>\n" +
                    "    <td colspan=\"2\" nowrap> " + gap + gap + "Min: ");
                writer.write(widgets.textField(paramName, 
                    "Specify the low value for the color bar (or leave blank for the default).", 
                   10, 40, palMin, ""));

                paramName = "pMax";
                String defaultMax = "";
                palMax = pParts.length > 4? pParts[4] : defaultMax;
                writer.write(gap + "Max: ");
                writer.write(widgets.textField(paramName, 
                    "Specify the high value for the color bar (or leave blank for the default).", 
                   10, 40, palMax, ""));

                paramName = "pSec";
                pSections = Math.max(0, String2.indexOf(EDStatic.paletteSections, pParts.length > 5? pParts[5] : ""));
                writer.write(gap + "N Sections: ");
                writer.write(widgets.select(paramName, 
                    "Specify the number of sections for the color bar" +
                    "<br>and the number of labels (-1)" +
                    "<br>(or leave blank for the default).", 
                    1, EDStatic.paletteSections, pSections, ""));
                writer.write("    </td>\n" +
                             "  </tr>\n");

                //add to graphQuery
                graphQuery.append(
                    "&.colorBar=" + EDStatic.palettes0[palette] + "|" +
                    (conDisAr[continuous].length() == 0? "" : conDisAr[continuous].charAt(0)) + "|" +
                    EDV.VALID_SCALES0[scale] + "|" +
                    palMin + "|" + palMax + "|" + EDStatic.paletteSections[pSections]);
            }

            //Vector Standard 
            String vec = "";
            if (drawVectors) {
                paramName = "vec";
                vec = String2.stringStartsWith(queryParts, partName = ".vec=");
                vec = vec == null? "" : vec.substring(partName.length());
                writer.write(
                    "  <tr>\n" +
                    "    <td nowrap>Vector Standard:&nbsp;</td>\n" +
                    "    <td nowrap>");
                writer.write(widgets.textField(paramName, 
                    "Specify the data vector length (in data units) to be " +
                    "<br>scaled to the size of the sample vector in the legend" +
                    "<br>(or leave blank for the default).", 
                   10, 20, vec, ""));
                writer.write(
                    "    </td>\n" +
                    "  </tr>\n");

                //add to graphQuery
                if (vec.length() > 0)
                    graphQuery.append("&.vec=" + vec);
            }

            //*** end of form         
            //end of graph settings table
            writer.write(widgets.endTable());

            //make javascript function to generate query
            writer.write(
                HtmlWidgets.PERCENT_ENCODE_JS +
                "<script type=\"text/javascript\"> \n" +
                "function makeQuery(varsToo) { \n" +
                "  try { \n" +
                "    var d = document; \n" +
                "    var tVar, c1 = \"\", c2 = \"\", q = \"\"; \n" + //constraint 1|2 q=query
                "    if (varsToo) { \n");
            //vars
            for (int v = 0; v < nVars; v++) {
                writer.write( 
                    "      tVar = d.f1.var" + v + ".options[d.f1.var" + v + ".selectedIndex].text; \n" +
                    "      if (tVar.length > 0) \n" +
                    "        q += (q.length == 0? \"\" : \",\") + tVar; \n");
            }
            writer.write(
                    "    } \n");
            //constraints
            for (int v = 0; v < nConRows; v++) {
                writer.write( 
                    "    tVar = d.f1.cVar" + v + ".options[d.f1.cVar" + v + ".selectedIndex].text; \n" +
                    "    if (tVar.length > 0) { \n" +
                    "      c1 = d.f1.cVal" + v + "a.value; \n" +
                    "      c2 = d.f1.cVal" + v + "b.value; \n" +
                    "      if (c1.length > 0) q += \"&\" + tVar + \n" +
                    "        d.f1.cOp" + v + "a.options[d.f1.cOp" + v + "a.selectedIndex].text + percentEncode(c1); \n" +
                    "      if (c2.length > 0) q += \"&\" + tVar + \n" +
                    "        d.f1.cOp" + v + "b.options[d.f1.cOp" + v + "b.selectedIndex].text + percentEncode(c2); \n" +
                    "    } \n");
            }
            //graph settings   
            writer.write(                
                "    q += \"&.draw=\" + d.f1.draw.options[d.f1.draw.selectedIndex].text; \n");
            if (drawLinesAndMarkers || drawMarkers) writer.write(
                "    q += \"&.marker=\" + d.f1.mType.selectedIndex + \"|\" + \n" +
                "      d.f1.mSize.options[d.f1.mSize.selectedIndex].text; \n");
            writer.write(
                "    q += \"&.color=0x\"; \n" +
                "    for (var rb = 0; rb < " + colors.length + "; rb++) \n" + 
                "      if (d.f1.colr[rb].checked) q += d.f1.colr[rb].value; \n"); //always: one will be checked
            if (drawLinesAndMarkers || drawMarkers) writer.write(
                "    var tpc = d.f1.pc.options[d.f1.pc.selectedIndex].text;\n" +
                "    q += \"&.colorBar=\" + d.f1.p.options[d.f1.p.selectedIndex].text + \"|\" + \n" +
                "      (tpc.length > 0? tpc.charAt(0) : \"\") + \"|\" + \n" +
                "      d.f1.ps.options[d.f1.ps.selectedIndex].text + \"|\" + \n" +
                "      d.f1.pMin.value + \"|\" + d.f1.pMax.value + \"|\" + \n" +
                "      d.f1.pSec.options[d.f1.pSec.selectedIndex].text; \n");
            if (drawVectors) writer.write(
                "    if (d.f1.vec.value.length > 0) q +=  \"&.vec=\" + d.f1.vec.value; \n");
            writer.write(
                "    return q; \n" +
                "  } catch (e) { \n" +
                "    alert(e); \n" +
                "    return \"\"; \n" +
                "  } \n" +
                "} \n" +
                "function mySubmit(varsToo) { \n" +
                "  var q = makeQuery(varsToo); \n" +
                "  if (q.length > 0) window.location=\"" + //javascript uses length, not length()
                    tErddapUrl + "/tabledap/" + datasetID() + 
                    ".graph?\" + q;\n" + 
                "} \n" +
                "</script> \n");  

            //submit
            writer.write(HtmlWidgets.ifJavaScriptDisabled);
            writer.write("<p>\n"); 
            writer.write(widgets.beginTable(0, 0, "")); 
            writer.write(
                "<tr><td nowrap>");
            writer.write(widgets.htmlButton("button", "", "",
                "Redraw the graph based on the settings above. " +
                "<br>Please be patient; sometimes it takes a long time " +
                "<br>to get the data from the source.", 
                "<b>Redraw the Graph</b>", 
                "onMouseUp='mySubmit(true);'")); 
            writer.write(
                "</td></tr>\n");

            //Download the Data
            writer.write(
                "<tr><td nowrap>Then set the File Type:\n");
            paramName = "fType";
            String xmlEncodedGraphQuery = XML.encodeAsXML(graphQuery.toString());
            writer.write(widgets.select(paramName, EDStatic.EDDSelectFileType, 1,
                allFileTypeNames, defaultFileTypeOption, 
                "onChange='f1.tUrl.value=\"" + tErddapUrl + "/tabledap/" + datasetID() + 
                    "\" + f1.fType.options[f1.fType.selectedIndex].text + " +
                    "\"?" + xmlEncodedGraphQuery + "\";'"));
            writer.write(" and\n");
            writer.write(widgets.button("button", "", 
                "Click here to download the data in the specified File Type. " +
                    "<br>Please be patient; sometimes it takes a long time " +
                    "<br>to get the data from the source.",
                "Download the Data or an Image", 
                "onMouseUp='window.location=\"" + 
                    tErddapUrl + "/tabledap/" + datasetID() + 
                    "\" + f1." + paramName + ".options[f1." + paramName + ".selectedIndex].text + \"?" + 
                    xmlEncodedGraphQuery +
                    "\";'")); //or open a new window: window.open(result);\n" +
            writer.write(
                "</td></tr>\n");

            //view the url
            writer.write(
                "<tr><td nowrap>&nbsp;&nbsp;or view the URL: \n");
            writer.write(widgets.textField("tUrl",   
                String2.replaceAll(EDStatic.justGenerateAndViewGraphUrlHtml, 
                    "&protocolName;", dapProtocol), 
                52, 1000, 
                tErddapUrl + "/tabledap/" + datasetID() + 
                    dataFileTypeNames[defaultFileTypeOption] + "?" + graphQuery.toString(), 
                ""));
            if (debugMode) String2.log("respondToGraphQuery 8");
            writer.write(
                "</td></tr>\n" +
                "</table>\n\n");

            //end form
            writer.write(widgets.endForm());

            //end of left half of big table
            writer.write("</td>\n" +
                "<td>" + gap + "</td>\n" + //gap in center
                "<td align=\"left\" valign=\"top\">\n"); //begin right half

            //display the graph
            if (verbose) 
                String2.log("graphQuery=" + graphQuery);
            writer.write("<img alt=\"The graph you specified. Please be patient. It may take a long time to get data from the source.\" " +
                "src=\"" + tErddapUrl + "/tabledap/" + //don't use \n for the following lines
                datasetID()+ ".png?" + xmlEncodedGraphQuery + "\">\n"); //no img end tag

            //end of right half of big table
            writer.write("</td></tr></table>\n");

            //do things with graphs
            writer.write("<hr noshade>\n" +
                "<h2><a name=\"uses\">Things</a> You Can Do With Your Graphs</h2>\n" +
                "<ul>\n" +
                "<li>Web page authors can \n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/embed.html\">embed a graph of the latest data in a web page</a> \n" +
                "  using HTML &lt;img&gt; tags.\n" +
                "<li>Anyone can use <a href=\"" + tErddapUrl + "/slidesorter.html\">Slide Sorter</a> \n" +
                "  to build a personal web page that displays graphs of the latest data, each in its own, draggable slide.\n" +
                "<li>Anyone can use or make\n" +
                "  <a href=\"http://coastwatch.pfeg.noaa.gov/erddap/images/gadgets/GoogleGadgets.html\">Google Gadgets</a>\n" +
                "  to display images with the latest data on their iGoogle home page. \n" +
                "  <br>&nbsp;\n" +
                "</ul>\n" +
                "\n");

            //end of document
            writer.write("<hr noshade>\n");
            writer.write("<h2>The Dataset Attribute Structure (.das) for this Dataset</h2>\n" +
                "<pre>\n");
            Table table = makeEmptyDestinationTable("/tabledap/" + datasetID() + ".das", "", true);
            table.writeDAS(writer, SEQUENCE_NAME, true); //useful so search engines find all relevant words
            writer.write("</pre>\n");
            writer.write("<br>&nbsp;\n");
            writer.write("<hr noshade>\n");
            writeGeneralDapHtmlInstructions(tErddapUrl, writer, false); 
            writer.write("<br>&nbsp;\n" +
                "<br><small>" + EDStatic.ProgramName + " Version " + EDStatic.erddapVersion + "</small>\n");
        } catch (Throwable t) {
            writer.write(EDStatic.htmlForException(t));
        }
        if (EDStatic.displayDiagnosticInfo) 
            EDStatic.writeDiagnosticInfoHtml(writer);
        writer.write(EDStatic.endBodyHtml(tErddapUrl));
        writer.write("\n</html>\n");

        //essential
        writer.flush(); 
        out.close();
    }

    /**
     * This looks for var,op in constraintVariable,constraintOp (or returns -1).
     *
     */
    protected static int indexOf(StringArray constraintVariables, StringArray constraintOps,
        String var, String op) {
        int n = constraintVariables.size();
        for (int i = 0; i < n; i++) {
            if (constraintVariables.get(i).equals(var) &&
                constraintOps.get(i).equals(op))
                return i;
        }
        return -1;
    }

    /**
     * This returns the number of offerings (e.g., stations).
     * Subclasses may overwrite this.
     *
     * @return the number of offerings (e.g., stations)
     *   (or 0 if this dataset can't be served via SOS because no offerings or no observedProperties)
     */
    public int sosNOfferings() {
        int result = idIndex >= 0 && lonIndex >= 0 && latIndex >= 0 && timeIndex >= 0 &&
            sosObservedProperties != null && sosObservedProperties.length > 0 &&
            sosOfferingName != null?  
            sosOfferingName.size() : 0;
        if (reallyVerbose) String2.log("sosNOffering(" + datasetID + ")=" + result + 
            "; indices: id=" + idIndex +
            " lon=" + lonIndex + " lat=" + latIndex + " time=" + timeIndex +
            "\n  nSosObsProp=" + (sosObservedProperties == null? -1 : sosObservedProperties.length) +
            " nSosOffName=" + (sosOfferingName == null? -1 : sosOfferingName.size()));
        return result;
    }

    /**
     * This returns the ith offering (e.g., station) name (e.g., 41012).
     * This throws an exception if i is invalid.
     */
    public String sosOfferingName(int i) {return sosOfferingName.getString(i); }

    /**
     * This returns the ith offering (e.g., station) minLon.
     * This throws an exception if i is invalid.
     * Subclasses may overwrite this.
     */
    public double sosMinLongitude(int i) {return sosMinLongitude.getNiceDouble(i); }

    /**
     * This returns the ith offering (e.g., station) maxLon.
     * This throws an exception if i is invalid.
     * Subclasses may overwrite this.
     */
    public double sosMaxLongitude(int i) {return sosMaxLongitude.getNiceDouble(i); }

    /**
     * This returns the ith offering (e.g., station) minLat.
     * This throws an exception if i is invalid.
     * Subclasses may overwrite this.
     */
    public double sosMinLatitude(int i) {return sosMinLatitude.getNiceDouble(i); }

    /**
     * This returns the ith offering (e.g., station) maxLat.
     * This throws an exception if i is invalid.
     * Subclasses may overwrite this.
     */
    public double sosMaxLatitude(int i) {return sosMaxLatitude.getNiceDouble(i); }

    /**
     * This returns the ith offering (e.g., station) minTime (epoch seconds).
     * This throws an exception if i is invalid.
     * Subclasses may overwrite this.
     */
    public double sosMinTime(int i) {return sosMinTime.getNiceDouble(i); }

    /**
     * This returns the ith offering (e.g., station) maxTime (epoch seconds).
     * This throws an exception if i is invalid.
     * Subclasses may overwrite this.
     */
    public double sosMaxTime(int i) {return sosMaxTime.getNiceDouble(i); }

    /**
     * This return the sorted list of SOS observedProperties for this dataset.
     * This return String[0] if none.
     * Subclasses may overwrite this.
     */
    public String[] sosObservedProperties() {return sosObservedProperties; }

    /**
     * This is called by EDDTable.ensureValid by subclass constructors,
     * to go through the metadata of the dataVariables and gather
     * all unique sosObservedProperties.
     */
    protected void gatherSosObservedProperties() {
        HashSet set = new HashSet();
        for (int dv = 0; dv < dataVariables.length; dv++) {
            String s = dataVariables[dv].combinedAttributes().getString("observedProperty");
            if (s != null)
               set.add(s);
        }
        sosObservedProperties = String2.toStringArray(set.toArray()); 
        Arrays.sort(sosObservedProperties);
    }

    /**
     * This writes the Google Visualization Query Language documentation .html page.

     */
    /*
       Google defined the services that a Google Data Source (GDS) needs to provide.
       It is a sophisticated, SQL-like query system.
       Unfortunately, the DAP query system that ERDDAP supports is not as 
       sophisticated as the GDS system. 
       Similarly, none of the remote data sources (with the exception of databases)
       have a query system as sophisticated as GDS.
       So ERDDAP can't process a GDS request by simply converting it into a DAP request,
       or by passing the GDS request to the remote server.

       Google recognized that it would be hard for people to create servers
       that could handle the GDS requests.
       So Google kindly released Java code to simplify creating a GDS server.
       It works by having the server program (e.g., ERDDAP) feed in the entire 
       dataset in order to process each request.
       It then does all of the hard work to filter and process the request.
       This works great for the vast majority of datasets on the web,
       where the dataset is small (a few hundred rows) and can be accessed quickly (i.e., is stored locally).
       Unfortunately, most datasets in ERDDAP are large (10's or 100's of thousands of rows)
       and are accessed slowly (because they are remote).
       Downloading the entire dataset just to process one request just isn't feasible.
       
       For now, ERDDAP's compromise solution is to just accept a subset of 
       the Google Query Language commands. 
       This way, ERDDAP can pass most of the constraints on to the remote data source
       (as it always has) and let the remote data source do most of the subset selection.
       Then ERDDAP can do the final subset selection and processing and 
       return the response data table to the client.
       This is not ideal, but there has to be some compromise to allow ERDDAP's
       large, remote datasets (accessible via the DAP query system) 
       to be accessible as Google Data Sources.
        
    */

    /**
     * This extracts a CSV list of variable/column names (gVis style)
     * 
     * @param start is the index in gVisQuery of white space or the first letter
     *    of the first name
     * @param names receives the list of variable/column names.
     *     It must be empty initially.
     * @return the new value of start (or -1 if names is an error message)
     */
    private int getGVisColumnNames(String gVisQuery, int start, StringBuffer names) {
        names.setLength(0);
        int gVisQueryLength = gVisQuery.length();
        do {
            if (names.length() > 0) {
                //eat the required ',' (see matching 'while' at bottom) 
                start++;

                //add , to names
                names.append(',');
            }

            //eat any spaces
            while (gVisQuery.startsWith(" ", start)) 
                start++;

            String name;
            if (gVisQuery.startsWith("'", start)) {
                //'colName'
                int po = gVisQuery.indexOf("'", start + 1);
                if (po < 0) {
                    names.setLength(0);
                    names.append("Query error: missing \"'\" after position=" + start);
                    return -1;
                }
                name = gVisQuery.substring(start + 1, po);
                start = po + 1;                        
            } else {
                //unquoted colName
                int po = start + 1;
                while (po < gVisQueryLength &&
                       !gVisQuery.startsWith(",", po) &&  
                       !gVisQuery.startsWith(" ", po))
                    po++;
                name = gVisQuery.substring(start, po);
                start = po;
            }
            //ensure the name is valid
            if (String2.indexOf(dataVariableDestinationNames, name) < 0) {
                names.setLength(0);
                names.append("Query error: unrecognized column name=\""+ name + 
                    "\" at position=" + (start - name.length()));
                return -1;
            }
            names.append(name);
            while (gVisQuery.startsWith(" ", start)) 
                start++;

        } while (gVisQuery.startsWith(",", start));
        return start;
    }

    /**
     * This generates a DAP query from a Google Visualization API Query Language string.
     * <br>See Query Language reference http://code.google.com/apis/visualization/documentation/querylanguage.html
     *
     * @param tgVisQuery  the case of the key words (e.g., SELECT) is irrelevant
     * @return dapQuery
     * @throws Throwable if trouble
     *  ???or if error, return gVis-style error message string?
     */
    public String gVisQueryToDapQuery(String gVisQuery) {
        //SELECT [DISTINCT] csvRequestColumns [WHERE constraint [AND constraint]] [ORDER BY csvOrderBy] 
        //csvRequestColumns[&constraint][&distinct()][&orderBy csvOrderBy
        int gVisQueryLength = gVisQuery.length();
        String gVisQueryLC = gVisQuery.toLowerCase();
        StringBuffer dapQuery = new StringBuffer();

        //SELECT
        int start = 0;
        if (gVisQueryLC.startsWith("select ", start)) {
            start += 7;
            while (gVisQueryLC.startsWith(" ", start)) 
                start++;
            if (gVisQueryLC.startsWith("*", start)) {
                start++;
                while (gVisQueryLC.startsWith(" ", start)) 
                    start++;
            } else {
                do {
                    if (dapQuery.length() > 0) {
                        //eat the required ',' and optional white space
                        start++;
                        while (gVisQueryLC.startsWith(" ", start)) 
                            start++;
                        //add , to dapQuery
                        dapQuery.append(',');
                    }
                    String name;
                    if (gVisQueryLC.startsWith("'", start)) {
                        //'colName'
                        int po = gVisQueryLC.indexOf("'", start + 1);
                        if (po < 0)
                            return "Query error: missing \"'\" after position=" + start;
                        name = gVisQuery.substring(start + 1, po);
                        start = po + 1;                        
                    } else {
                        //unquoted colName
                        int po = start + 1;
                        while (po < gVisQueryLength &&
                               !gVisQueryLC.startsWith(",", po) &&  
                               !gVisQueryLC.startsWith(" ", po))
                            po++;
                        name = gVisQuery.substring(start, po);
                        start = po;
                    }
                    if (String2.indexOf(dataVariableDestinationNames, name) < 0)
                        return "Query error: unrecognized column name=\""+ name + 
                            "\" at position=" + (start - name.length());
                    dapQuery.append(name);
                    while (gVisQueryLC.startsWith(" ", start)) 
                        start++;

                } while (gVisQueryLC.startsWith(",", start));
            }
        }

        //WHERE
        if (gVisQueryLC.startsWith("where ", start)) {
        }

        //ORDER BY
        if (gVisQueryLC.startsWith("order by ", start)) {
        }

        return dapQuery.toString();
    }




    /** This tests some of the methods in this class. 
     * @throws Throwable if trouble
     */
    public static void test() throws Throwable {
        //numeric testValueOpValue
        //"!=", REGEX_OP, "<=", ">=", "=", "<", ">"}; 
        double nan = Double.NaN;
        Test.ensureEqual(testValueOpValue(1,   "=",  1), true,  "");
        Test.ensureEqual(testValueOpValue(1,   "=",  2), false, "");
        Test.ensureEqual(testValueOpValue(1,   "=",  nan), false, "");
        Test.ensureEqual(testValueOpValue(nan, "=",  1), false, "");
        Test.ensureEqual(testValueOpValue(nan, "=",  nan), true, "");

        Test.ensureEqual(testValueOpValue(1,   "!=", 1), false,  "");
        Test.ensureEqual(testValueOpValue(1,   "!=", 2), true, "");
        Test.ensureEqual(testValueOpValue(1,   "!=", nan), true, "");
        Test.ensureEqual(testValueOpValue(nan, "!=", 1), true, "");
        Test.ensureEqual(testValueOpValue(nan, "!=", nan), false, "");

        Test.ensureEqual(testValueOpValue(1,   "<=", 1), true,  "");
        Test.ensureEqual(testValueOpValue(1,   "<=", 2), true, "");
        Test.ensureEqual(testValueOpValue(2,   "<=", 1), false, "");
        Test.ensureEqual(testValueOpValue(1,   "<=", nan), false, "");
        Test.ensureEqual(testValueOpValue(nan, "<=", 1), false, "");
        Test.ensureEqual(testValueOpValue(nan, "<=", nan), false, "");

        Test.ensureEqual(testValueOpValue(1,   "<",  1), false,  "");
        Test.ensureEqual(testValueOpValue(1,   "<",  2), true, "");

        Test.ensureEqual(testValueOpValue(1,   ">=", 1), true,  "");
        Test.ensureEqual(testValueOpValue(1,   ">=", 2), false, "");
        Test.ensureEqual(testValueOpValue(2,   ">=", 1), true, "");

        Test.ensureEqual(testValueOpValue(2,   ">",  1), true,  "");
        Test.ensureEqual(testValueOpValue(1,   ">",  2), false, "");
        //regex tests always via testValueOpValue(string)

        //string testValueOpValue
        //"!=", REGEX_OP, "<=", ">=", "=", "<", ">"}; 
        String s = "";
        Test.ensureEqual(testValueOpValue("a", "=",  "a"), true,  "");
        Test.ensureEqual(testValueOpValue("a", "=",  "B"), false, "");
        Test.ensureEqual(testValueOpValue("a", "=",  s), false, "");
        Test.ensureEqual(testValueOpValue(s,   "=",  "a"), false, "");
        Test.ensureEqual(testValueOpValue(s,   "=",  s), true, "");

        Test.ensureEqual(testValueOpValue("a", "!=", "a"), false,  "");
        Test.ensureEqual(testValueOpValue("a", "!=", "B"), true, "");
        Test.ensureEqual(testValueOpValue("a", "!=", s), true, "");
        Test.ensureEqual(testValueOpValue(s,   "!=", "a"), true, "");
        Test.ensureEqual(testValueOpValue(s,   "!=", s), false, "");

        Test.ensureEqual(testValueOpValue("a", "<=", "a"), true,  "");
        Test.ensureEqual(testValueOpValue("a", "<=", "B"), true, "");
        Test.ensureEqual(testValueOpValue("B", "<=", "a"), false, "");
        Test.ensureEqual(testValueOpValue("a", "<=", s), false, "");
        Test.ensureEqual(testValueOpValue(s,   "<=", "a"), true, "");
        Test.ensureEqual(testValueOpValue(s,   "<=", s), true, "");

        Test.ensureEqual(testValueOpValue("a", "<",  "a"), false,  "");
        Test.ensureEqual(testValueOpValue("a", "<",  "B"), true, "");

        Test.ensureEqual(testValueOpValue("a", ">=", "a"), true,  "");
        Test.ensureEqual(testValueOpValue("a", ">=", "B"), false, "");
        Test.ensureEqual(testValueOpValue("B", ">=", "a"), true, "");

        Test.ensureEqual(testValueOpValue("B", ">",  "a"), true,  "");
        Test.ensureEqual(testValueOpValue("a", ">",  "B"), false, "");

        Test.ensureEqual(testValueOpValue("12345", REGEX_OP, "[0-9]+"), true,  "");
        Test.ensureEqual(testValueOpValue("12a45", REGEX_OP, "[0-9]+"), false, "");

        //test speed
        long tTime = System.currentTimeMillis();
        int n = 1000000;
        for (int i = 0; i < n; i++) {
            Test.ensureEqual(testValueOpValue("abcdefghijk", "=",  "abcdefghijk"), true,  "");
            Test.ensureEqual(testValueOpValue("abcdefghijk", "!=", "abcdefghijk"), false,  "");
            Test.ensureEqual(testValueOpValue("abcdefghijk", "<=", "abcdefghijk"), true, "");
            Test.ensureEqual(testValueOpValue("abcdefghijk", "<",  "abcdefghijk"), false,  "");
            Test.ensureEqual(testValueOpValue("abcdefghijk", ">=", "abcdefghijk"), true,  "");
            Test.ensureEqual(testValueOpValue("abcdefghijk", ">",  "abcdefghijk"), false,  "");
            Test.ensureEqual(testValueOpValue("12345", REGEX_OP, "[0-9]+"), true,  "");
        }
        String2.log("time for " + (7 * n) + " testValueOpValue(string): " + (System.currentTimeMillis() - tTime));

        tTime = System.currentTimeMillis();
        for (int i = 0; i < n; i++) {
            Test.ensureEqual(testValueOpValue(1, "=",  1), true,  "");
            Test.ensureEqual(testValueOpValue(1, "!=", 1), false,  "");
            Test.ensureEqual(testValueOpValue(1, "<=", 1), true,  "");
            Test.ensureEqual(testValueOpValue(1, "<",  1), false,  "");
            Test.ensureEqual(testValueOpValue(1, ">=", 1), true,  "");
            Test.ensureEqual(testValueOpValue(2, ">",  1), true,  "");
            Test.ensureEqual(testValueOpValue(1, ">",  2), false, "");
            //regex tests always via testValueOpValue(string)
        }
        String2.log("time for " + (7 * n) + " testValueOpValue(double): " + (System.currentTimeMillis() - tTime));

        tTime = System.currentTimeMillis();
        for (int i = 0; i < 7*n; i++) {
            Test.ensureEqual(testValueOpValue(1, "<=",  1), true,  "");
        }
        String2.log("time for " + (7 * n) + " testValueOpValue(double <=): " + (System.currentTimeMillis() - tTime));
    }
}
