/* 
 * EDDTableFromHyraxFiles Copyright 2009, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.ShortArray;
import com.cohort.array.LongArray;
import com.cohort.array.NDimensionalIndex;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.String2;
import com.cohort.util.Test;

/** The Java DAP classes.  */
import dods.dap.*;

import gov.noaa.pfel.coastwatch.griddata.OpendapHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.*;

import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.List;

import org.joda.time.*;
import org.joda.time.format.*;

/** 
 * This class represents a directory tree with a collection of n-dimensional (1,2,3,4,...) 
 * opendap files, each of which is flattened into a table.
 * For example, http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ 
 * (the four dimensions there are e.g., time,depth,lat,lon).
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2009-06-08
 */
public class EDDTableFromHyraxFiles extends EDDTableFromFiles { 

    /** Indicates if data can be transmitted in a compressed form.
     * It is unlikely anyone would want to change this. */
    public static boolean acceptDeflate = true;

    /** Format for hyrax date time, e.g., 03-Apr-2009 17:15 */
    protected DateTimeFormatter dateTimeFormatter;

    //info from last getFileNames
    private StringArray lastFileDirs;
    //these three are have matching values for each file:
    private ShortArray  lastFileDirIndex; 
    private StringArray lastFileName; 
    private LongArray   lastFileLastMod; 

    private String[] sourceAxisNames;

    /** 
     * The constructor just calls the super constructor. 
     *
     * @param tAccessibleTo is a comma separated list of 0 or more
     *    roles which will have access to this dataset.
     *    <br>If null, everyone will have access to this dataset (even if not logged in).
     *    <br>If "", no one will have access to this dataset.
     * <p>The sortedColumnSourceName can't be for a char/String variable
     *   because NcHelper binary searches are currently set up for numeric vars only.
     */
    public EDDTableFromHyraxFiles(String tDatasetID, String tAccessibleTo,
        StringArray tOnChange, 
        int tNDimensions,
        Attributes tAddGlobalAttributes,
        double tAltMetersPerSourceUnit, 
        Object[][] tDataVariables,
        int tReloadEveryNMinutes,
        String tFileDir, boolean tRecursive, String tFileNameRegex, String tMetadataFrom,
        String tPreExtractRegex, String tPostExtractRegex, String tExtractRegex, 
        String tColumnNameForExtract,
        String tSortedColumnSourceName, String tSortFilesBySourceNames) 
        throws Throwable {

        super("EDDTableFromHyraxFiles", false, tDatasetID, tAccessibleTo, tOnChange, 
            tNDimensions, tAddGlobalAttributes, tAltMetersPerSourceUnit, 
            tDataVariables, tReloadEveryNMinutes,
            tFileDir, tRecursive, tFileNameRegex, tMetadataFrom,
            tPreExtractRegex, tPostExtractRegex, tExtractRegex, tColumnNameForExtract,
            tSortedColumnSourceName, tSortFilesBySourceNames);

    }

    /**
     * This is the default implementation of getFileNames, which
     * gets file names from a local directory.
     * This overrides the superclass version of this method.
     *
     * @param fileDir for this version of this method, fileDir is the
     *  url of the base web page
     *  e.g., http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/
     *
     * @returns an array with a list of full file names 
     * @throws Throwable if trouble
     */
    public String[] getFileNames(String fileDir, String fileNameRegex, boolean recursive) throws Throwable {
        if (reallyVerbose) String2.log("EDDTableFromHyraxFiles getFileNames");
        //clear the stored file info
        lastFileDirs     = new StringArray();
        lastFileDirIndex = new ShortArray(); 
        lastFileName     = new StringArray(); 
        lastFileLastMod  = new LongArray(); 

        //This is used by superclass' constructor, so must be made here, instead of where defined
        //Format for date time in hyrax .html lists of files, e.g., 03-Apr-2009 17:15 
        //timeZone doesn't matter as long as consistent for a given instance
        if (dateTimeFormatter == null)
            dateTimeFormatter = DateTimeFormat.forPattern("dd-MMM-yyyy HH:mm").withZone(DateTimeZone.UTC);

        //call the recursive method
        getFileNameInfo(fileDir, fileNameRegex, recursive);

        //assemble the fileNames
        int n = lastFileDirIndex.size();
        String tNames[] = new String[n];
        for (int i = 0; i < n; i++) 
            tNames[i] = lastFileDirs.array[lastFileDirIndex.array[i]] + lastFileName.array[i];
        return tNames;
    }

    /**
     * This does the work for getFileNames.
     * This calls itself recursively, adding into to fileNameInfo as it is found.
     *
     * @param url the url of the current web page (with a hyrax catalog)
     * @throws Throwable if trouble, e.g., if url doesn't respond
     */
    private void getFileNameInfo(String url, String fileNameRegex, boolean recursive) throws Throwable {
        if (reallyVerbose) String2.log("\ngetFileNameInfo url=" + url + " regex=" + fileNameRegex);
        String response = SSR.getUrlResponseString(url);

        //skip to start of file,dir listings
        int po = response.indexOf("<pre>");
        if (po < 0)
            return;
        po += 5;

        //skip header line and parent directory
        po = response.indexOf(">Parent Directory<", po);
        if (po < 0)
            return;
        po += 18;

        //go through file,dir listings
        while (true) {
            //<A HREF="http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/">ANO001/</a> 03-Apr-2009 17:15  673K
            po = response.indexOf("<A HREF=\"", po);
            int po2 = response.indexOf("\">", po + 9);
            int po3 = response.indexOf("</a>", po2 + 2);
            if (po < 0 || po2 < 0 || po3 < 0)
                return;  //no more href's
            po += 9;

            String tUrl = response.substring(po, po2);
            if (reallyVerbose) String2.log("  url=" + tUrl);
            if (tUrl.endsWith("/")) {
                //it's a directory
                if (recursive)
                    getFileNameInfo(tUrl, fileNameRegex, recursive);
            } else if (tUrl.endsWith(".html")) {

                //it's a web page (perhaps a DAP .html page)
                tUrl = tUrl.substring(0, tUrl.length() - 5);
                String tDir = File2.getDirectory(tUrl);
                String tName = File2.getNameAndExtension(tUrl);
                if (tName.matches(fileNameRegex)) {
                    //it's a match!
                    long tLastMod = 0;
                    String sourceTime = "";
                    try {
                        sourceTime = response.substring(po3 + 5, po3 + 22);
                        tLastMod = dateTimeFormatter.parseMillis(sourceTime);
                    } catch (Throwable t) {
                        if (verbose) String2.log("getFileName error while parsing sourceTime=\"" + 
                            sourceTime + "\"\n" + MustBe.throwableToString(t));
                    }
                    if (reallyVerbose) String2.log("    MatchedRegex=true!  sourceTime=\"" + 
                        sourceTime + "\" -> lastMod=" + tLastMod);
                    if (tLastMod > 0) {
                        if (verbose) String2.log("  found: " + tDir + tName);

                        //ensure strings aren't just substring of big document
                        tDir  = new String(tDir);
                        tName = new String(tName);

                        //add to file list
                        int tDirIndex = lastFileDirs.indexOf(tDir);
                        if (tDirIndex == -1) {
                            tDirIndex = lastFileDirs.size();
                            lastFileDirs.add(tDir);
                        }
                        lastFileDirIndex.add((short)tDirIndex);
                        lastFileName.add(tName); 
                        lastFileLastMod.add(tLastMod);
                    }
                } else {
                    if (reallyVerbose) String2.log("    MatchedRegex=false");
                }

            }
            po = po3 + 2;
        }
    }

    /**
     * This overrides the superclass's version and returns info for files/urls from last getFileNames.
     *
     * @return the time (millis since the start of the Unix epoch) 
     *    the file was last modified 
     *    (or 0 if trouble)
     */
    public long getFileLastModified(String tDir, String tName) {
        int dirIndex = lastFileDirs.indexOf(tDir);
        if (dirIndex < 0) {
            if (verbose) String2.log("hyrax.getFileLastModified can't find dir=" + tDir);
            return 0;
        }

        //linear search through fileNames
        int n = lastFileDirIndex.size();
        for (int i = 0; i < n; i++) {
            if (lastFileDirIndex.array[i] == dirIndex &&
                lastFileName.array[i].equals(tName)) 
                return lastFileLastMod.array[i];
        }
        if (verbose) String2.log("hyrax.getFileLastModified can't find file=" + tName );
        return 0;
    }


    /**
     * This gets source data from one file.
     * See documentation in EDDTableFromFiles.
     *
     */
    public Table getSourceDataFromFile(String fileDir, String fileName, 
        StringArray sourceDataNames, String sourceDataTypes[],
        double sortedSpacing, double minSorted, double maxSorted, 
        boolean getMetadata, boolean mustGetData) 
        throws Throwable {

        if (reallyVerbose) String2.log("getSourceDataFromFile " + fileDir + fileName);
        DConnect dConnect = new DConnect(fileDir + fileName, acceptDeflate, 1, 1);
        DAS das = null;
        DDS dds = null;
        Table table = new Table();
        if (getMetadata) {
            if (das == null)
                das = dConnect.getDAS(OpendapHelper.DEFAULT_TIMEOUT);
            OpendapHelper.getAttributes(das, "GLOBAL", table.globalAttributes());
        }

        //first time: get axisVariables, ensure all vars use same sourceAxisNames
        if (sourceAxisNames == null) {
            if (reallyVerbose) String2.log("one time: getSourceAxisNames");
            if (dds == null)
                dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);
            for (int v = 0; v < sourceDataNames.size(); v++) {
                //find a multidimensional var
                String tSourceAxisNames[] = null;
                String tName = sourceDataNames.get(v);
                BaseType baseType = dds.getVariable(tName);
                if (baseType instanceof DGrid) {
                    //dGrid has main dArray + dimensions
                    DGrid dGrid = (DGrid)baseType;
                    int nEl = dGrid.elementCount(true);
                    tSourceAxisNames = new String[nEl - 1];
                    for (int el = 1; el < nEl; el++) //1..
                        tSourceAxisNames[el - 1] = dGrid.getVar(el).getName();
                } else if (baseType instanceof DArray) {
                    //dArray is usually 1 dim, but may be multidimensional
                    DArray dArray = (DArray)baseType;
                    int nDim = dArray.numDimensions();
                    if (nDim > 1) {
                        tSourceAxisNames = new String[nDim];
                        for (int d = 0; d < nDim; d++) {//0..
                            tSourceAxisNames[d] = dArray.getDimension(d).getName();
                            if (tSourceAxisNames[d] == null || tSourceAxisNames[d].equals(""))
                                tSourceAxisNames[d] = "" + dArray.getDimension(d).getSize();
                        }
                    }
                }

                //ok?
                if (tSourceAxisNames == null) {
                    //this var isn't multidimensional
                } else if (sourceAxisNames == null) {
                    //first multidimensional var; use its sourceAxisNames
                    sourceAxisNames = tSourceAxisNames; 
                } else {
                    //verify same
                    Test.ensureEqual(tSourceAxisNames, sourceAxisNames, 
                        "The dimensions for all multidimensional variables must be the same (" +
                        tName + " is different)."); 

                }
            }
            if (sourceAxisNames == null)
                throw new RuntimeException("Error: no multidimensional variables in initial data request.");
            if (!sourceAxisNames[0].equals(sortedColumnSourceName))
                throw new RuntimeException("Error: sortedColumnSourceName must be axisVariable[0]'s name.");
        }

        //if don't have to get actual data, just get all values of axis0, min,max of other axes, and mv for others
        int tNDim = -1;
        if (!mustGetData) {
            if (dds == null)
                dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);

            //get all the axis values at once
            PrimitiveArray axisPa[] = OpendapHelper.getAxisValues(dConnect, sourceAxisNames, "");

            //go through the sourceDataNames
            for (int dv = 0; dv < sourceDataNames.size(); dv++) {
                String tName = sourceDataNames.get(dv);
                BaseType baseType;
                try {
                    baseType = dds.getVariable(tName);
                } catch (Throwable t) {
                    if (verbose) String2.log("variable=" + tName + " not found in this file.");
                    continue;
                }
                Attributes atts = new Attributes();
                if (getMetadata)
                    OpendapHelper.getAttributes(das, tName, atts);
                PrimitiveArray pa;
                int po = String2.indexOf(sourceAxisNames, tName);
                if (po >= 0) {
                    pa = axisPa[po];
                } else if (tName.equals(sortedColumnSourceName)) {
                    //sortedColumn    get all values
                    pa = OpendapHelper.getPrimitiveArray(dConnect, "?" + tName);
                } else if (baseType instanceof DGrid) {
                    //multidimensional   don't get any values
                    pa = PrimitiveArray.factory(PrimitiveArray.elementStringToType(sourceDataTypes[dv]), 2, false);
                    pa.addString(pa.minValue());
                    pa.addString(pa.maxValue());
                } else if (baseType instanceof DArray) {                   
                    DArray dArray = (DArray)baseType;
                    if (dArray.numDimensions() == 1) {
                        //if 1 dimension, get min and max
                        //This didn't work: tSize was 0!
                        //int tSize = dArray.getDimension(0).getSize();
                        //String query = "?" + tName + "[0:" + (tSize - 1) + ":" + (tSize - 1) + "]";
                        //so get all values
                        String query = "?" + tName;
                        if (reallyVerbose) String2.log("  query=" + query);
                        pa = OpendapHelper.getPrimitiveArray(dConnect, query);
                    } else {
                        //if multidimensional   don't get any values
                        pa = PrimitiveArray.factory(PrimitiveArray.elementStringToType(sourceDataTypes[dv]), 2, false);
                        pa.addString(pa.minValue());
                        pa.addString(pa.maxValue());
                    }
                } else {
                    String2.log("Ignoring variable=" + tName + " because of unexpected baseType=" + baseType);
                    continue;
                }
                //String pas = pa.toString();
                //String2.log("\n" + tName + "=" + pas.substring(0, Math.min(pas.length(), 60)));
                table.addColumn(table.nColumns(), tName, pa, atts);
            }
            table.ensureColumnsAreSameSize();
            //String2.log(table.toString("row", 5));            
            return table;        
        }

        //need to get the requested data
        //calculate firstRow, lastRow of leftmost axis from minSorted, maxSorted
        int firstRow = 0;
        int lastRow = -1; 
        if (sortedSpacing >= 0 && !Double.isNaN(minSorted)) {
            //min/maxSorted is active       
            //get axis0 values
            PrimitiveArray pa = OpendapHelper.getPrimitiveArray(dConnect, "?" + sourceAxisNames[0]);
            firstRow = pa.binaryFindClosest(minSorted);
            lastRow = minSorted == maxSorted? firstRow : pa.binaryFindClosest(maxSorted);
            if (reallyVerbose) String2.log("  binaryFindClosest n=" + pa.size() + 
                " reqMin=" + minSorted + " firstRow=" + firstRow + 
                " reqMax=" + maxSorted + " lastRow=" + lastRow);
        } else {
            if (reallyVerbose) String2.log("  getting all rows since sortedSpacing=" + 
                sortedSpacing + " and minSorted=" + minSorted);
        }

        //dds is always needed for stuff below
        if (dds == null)
            dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);

        //build String form of the constraint
        String axis0Constraint = lastRow == -1? "" : "[" + firstRow + ":" + lastRow + "]";
        StringBuffer constraintSB = new StringBuffer();
        if (lastRow != -1) {
            constraintSB.append(axis0Constraint);
            for (int av = 1; av < sourceAxisNames.length; av++) {
                //dap doesn't allow requesting all via [], so get each dim's size
                //String2.log("  dds.getVar " + sourceAxisNames[av]);
                BaseType bt = dds.getVariable(sourceAxisNames[av]);
                DArray dArray = (DArray)bt;
                constraintSB.append("[0:" + (dArray.getDimension(0).getSize()-1) + "]");
            }
        }
        String constraint = constraintSB.toString();

        //get non-axis variables
        for (int dv = 0; dv < sourceDataNames.size(); dv++) {
            //???why not get all the dataVariables at once?
            //thredds has (and other servers may have) limits to the size of a given request
            //so breaking into parts avoids the problem.
            //Also, I check if each var is in this file. Var missing from a given file is allowed.
            String tName = sourceDataNames.get(dv);
            if (String2.indexOf(sourceAxisNames, tName) >= 0)
                continue;

            //ensure the variable is in this file
            try {
                BaseType baseType = dds.getVariable(tName);
            } catch (Throwable t) {
                if (verbose) String2.log("variable=" + tName + " not found.");
                continue;
            }

            //get the data
            PrimitiveArray pa[] = OpendapHelper.getPrimitiveArrays(dConnect, 
                "?" + tName + constraint);

            if (table.nColumns() == 0) {
                //first var: make axis columns
                int shape[] = new int[sourceAxisNames.length];
                for (int av = 0; av < sourceAxisNames.length; av++) {
                    String aName = sourceAxisNames[av];
                    Attributes atts = new Attributes();
                    if (getMetadata)
                        OpendapHelper.getAttributes(das, aName, atts);
                    shape[av] = pa[av+1].size();
                    table.addColumn(av, aName, 
                        PrimitiveArray.factory(pa[av+1].getElementType(), 128, false), atts);
                }
                //expand axis values into results table via NDimensionalIndex
                NDimensionalIndex ndIndex = new NDimensionalIndex(shape);
                int current[] = ndIndex.getCurrent();
                while (ndIndex.increment()) {
                    for (int av = 0; av < sourceAxisNames.length; av++) 
                        table.getColumn(av).addDouble(pa[av+1].getDouble(current[av]));
                }
            }  //subsequent vars: I could check that axis values are consistent

            //store dataVar column
            if (pa[0].size() != table.nRows()) 
                throw new RuntimeException("Data source error: nValues returned for " + tName + 
                    " wasn't expectedNValues=" + table.nRows());
            Attributes atts = new Attributes();
            if (getMetadata)
                OpendapHelper.getAttributes(das, tName, atts);
            table.addColumn(table.nColumns(), tName, pa[0], atts);
        }
        if (table.nColumns() > 0)
            return table;

        //no data vars? create a table with columns for sourceAxisNames only
        PrimitiveArray axisPa[] = OpendapHelper.getAxisValues(dConnect, sourceAxisNames, axis0Constraint);
        int shape[] = new int[sourceAxisNames.length];
        for (int av = 0; av < sourceAxisNames.length; av++) {
            String aName = sourceAxisNames[av];
            Attributes atts = new Attributes();
            if (getMetadata)
                OpendapHelper.getAttributes(das, aName, atts);
            shape[av] = axisPa[av].size();
            table.addColumn(av, aName, 
                PrimitiveArray.factory(axisPa[av].getElementType(), 128, false), atts);
        }
        //expand axis values into results table via NDimensionalIndex
        NDimensionalIndex ndIndex = new NDimensionalIndex(shape);
        int current[] = ndIndex.getCurrent();
        while (ndIndex.increment()) {
            for (int av = 0; av < sourceAxisNames.length; av++) 
                table.getColumn(av).addDouble(axisPa[av].getDouble(current[av]));
        }
        return table;

    }

    /** 
     * This generates a rough draft of the datasets.xml entry for an EDDTableFromHyraxFiles.
     * The XML can then be edited by hand and added to the datasets.xml file.
     *
     * @param dirUrl  the starting directory with a Hyrax sub-catalog 
     * @param oneFileDapUrl  url for one file, without ending .das or .html
     * @param sortedColumnsByName
     * @throws Throwable if trouble
     */
    public static String generateDatasetsXml(String dirUrl, String oneFileDapUrl, 
        boolean sortColumnsByName) throws Throwable {

        String2.log("EDDTableFromHyraxFiles.generateDatasetsXml" +
            "\ndirUrl=" + dirUrl + 
            "\noneFileDapUrl=" + oneFileDapUrl);

        //*** basically, make a Table which has the dataset's info
        Table table = new Table();
        DConnect dConnect = new DConnect(oneFileDapUrl, acceptDeflate, 1, 1);
        DAS das = dConnect.getDAS(OpendapHelper.DEFAULT_TIMEOUT);;
        DDS dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);

        //global metadata
        OpendapHelper.getAttributes(das, "GLOBAL", table.globalAttributes());
        addDummyRequiredGlobalAttributesForDatasetsXml(table.globalAttributes(), null);
        table.globalAttributes().add("sourceUrl", dirUrl);

        //variables
        StringBuffer varInfo = new StringBuffer();
        Enumeration en = dds.getVariables();
        while (en.hasMoreElements()) {
            BaseType baseType = (BaseType)en.nextElement();
            String varName = baseType.getName();
            Attributes atts = new Attributes();
            OpendapHelper.getAttributes(das, varName, atts);
            addDummyRequiredVariableAttributesForDatasetsXml(atts, varName, false);  //add colorbarMinMax
            varInfo.append(String2.left(varName, 17));
            PrimitiveVector pv = null; //for determining data type
            if (baseType instanceof DGrid) {   //for multidim vars
                varInfo.append(" DGrid [");
                DGrid dGrid = (DGrid)baseType;
                BaseType bt0 = dGrid.getVar(0); //holds the data
                pv = bt0 instanceof DArray? ((DArray)bt0).getPrimitiveVector() : bt0.newPrimitiveVector();
                int nEl = dGrid.elementCount(true);
                for (int el = 1; el < nEl; el++) {
                    varInfo.append(dGrid.getVar(el).getName());
                    varInfo.append(el == nEl - 1? "]\n" : ",");
                }
            } else if (baseType instanceof DArray) {  //for the dimension vars
                varInfo.append(" DArray[");
                DArray dArray = (DArray)baseType;
                pv = dArray.getPrimitiveVector();
                int nDim = dArray.numDimensions();
                for (int d = 0; d < nDim; d++) {
                    varInfo.append(dArray.getDimension(d).getName());
                    varInfo.append(d == nDim - 1? "]\n" : ",");
                }
            } else {
                varInfo.append(" baseType=" + baseType.toString() + " isn't supported yet.\n");
            }
            //varInfo.append("  pv=" + pv.toString() + "\n");
            if (pv != null)
                table.addColumn(table.nColumns(), varName, 
                    PrimitiveArray.factory(OpendapHelper.getElementType(pv), 2, false),
                    atts);
        }

        //sort the column names?
        if (sortColumnsByName)
            table.sortColumnsByName();

        //write the information
        StringBuffer sb = new StringBuffer();
        sb.append(directionsForDatasetsXml());
        sb.append(
            "<!-- If the multidimensional variables from a data source don't all use the same dimensions,\n" +
            "     you need to separate the data source into more than one dataset.\n" +
            "     The variables for this data source are:\n" + varInfo + "-->\n");
        sb.append(
"    <dataset type=\"EDDTableFromHyraxFiles\" datasetID=\"???\">\n" +
"        <reloadEveryNMinutes>???" + DEFAULT_RELOAD_EVERY_N_MINUTES + "</reloadEveryNMinutes>\n" +  
"        <fileDir>???" + dirUrl + "</fileDir>\n" +
"        <recursive>???true</recursive>\n" +
"        <fileNameRegex>???.*\\" + File2.getExtension(oneFileDapUrl) + "</fileNameRegex>\n" +
"        <metadataFrom>???last</metadataFrom>\n" +
"        <preExtractRegex>???^preRegex</preExtractRegex>\n" +
"        <postExtractRegex>???postRegex$</postExtractRegex>\n" +
"        <extractRegex>???.*</extractRegex>\n" +
"        <columnNameForExtract>???station</columnNameForExtract>\n" +
"        <sortedColumnSourceName>???</sortedColumnSourceName>\n" +
"        <sortFilesBySourceNames>???</sortFilesBySourceNames>\n" +
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n");
        sb.append(attsForDatasetsXml(table.globalAttributes(), "        "));

        sb.append(variablesForDatasetsXml(table, "dataVariable", true));
        sb.append(
"    </dataset>\n");
        return sb.toString();
        
    }

    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void testWcosTemp(boolean deleteCachedInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromHyraxFiles.testWcosTemp() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        int po;
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
/*
        results = generateDatasetsXml(
            "http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/",
            "http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/2005/ANO001_021MTBD000R00_20050617.nc",
            true);
        String2.log(results);
        expected = 
"yearday           DGrid [Time,Depth,Latitude,Longitude]\n" +
"Latitude          DArray[Latitude]\n" +
"Time              DArray[Time]\n" +
"Depth             DArray[Depth]\n" +
"Longitude         DArray[Longitude]\n" +
"Temperature_flag  DGrid [Time,Depth,Latitude,Longitude]\n" +
"Temperature       DGrid [Time,Depth,Latitude,Longitude]\n" +
"yearday_flag      DGrid [Time,Depth,Latitude,Longitude]\n" +
"-->\n" +
"    <dataset type=\"EDDTableFromNcFiles\" datasetID=\"???\">\n" +
"        <reloadEveryNMinutes>???10080</reloadEveryNMinutes>\n" +
"        <fileDir>???http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/</fileDir>\n";
        po = results.indexOf("yearday   ");
        Test.ensureEqual(results.substring(po, po + expected.length()), expected, "");

        expected = 
"<sourceName>Longitude</sourceName>\n" +
"            <destinationName>???Longitude</destinationName>\n" +
"            <dataType>double</dataType>\n" +
"            <addAttributes>\n" +
"                <att name=\"_FillValue\" type=\"int\">9999</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">???Longitude</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">???</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n" +
"        <dataVariable>\n" +
"            <sourceName>Temperature</sourceName>\n" +
"            <destinationName>???Temperature</destinationName>\n" +
"            <dataType>double</dataType>\n" +
"            <addAttributes>\n" +
"                <att name=\"_FillValue\" type=\"int\">9999</att>\n" +
"                <att name=\"description\">Seawater temperature</att>\n" +
"                <att name=\"ioos_category\">???</att>\n" +
"                <att name=\"long_name\">SeaWater Temperature</att>\n" +
"                <att name=\"quantity\">Temperature</att>\n" +
"                <att name=\"standard_name\">???</att>\n" +
"                <att name=\"units\">Celsius</att>\n" +
"            </addAttributes>\n" +
"        </dataVariable>\n";
        po = results.indexOf(expected.substring(0, 21));
        Test.ensureEqual(results.substring(po, po + expected.length()), expected, "");

*/
        if (deleteCachedInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosTemp.dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosTemp.files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosTemp.bad.json");
        }
        EDDTable eddTable = (EDDTable)oneFromDatasetXml("mssWcosTemp"); 

        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromHyraxFiles testWcosTemp das and dds for entire dataset\n");
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Attributes {\n" +
" s {\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 _FillValue -9999.0;\n" +
"    Float64 actual_range 1.11897192e+9, 1.19196696e+9;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    Float64 _FillValue -9999.0;\n" +
"    Float64 actual_range -20.0, 0.0;\n" +
"    String axis \"Z\";\n" +
"    String description \"Relative to Mean Sea Level (MSL)\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 _FillValue 9999.0;\n" +
"    Float64 actual_range 37.13015, 37.13015;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 _FillValue 9999.0;\n" +
"    Float64 actual_range -122.361253, -122.361253;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  station {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station\";\n" +
"  }\n" +
"  Temperature {\n" +
"    Float64 _FillValue 9999.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Sea Water Temperature\";\n" +
"    String quantity \"Temperature\";\n" +
"    String standard_name \"sea_water_temperature\";\n" +
"    String units \"degrees_C\";\n" +
"  }\n" +
"  Temperature_flag {\n" +
"    String description \"flag for data column, 0: no problems, 1: bad data due to malfunction or fouling, 2: suspicious data, 9: missing data\";\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Temperature Flag\";\n" +
"  }\n" +
"  yearday {\n" +
"    Float32 _FillValue 9999.0;\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Yearday\";\n" +
"  }\n" +
"  yearday_flag {\n" +
"    String description \"flag for data column, 0: no problems, 1: bad data due to malfunction or fouling, 2: suspicious data, 9: missing data\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Yearday Flag\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting -122.361253;\n" +
"    Float64 geospatial_lat_max 37.13015;\n" +
"    Float64 geospatial_lat_min 37.13015;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max -122.361253;\n" +
"    Float64 geospatial_lon_min -122.361253;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    Float64 geospatial_vertical_max 0.0;\n" +
"    Float64 geospatial_vertical_min -20.0;\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String History \"created by the NCDDC PISCO Temperature Profile to NetCDF converter on 2009/00/22 03:00 CST. Original dataset URL: \";\n" +
"    String history \"" + today + " http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/\n" +
today + " http://127.0.0.1:8080/cwexperimental/tabledap/mssWcosTemp.das\";\n" +
"    String infoUrl \"http://www.ncddc.noaa.gov/interactivemaps/national-marine-sanctuaries-west-coast-observatories\";\n" +
"    String institution \"NCDDC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 37.13015;\n" +
"    String sourceUrl \"http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/\";\n" +
"    Float64 Southernmost_Northing 37.13015;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The West Coast Observing System (WCOS) project provides access to temperature and currents data collected at four of the five National Marine Sanctuary sites, including Olympic Coast, Gulf of the Farallones, Monterey Bay, and Channel Islands. A semi-automated end-to-end data management system transports and transforms the data from source to archive, making the data acessible for discovery, access and analysis from multiple Internet points of entry.\n" +
"\n" +
"The stations (and their code names) are Ano Nuevo (ANO001), San Miguel North (BAY), Santa Rosa North (BEA), Big Creek (BIG001), Bodega Head (BOD001), Cape Alava 15M (CA015), Cape Alava 42M (CA042), Cape Alava 65M (CA065), Cape Alava 100M (CA100), Cannery Row (CAN001), Cape Elizabeth 15M (CE015), Cape Elizabeth 42M (CE042), Cape Elizabeth 65M (CE065), Cape Elizabeth 100M (CE100), Cuyler Harbor (CUY), Esalen (ESA001), Point Joe (JOE001), Kalaloch 15M (KL015), Kalaloch 27M (KL027), La Cruz Rock (LAC001), Lopez Rock (LOP001), Makah Bay 15M (MB015), Makah Bay 42M (MB042), Pelican/Prisoners Area (PEL), Pigeon Point (PIG001), Plaskett Rock (PLA001), Southeast Farallon Island (SEF001), San Miguel South (SMS), Santa Rosa South (SRS), Sunset Point (SUN001), Teawhit Head 15M (TH015), Teawhit Head 31M (TH031), Teawhit Head 42M (TH042), Terrace Point 7 (TPT007), Terrace Point 8 (TPT008), Valley Anch (VAL), Weston Beach (WES001).\";\n" +
"    String time_coverage_end \"2007-10-09T21:56:00Z\";\n" +
"    String time_coverage_start \"2005-06-17T01:32:00Z\";\n" +
"    String title \"West Coast Observing System (WCOS) Temperature Data\";\n" +
"    String Version \"2\";\n" +
"    Float64 Westernmost_Easting -122.361253;\n" +
"  }\n" +
"}\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float64 time;\n" +
"    Float64 altitude;\n" +
"    Float64 latitude;\n" +
"    Float64 longitude;\n" +
"    String station;\n" +
"    Float64 Temperature;\n" +
"    Byte Temperature_flag;\n" +
"    Float32 yearday;\n" +
"    Byte yearday_flag;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //*** test make data files
        String2.log("\n****************** EDDTableFromHyraxFiles.testWcosTemp make DATA FILES\n");       

        //.csv    for one lat,lon,time
        userDapQuery = "station&distinct()";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_stationList", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"station\n" +
"\n" +
"ANO001\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //.csv    for one lat,lon,time, many depths (from different files)      via lon > <
        userDapQuery = "&station=\"ANO001\"&time=1122592440";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_1StationGTLT", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);   
//one depth, by hand via DAP form:   (note that depth is already negative!)
//Longitude, 37.13015 Time=1122592440 depth=-12 Latitude=-122.361253    
// yearday 208.968,  temp_flag 0, temp 10.66, yeardayflag  0
        expected = 
"time, altitude, latitude, longitude, station, Temperature, Temperature_flag, yearday, yearday_flag\n" +
"UTC, m, degrees_north, degrees_east, , degrees_C, , , \n" +
"2005-07-28T23:14:00Z, -20.0, 37.13015, -122.361253, ANO001, 10.51, 0, 208.96805, 0\n" +
"2005-07-28T23:14:00Z, -12.0, 37.13015, -122.361253, ANO001, 10.66, 0, 208.96805, 0\n" +
"2005-07-28T23:14:00Z, -4.0, 37.13015, -122.361253, ANO001, 12.04, 0, 208.96805, 0\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        /* */
    }

/*
Time              DArray[Time]
Depth             DArray[Depth]
Latitude          DArray[Latitude]
Longitude         DArray[Longitude]
Height            DGrid [Depth,Latitude,Longitude]
Height_flag       DGrid [Depth,Latitude,Longitude]
Pressure          DGrid [Time,Latitude,Longitude]
Pressure_flag     DGrid [Time,Latitude,Longitude]
Temperature       DGrid [Time,Latitude,Longitude]
Temperature_flag  DGrid [Time,Latitude,Longitude]
WaterDepth        DGrid [Time,Latitude,Longitude]
WaterDepth_flag   DGrid [Time,Latitude,Longitude]
YearDay           DGrid [Time,Latitude,Longitude]
YearDay_flag      DGrid [Time,Latitude,Longitude]
DataQuality       DGrid [Time,Depth,Latitude,Longitude]
DataQuality_flag  DGrid [Time,Depth,Latitude,Longitude]
Eastward          DGrid [Time,Depth,Latitude,Longitude]
Eastward_flag     DGrid [Time,Depth,Latitude,Longitude]
ErrorVelocity     DGrid [Time,Depth,Latitude,Longitude]
ErrorVelocity_flag DGrid [Time,Depth,Latitude,Longitude]
Intensity         DGrid [Time,Depth,Latitude,Longitude]
Intensity_flag    DGrid [Time,Depth,Latitude,Longitude]
Northward         DGrid [Time,Depth,Latitude,Longitude]
Northward_flag    DGrid [Time,Depth,Latitude,Longitude]
Upwards_flag      DGrid [Time,Depth,Latitude,Longitude]
Upwards           DGrid [Time,Depth,Latitude,Longitude]
*/

    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void testWcosAdcpS(boolean deleteCachedInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromHyraxFiles.testWcosAdcpS() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        int po;
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
        if (deleteCachedInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosAdcpS.dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosAdcpS.files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosAdcpS.bad.json");
        }
        EDDTable eddTable = (EDDTable)oneFromDatasetXml("mssWcosAdcpS"); 

        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromHyraxFiles testWcosAdcpS das and dds for entire dataset\n");
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Attributes {\n" +
" s {\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 _FillValue 9999.0;\n" +
"    Float64 actual_range 1.10184436e+9, 1.160603466e+9;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 34.04017, 34.04017;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -120.31121, -120.31121;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  station {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station\";\n" +
"  }\n" +
"  Pressure {\n" +
"    Float64 _FillValue 9999.0;\n" +
"    String description \"pressure measured at the ADCP sensor, measured in decibars with a precision defined as the effective resolution of the instrument.\";\n" +
"    String ioos_category \"Pressure\";\n" +
"    String standard_name \"sea_water_pressure\";\n" +
"    String units \"dbar\";\n" +
"  }\n" +
"  Pressure_flag {\n" +
"    String description \"flag for data column, 0: no problems, 1: bad data due to malfunction or fouling, 2: suspicious data, 9: missing data\";\n" +
"    String ioos_category \"Pressure\";\n" +
"    String long_name \"Pressure Flag\";\n" +
"  }\n" +
"  Temperature {\n" +
"    Float64 _FillValue 9999.0;\n" +
"    String description \"seawater temperature from ADCP sensor\";\n" +
"    String ioos_category \"Temperature\";\n" +
"    String quantity \"Temperature\";\n" +
"    String standard_name \"sea_water_temperature\";\n" +
"    String units \"degree_C\";\n" +
"  }\n" +
"  Temperature_flag {\n" +
"    String description \"flag for data column, 0: no problems, 1: bad data due to malfunction or fouling, 2: suspicious data, 9: missing data\";\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Temperature Flag\";\n" +
"  }\n" +
"  WaterDepth {\n" +
"    Float64 _FillValue 9999.0;\n" +
"    String description \"Fluctuating water depth (sea floor to sea surface distance) at measurement site, in meters, as determined by location of sea surface from ADCP pressure measurements,or if pressure is unavailable, the maximum in ADCP echo intensity.\";\n" +
"    String ioos_category \"Bathymetry\";\n" +
"    String long_name \"Water Depth\";\n" +
"    String positive \"down\";\n" +
"    String standard_name \"sea_floor_depth_below_sea_level\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  WaterDepth_flag {\n" +
"    String description \"flag for data column, 0: no problems, 1: bad data due to malfunction or fouling, 2: suspicious data, 9: missing data\";\n" +
"    String ioos_category \"Bathymetry\";\n" +
"    String long_name \"Water Depth Flag\";\n" +
"  }\n" +
"  YearDay {\n" +
"    Float32 _FillValue 9999.0;\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Year Day\";\n" +
"  }\n" +
"  YearDay_flag {\n" +
"    String description \"flag for data column, 0: no problems, 1: bad data due to malfunction or fouling, 2: suspicious data, 9: missing data\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Year Day Flag\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting -120.31121;\n" +
"    Float64 geospatial_lat_max 34.04017;\n" +
"    Float64 geospatial_lat_min 34.04017;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max -120.31121;\n" +
"    Float64 geospatial_lon_min -120.31121;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String History \"created by the NCDDC PISCO ADCP Profile to converter on 2009/42/09 13:42 CST. Original dataset URL: \";\n" +
"    String history \"" + today + " http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/BAY/\n" +
today + " http://127.0.0.1:8080/cwexperimental/tabledap/mssWcosAdcpS.das\";\n" +
"    String infoUrl \"http://www.ncddc.noaa.gov/interactivemaps/national-marine-sanctuaries-west-coast-observatories\";\n" +
"    String institution \"NCDDC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 34.04017;\n" +
"    String sourceUrl \"http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/BAY/\";\n" +
"    Float64 Southernmost_Northing 34.04017;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The West Coast Observing System (WCOS) project provides access to temperature and currents data collected at four of the five National Marine Sanctuary sites, including Olympic Coast, Gulf of the Farallones, Monterey Bay, and Channel Islands. A semi-automated end-to-end data management system transports and transforms the data from source to archive, making the data acessible for discovery, access and analysis from multiple Internet points of entry.\n" +
"\n" +
"The stations (and their code names) are Ano Nuevo (ANO001), San Miguel North (BAY), Santa Rosa North (BEA), Big Creek (BIG001), Bodega Head (BOD001), Cape Alava 15M (CA015), Cape Alava 42M (CA042), Cape Alava 65M (CA065), Cape Alava 100M (CA100), Cannery Row (CAN001), Cape Elizabeth 15M (CE015), Cape Elizabeth 42M (CE042), Cape Elizabeth 65M (CE065), Cape Elizabeth 100M (CE100), Cuyler Harbor (CUY), Esalen (ESA001), Point Joe (JOE001), Kalaloch 15M (KL015), Kalaloch 27M (KL027), La Cruz Rock (LAC001), Lopez Rock (LOP001), Makah Bay 15M (MB015), Makah Bay 42M (MB042), Pelican/Prisoners Area (PEL), Pigeon Point (PIG001), Plaskett Rock (PLA001), Southeast Farallon Island (SEF001), San Miguel South (SMS), Santa Rosa South (SRS), Sunset Point (SUN001), Teawhit Head 15M (TH015), Teawhit Head 31M (TH031), Teawhit Head 42M (TH042), Terrace Point 7 (TPT007), Terrace Point 8 (TPT008), Valley Anch (VAL), Weston Beach (WES001).\";\n" +
"    String time_coverage_end \"2006-10-11T21:51:06Z\";\n" +
"    String time_coverage_start \"2004-11-30T19:52:40Z\";\n" +
"    String title \"West Coast Observing System (WCOS) ADCP Station Data\";\n" +
"    String Version \"2\";\n" +
"    Float64 Westernmost_Easting -120.31121;\n" +
"  }\n" +
"}\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float64 time;\n" +
"    Float64 latitude;\n" +
"    Float64 longitude;\n" +
"    String station;\n" +
"    Float64 Pressure;\n" +
"    Byte Pressure_flag;\n" +
"    Float64 Temperature;\n" +
"    Byte Temperature_flag;\n" +
"    Float64 WaterDepth;\n" +
"    Byte WaterDepth_flag;\n" +
"    Float32 YearDay;\n" +
"    Byte YearDay_flag;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //*** test make data files
        String2.log("\n****************** EDDTableFromHyraxFiles.testWcosAdcpS make DATA FILES\n");       

        //.csv    for one lat,lon,time
        userDapQuery = "station&distinct()";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_stationList", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"station\n" +
"\n" +
"BAYXXX\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //.csv    for one lat,lon,time, many depths (from different files)      via lon > <
        userDapQuery = "&station=\"BAYXXX\"&time=1153860786";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_1StationGTLT", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);   
//one depth, by hand via DAP form:   (note that depth is already negative!)
//http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/BAY/2006/BAYXXX_015ADCP015R00_20060725.nc.ascii?WaterDepth_flag[100][0:1:0][0:1:0],YearDay_flag[100][0:1:0][0:1:0],WaterDepth[100][0:1:0][0:1:0],Pressure[100][0:1:0][0:1:0],Pressure_flag[100][0:1:0][0:1:0],Temperature[100][0:1:0][0:1:0]
//Longitude, 34.04017
//Time=1153860786
//Latitude=-120.31121
//WaterDepth_flag 0
//YearDay_flag 0
//WaterDepth 17.7
//Pressure 17.205
//Pressure_flag 0
//Temperature 12.06
expected = 
"time, latitude, longitude, station, Pressure, Pressure_flag, Temperature, Temperature_flag, WaterDepth, WaterDepth_flag, YearDay, YearDay_flag\n" +
"UTC, degrees_north, degrees_east, , dbar, , degree_C, , m, , , \n" +
"2006-07-25T20:53:06Z, 34.04017, -120.31121, BAYXXX, 17.205, 0, 12.06, 0, 17.7, 0, 205.87021, 0\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        /* */
    }

    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void testWcosAdcpC(boolean deleteCachedInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromHyraxFiles.testWcosAdcpC() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        int po;
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
        if (deleteCachedInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosAdcpC.dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosAdcpC.files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "mssWcosAdcpC.bad.json");
        }
        EDDTable eddTable = (EDDTable)oneFromDatasetXml("mssWcosAdcpC"); 

        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromHyraxFiles testWcosAdcpC das and dds for entire dataset\n");
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float64 time;\n" +
"    Float64 altitude;\n" +
"    Float64 latitude;\n" +
"    Float64 longitude;\n" +
"    String station;\n" +
"    Int32 DataQuality;\n" +
"    Byte DataQuality_flag;\n" +
"    Float64 Eastward;\n" +
"    Byte Eastward_flag;\n" +
"    Float64 ErrorVelocity;\n" +
"    Byte ErrorVelocity_flag;\n" +
"    Int32 Intensity;\n" +
"    Byte Intensity_flag;\n" +
"    Float64 Northward;\n" +
"    Byte Northward_flag;\n" +
"    Float64 Upwards;\n" +
"    Byte Upwards_flag;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //*** test make data files
        String2.log("\n****************** EDDTableFromHyraxFiles.testWcosAdcpC make DATA FILES\n");       

        //.csv    for one lat,lon,time
        userDapQuery = "station&distinct()";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_stationList", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"station\n" +
"\n" +
"BAYXXX\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //.csv    for one lat,lon,time, many depths (from different files)      via lon > <
        userDapQuery = "&station=\"BAYXXX\"&time=1153865226";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_1StationGTLT", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);   
//one depth, by hand via DAP form:   (note that depth is already negative!)
//http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/BAY/2006/BAYXXX_015ADCP015R00_20060725.nc.ascii?Intensity[137][0:1:18][0:1:0][0:1:0],Upwards_flag[137][0:1:18][0:1:0][0:1:0],Upwards[137][0:1:18][0:1:0][0:1:0],Northward_flag[137][0:1:18][0:1:0][0:1:0],Temperature_flag[137][0:1:0][0:1:0],Eastward[137][0:1:18][0:1:0][0:1:0],DataQuality[137][0:1:18][0:1:0][0:1:0],Northward[137][0:1:18][0:1:0][0:1:0]
//Intensity.Longitude, 34.04017
//Intensity.Time=1153865226
//[Intensity.Latitude=-120.31121]
//Depth=16], 9999  15: 125, 14: 128, 124,  118 113 113 115 112 103 100 104 102 100 113 171 180 123  -2: 95
//Upwards_flag 16: 9 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0  9  9
//Upwards.Upwards 16:  9999  0.003  -0.002 0.011 0.011 0.012 0.012 -0.001 0.006 0.002 0.011 0.018 0.027 0.017 -0.001 9999 9999 9999 9999
//Northward_flag 9,  0's, 9 9 9 9
//Eastward 16: 9999 -0.024 -0.019 -0.005 -0.006 -0.02 0.02 0.053 -0.002 0.008 0.039 -0.022 0.013 -0.033 -0.047 9999 9999 9999 9999
//DataQuality 16: 9999 97 97 85 97 97 97 97 100 100 100 60 47 100 90 5 65 5 82
//Northward.Northward 16: 9999 -0.017 -0.026 -0.076 ...
    expected = 
"time, altitude, latitude, longitude, station, DataQuality, DataQuality_flag, Eastward, Eastward_flag, ErrorVelocity, ErrorVelocity_flag, Intensity, Intensity_flag, Northward, Northward_flag, Upwards, Upwards_flag\n" +
"UTC, m, degrees_north, degrees_east, , %, , m s-1, , m s-1, , RDI counts, , m s-1, , m s-1, \n" +
"2006-07-25T22:07:06Z, -16.0, 34.04017, -120.31121, BAYXXX, NaN, 9, NaN, 9, NaN, 9, NaN, 9, NaN, 9, NaN, 9\n" +
"2006-07-25T22:07:06Z, -15.0, 34.04017, -120.31121, BAYXXX, 97, 0, -0.024, 0, -0.018, 0, 125, 0, -0.017, 0, 0.0030, 0\n" +
"2006-07-25T22:07:06Z, -14.0, 34.04017, -120.31121, BAYXXX, 97, 0, -0.019, 0, 0.011, 0, 128, 0, -0.026, 0, -0.0020, 0\n" +
"2006-07-25T22:07:06Z, -13.0, 34.04017, -120.31121, BAYXXX, 85, 0, -0.0050, 0, 0.038, 0, 124, 0, -0.076, 0, 0.011, 0\n" +
"2006-07-25T22:07:06Z, -12.0, 34.04017, -120.31121, BAYXXX, 97, 0, -0.0060, 0, -0.0070, 0, 118, 0, -0.063, 0, 0.011, 0\n" +
"2006-07-25T22:07:06Z, -11.0, 34.04017, -120.31121, BAYXXX, 97, 0, -0.02, 0, 0.01, 0, 113, 0, -0.053, 0, 0.012, 0\n" +
"2006-07-25T22:07:06Z, -10.0, 34.04017, -120.31121, BAYXXX, 97, 0, 0.02, 0, 0.02, 0, 113, 0, -0.073, 0, 0.012, 0\n" +
"2006-07-25T22:07:06Z, -9.0, 34.04017, -120.31121, BAYXXX, 97, 0, 0.053, 0, 0.036, 0, 115, 0, -0.089, 0, -0.0010, 0\n" +
"2006-07-25T22:07:06Z, -8.0, 34.04017, -120.31121, BAYXXX, 100, 0, -0.0020, 0, 0.016, 0, 112, 0, -0.02, 0, 0.0060, 0\n" +
"2006-07-25T22:07:06Z, -7.0, 34.04017, -120.31121, BAYXXX, 100, 0, 0.0080, 0, 0.0070, 0, 103, 0, -0.024, 0, 0.0020, 0\n" +
"2006-07-25T22:07:06Z, -6.0, 34.04017, -120.31121, BAYXXX, 100, 0, 0.039, 0, 0.029, 0, 100, 0, -0.045, 0, 0.011, 0\n" +
"2006-07-25T22:07:06Z, -5.0, 34.04017, -120.31121, BAYXXX, 60, 0, -0.022, 0, 0.051, 0, 104, 0, 0.034, 0, 0.018, 0\n" +
"2006-07-25T22:07:06Z, -4.0, 34.04017, -120.31121, BAYXXX, 47, 0, 0.013, 0, 0.074, 0, 102, 0, 0.039, 0, 0.027, 0\n" +
"2006-07-25T22:07:06Z, -3.0, 34.04017, -120.31121, BAYXXX, 100, 0, -0.033, 0, 0.05, 0, 100, 0, 0.112, 0, 0.017, 0\n" +
"2006-07-25T22:07:06Z, -2.0, 34.04017, -120.31121, BAYXXX, 90, 0, -0.047, 0, -0.024, 0, 113, 0, 0.134, 0, -0.0010, 0\n" +
"2006-07-25T22:07:06Z, -1.0, 34.04017, -120.31121, BAYXXX, 5, 0, NaN, 9, NaN, 9, 171, 0, NaN, 9, NaN, 9\n" +
"2006-07-25T22:07:06Z, 0.0, 34.04017, -120.31121, BAYXXX, 65, 0, NaN, 9, NaN, 9, 180, 0, NaN, 9, NaN, 9\n" +
"2006-07-25T22:07:06Z, 1.0, 34.04017, -120.31121, BAYXXX, 5, 0, NaN, 9, NaN, 9, 123, 0, NaN, 9, NaN, 9\n" +
"2006-07-25T22:07:06Z, 2.0, 34.04017, -120.31121, BAYXXX, 82, 0, NaN, 9, NaN, 9, 95, 0, NaN, 9, NaN, 9\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        /* */
    }

    public static void testGetSize() throws Throwable {
        DConnect dConnect = new DConnect(
            "http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/2005/ANO001_021MTBD000R00_20050617.nc", 
            acceptDeflate, 1, 1);
        DDS dds = dConnect.getDDS(OpendapHelper.DEFAULT_TIMEOUT);
        BaseType bt = dds.getVariable("Time");
        DArray dArray = (DArray)bt;
        String2.log("nDim=" + dArray.numDimensions());
        DArrayDimension dim = dArray.getDimension(0);
        int n = dim.getSize();
        String2.log("n=" + n);
        Test.ensureTrue(n > 10, "n=" + n);
    }

    public static void testGetAxisValues() throws Throwable {
        DConnect dConnect = new DConnect(
            "http://biloxi-bay.ssc.hpc.msstate.edu/dods-bin/nph-dods/WCOS/nmsp/wcos/ANO001/2005/ANO001_021MTBD000R00_20050617.nc", 
            acceptDeflate, 1, 1);
        PrimitiveArray pa[] = OpendapHelper.getAxisValues(dConnect, 
            new String[]{"Time","Latitude","Longitude","Depth"}, "[11:15]");
        for (int i = 0; i < pa.length; i++)
            String2.log("pa[" + i + "]=" + pa[i].toString());
        Test.ensureEqual(pa[0].toString(), "1.1189748E9, 1.11897504E9, 1.11897528E9, 1.11897552E9, 1.11897576E9", "");
//!!!lat and lon are reversed by the server!  This tests expected, but incorrect, response.
//!!!when this test fails, change it and change Latitude and Longitude in mssWcos datasets in datasets.xml
        Test.ensureEqual(pa[1].toString(), "-122.361253", "");
        Test.ensureEqual(pa[2].toString(), "37.13015", "");
        Test.ensureEqual(pa[3].toString(), "0.0", "");
    }


    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void test(boolean deleteCachedInfo) throws Throwable {

        //usually run
        testGetSize();
//lat and lon are reversed by the server!
        testGetAxisValues();
        testWcosTemp(deleteCachedInfo);  
        testWcosAdcpS(deleteCachedInfo);  
        testWcosAdcpC(deleteCachedInfo);  

        //not usually run

    }
}

