/* 
 * EDDTableFromNcFiles Copyright 2009, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.RegexFilenameFilter;

import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.*;

import java.util.List;

/**
 * Get netcdf-X.X.XX.jar from http://www.unidata.ucar.edu/software/netcdf-java/index.htm
 * and copy it to <context>/WEB-INF/lib renamed as netcdf-latest.jar.
 * Get slf4j-jdk14.jar from 
 * ftp://ftp.unidata.ucar.edu/pub/netcdf-java/slf4j-jdk14.jar
 * and copy it to <context>/WEB-INF/lib.
 * Put both of these .jar files in the classpath for the compiler and for Java.
 */
import ucar.nc2.*;
import ucar.nc2.dataset.NetcdfDataset;
import ucar.nc2.dods.*;
import ucar.nc2.util.*;
import ucar.ma2.*;

/** 
 * This class represents a table of data from a collection of n-dimensional (1,2,3,4,...) .nc data files.
 * The dimensions are e.g., time,depth,lat,lon.
 * In a given file, there are multiple values of the outermost dimension (e.g., time), 
 * but THERE MUST BE JUST ONE value for the other dimensions (e.g., depth, lat, and lon).
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2009-02-13
 */
public class EDDTableFromNcFiles extends EDDTableFromFiles { 


    /** Used to ensure that all non-axis variables in all files have the same leftmost dimension. */
    protected String dim0Name = null;


    /** 
     * The constructor just calls the super constructor. 
     *
     * @param tAccessibleTo is a comma separated list of 0 or more
     *    roles which will have access to this dataset.
     *    <br>If null, everyone will have access to this dataset (even if not logged in).
     *    <br>If "", no one will have access to this dataset.
     * <p>The sortedColumnSourceName can't be for a char/String variable
     *   because NcHelper binary searches are currently set up for numeric vars only.
     */
    public EDDTableFromNcFiles(String tDatasetID, String tAccessibleTo,
        StringArray tOnChange, 
        int tNDimensions,
        Attributes tAddGlobalAttributes,
        double tAltMetersPerSourceUnit, 
        Object[][] tDataVariables,
        int tReloadEveryNMinutes,
        String tFileDir, boolean tRecursive, String tFileNameRegex, String tMetadataFrom,
        String tPreExtractRegex, String tPostExtractRegex, String tExtractRegex, 
        String tColumnNameForExtract,
        String tSortedColumnSourceName, String tSortFilesBySourceNames) 
        throws Throwable {

        super("EDDTableFromNcFiles", true, tDatasetID, tAccessibleTo, tOnChange, 
            tNDimensions, tAddGlobalAttributes, tAltMetersPerSourceUnit, 
            tDataVariables, tReloadEveryNMinutes,
            tFileDir, tRecursive, tFileNameRegex, tMetadataFrom,
            tPreExtractRegex, tPostExtractRegex, tExtractRegex, tColumnNameForExtract,
            tSortedColumnSourceName, tSortFilesBySourceNames);

    }

    /**
     * This gets source data from one file.
     * See documentation in EDDTableFromFiles.
     *
     */
    public Table getSourceDataFromFile(String fileDir, String fileName, 
        StringArray sourceDataNames, String sourceDataTypes[],
        double sortedSpacing, double minSorted, double maxSorted, 
        boolean getMetadata, boolean mustGetData) 
        throws Throwable {

        NetcdfFile ncFile = NcHelper.openFile(fileDir + fileName); //may throw exception

        try {
//!!!this doesn't yet take advantage of sortedSpacing > 0
//if it does, it needs to be passed value0 or check it here
//and/or maybe it should verify lastMod time 

            Table table = new Table();
            int firstRow = 0;
            int lastRow = -1; 
            if (getMetadata) 
                NcHelper.getGlobalAttributes(ncFile, table.globalAttributes());

            if (sortedSpacing >= 0 && !Double.isNaN(minSorted)) {
                //min/maxSorted is active       
                //get the sortedVar
                //This requires sortedVar to be numeric, because currently NcHelper binary searches assume numeric
                Variable sortedVar = ncFile.findVariable(sortedColumnSourceName);  //null if not found
                int sortedVarSize = (int)sortedVar.getSize();
              
                //find the first and last rows -- round to nearest
                long tTime = System.currentTimeMillis();
                int oFirstRow = NcHelper.binaryFindClosest(sortedVar, 0, sortedVarSize - 1, minSorted);
                firstRow = NcHelper.findFirst(sortedVar, oFirstRow);
                int oLastRow  = minSorted == maxSorted? oFirstRow :
                    NcHelper.binaryFindClosest(sortedVar, oFirstRow, sortedVarSize - 1, maxSorted);
                lastRow = NcHelper.findLast(sortedVar, oLastRow);
                if (reallyVerbose) String2.log("  binaryFindClosest n=" + sortedVarSize + 
                    " reqMin=" + minSorted + " firstRow=" + firstRow + 
                    " reqMax=" + maxSorted + " lastRow=" + lastRow +
                    " time=" + (System.currentTimeMillis() - tTime));
            } else {
                if (reallyVerbose) String2.log("  getting all rows since sortedSpacing=" + 
                    sortedSpacing + " and minSorted=" + minSorted);
            }

            //for each sourceDataName
            PrimitiveArray pa;
            for (int sn = 0; sn < sourceDataNames.size(); sn++) {
                pa = null; //for safety
                Variable var = ncFile.findVariable(sourceDataNames.get(sn));  //null if not found
                Attributes atts = new Attributes();
                if (var == null) {
                    //this var not in this file
                    //create pa with no values; ensureColumnsAreSameSize_LastValue below fills in MVs
                    pa = PrimitiveArray.factory(
                        PrimitiveArray.elementStringToType(sourceDataTypes[sn]), 1, false);

                } else {
                    //get metadata
                    if (getMetadata)
                        NcHelper.getVariableAttributes(var, atts);

                    //ensure shape is appropriate
                    int rank = var.getRank();
                    boolean isChar = var.getDataType() == DataType.CHAR;
                    int shape[] = var.getShape();
                    boolean isAxis = //true for leftmost and inner axes
                        (!isChar && rank == 1) ||
                        ( isChar && rank == 2);
                    //String2.log("name=" + sourceDataNames.get(sn) + " isAxis=" + isAxis + " shape=" + String2.toCSVString(shape));
                    if (shape[0] == 1 && isAxis) {
                        //an axis variable with just one value -- it's an inner axis
                        //ensureColumnsAreSameSize_LastValue below fills replicates of the value
                        pa = NcHelper.getPrimitiveArray(var, 0, 0); 

                    } else {
                        //set/check dim0Name 
                        if (dim0Name == null)
                            dim0Name = var.getDimension(0).getName();
                        Test.ensureEqual(var.getDimension(0).getName(), dim0Name, 
                            "variable=" + sourceDataNames.get(sn) + 
                            " had an unexpected leftmost dimension.");

                        //ensure rank and shape are correct 
                        if (!isAxis) {  //if not the leftmost dimension  (other isAxis handled above)
                            if (rank != nDimensions + (isChar? 1 : 0))
                                throw new RuntimeException(
                                    (isChar? "String" : "Numeric") +
                                    " variable=" + sourceDataNames.get(sn) + 
                                    " had unexpected nDimensions=" + rank +
                                    " (dataset nDimensions=" + nDimensions + ").");
                            int last = nDimensions - (isChar? 1 : 0);
                            for (int i = 1; i < last; i++) {
                                if (shape[i] != 1) {
                                    throw new RuntimeException(
                                        "The non-leftmost " +
                                        (isChar? "and non-rightmost dimensions of String" : 
                                                 "dimensions of numeric") +
                                        " variables must have size=1, but variable=" + sourceDataNames.get(sn) + 
                                        " had shape=[" + String2.toCSVString(shape) + "].");
                                }
                            }
                        }

                        //get the data   
                        pa = NcHelper.getPrimitiveArray(var, firstRow, lastRow); //it throws Throwable if incorrect nValues
                    }
                }

                table.addColumn(table.nColumns(), sourceDataNames.get(sn), pa, atts);
            }
            //String2.log("table from getSourceDataFromFile=\n" + table.toString());
            //I care about this exception
            ncFile.close();
            table.ensureColumnsAreSameSize_LastValue(); //deals with missing variables in some files
            return table;

        } catch (Throwable t) {
            try {
                ncFile.close(); //make sure it is explicitly closed
            } catch (Throwable t2) {
                //don't care
            }
            throw t;
        }
    }

    /** 
     * This generates a rough draft of the datasets.xml entry for an EDDTableFromNcFiles.
     * The XML can then be edited by hand and added to the datasets.xml file.
     *
     * <p>This can't be made into a web service because it would allow any user
     * to looks at (possibly) private .nc files on the server.
     *
     * @param fullFileName one of the files in the collection
     * @param sortedColumnsByName
     * @throws Throwable if trouble
     */
    public static String generateDatasetsXml(String fullFileName, 
        boolean sortColumnsByName) throws Throwable {

        String2.log("EDDTableFromNcFiles.generateDatasetsXml" +
            "\nfullFileName=" + fullFileName);

        //*** basically, make a Table which has the dataset's info
        Table table = new Table();
        StringBuffer varInfo = new StringBuffer();
        NetcdfFile ncFile = NcHelper.openFile(fullFileName);
        try {
            List variables = ncFile.getVariables();
            for (int dv = 0; dv < variables.size(); dv++) {
                Variable var = (Variable)variables.get(dv);
                Class elementType = NcHelper.getElementType(var.getDataType());
                if (elementType == boolean.class) elementType = byte.class;  //NcHelper.getArray converts booleans to bytes
                else if (elementType == char.class) elementType = String.class;
                varInfo.append(var.getName() + " " + 
                    PrimitiveArray.elementTypeToString(elementType) +
                    " [" + 
                    String2.replaceAll(String2.toCSVString(var.getShape()), ", ", "][") +
                    "]\n");
                PrimitiveArray pa = PrimitiveArray.factory(elementType, 1, false);
                Attributes atts = new Attributes();
                NcHelper.getVariableAttributes(var, atts);
                table.addColumn(dv, var.getName(), pa, atts);
            }

            //load the global metadata
            NcHelper.getGlobalAttributes(ncFile, table.globalAttributes());

            //I do care if this throws Throwable
            ncFile.close(); 

        } catch (Throwable t) {
            //make sure ncFile is explicitly closed
            try {
                ncFile.close(); 
            } catch (Throwable t2) {
                //don't care
            }

            throw t;
        }


        //add dummyRequired attributes
        addDummyRequiredGlobalAttributesForDatasetsXml(table.globalAttributes(), null);
        table.globalAttributes().add("sourceUrl", "???");

        for (int col = 0; col < table.nColumns(); col++) 
            addDummyRequiredVariableAttributesForDatasetsXml(table.columnAttributes(col), 
                table.getColumnName(col), true);

        //sort the column names?
        if (sortColumnsByName)
            table.sortColumnsByName();

        //write the information
        StringBuffer sb = new StringBuffer();
        sb.append(directionsForDatasetsXml());
        sb.append(varInfo);
        sb.append(
"    <dataset type=\"EDDTableFromNcFiles\" datasetID=\"???\">\n" +
"        <reloadEveryNMinutes>???" + DEFAULT_RELOAD_EVERY_N_MINUTES + "</reloadEveryNMinutes>\n" +  
"        <fileDir>???" + File2.getDirectory(fullFileName) + "</fileDir>\n" +
"        <recursive>???false</recursive>\n" +
"        <fileNameRegex>???.*\\" + File2.getExtension(fullFileName) + "</fileNameRegex>\n" +
"        <metadataFrom>???last</metadataFrom>\n" +
"        <preExtractRegex>???^preRegex</preExtractRegex>\n" +
"        <postExtractRegex>???postRegex$</postExtractRegex>\n" +
"        <extractRegex>???.*</extractRegex>\n" +
"        <columnNameForExtract>???station</columnNameForExtract>\n" +
"        <sortedColumnSourceName>???</sortedColumnSourceName>\n" +
"        <sortFilesBySourceNames>???</sortFilesBySourceNames>\n" +
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n");
        sb.append(attsForDatasetsXml(table.globalAttributes(), "        "));

        sb.append(variablesForDatasetsXml(table, "dataVariable", true));
        sb.append(
"    </dataset>\n");
        return sb.toString();
        
    }

    /**
     * This tests the methods in this class with a 1D dataset.
     *
     * @throws Throwable if trouble
     */
    public static void test1D(boolean deleteCachedDatasetInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.test1D() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        if (deleteCachedDatasetInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + "erdCinpKfmSFNH.dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "erdCinpKfmSFNH.files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "erdCinpKfmSFNH.bad.json");
        }
        EDDTable eddTable = (EDDTable)oneFromDatasetXml("erdCinpKfmSFNH"); 

        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromNcFiles 1D test das and dds for entire dataset\n");
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".das"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -120.4, -118.4;\n" +
"    String axis \"X\";\n" +
"    Float64 colorBarMaximum -118.4;\n" +
"    Float64 colorBarMinimum -120.4;\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 32.8, 34.05;\n" +
"    String axis \"Y\";\n" +
"    Float64 colorBarMaximum 34.5;\n" +
"    Float64 colorBarMinimum 32.5;\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    Float64 actual_range -17.0, -5.0;\n" +
"    String axis \"Z\";\n" +
"    Float64 colorBarMaximum 0.0;\n" +
"    Float64 colorBarMinimum -20.0;\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 4.89024e+8, 1.183248e+9;\n" +
"    String axis \"T\";\n" +
"    Float64 colorBarMaximum 1.183248e+9;\n" +
"    Float64 colorBarMinimum 4.89024e+8;\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station Identifier\";\n" +
"  }\n" +
"  common_name {\n" +
"    String ioos_category \"Taxonomy\";\n" +
"    String long_name \"Common Name\";\n" +
"  }\n" +
"  species_name {\n" +
"    String ioos_category \"Taxonomy\";\n" +
"    String long_name \"Species Name\";\n" +
"  }\n" +
"  size {\n" +
"    Int16 actual_range 1, 385;\n" +
"    String ioos_category \"Biology\";\n" +
"    String long_name \"Size\";\n" +
"    String units \"mm\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String acknowledgement \"NOAA NESDIS COASTWATCH, NOAA SWFSC ERD, Channel Islands National Park, National Park Service\";\n" +
"    String cdm_data_type \"Station\";\n" +
"    String contributor_email \"David_Kushner@nps.gov\";\n" +
"    String contributor_name \"Channel Islands National Park, National Park Service\";\n" +
"    String contributor_role \"Source of data.\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Observation Dataset v1.0\";\n" +
"    String creator_email \"Roy.Mendelssohn@noaa.gov\";\n" +
"    String creator_name \"NOAA NMFS SWFSC ERD\";\n" +
"    String creator_url \"http://www.pfel.noaa.gov\";\n" +
"    String date_created \"2008-06-11T21:43:28Z\";\n" +
"    String date_issued \"2008-06-11T21:43:28Z\";\n" +
"    Float64 Easternmost_Easting -118.4;\n" +
"    Float64 geospatial_lat_max 34.05;\n" +
"    Float64 geospatial_lat_min 32.8;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max -118.4;\n" +
"    Float64 geospatial_lon_min -120.4;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    Float64 geospatial_vertical_max -5.0;\n" +
"    Float64 geospatial_vertical_min -17.0;\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"Channel Islands National Park, National Park Service\n" +
"2008-06-11T21:43:28Z NOAA CoastWatch (West Coast Node) and NOAA SFSC ERD\n" +
today + " (local files)\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/erdCinpKfmSFNH.das\";\n" +
"    String infoUrl \"http://www.nps.gov/chis/naturescience/index.htm\";\n" +
"    String institution \"CINP\";\n" +
"    String keywords \"EARTH SCIENCE > Oceans > Marine Biology > Marine Habitat\";\n" +
"    String keywords_vocabulary \"GCMD Science Keywords\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended for legal use, since it may contain inaccuracies. Neither the data Contributor, CoastWatch, NOAA, nor the United States Government, nor any of their employees or contractors, makes any warranty, express or implied, including warranties of merchantability and fitness for a particular purpose, or assumes any legal liability for the accuracy, completeness, or usefulness, of this information.  National Park Service Disclaimer: The National Park Service shall not be held liable for improper or incorrect use of the data described and/or contained herein. These data and related graphics are not legal documents and are not intended to be used as such. The information contained in these data is dynamic and may change over time. The data are not better than the original sources from which they were derived. It is the responsibility of the data user to use the data appropriately and consistent within the limitation of geospatial data in general and these data in particular. The related graphics are intended to aid the data user in acquiring relevant data; it is not appropriate to use the related graphics as data. The National Park Service gives no warranty, expressed or implied, as to the accuracy, reliability, or completeness of these data. It is strongly recommended that these data are directly acquired from an NPS server and not indirectly through other sources which may have changed the data in some way. Although these data have been processed successfully on computer systems at the National Park Service, no warranty expressed or implied is made regarding the utility of the data on other systems for general or scientific purposes, nor shall the act of distribution constitute any such warranty. This disclaimer applies both to individual use of the data and aggregate use with other data.\";\n" +
"    String naming_authority \"gov.noaa.pfel.coastwatch\";\n" +
"    Float64 Northernmost_Northing 34.05;\n" +
"    String observationDimension \"row\";\n" +
"    String project \"NOAA NMFS SWFSC ERD (http://www.pfel.noaa.gov/)\";\n" +
"    String references \"Channel Islands National Parks Inventory and Monitoring information: http://nature.nps.gov/im/units/medn . Kelp Forest Monitoring Protocols: http://www.nature.nps.gov/im/units/chis/Reports_PDF/Marine/KFM-HandbookVol1.pdf .\";\n" +
"    String sourceUrl \"(local files)\";\n" +
"    Float64 Southernmost_Northing 32.8;\n" +
"    String standard_name_vocabulary \"CF-1.0\";\n" +
"    String summary \"This dataset has measurements of the size of selected animal species at selected locations in the Channel Islands National Park. Sampling is conducted annually between the months of May-October, so the Time data in this file is July 1 of each year (a nominal value). The size frequency measurements were taken within 10 meters of the transect line at each site.  Depths at the site vary some, but we describe the depth of the site along the transect line where that station's temperature logger is located, a typical depth for the site.\";\n" +
"    String time_coverage_end \"2007-07-01T00:00:00Z\";\n" +
"    String time_coverage_start \"1985-07-01T00:00:00Z\";\n" +
"    String title \"Channel Islands, Kelp Forest Monitoring, Size and Frequency, Natural Habitat\";\n" +
"    Float64 Westernmost_Easting -120.4;\n" +
"  }\n" +
"}\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float64 longitude;\n" +
"    Float64 latitude;\n" +
"    Float64 altitude;\n" +
"    Float64 time;\n" +
"    String id;\n" +
"    String common_name;\n" +
"    String species_name;\n" +
"    Int16 size;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //*** test make data files
        String2.log("\n****************** EDDTableFromNcFiles.test 1D make DATA FILES\n");       

        //.csv    for one lat,lon,time
        userDapQuery = "" +
            "&longitude=-119.05&latitude=33.46666666666&time=2005-07-01T00:00:00";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_1Station", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, id, common_name, species_name, size\n" +
"degrees_east, degrees_north, m, UTC, , , , mm\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 57\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 41\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 55\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 60\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
        expected = //last 3 lines
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Purple sea urchin, Strongylocentrotus purpuratus, 15\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Purple sea urchin, Strongylocentrotus purpuratus, 23\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Purple sea urchin, Strongylocentrotus purpuratus, 19\n";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "\nresults=\n" + results);


        //.csv    for one lat,lon,time      via lon > <
        userDapQuery = "" +
            "&longitude>-119.06&longitude<=-119.04&latitude=33.46666666666&time=2005-07-01T00:00:00";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_1StationGTLT", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, id, common_name, species_name, size\n" +
"degrees_east, degrees_north, m, UTC, , , , mm\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 57\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 41\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 55\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Bat star, Asterina miniata, 60\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
        expected = //last 3 lines
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Purple sea urchin, Strongylocentrotus purpuratus, 15\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Purple sea urchin, Strongylocentrotus purpuratus, 23\n" +
"-119.05, 33.4666666666667, -14.0, 2005-07-01T00:00:00Z, Santa Barbara (Webster's Arch), Purple sea urchin, Strongylocentrotus purpuratus, 19\n";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "\nresults=\n" + results);


        //.csv for test requesting all stations, 1 time, 1 species
        userDapQuery = "" +
            "&time=2005-07-01&common_name=\"Red+abalone\"";
        long time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_eq", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, id, common_name, species_name, size\n" +
"degrees_east, degrees_north, m, UTC, , , , mm\n" +
"-120.35, 34.05, -5.0, 2005-07-01T00:00:00Z, San Miguel (Hare Rock), Red abalone, Haliotis rufescens, 13\n" +
"-120.4, 34.0166666666667, -10.0, 2005-07-01T00:00:00Z, San Miguel (Miracle Mile), Red abalone, Haliotis rufescens, 207\n" +
"-120.4, 34.0166666666667, -10.0, 2005-07-01T00:00:00Z, San Miguel (Miracle Mile), Red abalone, Haliotis rufescens, 203\n" +
"-120.4, 34.0166666666667, -10.0, 2005-07-01T00:00:00Z, San Miguel (Miracle Mile), Red abalone, Haliotis rufescens, 193\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
        expected = //last 3 lines
"-120.116666666667, 33.8833333333333, -13.0, 2005-07-01T00:00:00Z, Santa Rosa (South Point), Red abalone, Haliotis rufescens, 185\n" +
"-120.15, 33.9, -9.0, 2005-07-01T00:00:00Z, Santa Rosa (Trancion Canyon), Red abalone, Haliotis rufescens, 198\n" +
"-120.15, 33.9, -9.0, 2005-07-01T00:00:00Z, Santa Rosa (Trancion Canyon), Red abalone, Haliotis rufescens, 85\n";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "\nresults=\n" + results);


        //.csv for test requesting all stations, 1 time, 1 species    String !=
        userDapQuery = "" +
            "&time=2005-07-01&id!=\"San+Miguel+(Hare+Rock)\"&common_name=\"Red+abalone\"";
        time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_NE", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, id, common_name, species_name, size\n" +
"degrees_east, degrees_north, m, UTC, , , , mm\n" +
//hare rock removed
"-120.4, 34.0166666666667, -10.0, 2005-07-01T00:00:00Z, San Miguel (Miracle Mile), Red abalone, Haliotis rufescens, 207\n" +
"-120.4, 34.0166666666667, -10.0, 2005-07-01T00:00:00Z, San Miguel (Miracle Mile), Red abalone, Haliotis rufescens, 203\n" +
"-120.4, 34.0166666666667, -10.0, 2005-07-01T00:00:00Z, San Miguel (Miracle Mile), Red abalone, Haliotis rufescens, 193\n";
        Test.ensureEqual(results.substring(0, expected.length()), expected, "\nresults=\n" + results);
        expected = //last 3 lines
"-120.116666666667, 33.8833333333333, -13.0, 2005-07-01T00:00:00Z, Santa Rosa (South Point), Red abalone, Haliotis rufescens, 185\n" +
"-120.15, 33.9, -9.0, 2005-07-01T00:00:00Z, Santa Rosa (Trancion Canyon), Red abalone, Haliotis rufescens, 198\n" +
"-120.15, 33.9, -9.0, 2005-07-01T00:00:00Z, Santa Rosa (Trancion Canyon), Red abalone, Haliotis rufescens, 85\n";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, "\nresults=\n" + results);


        //.csv for test requesting all stations, 1 time, 1 species   String > <
        userDapQuery = "" +
            "&time=2005-07-01&id>\"San+Miguel+(G\"&id<=\"San+Miguel+(I\"&common_name=\"Red+abalone\"";
        time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_gtlt", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, id, common_name, species_name, size\n" +
"degrees_east, degrees_north, m, UTC, , , , mm\n" +
"-120.35, 34.05, -5.0, 2005-07-01T00:00:00Z, San Miguel (Hare Rock), Red abalone, Haliotis rufescens, 13\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //.csv for test requesting all stations, 1 time, 1 species     REGEX
        userDapQuery = "longitude,latitude,altitude,time,id,species_name,size" + //no common_name
            "&time=2005-07-01&id=~\"(zztop|.*Hare+Rock.*)\"&common_name=\"Red+abalone\"";   //but common_name here
        time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_regex", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, id, species_name, size\n" +
"degrees_east, degrees_north, m, UTC, , , mm\n" +
"-120.35, 34.05, -5.0, 2005-07-01T00:00:00Z, San Miguel (Hare Rock), Haliotis rufescens, 13\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

    }

    /**
     * This tests the methods in this class with a 2D dataset.
     *
     * @throws Throwable if trouble
     */
    public static void test2D(boolean deleteCachedDatasetInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.test2D() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        //the test files were made with makeTestFiles();
        String datasetName = "testNc2D";       
        if (deleteCachedDatasetInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + datasetName + ".dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + datasetName + ".files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + datasetName + ".bad.json");
        }

        //touch a good and a bad file, so they are checked again
        File2.touch("/u00/data/points/nc2d/NDBC_32012_met.nc");
        File2.touch("/u00/data/points/nc2d/NDBC_4D_met.nc");

        EDDTable eddTable = (EDDTable)oneFromDatasetXml(datasetName); 
        //just comment out when working on datasets below
/* currently not active
        //test sos-server values
        String2.log("nOfferings=" + eddTable.sosNOfferings());
        String2.log(eddTable.sosOfferingName(0) + "  lon=" +
            eddTable.sosMinLongitude(0) + ", " +
            eddTable.sosMaxLongitude(0) + " lat=" +
            eddTable.sosMinLatitude(0) + ", " +
            eddTable.sosMaxLatitude(0) + " time=" +
            eddTable.sosMinTime(0) + ", " +
            eddTable.sosMaxTime(0));
        String2.log(String2.toCSVString(eddTable.sosObservedProperties()));

        Test.ensureTrue(eddTable.sosNOfferings() >=675, "n="+eddTable.sosNOfferings());
        Test.ensureEqual(eddTable.sosOfferingName(0), "31201", "");
        Test.ensureEqual(eddTable.sosMinLongitude(0), -48.13, "");
        Test.ensureEqual(eddTable.sosMaxLongitude(0), -48.13, "");
        Test.ensureEqual(eddTable.sosMinLatitude(0), -27.7, "");
        Test.ensureEqual(eddTable.sosMaxLatitude(0), -27.7, "");
        Test.ensureEqual(eddTable.sosMinTime(0), 1.1138688E9, "");
        Test.ensureEqual(eddTable.sosMaxTime(0), 1.1244636E9, "");
        Test.ensureEqual(eddTable.sosObservedProperties().length, 10, ""); 
        Test.ensureEqual(eddTable.sosObservedProperties()[0], 
            "http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.0/dictionaries/phenomenaDictionary.xml#AverageWavePeriod", 
            "");
*/
        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromNcFiles 2D test das dds for entire dataset\n");
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".das"); 
        results = String2.annotatedString(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"  time {[10]\n" +
"    String _CoordinateAxisType \"Time\";[10]\n" +
"    Float64 actual_range 8.67456e+7, 1.2075984e+9;[10]\n" +
"    String axis \"T\";[10]\n" +
"    String comment \"Time in seconds since 1970-01-01T00:00:00Z. The original times are rounded to the nearest hour.\";[10]\n" +
"    String ioos_category \"Time\";[10]\n" +
"    String long_name \"Time\";[10]\n" +
"    String standard_name \"time\";[10]\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";[10]\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";[10]\n" +
"  }[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  wd {[10]\n" +
"    Int16 _FillValue 32767;[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  wspv {[10]\n" +
"    Float32 _FillValue -9999999.0;[10]\n" +
"    Float32 actual_range"; //varies with subset -6.1, 11.0;[10]  
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"    String comment \"The meridional wind speed (m/s) indicates the v component of where the wind is going, derived from Wind Direction and Wind Speed.\";[10]\n" +
"    String ioos_category \"Wind\";[10]\n" +
"    String long_name \"Wind Speed, Meridional\";[10]\n" +
"    Float32 missing_value -9999999.0;[10]\n" +
"    String standard_name \"northward_wind\";[10]\n" +
"    String units \"m s-1\";[10]\n" +
"  }[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float32 latitude;\n" +
"    Float64 time;\n" +      //no altitude or longitude
"    String station;\n" +
"    Int16 wd;\n" +
"    Float32 wspd;\n" +
"    Float32 gst;\n" +
"    Float32 wvht;\n" +
"    Float32 dpd;\n" +
"    Float32 apd;\n" +
"    Int16 mwd;\n" +
"    Float32 bar;\n" +
"    Float32 atmp;\n" +
"    Float32 wtmp;\n" +
"    Float32 dewp;\n" +
"    Float32 vis;\n" +
"    Float32 ptdy;\n" +
"    Float32 tide;\n" +
"    Float32 wspu;\n" +
"    Float32 wspv;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //*** test make data files
        String2.log("\n****************** EDDTableFromNcFiles.test2D make DATA FILES\n");       

        //.csv
        //from NdbcMetStation.test31201
        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 19 00 00 999 99.0 99.0  1.40  9.00 99.00 999 9999.0 999.0  24.4 999.0 99.0 99.00 first available
        //double seconds = Calendar2.isoStringToEpochSeconds("2005-04-19T00");
        //int row = table.getColumn(timeIndex).indexOf("" + seconds, 0);
        //Test.ensureEqual(table.getStringData(idIndex, row), "31201", "");
        //Test.ensureEqual(table.getFloatData(latIndex, row), -27.7f, "");
        //Test.ensureEqual(table.getFloatData(lonIndex, row), -48.13f, "");

        userDapQuery = "latitude,time,station,wvht,dpd,wtmp,dewp" +
            "&latitude=-27.7&time=2005-04-19T00";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data1", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"latitude, time, station, wvht, dpd, wtmp, dewp\n" +
"degrees_north, UTC, , m, s, degree_C, degree_C\n" +
"-27.7, 2005-04-19T00:00:00Z, 31201, 1.4, 9.0, 24.4, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 25 18 00 999 99.0 99.0  3.90  8.00 99.00 999 9999.0 999.0  23.9 999.0 99.0 99.00
        userDapQuery = "latitude,time,station,wvht,dpd,wtmp,dewp" +
            "&latitude=-27.7&time>=2005-04-01&time<=2005-04-26";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = "latitude, time, station, wvht, dpd, wtmp, dewp\n";
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "degrees_north, UTC, , m, s, degree_C, degree_C\n";
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "-27.7, 2005-04-19T00:00:00Z, 31201, 1.4, 9.0, 24.4, NaN\n"; //time above
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "-27.7, 2005-04-25T18:00:00Z, 31201, 3.9, 8.0, 23.9, NaN\n"; //this time
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);

        //test requesting a lat area
        userDapQuery = "latitude,time,station,wvht,dpd,wtmp,dewp" +
            "&latitude>35&latitude<39&time=2005-04-01";
        long time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data3", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"latitude, time, station, wvht, dpd, wtmp, dewp\n" +
"degrees_north, UTC, , m, s, degree_C, degree_C\n" +
"35.01, 2005-04-01T00:00:00Z, 41025, 1.34, 10.0, 9.1, 14.9\n" +
"38.47, 2005-04-01T00:00:00Z, 44004, 2.04, 11.43, 9.8, 4.9\n" +
"38.46, 2005-04-01T00:00:00Z, 44009, 1.3, 10.0, 5.0, 5.7\n" +
"36.61, 2005-04-01T00:00:00Z, 44014, 1.67, 11.11, 6.5, 8.6\n" +
"37.36, 2005-04-01T00:00:00Z, 46012, 2.55, 12.5, 13.7, NaN\n" +
"38.23, 2005-04-01T00:00:00Z, 46013, 2.3, 12.9, 13.9, NaN\n" +
"37.75, 2005-04-01T00:00:00Z, 46026, 1.96, 12.12, 14.0, NaN\n" +
"35.74, 2005-04-01T00:00:00Z, 46028, 2.57, 12.9, 16.3, NaN\n" +
"36.75, 2005-04-01T00:00:00Z, 46042, 2.21, 17.39, 14.5, NaN\n" +
"37.98, 2005-04-01T00:00:00Z, 46059, 2.51, 14.29, 12.9, NaN\n" +
"36.83, 2005-04-01T00:00:00Z, 46091, NaN, NaN, NaN, NaN\n" +
"36.75, 2005-04-01T00:00:00Z, 46092, NaN, NaN, NaN, NaN\n" +
"36.69, 2005-04-01T00:00:00Z, 46093, NaN, NaN, 14.3, NaN\n" +
"37.57, 2005-04-01T00:00:00Z, 46214, 2.5, 9.0, 12.8, NaN\n" +
"35.21, 2005-04-01T00:00:00Z, 46215, 1.4, 10.0, 11.4, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //test that constraint vars are sent to low level data request
        userDapQuery = "latitude,station,wvht,dpd,wtmp,dewp" + //no "time" here
            "&latitude>35&latitude<39&time=2005-04-01"; //"time" here
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data4", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"latitude, station, wvht, dpd, wtmp, dewp\n" +
"degrees_north, , m, s, degree_C, degree_C\n" +
"35.01, 41025, 1.34, 10.0, 9.1, 14.9\n" +
"38.47, 44004, 2.04, 11.43, 9.8, 4.9\n" +
"38.46, 44009, 1.3, 10.0, 5.0, 5.7\n" +
"36.61, 44014, 1.67, 11.11, 6.5, 8.6\n" +
"37.36, 46012, 2.55, 12.5, 13.7, NaN\n" +
"38.23, 46013, 2.3, 12.9, 13.9, NaN\n" +
"37.75, 46026, 1.96, 12.12, 14.0, NaN\n" +
"35.74, 46028, 2.57, 12.9, 16.3, NaN\n" +
"36.75, 46042, 2.21, 17.39, 14.5, NaN\n" +
"37.98, 46059, 2.51, 14.29, 12.9, NaN\n" +
"36.83, 46091, NaN, NaN, NaN, NaN\n" +
"36.75, 46092, NaN, NaN, NaN, NaN\n" +
"36.69, 46093, NaN, NaN, 14.3, NaN\n" +
"37.57, 46214, 2.5, 9.0, 12.8, NaN\n" +
"35.21, 46215, 1.4, 10.0, 11.4, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //test that constraint vars are sent to low level data request
        //and that constraint causing 0rows for a station doesn't cause problems
        userDapQuery = "latitude,wtmp&time>=2008-03-14T18:00:00Z&time<=2008-03-14T18:00:00Z&wtmp>20";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data5", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"latitude, wtmp\n" +
"degrees_north, degree_C\n" +
"32.31, 23.5\n" +
"28.5, 21.3\n" +
"28.95, 23.7\n" +
"30.0, 20.1\n" +
"14.53, 25.3\n" +
"20.99, 25.7\n" +
"27.47, 23.8\n" +
"31.9784, 22.0\n" +
"28.4, 21.0\n" +
"27.55, 21.8\n" +
"25.9, 24.1\n" +
"25.17, 23.9\n" +
"26.07, 26.1\n" +
"22.01, 24.4\n" +
"19.87, 26.8\n" +
"15.01, 26.4\n" +
"27.3403, 20.2\n" +
"29.06, 21.8\n" +
"38.47, 20.4\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);
    }

    /**
     * This tests the methods in this class with a 3D dataset.
     *
     * @throws Throwable if trouble
     */
    public static void test3D(boolean deleteCachedDatasetInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.test3D() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
        String datasetName = "testNc3D";
        
        if (deleteCachedDatasetInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + datasetName + ".dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + datasetName + ".files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + datasetName + ".bad.json");
        }

        //touch a good and a bad file, so they are checked again
        File2.touch("/u00/data/points/nc3d/NDBC_32012_met.nc");
        File2.touch("/u00/data/points/nc3d/NDBC_4D_met.nc");

        EDDTable eddTable = (EDDTable)oneFromDatasetXml(datasetName); 
        //just comment out when working on datasets below
/* currently not active
        //test sos-server values
        String2.log("nOfferings=" + eddTable.sosNOfferings());
        String2.log(eddTable.sosOfferingName(0) + "  lon=" +
            eddTable.sosMinLongitude(0) + ", " +
            eddTable.sosMaxLongitude(0) + " lat=" +
            eddTable.sosMinLatitude(0) + ", " +
            eddTable.sosMaxLatitude(0) + " time=" +
            eddTable.sosMinTime(0) + ", " +
            eddTable.sosMaxTime(0));
        String2.log(String2.toCSVString(eddTable.sosObservedProperties()));

        Test.ensureTrue(eddTable.sosNOfferings() >=675, "n="+eddTable.sosNOfferings());
        Test.ensureEqual(eddTable.sosOfferingName(0), "31201", "");
        Test.ensureEqual(eddTable.sosMinLongitude(0), -48.13, "");
        Test.ensureEqual(eddTable.sosMaxLongitude(0), -48.13, "");
        Test.ensureEqual(eddTable.sosMinLatitude(0), -27.7, "");
        Test.ensureEqual(eddTable.sosMaxLatitude(0), -27.7, "");
        Test.ensureEqual(eddTable.sosMinTime(0), 1.1138688E9, "");
        Test.ensureEqual(eddTable.sosMaxTime(0), 1.1244636E9, "");
        Test.ensureEqual(eddTable.sosObservedProperties().length, 10, ""); 
        Test.ensureEqual(eddTable.sosObservedProperties()[0], 
            "http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.0/dictionaries/phenomenaDictionary.xml#AverageWavePeriod", 
            "");
*/
        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromNcFiles test3D das dds for entire dataset\n");
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".das"); 
        results = String2.annotatedString(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"  time {[10]\n" +
"    String _CoordinateAxisType \"Time\";[10]\n" +
"    Float64 actual_range 8.67456e+7, 1.2075984e+9;[10]\n" +
"    String axis \"T\";[10]\n" +
"    String comment \"Time in seconds since 1970-01-01T00:00:00Z. The original times are rounded to the nearest hour.\";[10]\n" +
"    String ioos_category \"Time\";[10]\n" +
"    String long_name \"Time\";[10]\n" +
"    String standard_name \"time\";[10]\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";[10]\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";[10]\n" +
"  }[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  wd {[10]\n" +
"    Int16 _FillValue 32767;[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  wspv {[10]\n" +
"    Float32 _FillValue -9999999.0;[10]\n" +
"    Float32 actual_range"; //varies with subset -6.1, 11.0;[10]  
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"    String comment \"The meridional wind speed (m/s) indicates the v component of where the wind is going, derived from Wind Direction and Wind Speed.\";[10]\n" +
"    String ioos_category \"Wind\";[10]\n" +
"    String long_name \"Wind Speed, Meridional\";[10]\n" +
"    Float32 missing_value -9999999.0;[10]\n" +
"    String standard_name \"northward_wind\";[10]\n" +
"    String units \"m s-1\";[10]\n" +
"  }[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float32 longitude;\n" +
"    Float32 latitude;\n" +
"    Float64 time;\n" +      //no altitude
"    String station;\n" +
"    Int16 wd;\n" +
"    Float32 wspd;\n" +
"    Float32 gst;\n" +
"    Float32 wvht;\n" +
"    Float32 dpd;\n" +
"    Float32 apd;\n" +
"    Int16 mwd;\n" +
"    Float32 bar;\n" +
"    Float32 atmp;\n" +
"    Float32 wtmp;\n" +
"    Float32 dewp;\n" +
"    Float32 vis;\n" +
"    Float32 ptdy;\n" +
"    Float32 tide;\n" +
"    Float32 wspu;\n" +
"    Float32 wspv;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //*** test make data files
        String2.log("\n****************** EDDTableFromNcFiles.test3D make DATA FILES\n");       

        //.csv
        //from NdbcMetStation.test31201
        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 19 00 00 999 99.0 99.0  1.40  9.00 99.00 999 9999.0 999.0  24.4 999.0 99.0 99.00 first available
        //double seconds = Calendar2.isoStringToEpochSeconds("2005-04-19T00");
        //int row = table.getColumn(timeIndex).indexOf("" + seconds, 0);
        //Test.ensureEqual(table.getStringData(idIndex, row), "31201", "");
        //Test.ensureEqual(table.getFloatData(latIndex, row), -27.7f, "");
        //Test.ensureEqual(table.getFloatData(lonIndex, row), -48.13f, "");

        userDapQuery = "longitude,latitude,time,station,wvht,dpd,wtmp,dewp" +
            "&longitude=-48.13&latitude=-27.7&time=2005-04-19T00";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data1", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, time, station, wvht, dpd, wtmp, dewp\n" +
"degrees_east, degrees_north, UTC, , m, s, degree_C, degree_C\n" +
"-48.13, -27.7, 2005-04-19T00:00:00Z, 31201, 1.4, 9.0, 24.4, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 25 18 00 999 99.0 99.0  3.90  8.00 99.00 999 9999.0 999.0  23.9 999.0 99.0 99.00
        userDapQuery = "longitude,latitude,time,station,wvht,dpd,wtmp,dewp" +
            "&longitude=-48.13&latitude=-27.7&time>=2005-04-01&time<=2005-04-26";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = "longitude, latitude, time, station, wvht, dpd, wtmp, dewp\n";
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "degrees_east, degrees_north, UTC, , m, s, degree_C, degree_C\n";
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "-48.13, -27.7, 2005-04-19T00:00:00Z, 31201, 1.4, 9.0, 24.4, NaN\n"; //time above
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "-48.13, -27.7, 2005-04-25T18:00:00Z, 31201, 3.9, 8.0, 23.9, NaN\n"; //this time
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);

        //test requesting a lat lon area
        userDapQuery = "longitude,latitude,time,station,wvht,dpd,wtmp,dewp" +
            "&longitude>-125&longitude<-121&latitude>35&latitude<39&time=2005-04-01";
        long time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data3", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, time, station, wvht, dpd, wtmp, dewp\n" +
"degrees_east, degrees_north, UTC, , m, s, degree_C, degree_C\n" +
"-122.88, 37.36, 2005-04-01T00:00:00Z, 46012, 2.55, 12.5, 13.7, NaN\n" +
"-123.32, 38.23, 2005-04-01T00:00:00Z, 46013, 2.3, 12.9, 13.9, NaN\n" +
"-122.82, 37.75, 2005-04-01T00:00:00Z, 46026, 1.96, 12.12, 14.0, NaN\n" +
"-121.89, 35.74, 2005-04-01T00:00:00Z, 46028, 2.57, 12.9, 16.3, NaN\n" +
"-122.42, 36.75, 2005-04-01T00:00:00Z, 46042, 2.21, 17.39, 14.5, NaN\n" +
"-121.9, 36.83, 2005-04-01T00:00:00Z, 46091, NaN, NaN, NaN, NaN\n" +
"-122.02, 36.75, 2005-04-01T00:00:00Z, 46092, NaN, NaN, NaN, NaN\n" +
"-122.41, 36.69, 2005-04-01T00:00:00Z, 46093, NaN, NaN, 14.3, NaN\n" +
"-123.28, 37.57, 2005-04-01T00:00:00Z, 46214, 2.5, 9.0, 12.8, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //test that constraint vars are sent to low level data request
        userDapQuery = "longitude,latitude,station,wvht,dpd,wtmp,dewp" + //no "time" here
            "&longitude>-125&longitude<-121&latitude>35&latitude<39&time=2005-04-01"; //"time" here
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data4", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, station, wvht, dpd, wtmp, dewp\n" +
"degrees_east, degrees_north, , m, s, degree_C, degree_C\n" +
"-122.88, 37.36, 46012, 2.55, 12.5, 13.7, NaN\n" +
"-123.32, 38.23, 46013, 2.3, 12.9, 13.9, NaN\n" +
"-122.82, 37.75, 46026, 1.96, 12.12, 14.0, NaN\n" +
"-121.89, 35.74, 46028, 2.57, 12.9, 16.3, NaN\n" +
"-122.42, 36.75, 46042, 2.21, 17.39, 14.5, NaN\n" +
"-121.9, 36.83, 46091, NaN, NaN, NaN, NaN\n" +
"-122.02, 36.75, 46092, NaN, NaN, NaN, NaN\n" +
"-122.41, 36.69, 46093, NaN, NaN, 14.3, NaN\n" +
"-123.28, 37.57, 46214, 2.5, 9.0, 12.8, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //test that constraint vars are sent to low level data request
        //and that constraint causing 0rows for a station doesn't cause problems
        userDapQuery = "longitude,latitude,wtmp&time>=2008-03-14T18:00:00Z&time<=2008-03-14T18:00:00Z&wtmp>20";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data5", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, wtmp\n" +
"degrees_east, degrees_north, degree_C\n" +
"-75.35, 32.31, 23.5\n" +
"-80.17, 28.5, 21.3\n" +
"-78.47, 28.95, 23.7\n" +
"-80.6, 30.0, 20.1\n" +
"-46.0, 14.53, 25.3\n" +
"-65.01, 20.99, 25.7\n" +
"-71.49, 27.47, 23.8\n" +
"-69.649, 31.9784, 22.0\n" +
"-80.53, 28.4, 21.0\n" +
"-80.22, 27.55, 21.8\n" +
"-89.67, 25.9, 24.1\n" +
"-94.42, 25.17, 23.9\n" +
"-85.94, 26.07, 26.1\n" +
"-94.05, 22.01, 24.4\n" +
"-85.06, 19.87, 26.8\n" +
"-67.5, 15.01, 26.4\n" +
"-84.245, 27.3403, 20.2\n" +
"-88.09, 29.06, 21.8\n" +
"-70.56, 38.47, 20.4\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);
    }

    /**
     * One time: make C:/u00/data/points/nc2d and C:/u00/data/points/nc3d test files 
     * from C:/u00/data/points/ndbcMet nc4d files.
     */
    public static void makeTestFiles() {
        //get list of files
        String fromDir = "C:/u00/data/points/ndbcMet/";
        String dir3 = "C:/u00/data/points/nc3d/";
        String dir2 = "C:/u00/data/points/nc2d/";
        String[] names = RegexFilenameFilter.list(fromDir, "NDBC_(3|4).*nc");        
        
        //for each file
        for (int f = 0; f < names.length; f++) {
            NetcdfFile in = null;
            NetcdfFileWriteable out2 = null;
            NetcdfFileWriteable out3 = null;

            try {
                String2.log("in #" + f + "=" + fromDir + names[f]);
                if (f == 0) 
                    String2.log(NcHelper.dumpString(fromDir + names[f], false));

                in = NcHelper.openFile(fromDir + names[f]);
                out2 = NetcdfFileWriteable.createNew(dir2 + names[f], false);
                out3 = NetcdfFileWriteable.createNew(dir3 + names[f], false);

                //write the globalAttributes
                Attributes atts = new Attributes();
                NcHelper.getGlobalAttributes(in, atts);
                String attNames[] = atts.getNames();
                for (int a = 0; a < attNames.length; a++) { 
                    out2.addGlobalAttribute(NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                    out3.addGlobalAttribute(NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                }              

                List vars = NcHelper.find4DVariables(in, null);
                Variable timeVar = in.findVariable("TIME");
                Variable latVar = in.findVariable("LAT");
                Variable lonVar = in.findVariable("LON");
                Dimension tDim2 = out2.addDimension("TIME", (int)timeVar.getSize());
                Dimension tDim3 = out3.addDimension("TIME", (int)timeVar.getSize());
                Dimension yDim2 = out2.addDimension("LAT", 1);
                Dimension yDim3 = out3.addDimension("LAT", 1);
                Dimension xDim3 = out3.addDimension("LON", 1);
                
                //create axis variables
                out2.addVariable("TIME", timeVar.getDataType(), new Dimension[]{tDim2}); 
                out2.addVariable("LAT",  latVar.getDataType(),  new Dimension[]{yDim2}); 

                out3.addVariable("TIME", timeVar.getDataType(), new Dimension[]{tDim3}); 
                out3.addVariable("LAT",  latVar.getDataType(),  new Dimension[]{yDim3}); 
                out3.addVariable("LON",  lonVar.getDataType(),  new Dimension[]{xDim3}); 

                //write the axis variable attributes
                atts.clear();
                NcHelper.getVariableAttributes(timeVar, atts);
                attNames = atts.getNames();
                for (int a = 0; a < attNames.length; a++) {
                    out2.addVariableAttribute("TIME", NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                    out3.addVariableAttribute("TIME", NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                }

                atts.clear();
                NcHelper.getVariableAttributes(latVar, atts);
                attNames = atts.getNames();
                for (int a = 0; a < attNames.length; a++) {
                    out2.addVariableAttribute("LAT", NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                    out3.addVariableAttribute("LAT", NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                }

                atts.clear();
                NcHelper.getVariableAttributes(lonVar, atts);
                attNames = atts.getNames();
                for (int a = 0; a < attNames.length; a++) {
                    //not out2
                    out3.addVariableAttribute("LON", NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                }

                //create data variables
                for (int col = 0; col < vars.size(); col++) {
                    //create the data variables
                    Variable var = (Variable)vars.get(col);
                    String varName = var.getName();
                    Array ar = var.read();
                    out2.addVariable(varName, var.getDataType(), new Dimension[]{tDim2, yDim2}); 
                    out3.addVariable(varName, var.getDataType(), new Dimension[]{tDim3, yDim3, xDim3}); 

                    //write the data variable attributes
                    atts.clear();
                    NcHelper.getVariableAttributes(var, atts);
                    attNames = atts.getNames();
                    for (int a = 0; a < attNames.length; a++) {
                        out2.addVariableAttribute(varName, NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                        out3.addVariableAttribute(varName, NcHelper.getAttribute(attNames[a], atts.get(attNames[a])));
                    }
                }

                //leave "define" mode
                out2.create();
                out3.create();

                //write axis data
                Array ar = in.findVariable("TIME").read();
                out2.write("TIME", ar);
                out3.write("TIME", ar);
                ar = in.findVariable("LAT").read();
                out2.write("LAT", ar);
                out3.write("LAT", ar);
                ar = in.findVariable("LON").read();
                out3.write("LON", ar);
                 
                for (int col = 0; col < vars.size(); col++) {
                    //write the data for each var
                    Variable var = (Variable)vars.get(col);
                    String varName = var.getName();
                    ar = var.read();
                    int oldShape[] = ar.getShape();
                    int newShape2[] = {oldShape[0], 1};
                    int newShape3[] = {oldShape[0], 1, 1};
                    out2.write(varName, ar.reshape(newShape2));
                    out3.write(varName, ar.reshape(newShape3));
                }

                in.close();
                out2.close();
                out3.close();

                if (f == 0) {
                    String2.log("\nout2=" + NcHelper.dumpString(dir2 + names[f], false));
                    String2.log("\nout3=" + NcHelper.dumpString(dir3 + names[f], false));
                }

            } catch (Throwable t) {
                String2.log(MustBe.throwableToString(t));
                try { if (in != null)  in.close();    } catch (Exception t2) {}
                try { if (out2 != null) out2.close(); } catch (Exception t2) {}
                try { if (out3 != null) out3.close(); } catch (Exception t2) {}
            }
        }
    }

    /**
     * This tests the methods in this class with a 4D dataset.
     *
     * @throws Throwable if trouble
     */
    public static void test4D(boolean deleteCachedDatasetInfo) throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.test4D() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        if (deleteCachedDatasetInfo) {
            File2.delete(EDStatic.fullDatasetInfoDirectory + "cwwcNDBCMet.dirs.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "cwwcNDBCMet.files.json");
            File2.delete(EDStatic.fullDatasetInfoDirectory + "cwwcNDBCMet.bad.json");
        }
        //touch a good and a bad file, so they are checked again
        File2.touch("/u00/data/points/ndbcMet/NDBC_32012_met.nc");
        File2.touch("/u00/data/points/ndbcMet/NDBC_3D_met.nc");

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("cwwcNDBCMet"); 
        //just comment out when working on datasets below
/* currently not active
        //test sos-server values
        String2.log("nOfferings=" + eddTable.sosNOfferings());
        String2.log(eddTable.sosOfferingName(0) + "  lon=" +
            eddTable.sosMinLongitude(0) + ", " +
            eddTable.sosMaxLongitude(0) + " lat=" +
            eddTable.sosMinLatitude(0) + ", " +
            eddTable.sosMaxLatitude(0) + " time=" +
            eddTable.sosMinTime(0) + ", " +
            eddTable.sosMaxTime(0));
        String2.log(String2.toCSVString(eddTable.sosObservedProperties()));

        Test.ensureTrue(eddTable.sosNOfferings() >=675, "n="+eddTable.sosNOfferings());
        Test.ensureEqual(eddTable.sosOfferingName(0), "31201", "");
        Test.ensureEqual(eddTable.sosMinLongitude(0), -48.13, "");
        Test.ensureEqual(eddTable.sosMaxLongitude(0), -48.13, "");
        Test.ensureEqual(eddTable.sosMinLatitude(0), -27.7, "");
        Test.ensureEqual(eddTable.sosMaxLatitude(0), -27.7, "");
        Test.ensureEqual(eddTable.sosMinTime(0), 1.1138688E9, "");
        Test.ensureEqual(eddTable.sosMaxTime(0), 1.1244636E9, "");
        Test.ensureEqual(eddTable.sosObservedProperties().length, 10, ""); 
        Test.ensureEqual(eddTable.sosObservedProperties()[0], 
            "http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.0/dictionaries/phenomenaDictionary.xml#AverageWavePeriod", 
            "");
*/
        //*** test getting das for entire dataset
        String2.log("\n****************** EDDTableFromNcFiles test4D das dds for entire dataset\n");
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".das"); 
        results = String2.annotatedString(new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray()));
        //String2.log(results);
        expected = 
"  altitude {[10]\n" +
"    String _CoordinateAxisType \"Height\";[10]\n" +
"    String _CoordinateZisPositive \"up\";[10]\n" +
"    Float32 actual_range 0.0, 0.0;[10]\n" +
"    String axis \"Z\";[10]\n" +
"    String comment \"The depth of the station, nominally 0 (see station information for details).\";[10]\n" +
"    String ioos_category \"Location\";[10]\n" +
"    String long_name \"Altitude\";[10]\n" +
"    String positive \"up\";[10]\n" +
"    String standard_name \"altitude\";[10]\n" +
"    String units \"m\";[10]\n" +
"  }[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  wd {[10]\n" +
"    Int16 _FillValue 32767;[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"  wspv {[10]\n" +
"    Float32 _FillValue -9999999.0;[10]\n" +
"    Float32 actual_range"; //varies with subset -6.1, 11.0;[10]  
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        expected = 
"    String comment \"The meridional wind speed (m/s) indicates the v component of where the wind is going, derived from Wind Direction and Wind Speed.\";[10]\n" +
"    String ioos_category \"Wind\";[10]\n" +
"    String long_name \"Wind Speed, Meridional\";[10]\n" +
"    Float32 missing_value -9999999.0;[10]\n" +
"    String standard_name \"northward_wind\";[10]\n" +
"    String units \"m s-1\";[10]\n" +
"  }[10]\n";
        Test.ensureTrue(results.indexOf(expected) > 0, "\nresults=\n" + results);
        
        //*** test getting dds for entire dataset
        tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Entire", ".dds"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"Dataset {\n" +
"  Sequence {\n" +
"    Float32 longitude;\n" +
"    Float32 latitude;\n" +
"    Float32 altitude;\n" +
"    Float64 time;\n" +
"    String station;\n" +
"    Int16 wd;\n" +
"    Float32 wspd;\n" +
"    Float32 gst;\n" +
"    Float32 wvht;\n" +
"    Float32 dpd;\n" +
"    Float32 apd;\n" +
"    Int16 mwd;\n" +
"    Float32 bar;\n" +
"    Float32 atmp;\n" +
"    Float32 wtmp;\n" +
"    Float32 dewp;\n" +
"    Float32 vis;\n" +
"    Float32 ptdy;\n" +
"    Float32 tide;\n" +
"    Float32 wspu;\n" +
"    Float32 wspv;\n" +
"  } s;\n" +
"} s;\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //*** test make data files
        String2.log("\n****************** EDDTableFromNcFiles.test4D make DATA FILES\n");       

        //.csv
        //from NdbcMetStation.test31201
        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 19 00 00 999 99.0 99.0  1.40  9.00 99.00 999 9999.0 999.0  24.4 999.0 99.0 99.00 first available
        //double seconds = Calendar2.isoStringToEpochSeconds("2005-04-19T00");
        //int row = table.getColumn(timeIndex).indexOf("" + seconds, 0);
        //Test.ensureEqual(table.getStringData(idIndex, row), "31201", "");
        //Test.ensureEqual(table.getFloatData(latIndex, row), -27.7f, "");
        //Test.ensureEqual(table.getFloatData(lonIndex, row), -48.13f, "");

        userDapQuery = "longitude,latitude,altitude,time,station,wvht,dpd,wtmp,dewp" +
            "&longitude=-48.13&latitude=-27.7&time=2005-04-19T00";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data1", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, station, wvht, dpd, wtmp, dewp\n" +
"degrees_east, degrees_north, m, UTC, , m, s, degree_C, degree_C\n" +
"-48.13, -27.7, 0.0, 2005-04-19T00:00:00Z, 31201, 1.4, 9.0, 24.4, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 25 18 00 999 99.0 99.0  3.90  8.00 99.00 999 9999.0 999.0  23.9 999.0 99.0 99.00
        userDapQuery = "longitude,latitude,altitude,time,station,wvht,dpd,wtmp,dewp" +
            "&longitude=-48.13&latitude=-27.7&time>=2005-04-01&time<=2005-04-26";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = "longitude, latitude, altitude, time, station, wvht, dpd, wtmp, dewp\n";
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "degrees_east, degrees_north, m, UTC, , m, s, degree_C, degree_C\n";
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "-48.13, -27.7, 0.0, 2005-04-19T00:00:00Z, 31201, 1.4, 9.0, 24.4, NaN\n"; //time above
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);
        expected = "-48.13, -27.7, 0.0, 2005-04-25T18:00:00Z, 31201, 3.9, 8.0, 23.9, NaN\n"; //this time
        Test.ensureTrue(results.indexOf(expected) >= 0, "\nresults=\n" + results);

        //test requesting a lat lon area
        userDapQuery = "longitude,latitude,altitude,time,station,wvht,dpd,wtmp,dewp" +
            "&longitude>-125&longitude<-121&latitude>35&latitude<39&time=2005-04-01";
        long time = System.currentTimeMillis();
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data3", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, time, station, wvht, dpd, wtmp, dewp\n" +
"degrees_east, degrees_north, m, UTC, , m, s, degree_C, degree_C\n" +
"-122.88, 37.36, 0.0, 2005-04-01T00:00:00Z, 46012, 2.55, 12.5, 13.7, NaN\n" +
"-123.32, 38.23, 0.0, 2005-04-01T00:00:00Z, 46013, 2.3, 12.9, 13.9, NaN\n" +
"-122.82, 37.75, 0.0, 2005-04-01T00:00:00Z, 46026, 1.96, 12.12, 14.0, NaN\n" +
"-121.89, 35.74, 0.0, 2005-04-01T00:00:00Z, 46028, 2.57, 12.9, 16.3, NaN\n" +
"-122.42, 36.75, 0.0, 2005-04-01T00:00:00Z, 46042, 2.21, 17.39, 14.5, NaN\n" +
"-121.9, 36.83, 0.0, 2005-04-01T00:00:00Z, 46091, NaN, NaN, NaN, NaN\n" +
"-122.02, 36.75, 0.0, 2005-04-01T00:00:00Z, 46092, NaN, NaN, NaN, NaN\n" +
"-122.41, 36.69, 0.0, 2005-04-01T00:00:00Z, 46093, NaN, NaN, 14.3, NaN\n" +
"-123.28, 37.57, 0.0, 2005-04-01T00:00:00Z, 46214, 2.5, 9.0, 12.8, NaN\n" +
"-122.3, 37.77, 0.0, 2005-04-01T00:00:00Z, AAMC1, NaN, NaN, 15.5, NaN\n" +
"-123.71, 38.91, 0.0, 2005-04-01T00:00:00Z, ANVC1, NaN, NaN, NaN, NaN\n" +
"-122.47, 37.81, 0.0, 2005-04-01T00:00:00Z, FTPC1, NaN, NaN, NaN, NaN\n" +
"-121.89, 36.61, 0.0, 2005-04-01T00:00:00Z, MTYC1, NaN, NaN, 15.1, NaN\n" +
"-122.04, 38.06, 0.0, 2005-04-01T00:00:00Z, PCOC1, NaN, NaN, 14.9, NaN\n" +
"-123.74, 38.96, 0.0, 2005-04-01T00:00:00Z, PTAC1, NaN, NaN, NaN, NaN\n" +
"-122.4, 37.93, 0.0, 2005-04-01T00:00:00Z, RCMC1, NaN, NaN, 14.0, NaN\n" +
"-122.21, 37.51, 0.0, 2005-04-01T00:00:00Z, RTYC1, NaN, NaN, 14.2, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //test that constraint vars are sent to low level data request
        userDapQuery = "longitude,latitude,altitude,station,wvht,dpd,wtmp,dewp" + //no "time" here
            "&longitude>-125&longitude<-121&latitude>35&latitude<39&time=2005-04-01"; //"time" here
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data4", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, altitude, station, wvht, dpd, wtmp, dewp\n" +
"degrees_east, degrees_north, m, , m, s, degree_C, degree_C\n" +
"-122.88, 37.36, 0.0, 46012, 2.55, 12.5, 13.7, NaN\n" +
"-123.32, 38.23, 0.0, 46013, 2.3, 12.9, 13.9, NaN\n" +
"-122.82, 37.75, 0.0, 46026, 1.96, 12.12, 14.0, NaN\n" +
"-121.89, 35.74, 0.0, 46028, 2.57, 12.9, 16.3, NaN\n" +
"-122.42, 36.75, 0.0, 46042, 2.21, 17.39, 14.5, NaN\n" +
"-121.9, 36.83, 0.0, 46091, NaN, NaN, NaN, NaN\n" +
"-122.02, 36.75, 0.0, 46092, NaN, NaN, NaN, NaN\n" +
"-122.41, 36.69, 0.0, 46093, NaN, NaN, 14.3, NaN\n" +
"-123.28, 37.57, 0.0, 46214, 2.5, 9.0, 12.8, NaN\n" +
"-122.3, 37.77, 0.0, AAMC1, NaN, NaN, 15.5, NaN\n" +
"-123.71, 38.91, 0.0, ANVC1, NaN, NaN, NaN, NaN\n" +
"-122.47, 37.81, 0.0, FTPC1, NaN, NaN, NaN, NaN\n" +
"-121.89, 36.61, 0.0, MTYC1, NaN, NaN, 15.1, NaN\n" +
"-122.04, 38.06, 0.0, PCOC1, NaN, NaN, 14.9, NaN\n" +
"-123.74, 38.96, 0.0, PTAC1, NaN, NaN, NaN, NaN\n" +
"-122.4, 37.93, 0.0, RCMC1, NaN, NaN, 14.0, NaN\n" +
"-122.21, 37.51, 0.0, RTYC1, NaN, NaN, 14.2, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);


        //test that constraint vars are sent to low level data request
        //and that constraint causing 0rows for a station doesn't cause problems
        userDapQuery = "longitude,latitude,wtmp&time>=2008-03-14T18:00:00Z&time<=2008-03-14T18:00:00Z&wtmp>20";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_Data5", ".csv"); 
        String2.log("queryTime=" + (System.currentTimeMillis() - time));
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, wtmp\n" +
"degrees_east, degrees_north, degree_C\n" +
"-80.17, 28.5, 21.2\n" +
"-78.47, 28.95, 23.7\n" +
"-80.6, 30.0, 20.1\n" +
"-46.0, 14.53, 25.3\n" +
"-65.01, 20.99, 25.7\n" +
"-71.49, 27.47, 23.8\n" +
"-69.649, 31.9784, 22.0\n" +
"-80.53, 28.4, 21.6\n" +
"-80.22, 27.55, 21.7\n" +
"-89.67, 25.9, 24.1\n" +
"-94.42, 25.17, 23.9\n" +
"-85.94, 26.07, 26.1\n" +
"-94.05, 22.01, 24.4\n" +
"-85.06, 19.87, 26.8\n" +
"-67.5, 15.01, 26.4\n" +
"-84.245, 27.3403, 20.2\n" +
"-88.09, 29.06, 21.8\n" +
"-70.56, 38.47, 20.4\n" +
"-157.79, 17.14, 24.3\n" +
"-160.74, 19.16, 24.7\n" +
"-152.48, 17.52, 24.0\n" +
"-153.87, 0.02, 25.0\n" +
"-158.12, 21.67, 24.2\n" +
"-157.68, 21.42, 24.3\n" +
"144.79, 13.54, 28.1\n" +
"-90.42, 29.78, 20.6\n" +
"-64.92, 18.34, 27.7\n" +
"-81.87, 26.65, 22.4\n" +
"-80.1, 25.59, 23.5\n" +
"-156.47, 20.9, 25.0\n" +
"167.74, 8.74, 27.6\n" +
"-81.81, 24.55, 23.8\n" +
"-80.86, 24.84, 23.8\n" +
"-64.75, 17.7, 26.0\n" +
"-67.05, 17.97, 27.2\n" +
"-80.38, 25.01, 24.2\n" +
"-81.81, 26.13, 24.0\n" +
"-170.688, -14.28, 29.6\n" +
"-157.87, 21.31, 25.5\n" +
"-96.4, 28.45, 20.1\n" +
"-82.77, 24.69, 22.8\n" +
"-82.63, 27.76, 21.7\n" +
"-66.12, 18.46, 28.3\n" +
"-177.36, 28.21, 21.8\n" +
"-80.59, 28.42, 22.9\n" +
"166.62, 19.29, 27.8\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

    }

    /**
     * This is run by hand (special setup) to test dealing with the last 24 hours.
     * Before running this, run NDBCMet updateLastNDays, then copy some files to /u00/data/points/ndbcMet
     * so files have very recent data.
     *
     * @throws Throwable if trouble
     */
    public static void test24Hours() throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.test24Hours() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("cwwcNDBCMet"); 

        //!!!change time to be ~nowLocal+16 (= nowZulu+8);  e.g., T20 for local time 4pm
        userDapQuery = "longitude,latitude,time,station,wd,wtmp&time%3E=2009-03-12T20"; 
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_24hours", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        String2.log(results);

        //in log output, look at end of constructor for "maxTime is within last 24hrs, so setting maxTime to NaN (i.e., Now)."
        //in log output, look for stations saying "file maxTime is within last 24hrs, so ERDDAP is pretending file maxTime is now+4hours."
    }


    /**
     * This test &amp;distinct().
     *
     * @throws Throwable if trouble
     */
    public static void testDistinct() throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.testDistinct() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("cwwcNDBCMet"); 

        //time constraints force erddap to get actual data, (not just station variables)
        //  and order of variables says to sort by lon first
        userDapQuery = "longitude,latitude,station&station=~\"5.*\"&time>=2008-03-11&time<2008-03-12&distinct()"; 
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_distincts", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = 
"longitude, latitude, station\n" +
"degrees_east, degrees_north, \n" +
"-162.21, 23.43, 51001\n" +
"-160.74, 19.16, 51003\n" +
"-158.12, 21.67, 51201\n" +
"-157.79, 17.14, 51002\n" +
"-157.68, 21.42, 51202\n" +
"-153.87, 0.02, 51028\n" +
"-152.48, 17.52, 51004\n" +
"144.79, 13.54, 52200\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

    }


    /** This test getting just station ids. */
    public static void testId() throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.testId() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";
        EDV edv;
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("cwwcNDBCMet"); 

        userDapQuery = "station&station>\"5\"&station<\"6\""; 
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_id", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"station\n" +
"\n" +
"51001\n" +
"51002\n" +
"51003\n" +
"51004\n" +
"51005\n" +
"51026\n" +
"51027\n" +
"51028\n" +
"51101\n" +
"51201\n" +
"51202\n" +
"51203\n" +
"52009\n" +
"52200\n";
        Test.ensureEqual(results, expected, "results=" + results);
    }
     
    /**
     * This tests orderBy.
     *
     * @throws Throwable if trouble
     */
    public static void testOrderBy() throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.testOrderBy() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("cwwcNDBCMet"); 

        //.csv
        //from NdbcMetStation.test31201
        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 19 00 00 999 99.0 99.0  1.40  9.00 99.00 999 9999.0 999.0  24.4 999.0 99.0 99.00 first available
        userDapQuery = "time,station,wtmp,atmp&station>\"5\"&station<\"6\"" +
            "&time>=2005-04-19T21&time<2005-04-20&orderBy(\"station,time\")";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_ob", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"time, station, wtmp, atmp\n" +
"UTC, , degree_C, degree_C\n" +
"2005-04-19T21:00:00Z, 51001, 24.1, 23.5\n" +
"2005-04-19T22:00:00Z, 51001, 24.2, 23.6\n" +
"2005-04-19T23:00:00Z, 51001, 24.2, 22.1\n" +
"2005-04-19T21:00:00Z, 51002, 25.1, 25.4\n" +
"2005-04-19T22:00:00Z, 51002, 25.2, 25.4\n" +
"2005-04-19T23:00:00Z, 51002, 25.2, 24.8\n" +
"2005-04-19T21:00:00Z, 51003, 25.3, 23.9\n" +
"2005-04-19T22:00:00Z, 51003, 25.4, 24.3\n" +
"2005-04-19T23:00:00Z, 51003, 25.4, 24.7\n" +
"2005-04-19T21:00:00Z, 51004, 25.0, 24.0\n" +
"2005-04-19T22:00:00Z, 51004, 25.0, 23.8\n" +
"2005-04-19T23:00:00Z, 51004, 25.0, 24.3\n" +
"2005-04-19T21:00:00Z, 51028, 27.7, 27.6\n" +
"2005-04-19T22:00:00Z, 51028, 27.8, 27.1\n" +
"2005-04-19T23:00:00Z, 51028, 27.8, 27.5\n" +
"2005-04-19T21:00:00Z, 51201, 25.0, NaN\n" +
"2005-04-19T22:00:00Z, 51201, 24.9, NaN\n" +
"2005-04-19T23:00:00Z, 51201, 25.0, NaN\n" +
"2005-04-19T21:00:00Z, 51202, 24.5, NaN\n" +
"2005-04-19T22:00:00Z, 51202, 24.6, NaN\n" +
"2005-04-19T23:00:00Z, 51202, 24.5, NaN\n" +
"2005-04-19T21:00:00Z, 52200, 28.0, NaN\n" +
"2005-04-19T22:00:00Z, 52200, 28.0, NaN\n" +
"2005-04-19T23:00:00Z, 52200, 28.0, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        //.csv
        //from NdbcMetStation.test31201
        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 19 00 00 999 99.0 99.0  1.40  9.00 99.00 999 9999.0 999.0  24.4 999.0 99.0 99.00 first available
        userDapQuery = "time,station,wtmp,atmp&station>\"5\"&station<\"6\"" +
            "&time>=2005-04-19T21&time<2005-04-20&orderBy(\"time,station\")";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_ob2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"time, station, wtmp, atmp\n" +
"UTC, , degree_C, degree_C\n" +
"2005-04-19T21:00:00Z, 51001, 24.1, 23.5\n" +
"2005-04-19T21:00:00Z, 51002, 25.1, 25.4\n" +
"2005-04-19T21:00:00Z, 51003, 25.3, 23.9\n" +
"2005-04-19T21:00:00Z, 51004, 25.0, 24.0\n" +
"2005-04-19T21:00:00Z, 51028, 27.7, 27.6\n" +
"2005-04-19T21:00:00Z, 51201, 25.0, NaN\n" +
"2005-04-19T21:00:00Z, 51202, 24.5, NaN\n" +
"2005-04-19T21:00:00Z, 52200, 28.0, NaN\n" +
"2005-04-19T22:00:00Z, 51001, 24.2, 23.6\n" +
"2005-04-19T22:00:00Z, 51002, 25.2, 25.4\n" +
"2005-04-19T22:00:00Z, 51003, 25.4, 24.3\n" +
"2005-04-19T22:00:00Z, 51004, 25.0, 23.8\n" +
"2005-04-19T22:00:00Z, 51028, 27.8, 27.1\n" +
"2005-04-19T22:00:00Z, 51201, 24.9, NaN\n" +
"2005-04-19T22:00:00Z, 51202, 24.6, NaN\n" +
"2005-04-19T22:00:00Z, 52200, 28.0, NaN\n" +
"2005-04-19T23:00:00Z, 51001, 24.2, 22.1\n" +
"2005-04-19T23:00:00Z, 51002, 25.2, 24.8\n" +
"2005-04-19T23:00:00Z, 51003, 25.4, 24.7\n" +
"2005-04-19T23:00:00Z, 51004, 25.0, 24.3\n" +
"2005-04-19T23:00:00Z, 51028, 27.8, 27.5\n" +
"2005-04-19T23:00:00Z, 51201, 25.0, NaN\n" +
"2005-04-19T23:00:00Z, 51202, 24.5, NaN\n" +
"2005-04-19T23:00:00Z, 52200, 28.0, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

    }

    /**
     * This tests orderByMax.
     *
     * @throws Throwable if trouble
     */
    public static void testOrderByMax() throws Throwable {
        String2.log("\n****************** EDDTableFromNcFiles.testOrderBy() *****************\n");
        testVerboseOn();
        String name, tName, results, tResults, expected, userDapQuery, tQuery;
        String error = "";

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("cwwcNDBCMet"); 

        //.csv
        //from NdbcMetStation.test31201
        //YYYY MM DD hh mm  WD WSPD  GST  WVHT   DPD   APD MWD  BARO   ATMP  WTMP  DEWP  VIS  TIDE
        //2005 04 19 00 00 999 99.0 99.0  1.40  9.00 99.00 999 9999.0 999.0  24.4 999.0 99.0 99.00 first available
        userDapQuery = "time,station,wtmp,atmp&station>\"5\"&station<\"6\"" +
            "&time>=2005-04-19T21&time<2005-04-20&orderByMax(\"station,time\")";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_obm", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"time, station, wtmp, atmp\n" +
"UTC, , degree_C, degree_C\n" +
"2005-04-19T23:00:00Z, 51001, 24.2, 22.1\n" +
"2005-04-19T23:00:00Z, 51002, 25.2, 24.8\n" +
"2005-04-19T23:00:00Z, 51003, 25.4, 24.7\n" +
"2005-04-19T23:00:00Z, 51004, 25.0, 24.3\n" +
"2005-04-19T23:00:00Z, 51028, 27.8, 27.5\n" +
"2005-04-19T23:00:00Z, 51201, 25.0, NaN\n" +
"2005-04-19T23:00:00Z, 51202, 24.5, NaN\n" +
"2005-04-19T23:00:00Z, 52200, 28.0, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

        userDapQuery = "time,station,wtmp,atmp&station>\"5\"&station<\"6\"" +
            "&time>=2005-04-19T21&time<2005-04-20&orderByMax(\"time,station\")";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.className() + "_obm2", ".csv"); 
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        expected = 
"time, station, wtmp, atmp\n" +
"UTC, , degree_C, degree_C\n" +
"2005-04-19T21:00:00Z, 52200, 28.0, NaN\n" +
"2005-04-19T22:00:00Z, 52200, 28.0, NaN\n" +
"2005-04-19T23:00:00Z, 52200, 28.0, NaN\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);

    }

    /**
     * This tests the methods in this class.
     *
     * @throws Throwable if trouble
     */
    public static void test() throws Throwable {
        test1D(false); //deleteCachedDatasetInfo
        test2D(true); 
        test3D(false);
        test4D(false);
        testId();
        testDistinct();
        testOrderBy();
        testOrderByMax();

        //not usually run
        //test24Hours();  //requires special set up
    }
}

