/* 
 * EDDTableFromSOS Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.ByteArray;
import com.cohort.array.DoubleArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.String2;
import com.cohort.util.Test;

import gov.noaa.pfel.coastwatch.griddata.FileNameUtility;
import gov.noaa.pfel.coastwatch.griddata.NcHelper;
import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.coastwatch.util.SimpleXMLReader;
import gov.noaa.pfel.coastwatch.util.SSR;

import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.*;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;


/** 
 * This class aggregates data from a group of stations, all served by
 * one SOS server. The stations all serve the same set of variables
 * (although the source for each station doesn't have to serve all variables).
 * <p>The SWE (Sensor Web Enablement) and SOS (Sensor Observation Service) home page
 *   is  http://www.opengeospatial.org/projects/groups/sensorweb .
 * <p>See OpenGIS Sensor Observation Service Implementation Specification
 *   (OGC 06-009r6) Version: 1.0.0
 *   from http://www.opengeospatial.org/projects/groups/sensorweb .
 *   (Bob has a copy in c:/projects/sos)
 * <p>See OGC Web Services Common Specification ver 1.1.0 (OGC 06-121r3)
 *    from http://www.opengeospatial.org/standards/common
 *    which covers construction of GET and POST queries (e.g., section 7.2.3 
 *    and section 9).
 *   (Bob has a copy in c:/projects/sos)
 * <p>SOS overview:
 *   <ul>
 *   <li>If you send a getCapabilities xml request to a SOS server
 *      (sourceUrl + "?service=SOS&request=GetCapabilities"), 
 *      you get an xml result
 *      with a list of stations and the observedProperties that they have data for.
 *   <li>An observedProperty is a formal URI reference to a property
 *      e.g., urn:ogc:phenomenon:longitude:wgs84 or http://marinemetadata.org/cf#sea_water_temperature
 *   <li>An observedProperty isn't a variable.
 *   <li>More than one variable may have the same observedProperty 
 *     (e.g., insideTemp and outsideTemp might both have 
 *     observedProperty http://marinemetadata.org/cf#air_temperature)
 *     See the observedProperty schema to find the variables associated
 *     with each observedProperty.
 *   <li>If you send a getObservation xml request to a SOS server, you get an xml result
 *      with descriptions of field names in the response, field units, and the data.
 *      The field names will include longitude, latitude, depth(perhaps), and time.
 *   <li>This class just gets data for one station at a time to avoid
 *      requests for huge amounts of data at one time (which ties up memory
 *      and is slow getting initial response to user).
 *   <li>Currently, some SOS servers (e.g., GoMOOS) respond to getObservation requests
 *      for more than one observedProperty by just returning results for
 *      the first of the observedProperties. (No error message!)
 *      See the constructor parameter tRequestObservedPropertiesSeparately.  
 *   <li>!!!The getCapabilities result doesn't include the results variables (they say field names).
 *       The only way to know them ahead of time is to request some data,
 *       look at the xml, and note the field name (and units).
 *       E.g., For a given station (e.g., D01) and observedProperty (e.g., ...sea_water_temperature)
 *       look at the results from the getObservation url:
 *       http://dev.gomoos.org/cgi-bin/sos/oostethys_sos?REQUEST=GetObservation&OFFERING=D01&OBSERVEDPROPERTY=sea_water_temperature&TIME=2007-06-01T00:00:00/2007-06-24T23:59:00
 *       Similarly, getCapabilities doesn't include altitude/depth information.
 *       But maybe this info is in the getDescription results (info for specific station).
 *   </ul>
 * <p>See Phenomenon dictionary at
 *   https://www.seegrid.csiro.au/subversion/xmml/OGC/branches/SWE_gml2/sweCommon/current/examples/phenomena.xml
 * <p>Info about a sample server: http://www.csc.noaa.gov/DTL/dtl_proj3_oostethys.html
 *   <br>The csc server uses code from
 *     http://www.oostethys.org/ogc-oceans-interoperability-experiment/experiment-1 .
 *   <br>It seems to be set up to just serve several instances of lon,lat,alt,time,1variable
 *     instead of lon,lat,alt,time,manyVariables.(?)
 *
 * <p>Sample queries for different time periods
 * http://www.oostethys.org/Members/tcook/sos-xml-time-encodings/?searchterm=Time
 *
 * <p>In general, if there is an error in processing the getCapabilities xml
 * for one station, only that station is rejected (with a message to the log.txt file).
 * So its is good to read the logs periodically to see if there are unexpected errors.
 *
 * <p>This class insists that each station is indeed a station at one lat lon point:
 *    lowerCorner's lat lon must equal upperCorner's lat lon or the station is ignored.
 *    
 * @author Bob Simons (bob.simons@noaa.gov) 2007-09-21
 */
public class EDDTableFromSOS extends EDDTable{ 

    /** stationTable */
    protected final static int stationLonCol = 0; 
    protected final static int stationLatCol = 1; 
    protected final static int stationBeginTimeCol = 2; //epochSeconds
    protected final static int stationEndTimeCol = 3;   //epochSeconds
    protected final static int stationIDCol = 4; 
    protected final static int stationProcedureCol = 5; 

    /** The first nFixedVariables dataVariables are always created automatically 
     * (don't include in constructor tDataVariables): 
     * longitude, latitude, stationID, altitude, time. */
    protected final static int nFixedVariables = 5; 

    public static boolean reallyReallyVerbose = false;


    /** Variables set by the constructor. */
    protected String sosVersion;
    protected boolean requestObservedPropertiesSeparately;
    protected StringArray uniqueSourceObservedProperties;
    protected Table stationTable;
    protected boolean[][] stationHasObsProp; //[station#][uniqueSourceObservedProperties index]
    protected boolean ioosServer = false;

    /**
     * This constructs an EDDTableFromSOS based on the information in an .xml file.
     * 
     * @param xmlReader with the &lt;erddapDatasets&gt;&lt;dataset type="EDDTableFromSOS"&gt;
     *    having just been read.  
     * @return an EDDTableFromSOS.
     *    When this returns, xmlReader will have just read &lt;erddapDatasets&gt;&lt;/dataset&gt; .
     * @throws Throwable if trouble
     */
    public static EDDTableFromSOS fromXml(SimpleXMLReader xmlReader) throws Throwable {

        //data to be obtained (or not)
        if (verbose) String2.log("\n*** constructing EDDTableFromSOS(xmlReader)...");
        String tDatasetID = xmlReader.attributeValue("datasetID"); 
        Attributes tGlobalAttributes = null;
        String tLongitudeSourceName = null;  
        String tLatitudeSourceName = null;  
        String tAltitudeSourceName = null;  double tAltitudeMetersPerSourceUnit = 1; 
        double tAltitudeSourceMinimum = Double.NaN;  
        double tAltitudeSourceMaximum = Double.NaN;  
        String tTimeSourceName = null; String tTimeSourceFormat = null;
        ArrayList tDataVariables = new ArrayList();
        int tReloadEveryNMinutes = Integer.MAX_VALUE;
        String tAccessibleTo = null;
        StringArray tOnChange = new StringArray();
        String tSourceUrl = null, tObservationOfferingIdRegex = null;
        boolean tRequestObservedPropertiesSeparately = false;

        //process the tags
        String startOfTags = xmlReader.allTags();
        int startOfTagsN = xmlReader.stackSize();
        int startOfTagsLength = startOfTags.length();
        while (true) {
            xmlReader.nextTag();
            String tags = xmlReader.allTags();
            String content = xmlReader.content();
            //if (reallyVerbose) String2.log("  tags=" + tags + content);
            if (xmlReader.stackSize() == startOfTagsN) 
                break; //the </dataset> tag
            String localTags = tags.substring(startOfTagsLength);

            //try to make the tag names as consistent, descriptive and readable as possible
            if      (localTags.equals("<addAttributes>"))
                tGlobalAttributes = getAttributesFromXml(xmlReader);
            else if (localTags.equals( "<longitudeSourceName>")) {}
            else if (localTags.equals("</longitudeSourceName>")) tLongitudeSourceName = content; 
            else if (localTags.equals( "<latitudeSourceName>"))  {}
            else if (localTags.equals("</latitudeSourceName>"))  tLatitudeSourceName = content; 
            else if (localTags.equals( "<altitudeSourceName>"))  {}
            else if (localTags.equals("</altitudeSourceName>"))  tAltitudeSourceName = content; 
            else if (localTags.equals( "<altitudeSourceMinimum>")) {}
            else if (localTags.equals("</altitudeSourceMinimum>")) tAltitudeSourceMinimum = String2.parseDouble(content); 
            else if (localTags.equals( "<altitudeSourceMaximum>")) {}
            else if (localTags.equals("</altitudeSourceMaximum>")) tAltitudeSourceMaximum = String2.parseDouble(content); 
            else if (localTags.equals( "<altitudeMetersPerSourceUnit>")) {}
            else if (localTags.equals("</altitudeMetersPerSourceUnit>")) 
                tAltitudeMetersPerSourceUnit = String2.parseDouble(content); 
            else if (localTags.equals( "<timeSourceName>"))   {}
            else if (localTags.equals("</timeSourceName>"))   tTimeSourceName = content; 
            else if (localTags.equals( "<timeSourceFormat>")) {}
            else if (localTags.equals("</timeSourceFormat>")) tTimeSourceFormat = content; 
            else if (localTags.equals( "<dataVariable>")) 
                tDataVariables.add(getSDADVariableFromXml(xmlReader));           
            else if (localTags.equals( "<accessibleTo>")) {}
            else if (localTags.equals("</accessibleTo>")) tAccessibleTo = content;
            else if (localTags.equals( "<reloadEveryNMinutes>")) {}
            else if (localTags.equals("</reloadEveryNMinutes>")) tReloadEveryNMinutes = String2.parseInt(content); 
            else if (localTags.equals( "<sourceUrl>")) {}
            else if (localTags.equals("</sourceUrl>")) tSourceUrl = content; 
            else if (localTags.equals( "<observationOfferingIdRegex>")) {}
            else if (localTags.equals("</observationOfferingIdRegex>")) tObservationOfferingIdRegex = content; 
            else if (localTags.equals( "<requestObservedPropertiesSeparately>")) {}
            else if (localTags.equals("</requestObservedPropertiesSeparately>")) 
                tRequestObservedPropertiesSeparately = content.equals("true"); 

            //onChange
            else if (localTags.equals( "<onChange>")) {}
            else if (localTags.equals("</onChange>")) 
                tOnChange.add(content); 

            else xmlReader.unexpectedTagException();
        }
        int ndv = tDataVariables.size();
        Object ttDataVariables[][] = new Object[ndv][];
        for (int i = 0; i < tDataVariables.size(); i++)
            ttDataVariables[i] = (Object[])tDataVariables.get(i);

        return new EDDTableFromSOS(tDatasetID, tAccessibleTo,
            tOnChange, tGlobalAttributes,
            tLongitudeSourceName, tLatitudeSourceName,
            tAltitudeSourceName, tAltitudeSourceMinimum, tAltitudeSourceMaximum, 
            tAltitudeMetersPerSourceUnit,
            tTimeSourceName, tTimeSourceFormat,
            ttDataVariables,
            tReloadEveryNMinutes, tSourceUrl,
            tObservationOfferingIdRegex, tRequestObservedPropertiesSeparately);
    }

    /**
     * The constructor.
     *
     * @param tDatasetID is a very short string identifier 
     *   (required: just safe characters: A-Z, a-z, 0-9, _, -, or .)
     *   for this dataset. See EDD.datasetID().
     * @param tAccessibleTo is a comma separated list of 0 or more
     *    roles which will have access to this dataset.
     *    <br>If null, everyone will have access to this dataset (even if not logged in).
     *    <br>If "", no one will have access to this dataset.
     * @param tOnChange 0 or more actions (starting with "http://" or "mailto:")
     *    to be done whenever the dataset changes significantly
     * @param tAddGlobalAttributes are global attributes which will
     *   be added to (and take precedence over) the data source's global attributes.
     *   This may be null if you have nothing to add.
     *   The combined global attributes must include:
     *   <ul>
     *   <li> "title" - the short (&lt; 80 characters) description of the dataset 
     *   <li> "summary" - the longer description of the dataset.
     *      It may have newline characters (usually at &lt;= 72 chars per line). 
     *   <li> "institution" - the source of the data 
     *      (best if &lt; 50 characters so it fits in a graph's legend).
     *   <li> "infoUrl" - the url with information about this data set 
     *   <li> "cdm_data_type" - one of the EDD.CDM_xxx options
     *   </ul>
     *   Special case: value="null" causes that item to be removed from combinedGlobalAttributes.
     *   Special case: if addGlobalAttributes name="license" value="[standard]",
     *     the EDStatic.standardLicense will be used.
     *   Special case: if addGlobalAttributes name="summary" value="[standard]",
     *     the standardSummary (from this class) will be used.
     * @param tLonSourceName the results field name for the longitude variable 
     *    (e.g., longitude).
     *    The name can vary, but the basic meaning must be longitude degrees East.
     *    If lon isn't in the results, use "longitude" here as a placeholder.
     * @param tLatSourceName the results field name for the latitude variable 
     *    (e.g., latitude)
     *    The name can vary, but the basic meaning must be latitude degrees North.
     *    If lat isn't in the results, use "latitude" here as a placeholder.
     * @param tAltSourceName the results field name for the altitude variable 
     *    (e.g., depth) (use this even if no alt variable)
     *    If alt isn't in the results, use "altitude" here as a placeholder.
     * @param tSourceMinAlt (in source units) or NaN if not known.
     *    This info is explicitly supplied because it isn't in getCapabilities.
     *    [I use eddTable.getEmpiricalMinMax("2007-02-01", "2007-02-01", false, true); below to get it.]
     * @param tSourceMaxAlt (in source units) or NaN if not known
     * @param tAltMetersPerSourceUnit the factor needed to convert the source
     *    alt values to/from meters above sea level.
     * @param tTimeSourceName the results field name for the time variable 
     *    (e.g., time)
     *    If time isn't in the results, use "time" here as a placeholder.
     * @param tTimeSourceFormat is either<ul>
     *    <li> a udunits string (containing " since ")
     *      describing how to interpret numeric values 
     *      (e.g., "seconds since 1970-01-01T00:00:00Z"),
     *    <li> a java.text.SimpleDateFormat string describing how to interpret string times  
     *    (see http://java.sun.com/j2se/1.4.2/docs/api/java/text/SimpleDateFormat.html),
     *    </ul>
     * @param tDataVariables is an Object[nDataVariables][4]: 
     *    <br>[0]=String sourceName (the field name of the data variable in the tabular results,
     *        e.g., Salinity, not the ObservedProperty name),
     *    <br>[1]=String destinationName (the name to be presented to the ERDDAP user, 
     *        or null to use the sourceName),
     *    <br>[2]=Attributes addAttributes (at ERD, this must have "ioos_category" - 
     *        a category from EDV.ioosCategories). 
     *        Special case: value="null" causes that item to be removed from combinedAttributes.
     *    <br>[3]=String source dataType (e.g., "int", "float", "String"). 
     *        Some data sources have ambiguous data types, so it needs to be specified here.
     *    <br>!!!Unique to EDDTableFromSOS: the longitude, latitude, and stationID 
     *       variables are created automatically.
     *    <br>Since these datasets can be served via ERDDAP's SOS server,
     *       each tDataVariable will have an "observedProperty" attribute.
     *       "observedProperty" defines the observedProperty SOS clients request in order
     *       to get data for that variable.
     *       For non-composite observedProperty, this is the observedProperty's 
     *         xlink:href attribute value, e.g., "http://marinemetadata.org/cf#sea_water_temperature".
     *       For composite observedProperty,  this is the CompositePhenomenon's
     *         gml:id attribute value, e.g., "WEATHER_OBSERVABLES".
     *    <br>!!!Unique to EDDTableFromSOS: if you need to set up a different
     *       observedProperty for the source and for what ERDDAP's SOS server will serve,
     *       use sourceObservedProperty to define the name for the source observedProperty.
     *    <br>!!!You can get station information by visiting the tSourceUrl
     *       (which by default returns the sos:Capabilities XML document).
     * @param tReloadEveryNMinutes indicates how often the source should
     *    be checked for new data.
     * @param tSourceUrl the url to which queries are sent
     * @param tObservationOfferingIdRegex only observationOfferings with IDs 
     *    (usually the station names) which match this regular expression
     *    are included in the dataset
     *    (".+" will catch all station names)
     * @param tRequestObservedPropertiesSeparately if true, the observedProperties
     *   will be requested separately. If false, they will be requested all at once.
     * @throws Throwable if trouble
     */
    public EDDTableFromSOS(String tDatasetID, String tAccessibleTo,
        StringArray tOnChange, 
        Attributes tAddGlobalAttributes,
        String tLonSourceName,
        String tLatSourceName,
        String tAltSourceName, double tSourceMinAlt, double tSourceMaxAlt, double tAltMetersPerSourceUnit, 
        String tTimeSourceName, String tTimeSourceFormat,
        Object[][] tDataVariables,
        int tReloadEveryNMinutes,
        String tSourceUrl, String tObservationOfferingIdRegex, 
        boolean tRequestObservedPropertiesSeparately) throws Throwable {

        if (verbose) String2.log(
            "\n*** constructing EDDTableFromSOS " + tDatasetID); 
        long constructionStartMillis = System.currentTimeMillis();
        String errorInMethod = "Error in EDDTableFromSOS(" + 
            tDatasetID + ") constructor:\n";
            
        //save some of the parameters
        className = "EDDTableFromSOS"; 
        datasetID = tDatasetID;
        setAccessibleTo(tAccessibleTo);
        onChange = tOnChange;
        if (tAddGlobalAttributes == null)
            tAddGlobalAttributes = new Attributes();
        addGlobalAttributes = tAddGlobalAttributes;
        String tLicense = addGlobalAttributes.getString("license");
        if (tLicense != null && tLicense.equals("[standard]"))
            addGlobalAttributes.set("license", EDStatic.standardLicense);
        addGlobalAttributes.set("sourceUrl", tSourceUrl);
        String tSummary = addGlobalAttributes.getString("summary");
        if (tSummary != null && tSummary.equals("[standard]"))
            addGlobalAttributes.set("summary", standardSummary);
        setReloadEveryNMinutes(tReloadEveryNMinutes);
        sourceUrl = tSourceUrl;
        Test.ensureNotNothing(sourceUrl, "sourceUrl wasn't specified.");
        requestObservedPropertiesSeparately = tRequestObservedPropertiesSeparately;

        sourceCanConstrainNumericData = CONSTRAIN_PARTIAL; //just lat and lon
        sourceCanConstrainStringData  = CONSTRAIN_PARTIAL; //time (but not != or regex), station_id (even REGEX_OP), but nothing else
        sourceCanConstrainStringRegex = REGEX_OP; //just for station_id

      
        //set source attributes (none available from source)
        sourceGlobalAttributes = new Attributes();
        combinedGlobalAttributes = new Attributes(addGlobalAttributes, sourceGlobalAttributes); //order is important
        combinedGlobalAttributes.removeValue("null"); 

        //get all dv sourceObservedProperties
        uniqueSourceObservedProperties = new StringArray();
        for (int dv = 0; dv < tDataVariables.length; dv++) {
            //no sourceAtt
            String tSourceName = (String)tDataVariables[dv][0];
            Attributes tAddAtt = (Attributes)tDataVariables[dv][2];
            String op = tAddAtt.getString("sourceObservedProperty"); //preference for sourceObservedProperty
            if (op == null || op.length() == 0) 
                op = tAddAtt.getString(EDV.observedProperty); //otherwise, source = regular observedProperty
            if (op == null || op.length() == 0) 
                throw new IllegalArgumentException("Neither 'sourceObservedProperty' nor 'observervedProperty' " +
                    "attributes were set for dataVariable sourceName=" + tSourceName + "."); 
            if (uniqueSourceObservedProperties.indexOf(op) < 0)
                uniqueSourceObservedProperties.add(op);
        }

        //*** read the getCapabilities xml and set up stationTable
        stationTable = new Table();
        stationTable.addColumn("Lon", new DoubleArray());
        stationTable.addColumn("Lat", new DoubleArray());
        stationTable.addColumn("BeginTime", new DoubleArray());
        stationTable.addColumn("EndTime", new DoubleArray());
        stationTable.addColumn("ID", new StringArray());
        stationTable.addColumn("Procedure", new StringArray());
        ArrayList tStationHasObsPropAL = new ArrayList();  //for each valid station, the boolean[]
        //values that persist for a while
        double tLon = Double.NaN, tLat = Double.NaN, 
            tBeginTime = Double.NaN, tEndTime = Double.NaN;
        String tIndeterminateEnd = null, tStationID = "", tStationProcedure = "";     
        double currentEpochSeconds = Calendar2.gcToEpochSeconds(Calendar2.newGCalendarZulu());
        boolean tStationHasObsProp[] = new boolean[uniqueSourceObservedProperties.size()];
        String tDVNames[] = new String[tDataVariables.length];
        for (int dv = 0; dv < tDataVariables.length; dv++)
            tDVNames[dv] = (String)tDataVariables[dv][0];
        //use KVP (KeyValuePair) HTTP GET request to getCapabilities
        //see section 7.2.3 of OGC 06-121r3 (OGC Web Services Common Specification) ver 1.1.0
        String tUrl = sourceUrl + "?service=SOS&request=GetCapabilities"; //&version=0.0.31"; or 1.0.0
        if (reallyVerbose) String2.log("getCapabilities Url=" + tUrl); 
        //String2.log(SSR.getUrlResponseString(tUrl));
        SimpleXMLReader xmlReader;
//when testing nos sos (since GetCapabilities is very slow).
//if (sourceUrl.equals("http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos/SOS")) {
//String2.log("\n!!! Using file version of NOS SOS until actual is faster!!!"); 
//Math2.sleep(3000);
//xmlReader = new SimpleXMLReader(new FileInputStream("C:/programs/sos/nosSosGetCapabilitiesCleanedByHand.xml"));
//} else 
        xmlReader = new SimpleXMLReader(SSR.getUrlInputStream(tUrl));

        xmlReader.nextTag();
        String tags = xmlReader.allTags();
        String sosPrefix = "";
        if (xmlReader.tag(0).equals("Capabilities")) {
            sosPrefix = "";              //ioosServer
            ioosServer = true;
        } else if (xmlReader.tag(0).equals("sos:Capabilities")) {
            sosPrefix = "sos:"; //oostethys
        } else {
            xmlReader.close();
            throw new IllegalArgumentException("First SOS capabilities tag=" + 
                tags + " should have been <Capabilities> or <sos:Capabilities>.");
        }
        sosVersion = xmlReader.attributeValue("version"); //e.g., "0.0.31" or "1.0.0"
        if (verbose) String2.log("  sosPrefix=" + sosPrefix + " ioosServer=" + ioosServer +
            " sosVersion=" + sosVersion); 
        String offeringTag = "<" + 
            sosPrefix + "Capabilities><" + 
            sosPrefix + "Contents><" + 
            sosPrefix + "ObservationOfferingList><" + 
            sosPrefix + "ObservationOffering>";
        String offeringEndTag = "<" + 
            sosPrefix + "Capabilities><" + 
            sosPrefix + "Contents><" + 
            sosPrefix + "ObservationOfferingList></" + 
            sosPrefix + "ObservationOffering>";
        do {
            //process the tags
            //String2.log("tags=" + tags + xmlReader.content());

            if (tags.startsWith(offeringTag)) {
                String endOfTag = tags.substring(offeringTag.length());
                String content = xmlReader.content();
                String fatalError = null;
                //String2.log("endOfTag=" + endOfTag + xmlReader.content());

/* separate phenomena
            <sos:ObservationOffering xmlns:xlink="http://www.w3.org/1999/xlink" gml:id="A01">
				<gml:description> Latest data from Mooring A01 from the Gulf of Maine Ocean Observing System (GoMOOS), located in the Gulf of Maine Massachusetts Bay </gml:description>
				<gml:name>A01</gml:name>
				<gml:boundedBy>
					<gml:Envelope>
						<gml:lowerCorner srsName="urn:ogc:def:crs:EPSG:6.5:4329">42.5277 -70.5665</gml:lowerCorner>
						<gml:upperCorner srsName="urn:ogc:def:crs:EPSG:6.5:4329">42.5277 -70.5665</gml:upperCorner>
					</gml:Envelope>
				</gml:boundedBy>
				<sos:time>
					<gml:TimePeriod gml:id="AVAILABLE_OFFERING_TIME">
						<gml:beginPosition>2001-07-10T06:30:00Z</gml:beginPosition>
						<gml:endPosition indeterminatePosition="now"/>
						<gml:timeInterval unit="hour">1</gml:timeInterval>
					</gml:TimePeriod>
				</sos:time>
				<sos:procedure xlink:href="urn:gomoos.org:source.mooring#A01"/>
                <sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_temperature"/>
                <sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_salinity"/>
				<sos:featureOfInterest xlink:href="urn:something:bodyOfWater"/>
				<sos:responseFormat>application/com-xml</sos:responseFormat>
   			</sos:ObservationOffering>
*/
/* or compositePhenomenon...
                <sos:observedProperty>
                    <swe:CompositePhenomenon gml:id="WEATHER_OBSERVABLES">
                        <gml:name>Weather measurements</gml:name>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#AirTemperature"/>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#AtmosphericPressure"/>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#WindSpeed"/>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#WindDirection"/>
                    </swe:CompositePhenomenon>
                </sos:observedProperty>
*/

                //if (endOfTag.equals("")) {  //ioosServer has different gml:id than gml:name; requests need gml:name.
                //    //e.g., 44004
                //    if (ioosServer) tStationID = xmlReader.attributeValue("gml:id");

                if (endOfTag.equals("</gml:name>")) { //ioosServer and OOSTethys have this
                    //e.g., 44004
                    tStationID = content;

                    //if (ioosServer) { 
                    //    //remove 
                    //    String pre = ":station:";
                    //    int prePo = tStationID.indexOf(pre);
                    //    if (prePo >= 0)
                    //        tStationID = tStationID.substring(prePo + pre.length());
                    //}

                } else if (endOfTag.equals(               "<gml:boundedBy><gml:Envelope></gml:lowerCorner>") || //OOSTethys has this
                           endOfTag.equals("<gml:boundedBy><gml:boundedBy><gml:Envelope></gml:lowerCorner>")) { //VAST does this
                    //e.g., 38.48 -70.43    order is lat lon
                    int spo = content.indexOf(' ');                
                    if (spo < 0) spo = 0; //push error to handler below 
                    tLat = String2.parseDouble(content.substring(0, spo));
                    tLon = String2.parseDouble(content.substring(spo + 1));
                    if (Double.isNaN(tLon) || Double.isNaN(tLat))
                        String2.log("Error while parsing gml: Invalid <gml:lowerCorner>=" + content);

                } else if (endOfTag.equals(               "<gml:boundedBy><gml:Envelope></gml:upperCorner>") || //OOSTethys has this
                           endOfTag.equals("<gml:boundedBy><gml:boundedBy><gml:Envelope></gml:upperCorner>")) { //VAST has this
                    //ensure upperCorner = lowerCorner
                    //e.g., 38.48 -70.43    order is lat lon
                    int spo = content.indexOf(' ');                
                    if (spo < 0) spo = 0; //push error to handler below 
                    double ttLat = String2.parseDouble(content.substring(0, spo));
                    double ttLon = String2.parseDouble(content.substring(spo + 1));
                    //disable test for VAST development?!   
                    //VAST has tLat=34.723 ttLat=34.725 tLon=-86.646 ttLon=-86.644
                    if (Double.isNaN(ttLon) || Double.isNaN(ttLat))
                        String2.log("Error while parsing gml: Invalid <gml:upperCorner>=" + content);
                    if (tLat != ttLat || tLon != ttLon) {
                        String2.log("Error while parsing gml: lowerCorner!=upperCorner" +
                            " tLat=" + tLat + " ttLat=" + ttLat + 
                            " tLon=" + tLon + " ttLon=" + ttLon);
                        tLon = Double.NaN; tLat = Double.NaN; //invalidate this station
                    }                   
                } else if (endOfTag.equals(         "<time><gml:TimePeriod></gml:beginPosition>") || //ioosServer has this
                           endOfTag.equals(     "<sos:time><gml:TimePeriod></gml:beginPosition>") || //OOSTethys has this
                           endOfTag.equals("<sos:eventTime><gml:TimePeriod></gml:beginPosition>")) { //VAST has this
                    //e.g., 2001-07-10T06:30:00Z
                    //This is workaround for invalid value "TZ" in NOS SOS GetCapabilities.
                    try {
                        tBeginTime = Calendar2.isoStringToEpochSeconds(content);
                    } catch (Throwable t) {
                        String2.log("Error while parsing gml:beginPosition ISO time: " + t.toString());
                        tBeginTime = Double.NaN;
                    }

                } else if (endOfTag.equals(         "<time><gml:TimePeriod><gml:endPosition>") || //ioosServer has this
                           endOfTag.equals(     "<sos:time><gml:TimePeriod><gml:endPosition>") || //OOSTethys has this
                           endOfTag.equals("<sos:eventTime><gml:TimePeriod><gml:endPosition>")) { //VAST has this
                    tIndeterminateEnd = xmlReader.attributeValue("indeterminatePosition");

                } else if (endOfTag.equals(         "<time><gml:TimePeriod></gml:endPosition>") || //ioosServer has this
                           endOfTag.equals(     "<sos:time><gml:TimePeriod></gml:endPosition>") || //OOSTethys has this
                           endOfTag.equals("<sos:eventTime><gml:TimePeriod></gml:endPosition>")) { //VAST has this) {
                    //usually:  <gml:endPosition indeterminatePosition="now"/>
                    //could be: e.g., 2001-07-10T06:30:00Z
                    if (content.length() > 0) {
                        try {
                            tEndTime = Calendar2.isoStringToEpochSeconds(content);
                        } catch (Throwable t) {
                            String2.log("Data source error while parsing gml:endPosition ISO time: " + t.toString());
                            tEndTime = Double.NaN;
                        }
                        //are they being precise about recent end time?  change to NaN
                        if (!Double.isNaN(tEndTime) && 
                            currentEpochSeconds - tEndTime < 2*Calendar2.SECONDS_PER_DAY)
                            tEndTime = Double.NaN;   
                    } else if (tIndeterminateEnd != null && 
                        (tIndeterminateEnd.equals("now") ||       //ioosServer has this
                         tIndeterminateEnd.equals("unknown"))) {  //OOSTethys has this
                        //leave as NaN...
                        //tEndTime = Calendar2.gcToEpochSeconds(Calendar2.newGCalendarZulu() + 
                        //    Calendar2.SECONDS_PER_HOUR); //buffer
                    } else {
                        String2.log("Data source error while parsing TimePeriod /gml:endPosition; content=\"\" tIndeterminateEnd=" + 
                            tIndeterminateEnd);
                        tBeginTime = Double.NaN; //mark as invalid; use tBeginTime since tIndeterminateEnd isn't required.
                    }

                } else if (endOfTag.equals("<" + sosPrefix + "procedure>")) {
                    //<sos:procedure xlink:href="urn:gomoos.org:source.mooring#A01"/>
                    //attribute is without quotes
                    String tXlink = xmlReader.attributeValue("xlink:href");
                    if (tXlink == null) 
                        String2.log("Error while parsing 'procedure': no xlink:href attribute!");
                    else 
                        tStationProcedure = tXlink;                    

                } else if (endOfTag.equals("<" + sosPrefix + "observedProperty>")) {
                    //handle non-composite observedProperty
                    //<sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_temperature"/>
                    String tXlink = xmlReader.attributeValue("xlink:href"); //without quotes
                    if (tXlink != null) {
                        int opPo = uniqueSourceObservedProperties.indexOf(tXlink);   
                        if (opPo >= 0) 
                            tStationHasObsProp[opPo] = true;
                        else if (reallyVerbose) {
                            //String2.log("  Warning: tStationID=" + tStationID + 
                            //    " unexpected <observedProperty>=" + tXlink);
                        }
                    }
                } else if (endOfTag.equals("<" + sosPrefix + "observedProperty><swe:CompositePhenomenon>")) {
                    //handle composite observedProperty
                    //<swe:CompositePhenomenon gml:id="WEATHER_OBSERVABLES">
                    String tid = xmlReader.attributeValue("gml:id"); //without quotes
                    if (tid != null) {
                        int opPo = uniqueSourceObservedProperties.indexOf(tid);   
                        if (opPo >= 0) 
                            tStationHasObsProp[opPo] = true;
                        else if (reallyVerbose) {
                            //String2.log("  Warning: tStationID=" + tStationID + 
                            //" unexpected <observedProperty><swe:CompositePhenomenon>=" + tid);
                        }
                    }
                }

                //handle the error
                //but this isn't used; problems above are logged, but only cause this station not to be used (see 'invalid' below)
                if (fatalError != null)
                    throw new IllegalArgumentException(
                        "Error on xml line #" + xmlReader.lineNumber() + 
                        " stationID=" + tStationID + ": " + fatalError);

            //all data gathered; create the station
            } else if (tags.equals(offeringEndTag)) {

                //look for invalid values
                boolean hasObservedProperties = false;
                for (int i = 0; i < uniqueSourceObservedProperties.size(); i++) {
                    if (tStationHasObsProp[i]) {
                        hasObservedProperties = true; 
                        break;
                    }
                }
                String invalid = null;
                if (Double.isNaN(tLon))                   invalid = "longitude";
                else if (Double.isNaN(tLat))              invalid = "latitude";
                else if (Double.isNaN(tBeginTime))        invalid = "beginTime";
                //endTime may be NaN
                else if (tStationID.length() == 0)        invalid = "station_id";
                else if (tStationProcedure.length() == 0) invalid = "stationProcedure";
                else if (!hasObservedProperties)          invalid = "stationHasRelevantObservedProperties";

                if (tStationID.length() > 0 && 
                    !tStationID.matches(tObservationOfferingIdRegex)) {
                    if (reallyVerbose) String2.log("  station_id=\"" + tStationID + 
                        "\" rejected: didn't match the observationOfferingIdRegex=" + tObservationOfferingIdRegex);

                } else if (invalid != null) {
                    if (reallyVerbose) String2.log("  station_id=\"" + tStationID + 
                        "\" rejected: The " + invalid + " value wasn't set.");

                } else {
                    //add the station;  add a row of data to stationTable
                    stationTable.getColumn(stationLonCol).addDouble(tLon);
                    stationTable.getColumn(stationLatCol).addDouble(tLat);
                    stationTable.getColumn(stationBeginTimeCol).addDouble(tBeginTime); //epochSeconds
                    stationTable.getColumn(stationEndTimeCol).addDouble(tEndTime);     //epochSeconds
                    stationTable.getColumn(stationIDCol).addString(tStationID);
                    stationTable.getColumn(stationProcedureCol).addString(tStationProcedure);
                    tStationHasObsPropAL.add(tStationHasObsProp);
                }

                //reset values for next station
                tLon = Double.NaN;
                tLat = Double.NaN;
                tBeginTime = Double.NaN;
                tEndTime = Double.NaN;
                tIndeterminateEnd = null;
                tStationID = "";                
                tStationProcedure = "";                
                tStationHasObsProp = new boolean[uniqueSourceObservedProperties.size()];
            }           

            //get the next tag
            xmlReader.nextTag();
            tags = xmlReader.allTags();
        } while (!tags.startsWith("</"));

        xmlReader.close();      
        if (stationTable.nRows() == 0)
            throw new RuntimeException(datasetID() + " has no valid stations.");
        stationHasObsProp = new boolean[tStationHasObsPropAL.size()][];
        for (int i = 0; i < tStationHasObsPropAL.size(); i++)
            stationHasObsProp[i] = (boolean[])tStationHasObsPropAL.get(i);
        String2.log("Station Table=\n" + stationTable.saveAsJsonString(stationBeginTimeCol, true));

        //make the fixedVariables
        dataVariables = new EDV[nFixedVariables + tDataVariables.length];

        lonIndex = 0;
        double stats[] = stationTable.getColumn(stationLonCol).calculateStats();
        dataVariables[lonIndex] = new EDVLon(tLonSourceName, null, null,
            "double", stats[PrimitiveArray.STATS_MIN], stats[PrimitiveArray.STATS_MAX]);  

        latIndex = 1;
        stats = stationTable.getColumn(stationLatCol).calculateStats();
        dataVariables[latIndex] = new EDVLat(tLatSourceName, null, null,
            "double", stats[PrimitiveArray.STATS_MIN], stats[PrimitiveArray.STATS_MAX]);  

        idIndex = 2; 
        dataVariables[idIndex] = new EDV("station_id", null, 
            null, (new Attributes()) 
                .add("long_name", "Station ID")
                .add("ioos_category", "Identifier"),
            "String");//the constructor that reads actual_range
        //no need to call setActualRangeFromDestinationMinMax() since they are NaNs

        altIndex = 3;
        dataVariables[altIndex] = new EDVAlt(tAltSourceName, null, null,
            "double", tSourceMinAlt, tSourceMaxAlt, tAltMetersPerSourceUnit);

        timeIndex = 4;  //times in epochSeconds
        stats = stationTable.getColumn(stationBeginTimeCol).calculateStats();         //to get the minimum begin value
        double stats2[] = stationTable.getColumn(stationEndTimeCol).calculateStats(); //to get the maximum end value
        double tTimeMin = stats[PrimitiveArray.STATS_MIN];
        double tTimeMax = stats2[PrimitiveArray.STATS_N] < stationTable.getColumn(stationEndTimeCol).size()? //some indeterminant
            Double.NaN : stats2[PrimitiveArray.STATS_MAX]; 

        EDVTime edvTime = new EDVTime(tTimeSourceName, null, 
            (new Attributes()).add("units", tTimeSourceFormat),
            "String");
        if (!Double.isNaN(tTimeMin)) edvTime.setDestinationMin(tTimeMin);
        if (!Double.isNaN(tTimeMax)) edvTime.setDestinationMax(tTimeMax);
        edvTime.setActualRangeFromDestinationMinMax();
        dataVariables[timeIndex] = edvTime;

        //create non-fixed dataVariables[]
        for (int dv = 0; dv < tDataVariables.length; dv++) {
            String tSourceName = (String)tDataVariables[dv][0];
            String tDestName = (String)tDataVariables[dv][1];
            if (tDestName == null || tDestName.trim().length() == 0)
                tDestName = tSourceName;
            Attributes tSourceAtt = null; //(none available from source)
            Attributes tAddAtt = (Attributes)tDataVariables[dv][2];
            String tSourceType = (String)tDataVariables[dv][3];

            if (EDVTimeStamp.hasTimeUnits(tSourceAtt, tAddAtt)) {
                dataVariables[nFixedVariables + dv] = new EDVTimeStamp(tSourceName, tDestName, 
                    tSourceAtt, tAddAtt, tSourceType); //the constructor that reads actual_range
                dataVariables[nFixedVariables + dv].setActualRangeFromDestinationMinMax();
            } else {
                dataVariables[nFixedVariables + dv] = new EDV(tSourceName, tDestName, 
                    tSourceAtt, tAddAtt, tSourceType); //the constructor that reads actual_range
                dataVariables[nFixedVariables + dv].setActualRangeFromDestinationMinMax();
            }
            Test.ensureNotNothing(
                dataVariables[nFixedVariables + dv].combinedAttributes().getString(EDV.observedProperty),
                "\"" + EDV.observedProperty + "\" attribute not assigned for variable sourceName=" + tSourceName);
        }

        //sos data
        sosOfferingName = stationTable.getColumn(stationIDCol);
        sosMinLongitude = stationTable.getColumn(stationLonCol);
        sosMaxLongitude = stationTable.getColumn(stationLonCol);
        sosMinLatitude  = stationTable.getColumn(stationLatCol);
        sosMaxLatitude  = stationTable.getColumn(stationLatCol);
        sosMinTime      = stationTable.getColumn(stationBeginTimeCol);
        sosMaxTime      = stationTable.getColumn(stationEndTimeCol);
        gatherSosObservedProperties();

        //ensure the setup is valid
        ensureValid();

        //done
        if (verbose) String2.log(
            (reallyVerbose? "\n" + toString() : "") +
            "\n*** EDDTableFromSOS " + datasetID + 
            " constructor finished. TIME=" + 
            (System.currentTimeMillis() - constructionStartMillis) + "\n"); 

    }


    /** 
     * This gets the data (chunk by chunk) from this EDDTable for the 
     * OPeNDAP DAP-style query and writes it to the TableWriter. 
     * See the EDDTable method documentation.
     *
     * @param userDapQuery the part after the '?', still percentEncoded, may be null.
     */
    public void getDataForDapQuery(String requestUrl, String userDapQuery, 
        TableWriter tableWriter) throws Throwable {
        long getTime = System.currentTimeMillis();

        //get the sourceDapQuery (a query that the source can handle)
        StringArray resultsVariables    = new StringArray();
        StringArray constraintVariables = new StringArray();
        StringArray constraintOps       = new StringArray();
        StringArray constraintValues    = new StringArray();
        getSourceQueryFromDapQuery(userDapQuery,
            resultsVariables,
            constraintVariables, constraintOps, constraintValues);
        int nConstraints = constraintVariables.size();

        //further prune constraints 
        //sourceCanConstrainNumericData = CONSTRAIN_PARTIAL; //just lat and lon
        //sourceCanConstrainStringData  = CONSTRAIN_PARTIAL; //time (but not != or regex), station_id (even REGEX_OP), but nothing else
        //sourceCanConstrainStringRegex = REGEX_OP; //just for station_id
        //work backwards since deleting some
        int conDVI[] = new int[nConstraints];
        boolean justStationTableInfo = true;
        for (int c = constraintVariables.size() - 1; c >= 0; c--) { 
            String constraintVariable = constraintVariables.get(c);
            String constraintOp       = constraintOps.get(c);
            int dv = String2.indexOf(dataVariableSourceNames(), constraintVariable);
            conDVI[c] = dv;
            if (dv != lonIndex && dv != latIndex && dv != idIndex)
                justStationTableInfo = false;

            if (dv == latIndex || dv == lonIndex || dv == idIndex) {
                //all ok

            } else if (dv == timeIndex) {
                //remove if != or regex
                if (constraintOp.equals("!=") || constraintOp.equals(REGEX_OP)) {
                    //remove constraint
                    constraintVariables.remove(c);
                    constraintOps.remove(c);
                    constraintValues.remove(c);
                }

            } else {
                //remove all other constraints
                constraintVariables.remove(c);
                constraintOps.remove(c);
                constraintValues.remove(c);
            }
        }

        //make tableDVI -- the dataVariable indices corresponding to the columns needed for the results table.
        //Always include lon,lat,alt,time,id to facilitate merging observedProperties correctly.
        IntArray tableDVI = new IntArray(); //.get(tableColumnIndex) -> dataVariableIndex
        //add the nFixedVariables
        int tableLonCol = 0, tableLatCol = 1, tableAltCol = 2, 
            tableTimeCol = 3, 
            tableStationIdCol = 4;
        tableDVI.add(lonIndex);
        tableDVI.add(latIndex);
        tableDVI.add(altIndex);
        tableDVI.add(timeIndex);
        tableDVI.add(idIndex);
        StringArray requestObservedProperties = new StringArray();
        for (int rv = 0; rv < resultsVariables.size(); rv++) {
            int dvi = String2.indexOf(dataVariableSourceNames(), resultsVariables.get(rv));
            //only add if not already included
            if (tableDVI.indexOf(dvi, 0) < 0) 
                tableDVI.add(dvi);
            if (dvi != lonIndex && dvi != latIndex && dvi != idIndex)
                justStationTableInfo = false;
            //make list of desired observedProperties
            //remember that >1 var may refer to same obsProp (eg, insideTemp and outsideTemp refer to ...Temperature)
            if (dvi >= nFixedVariables) {
                String top = dataVariables[dvi].combinedAttributes().getString("sourceObservedProperty"); //preferred
                if (top == null)
                    top = dataVariables[dvi].combinedAttributes().getString(EDV.observedProperty);
                if (requestObservedProperties.indexOf(top) < 0)
                    requestObservedProperties.add(top);
            }
        }

        //always at least one observedProperty
        //If request is for lon,lat,id with time constraints
        //  you get no data if you don't ask for a property.
        //So add the first var after nFixedVariables
        //This is not a perfect solution, because results might be different if you chose a different property.
        if (!justStationTableInfo && requestObservedProperties.size() == 0) {
            int dvi = nFixedVariables;
            tableDVI.add(dvi);
            String top = dataVariables[dvi].combinedAttributes().getString("sourceObservedProperty"); //preferred
            if (top == null)
                top = dataVariables[dvi].combinedAttributes().getString(EDV.observedProperty);
            if (requestObservedProperties.indexOf(top) < 0)
                requestObservedProperties.add(top);
        }

        //get requestedMin,Max  
        double requestedDestinationMin[] = new double[4]; //LLAT   time in userDapQuery is epochSeconds
        double requestedDestinationMax[] = new double[4];
        getRequestedDestinationMinMax(userDapQuery, true,
            requestedDestinationMin, requestedDestinationMax);        

        //makeTable and find fixedVariables in table
        if (reallyVerbose) String2.log("  justStationTableInfo=" + justStationTableInfo);
        if (justStationTableInfo) {
            //make a table of just lon,lat,id
            tableDVI = new IntArray(); //.get(tableColumnIndex) -> dataVariableIndex
            tableLonCol = 0; tableLatCol = 1; tableAltCol = -1; tableTimeCol = -1; tableStationIdCol = 2;
            tableDVI.add(lonIndex);
            tableDVI.add(latIndex);
            tableDVI.add(idIndex);
        }
        Table table = makeTable(tableDVI);
        //IntArray fixedColumnsInTable = new IntArray();
        //for (int col = 0; col < tableDVI.size(); col++)
        //    if (tableDVI.get(col) < nFixedVariables)
        //        fixedColumnsInTable.add(col);

        //Go through the stations, request data if station is relevant.
        //SOS queries may be limited to 1 station at a time (based on 'procedure'?)
        //  But that's a good thing because I want to limit response chunk size.
        //  ???Extreme cases: may need to further limit to chunks of time.
        int nStations = stationTable.nRows();
        boolean aConstraintShown = false;
        for (int station = 0; station < nStations; station++) {

            //make sure this station is relevant
            //test lon,lat,time   (station alt not available)
            boolean stationOK = true;
            for (int i = 0; i < 4; i++) {
                String tName = "???";
                int stationMinCol = -1, stationMaxCol = -1, precision = 4;
                if      (i == 0) {tName = "longitude";  stationMinCol = stationLonCol;       stationMaxCol = stationLonCol;}
                else if (i == 1) {tName = "latitude";   stationMinCol = stationLatCol;       stationMaxCol = stationLatCol;}
                else if (i == 3) {tName = "time";       stationMinCol = stationBeginTimeCol; stationMaxCol = stationEndTimeCol; 
                                  precision = 9;}
                if (stationMinCol >= 0) {
                    double rMin = requestedDestinationMin[i];
                    double rMax = requestedDestinationMax[i];
                    double sMin = stationTable.getDoubleData(stationMinCol, station);
                    double sMax = stationTable.getDoubleData(stationMaxCol, station);
                    if ((Math2.isFinite(rMax) && Math2.isFinite(sMin) && !Math2.greaterThanAE(precision, rMax, sMin)) || 
                        (Math2.isFinite(rMin) && Math2.isFinite(sMax) && !Math2.lessThanAE(   precision, rMin, sMax))) {
                        if (reallyVerbose) String2.log("  reject station=" + station + 
                            ": no overlap between station " + tName + " (min=" + sMin + ", max=" + sMax +
                            ") and requested range (min=" + rMin + ", max=" + rMax + ").");
                        stationOK = false; 
                        break;
                    }
                }
            }
            if (!stationOK)
                continue;

            //test stationID constraint
            String tStationID  = stationTable.getStringData(stationIDCol, station);
            for (int con = 0; con < nConstraints; con++) {
                if (conDVI[con] == idIndex) {
                    String op = constraintOps.get(con);
                    boolean pass = testValueOpValue(tStationID, op, constraintValues.get(con));
                    if (!pass) {
                        if (reallyVerbose) String2.log("  rejecting station=" + station + 
                            " because stationID=\"" + tStationID + "\" isn't " + op + 
                            " \"" + constraintValues.get(con) + "\".");
                        stationOK = false;
                        break;  //constraint loop
                    }
                }
            }
            if (!stationOK)
                continue;
            if (reallyVerbose) String2.log("\nquerying stationID=" + tStationID);

            //The station is in the bounding box!
            double tStationLon = stationTable.getDoubleData(stationLonCol, station); 
            double tStationLat = stationTable.getDoubleData(stationLatCol, station); 
            double tStationAlt = Double.NaN;
            String tStationLonString = "" + tStationLon;
            String tStationLatString = "" + tStationLat;
            String tStationAltString = "";

            //just stationTable info?
            if (justStationTableInfo) {
                //there is a column in table for each tableDVI
                for (int col = 0; col < tableDVI.size(); col++) {
                    int dvi = tableDVI.get(col);
                    if      (dvi == lonIndex) table.getColumn(col).addDouble(tStationLon); 
                    else if (dvi == latIndex) table.getColumn(col).addDouble(tStationLat); 
                    else if (dvi == idIndex)  table.getColumn(col).addString(tStationID); 
                    else throw new Exception("JustStationTable: Unexpected dvi=" + 
                        dvi + " = " + dataVariableSourceNames()[dvi]);
                }
                continue;
            }

            //request the observedProperties separately
            //see class javadocs above explaining why this is needed
            HashMap llatHash = new HashMap(); //llat info -> table row number (as a String)
            for (int obsProp = 0; obsProp < requestObservedProperties.size(); obsProp++) {
                
                StringArray tRequestObservedProperties = new StringArray();
                if (requestObservedPropertiesSeparately) {  
                    tRequestObservedProperties.add(requestObservedProperties.get(obsProp));
                } else {
                    //get all at once
                    tRequestObservedProperties = (StringArray)requestObservedProperties.clone();

                    //or we're already done
                    if (obsProp == 1) 
                        break;
                }
                if (reallyVerbose) String2.log("\nobsProp=" + obsProp + " ttObsProp=" + tRequestObservedProperties);

                //remove obsProp from tRequestObservedProperties if station doesn't support it 
                for (int i = tRequestObservedProperties.size() - 1; i >= 0; i--) {
                    int po = uniqueSourceObservedProperties.indexOf(tRequestObservedProperties.get(i));
                    if (!stationHasObsProp[station][po])
                        tRequestObservedProperties.remove(i);
                }
                if (tRequestObservedProperties.size() == 0) {
                    if (reallyVerbose) String2.log("  reject station=" + station + 
                        ": It has none of the requested observedProperties.");
                    continue;
                }

                if (ioosServer) {
                 
                    //make kvp   -- no examples or info for XML request
                    //example from http://sdf.ndbc.noaa.gov/sos/
                    //if (reallyVerbose && obsProp == 0) String2.log("\n  sample=" +
                    //  "http://sdf.ndbc.noaa.gov/sos/server.php?request=GetObservation" +
                    //  "&service=SOS&offering=NDBC:46088&observedproperty=currents" +
                    //  "&responseformat=text/xml;schema=%22ioos/0.6.1%22&eventtime=2008-06-01T00:00Z/2008-06-02T00:00Z");
                    //???How should a client know how to convert GetCapabilities observedProperty (long)
                    //  into GetObservation observedProperty (short)?
                    int hashPo = tRequestObservedProperties.get(obsProp).lastIndexOf('#'); //for NDBC SOS
                    if (hashPo < 0)
                        hashPo = tRequestObservedProperties.get(obsProp).lastIndexOf(':'); //for NOS SOS
                    String kvp = 
                        "?service=SOS" +
                        "&version=" + sosVersion +
                        "&request=GetObservation" + 
                        "&offering=" + String2.replaceAll(tStationID, ' ', '+') + 
                        "&observedProperty=" + tRequestObservedProperties.get(obsProp).substring(hashPo + 1) + 
                        "&responseFormat=text/xml;schema=%22ioos/0.6.1%22" +  
                        "&eventTime=" +   //requestedDestination times are epochSeconds
                            Calendar2.epochSecondsToIsoStringT(requestedDestinationMin[3]) + "Z" + 
                        (requestedDestinationMin[3] == requestedDestinationMax[3]? "" : 
                            "/" + Calendar2.epochSecondsToIsoStringT(requestedDestinationMax[3]) + "Z");
                    if (reallyVerbose) {
                        String2.log("\n  url+kvp=\n" + sourceUrl + kvp);
                        aConstraintShown = true;
                    }
                    //SSR.copy(SSR.getUrlInputStream(sourceUrl + kvp), System.out);
                    InputStream in = SSR.getUrlInputStream(sourceUrl + kvp);
        
                    //values that need to be parsed from the xml and held
                    IntArray fieldToCol = new IntArray(); //converts results field# to table col#
                    String tokenSeparator = null, blockSeparator = null, decimalSeparator = null;

                    //request the data
                    //???Future: need to break the request up into smaller time chunks???
                    SimpleXMLReader xmlReader = new SimpleXMLReader(in);
                    xmlReader.nextTag();
                    String tags = xmlReader.allTags();
                    String ofInterest1 = null;
                    String ofInterest2 = null;
                    String ofInterest3 = null;

                    //response is error message
                    if (tags.equals("<ServiceExceptionReport>") ||
                        tags.equals("<ExceptionReport>")) { //ioosService
                        //<?xml version="1.0"?>
                        //<ExceptionReport xmlns="http://www.opengis.net/ows" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.opengis.net/ows owsExceptionReport.xsd" version="1.0.0" language="en">
                        //<Exception locator="service" exceptionCode="InvalidParamterValue">
                        //<ExceptionText>Unknown observedProperty parameter: sea_water_temperature,sea_water_salinity</ExceptionText></Exception>
                        //</ExceptionReport>
                        String errorText = "";
                        String2.log("\n  Error from " + sourceUrl + kvp);
                        do {
                            //log full exception report
                            String content = xmlReader.content();
                            String2.log(tags);
                            if (content.length() > 0) String2.log("  content=" + content);
                            String atts = xmlReader.attributesCSV();
                            if (atts.length() > 0) String2.log("  atts=" + atts);

                            //process the tags
                            //String2.log("tags=" + tags + xmlReader.content());
                            if (tags.equals("<ExceptionReport><Exception>")) {
                                if (xmlReader.attributeValue("exceptionCode") != null)
                                    errorText += xmlReader.attributeValue("exceptionCode") + ": ";
                                if (xmlReader.attributeValue("locator") != null)
                                    errorText += xmlReader.attributeValue("locator") + ": ";
                            }

                            if (tags.equals("<ServiceExceptionReport></ServiceException>") ||
                                tags.equals("<ExceptionReport><Exception></ExceptionText>")) {
                                errorText = "Source Exception=\"" + errorText + content + "\".";
                                if (content.indexOf("No ") == 0 &&
                                    content.indexOf(" data found for this station") > 0) {
                                    //content is now something like
                                    //No currents data found for this station and date/time".
                                    //but "currents" will change
                                } else if (errorText.indexOf("NoApplicableCode:") >= 0) {
                                    //occurs if station advertises data, but GetObservation returns error
                                    //java.lang.Exception: Source Exception="NoApplicableCode: 1619910: 
                                    //There is no Air Temperature sensor installed at station 
                                    //urn:x-noaa:def:station:NOAA.NOS.CO-OPS:1619910 or dissemination has been stopped by CORMS.".
                                } else {
                                    throw new RuntimeException(errorText);
                                }
                            }

                            //get the next tag
                            xmlReader.nextTag();
                            tags = xmlReader.allTags();
                        } while (!tags.startsWith("</"));
                        if (errorText == null)
                            throw new RuntimeException("Source sent an ExceptionReport (no text).");

                    //response is normal
                    } else if (tags.equals("<om:CompositeObservation>")) {
                        ofInterest1 = "<om:CompositeObservation><om:result>";                        
                        //ignore optional intervening tags 
                        //  "[<ioos:Composite><gml:valueComponents>[<ioos:Array><gml:valueComponents>]]"
                        ofInterest2 = "<ioos:ValueArray><gml:valueComponents><ioos:CompositeValue><gml:valueComponents>";
                        ofInterest3 = "<ioos:Composite><gml:valueComponents><ioos:CompositeValue><gml:valueComponents>"; 

                    //response is unexpected
                    } else {
                        throw new RuntimeException("Error when reading source xml: First tag=" + tags + 
                            " should have been <om:CompositeObservation>.");
                    }

                    int nCols = table.nColumns();
                    String rowValues[] = new String[nCols];
                    String altitudeSourceName = altIndex >= 0? dataVariables[altIndex].sourceName() : null;
                    String fieldName = null;
                    String uom = null;
                    String ofInterestAltitude = 
                        "<om:CompositeObservation><om:procedure><om:Process><ioos:CompositeContext><gml:valueComponents>" +
                        "<ioos:ContextArray><gml:valueComponents><ioos:CompositeContext><gml:valueComponents>" +
                        "</" + altitudeSourceName + ">";

                    while (!tags.startsWith("</")) { //note that "No data found" error causes this block to be skipped
                        /* process the tags
                        //String2.log("tags=" + tags + xmlReader.content());
  <om:observedProperty xlink:href="http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Currents"/>
  <om:featureOfInterest xlink:href="urn:x-noaa:def:station:ndbc::42376"/>
  <om:result>
    <ioos:Composite gml:id="CurrentsVerticalProfileCollectionTimeSeriesDataObservation" recDef="http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/recordDefinitions/CurrentsProfileDataRecordDefinition.xml">
      <gml:valueComponents>
        <ioos:Count name="NumberOfObservationsPoints">3</ioos:Count>
        <ioos:Array gml:id="CurrentsVerticalProfileCollectionTimeSeries">
          <gml:valueComponents>
            <ioos:Composite gml:id="Station1TimeSeriesRecord">
              <gml:valueComponents>
                <ioos:Count name="Station1NumberOfObservationTimes">3</ioos:Count>
                <ioos:Array gml:id="Station1CurrentsProfileTimeSeries">
                  <gml:valueComponents>
                    <ioos:Composite gml:id="Station1T1Profile">
                      <gml:valueComponents>
                        <ioos:CompositeContext gml:id="Station1T1ObservationConditions" processDef="#Station1Info">
                          <gml:valueComponents>
                            <gml:TimeInstant gml:id="Station1T1Time">
                              <gml:timePosition>2008-06-01T14:03Z</gml:timePosition>
                            </gml:TimeInstant>
                            <ioos:Context name="Heading" uom="deg" xsi:nil="true" xsi:nilReason="missing"/>
                            <ioos:Context name="Pitch" uom="deg" xsi:nil="true" xsi:nilReason="missing"/>
                            <ioos:Context name="Roll" uom="deg" xsi:nil="true" xsi:nilReason="missing"/>
                            <ioos:Context name="WaterTemperature" uom="C" xsi:nil="true" xsi:nilReason="missing"/>
                          </gml:valueComponents>
                        </ioos:CompositeContext>
                        <ioos:Count name="Station1T1NumberOfBinObservations">32</ioos:Count>
                        <ioos:ValueArray gml:id="Station1T1ProfileObservations">
                          <gml:valueComponents>
                            <ioos:CompositeValue gml:id="Station1T1Bin1Obs" processDef="#Station1Sensor1Info">
                              <gml:valueComponents>
                                <ioos:Context name="Depth" uom="m">56.80</ioos:Context>
                                <ioos:Quantity name="CurrentDirection" uom="deg">83</ioos:Quantity>
                                <ioos:Quantity name="CurrentSpeed" uom="cm/s">30.2</ioos:Quantity>
                                <ioos:Context name="VerticalVelocity" uom="cm/s">-2.6</ioos:Context>
                                <ioos:Context name="QualityFlags">393333330</ioos:Context>
                                ...
                              </gml:valueComponents>
                            </ioos:CompositeValue>
                            ...
                          </gml:valueComponents>
                        </ioos:ValueArray>
                      </gml:valueComponents>
                    </ioos:Composite>
                        */

                        //if (reallyVerbose) String2.log(tags + xmlReader.content());

                        //get depth 
                        if (tags.equals(ofInterestAltitude)) {
                            tStationAltString = xmlReader.content();
                            if (reallyVerbose) {String2.log("  found altitude=" + tStationAltString); //Math2.sleep(3000);
                            }

                        } else if (tags.startsWith(ofInterest1)) { 
                            //remember there are optional intervening tags: 
                            //  "[<ioos:Composite><gml:valueComponents>[<ioos:Array><gml:valueComponents>]]"
                            //if (reallyVerbose) String2.log(tags + xmlReader.content());
                            //quick tests via topTag 
                            String topTag = xmlReader.topTag();

                            //gather a row of data
                            if (topTag.equals("/gml:timePosition") &&
                                tags.endsWith("<ioos:Composite><gml:valueComponents><ioos:CompositeContext>" +
                                "<gml:valueComponents><gml:TimeInstant></gml:timePosition>")) {
                                rowValues[tableTimeCol] = xmlReader.content(); 
                                if (reallyVerbose) String2.log("  found time=" + rowValues[tableTimeCol]);

                            //for list of options, see 
                            //www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/schemas/ioosTypes.xsd
                            } else if (tags.endsWith(ofInterest2 + "<ioos:Context>") ||
                                       tags.endsWith(ofInterest3 + "<ioos:Context>") ||
                                       tags.endsWith(ofInterest2 + "<ioos:Count>") || 
                                       tags.endsWith(ofInterest3 + "<ioos:Count>") ||
                                       tags.endsWith(ofInterest2 + "<ioos:Integer>") || 
                                       tags.endsWith(ofInterest3 + "<ioos:Integer>") ||
                                       tags.endsWith(ofInterest2 + "<ioos:NamedQuantityList>") || 
                                       tags.endsWith(ofInterest3 + "<ioos:NamedQuantityList>") ||
                                       tags.endsWith(ofInterest2 + "<ioos:Quantity>") || 
                                       tags.endsWith(ofInterest3 + "<ioos:Quantity>") ||
                                       tags.endsWith(ofInterest2 + "<ioos:Text>") || 
                                       tags.endsWith(ofInterest3 + "<ioos:Text>")) {
                                    //get the fieldName
                                    fieldName = xmlReader.attributeValue("name");
                                    uom = xmlReader.attributeValue("uom");

                            } else if (tags.endsWith(ofInterest2 + "</ioos:Context>") ||
                                       tags.endsWith(ofInterest3 + "</ioos:Context>") ||
                                       tags.endsWith(ofInterest2 + "</ioos:Count>") ||
                                       tags.endsWith(ofInterest3 + "</ioos:Count>") ||
                                       tags.endsWith(ofInterest2 + "</ioos:Integer>") ||
                                       tags.endsWith(ofInterest3 + "</ioos:Integer>") ||
                                       tags.endsWith(ofInterest2 + "</ioos:NamedQuantityList>") ||
                                       tags.endsWith(ofInterest3 + "</ioos:NamedQuantityList>") ||
                                       tags.endsWith(ofInterest2 + "</ioos:Quantity>") ||
                                       tags.endsWith(ofInterest3 + "</ioos:Quantity>") ||
                                       tags.endsWith(ofInterest2 + "</ioos:Text>") ||
                                       tags.endsWith(ofInterest3 + "</ioos:Text>")) {
                                    //get the value associated with the fieldName
                                    int col = table.findColumnNumber(fieldName);
                                    if (col >= 0) {
                                        String content = xmlReader.content();
                                        rowValues[col] = content;
                                        if (reallyVerbose) String2.log("  found value name=" + fieldName + 
                                            " col=" + col + " content=" + content + " uom=" + uom);
                                    }
                                    fieldName = null;
                                    uom = null;

                            //process a row of data
                            } else if (tags.endsWith("<ioos:CompositeValue></gml:valueComponents>")) {
                                //if (reallyVerbose) String2.log("  pre  rowValues=" + String2.toCSVString(rowValues));

                                //add lat lon alt data
                                //it's usually in the result fields, but not always
                                if (rowValues[tableLonCol] == null)
                                    rowValues[tableLonCol] = tStationLonString;
                                if (rowValues[tableLatCol] == null)
                                    rowValues[tableLatCol] = tStationLatString;
                                if (rowValues[tableAltCol] == null)
                                    rowValues[tableAltCol] = tStationAltString;

                                //add tStationID
                                rowValues[tableStationIdCol] = tStationID;
                                if (reallyVerbose) String2.log("  post rowValues=" + String2.toCSVString(rowValues));

                                //does a row with identical LonLatAltTimeID exist in table?
                                String tHash = rowValues[tableLonCol] + "," + rowValues[tableLatCol] + 
                                         "," + rowValues[tableAltCol] + "," + rowValues[tableTimeCol] +
                                         "," + rowValues[tableStationIdCol];
                                int tRow = String2.parseInt((String)llatHash.get(tHash));
                                if (tRow < Integer.MAX_VALUE) {
                                    //merge this data into that row
                                    for (int col = 0; col < nCols; col++) {
                                        String ts = rowValues[col];
                                        if (ts != null) {
                                            PrimitiveArray pa = table.getColumn(col);
                                            if (true || verbose) { 
                                                //if there was an old value, ensure that old value = new value
                                                //???leave this test in as insurance?
                                                String tso = pa.getString(tRow);
                                                pa.setString(tRow, ts);
                                                ts = pa.getString(tRow); //setting a number changes it, e.g., 1 -> 1.0
                                                if (tso.length() > 0 && !tso.equals(ts))
                                                    throw new RuntimeException("Error while merging row=" + tRow +
                                                        ": col=" + col + ", old=" + tso + " != new=" + ts);
                                            } else {
                                                pa.setString(tRow, ts);
                                            }
                                        }
                                    }
                                } else {
                                    //add this row
                                    for (int col = 0; col < nCols; col++) {
                                        String ts = rowValues[col];
                                        table.getColumn(col).addString(ts == null? "" : ts);
                                    }

                                    llatHash.put(tHash, "" + (table.nRows() - 1));
                                }

                                //reset rowValues   (but keep shared time)
                                String tTime = rowValues[tableTimeCol]; 
                                Arrays.fill(rowValues, null);
                                rowValues[tableTimeCol] = tTime; 

                            } else if (topTag.equals("/ioos:ValueArray")) {
                                //time is no longer valid
                                Arrays.fill(rowValues, null);
                                if (reallyVerbose) String2.log("  time no longer valid\n");
                            }

                        }

                        //get the next tag
                        xmlReader.nextTag();
                        tags = xmlReader.allTags();
                    } //end of tag reading loop

                    xmlReader.close();  
                
                } else { //non-ioosServer
                    //make the xml constraint
                    //important help getting this to work from c:/programs/sos/oie_sos_time_range_obs.cgi
                    //which is from http://www.oostethys.org/ogc-oceans-interoperability-experiment/experiment-1/sos-client-testing-source-code
                    StringBuffer constraintSB = new StringBuffer();
                    constraintSB.append(
                        "<?xml version=\"1.0\" encoding=\"UTF-8\"?> \n" +
                        "<GetObservation\n" +
                        "  xmlns=\"http://www.opengis.net/sos\"\n" +  //from oostethys;  sos doc has http://www.opengeospatial.net/sos
                        "  xmlns:gml=\"http://www.opengis.net/gml\"\n" +
                        "  xmlns:ogc=\"http://www.opengis.net/ogc\"\n" +
                        "  xmlns:ows=\"http://www.opengeospatial.net/ows\"\n" +
                        "  xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n" +
                        "  xmlns:swe=\"http://www.opengis.net/swe\"\n" +
                        "  service=\"SOS\" version=\"" + sosVersion + "\" >\n" +
                        "  <offering>" + tStationID + "</offering>\n");
                    for (int top = 0; top < tRequestObservedProperties.size(); top++) 
                        constraintSB.append(
                          "  <observedProperty>" + tRequestObservedProperties.get(top) + "</observedProperty>\n");
                    //older SOS spec has <ogc:During>; v1.0 draft has <ogc:T_During>
                    //new oostethys server ignores <ogc:During>, wants T_During 
                    //see http://www.oostethys.org/Members/tcook/sos-xml-time-encodings/
                    if (requestedDestinationMin[3] == requestedDestinationMax[3]) 
                        constraintSB.append(
                            "  <eventTime>\n" +  
                            "    <ogc:T_During>\n" + 
                            "      <gml:TimeInstant>\n" +
                            "        <gml:timePosition>" + Calendar2.epochSecondsToIsoStringT(requestedDestinationMin[3]) + 
                                   "</gml:timePosition>\n" + //their example has no Z
                            "      </gml:TimeInstant>\n" +
                            "    </ogc:T_During>\n" +
                            "  </eventTime>\n");
                    else
                        constraintSB.append(
                            "  <eventTime>\n" +  
                            "    <ogc:T_During>\n" + 
                            "      <gml:TimePeriod>\n" +
                            "        <gml:beginPosition>" + Calendar2.epochSecondsToIsoStringT(requestedDestinationMin[3]) + 
                                   "</gml:beginPosition>\n" + //their example has no Z
                            "        <gml:endPosition>"   + Calendar2.epochSecondsToIsoStringT(requestedDestinationMax[3]) + 
                                   "</gml:endPosition>\n" +
                            "      </gml:TimePeriod>\n" +
                            "    </ogc:T_During>\n" +
                            "  </eventTime>\n");
                    //constraintSB.append( //not needed
                    //    "  <procedure>" + stationTable.getStringData(stationProcedureCol, station) + "</procedure>\n"); 
                    constraintSB.append(
                        //"  <featureOfInterest>\n" + //SOS doc has this; oostethys doesn't (and not needed since station already specified)
                        //"    <ogc:BBOX>\n" +
                        //"      <gml:Envelope srsName=\"EPSG:4326\">\n" + //standard geographic projection (from their example)
                        //         //e.g., 38.48 -70.43    order is lat lon
                        //"        <gml:lowerCorner srsName=\"urn:ogc:def:crs:EPSG:6.5:4329\">" + 
                        //    String2.genEFormat6(requestedDestinationMin[1]) + " " +
                        //    String2.genEFormat6(requestedDestinationMin[0]) + "</gml:lowerCorner>\n" +
                        //"        <gml:upperCorner srsName=\"urn:ogc:def:crs:EPSG:6.5:4329\">" + 
                        //    String2.genEFormat6(requestedDestinationMax[1]) + " " +
                        //    String2.genEFormat6(requestedDestinationMax[0]) + "</gml:upperCorner>\n" +
                        //"      </gml:Envelope>\n" +
                        //"    </ogc:BBOX>\n" +
                        //"  </featureOfInterest>\n" +
                        //"  <resultFormat>text/xml;subtype=\"OM/1.0.30\"</resultFormat>\n" + //?
                        "  <resultFormat>application/com-xml</resultFormat>\n" + //?
                        "</GetObservation>\n");
                    if (reallyVerbose && !aConstraintShown) {
                        String2.log("\n  constraint=" + constraintSB.toString());
                        aConstraintShown = true;
                    }
                    if (reallyReallyVerbose) { 
                        String2.log("*** Begin response");
                        SSR.copy(SSR.getPostInputStream(sourceUrl, "text/xml", "UTF-8", constraintSB.toString()),
                            System.out);
                        String2.log("*** End response");
                    }
                    InputStream in = SSR.getPostInputStream(sourceUrl, "text/xml", "UTF-8", constraintSB.toString());

                    //values that need to be parsed from the xml and held
                    IntArray fieldToCol = new IntArray(); //converts results field# to table col#
                    String tokenSeparator = null, blockSeparator = null, decimalSeparator = null;

                    //request the data
                    //???Future: need to break the request up into smaller time chunks???
                    SimpleXMLReader xmlReader = new SimpleXMLReader(in);
                    xmlReader.nextTag();
                    String tags = xmlReader.allTags();
                    if (tags.equals("<ServiceExceptionReport>") ||
                        tags.equals("<ExceptionReport>")) {
                        //<?xml version="1.0"?>
                        //<ExceptionReport xmlns="http://www.opengis.net/ows" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:schemaLocation="http://www.opengis.net/ows owsExceptionReport.xsd" version="1.0.0" language="en">
                        //<Exception locator="service" exceptionCode="InvalidParamterValue">
                        //<ExceptionText>Unknown observedProperty parameter: sea_water_temperature,sea_water_salinity</ExceptionText></Exception>
                        //</ExceptionReport>
                        do {
                            //process the tags
                            //String2.log("tags=" + tags + xmlReader.content());
                            if (tags.equals("<ServiceExceptionReport></ServiceException>")) 
                                throw new RuntimeException("Source Exception=\"" + xmlReader.content() + "\".");
                            if (tags.equals("<ExceptionReport><Exception></ExceptionText>")) 
                                throw new RuntimeException("Source Exception=\"" + xmlReader.content() + "\".");

                            //get the next tag
                            xmlReader.nextTag();
                            tags = xmlReader.allTags();
                        } while (!tags.startsWith("</"));
            
                        xmlReader.close();  
                        throw new RuntimeException("Source sent an ExceptionReport (no text).");
                    }

                    String ofInterest = null;
                    if (tags.equals("<om:Observation>")) 
                        ofInterest = tags;
                    else if (tags.equals("<om:ObservationCollection>")) 
                        ofInterest = "<om:ObservationCollection><om:member><om:Observation>";
                    else throw new RuntimeException("Data source error when reading source xml: First tag=" + tags + 
                        " should have been <om:Observation> or <om:ObservationCollection>.");

                    do {
                        //process the tags
                        //String2.log("tags=" + tags + xmlReader.content());
                        /* e.g., from oostethys
                        <?xml version="1.0" encoding="UTF-8"?>
                        <om:Observation xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:swe="http://www.opengis.net/swe/0" xmlns:gml="http://www.opengis.net/gml" xmlns:om="http://www.opengis.net/om" xmlns:xlink="http://www.w3.org/1999/xlink" xsi:schemaLocation="http://www.opengis.net/om ../../../../../../ows4/schema0/swe/branches/swe-ows4-demo/om/current/commonObservation.xsd" gml:id="SEA_WATER_TEMPERATURE">
                            <gml:description>latest seaTemperature measurement from NDBC buoy 41001</gml:description>
                            <gml:name>NDBC Buoy 41001 seaTemperature Data</gml:name>
                            <gml:location>
                                <gml:Point gml:id="OBSERVATION_LOCATION" srsName="urn:ogc:def:crs:EPSG:6.5:4329">
                                    <!-- use lat lon depth in deg deg m -->
                                    <gml:coordinates>34.68 -72.66 0</gml:coordinates>
                                </gml:Point>
                            </gml:location>
                            <om:time>
                                <gml:TimePeriod gml:id="DATA_TIME">
                                    <gml:beginPosition>2007-06-18T00:50:00Z</gml:beginPosition>
                                    <gml:endPosition>2007-06-18T23:50:00Z</gml:endPosition>
                                </gml:TimePeriod>
                            </om:time>
                            <om:procedure xlink:href="urn:ndbc.noaa.org:source.mooring#41001"/>
                            <om:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_temperature"/>
                            <om:featureOfInterest xlink:href="urn:something:bodyOfWater"/>
                            <om:resultDefinition>
                                <swe:DataBlockDefinition>
                                    <swe:components name="SeaTemperatureDataOf41001">
                                        <swe:DataRecord>
                                            <swe:field name="time">
                                                <swe:Time definition="urn:ogc:phenomenon:time:iso8601"/>
                                            </swe:field>
                                            <swe:field name="latitude">
                                                <swe:Quantity definition="urn:ogc:phenomenon:latitude:wgs84">
                                                    <swe:uom code="deg"/>
                                                </swe:Quantity>
                                            </swe:field>
                                            <swe:field name="longitude">
                                                <swe:Quantity definition="urn:ogc:phenomenon:longitude:wgs84">
                                                    <swe:uom code="deg"/>
                                                </swe:Quantity>
                                            </swe:field>
                                            <swe:field name="depth">
                                                <swe:Quantity definition="urn:ogc:phenomenon:depth">
                                                    <swe:uom code="m"/>
                                                </swe:Quantity>
                                            </swe:field>
                                            <swe:field name="seaTemperature">
                                                <swe:Quantity definition="http://marinemetadata.org/cf#sea_water_temperature">
                                                    <swe:uom xlink:href="urn:mm.def:units#celsius"/>
                                                </swe:Quantity>
                                            </swe:field>
                                        </swe:DataRecord>
                                    </swe:components>
                                    <swe:encoding>
                                        <swe:AsciiBlock tokenSeparator="," blockSeparator=" " decimalSeparator="."/>
                                    </swe:encoding>
                                </swe:DataBlockDefinition>
                            </om:resultDefinition>
                            <om:result>2007-06-18T00:50:00Z,34.68,-72.66,0,24.3 2007-06-18T01:50:00Z,34.68,-72.66,0,24.3 2007-06-18T02:50:00Z,34.68,-72.66,0,24.3 2007-06-18T03:50:00Z,34.68,-72.66,0,24.2 2007-06-18T04:50:00Z,34.68,-72.66,0,23.8 2007-06-18T05:50:00Z,34.68,-72.66,0,24.2 2007-06-18T06:50:00Z,34.68,-72.66,0,23.8 2007-06-18T07:50:00Z,34.68,-72.66,0,23.8 2007-06-18T08:50:00Z,34.68,-72.66,0,23.7 2007-06-18T09:50:00Z,34.68,-72.66,0,23.8 2007-06-18T10:50:00Z,34.68,-72.66,0,23.8 2007-06-18T11:50:00Z,34.68,-72.66,0,23.8 2007-06-18T12:50:00Z,34.68,-72.66,0,24.2 2007-06-18T13:50:00Z,34.68,-72.66,0,24.3 2007-06-18T14:50:00Z,34.68,-72.66,0,24.6 2007-06-18T15:50:00Z,34.68,-72.66,0,24.8 2007-06-18T16:50:00Z,34.68,-72.66,0,24.6 2007-06-18T17:50:00Z,34.68,-72.66,0,25 2007-06-18T18:50:00Z,34.68,-72.66,0,25.1 2007-06-18T19:50:00Z,34.68,-72.66,0,25 2007-06-18T20:50:00Z,34.68,-72.66,0,25 2007-06-18T21:50:00Z,34.68,-72.66,0,25 2007-06-18T22:50:00Z,34.68,-72.66,0,25 2007-06-18T23:50:00Z,34.68,-72.66,0,25</om:result>
                        </om:Observation>

                        VAST has om:Observation wrapped in om:ObservationCollection:
                        <om:ObservationCollection gml:id="WEAttp://www.opengis.net/swe/1.0" xmlns: ...
                          <gml:name>Weather Data</gml:name>
                          <om:member>
                            <om:Observation ... similar to above

                        VAST has lat lon alt info right before resultDefinition:
                        <om:featureOfInterest>
                            <swe:GeoReferenceableFeature>
                                <gml:name>NSSTC Weather Station</gml:name>
                                <gml:location>
                                    <gml:Point srsName="urn:ogc:def:crs:EPSG:6.1:4329">
                                        <gml:coordinates>34.7 -86.6 15.0</gml:coordinates>
                                    </gml:Point>
                                </gml:location>
                            </swe:GeoReferenceableFeature>
                        </om:featureOfInterest>
                        */

                        if (reallyReallyVerbose) String2.log(tags + xmlReader.content());

                        if (tags.startsWith(ofInterest)) { //i.e., within <om:Observation>
                            String endOfTag = tags.substring(ofInterest.length());
                            String content = xmlReader.content();
                            String error = null;

                            if (endOfTag.equals("<om:featureOfInterest><swe:GeoReferenceableFeature>" +
                                "<gml:location><gml:Point></gml:coordinates>")) {
                                //lat lon alt    if present, has precedence over station table
                                //VAST has this; others don't
                                String lla[] = String2.split(content, ' ');
                                if (lla.length >= 2) {
                                    tStationLatString = lla[0];
                                    tStationLonString = lla[1];
                                    if (lla.length >= 3)
                                        tStationAltString = lla[2];
                                }
                            } else if (endOfTag.equals("<om:resultDefinition><swe:DataBlockDefinition>" +
                                                "<swe:components><swe:DataRecord><swe:field>")) {
                                //field
                                String fieldName = xmlReader.attributeValue("name");
                                int col = table.findColumnNumber(fieldName);
                                fieldToCol.add(col);
                                //if (reallyVerbose) String2.log("  col=" + col + " fieldName=" + fieldName);

                            } else if (endOfTag.equals("<om:resultDefinition><swe:DataBlockDefinition>" +
                                                "<swe:encoding><swe:AsciiBlock>") ||  //oostethys has this
                                       endOfTag.equals("<om:resultDefinition><swe:DataBlockDefinition>" +
                                                "<swe:encoding><swe:TextBlock>")) {   //VAST has this
                                //encoding indicates how the data is stored
                                //tokenSeparator="," blockSeparator=" " decimalSeparator="."
                                tokenSeparator   = xmlReader.attributeValue("tokenSeparator");
                                blockSeparator   = xmlReader.attributeValue("blockSeparator");
                                decimalSeparator = xmlReader.attributeValue("decimalSeparator");

                            } else if (endOfTag.equals("</om:result>")) {
                                //the results in one big block
                                //first, ensure fieldToCol doesn't have 2 references to same column
                                int nFields = fieldToCol.size();
                                if (reallyVerbose) String2.log("fieldToCol=" + fieldToCol);
                                for (int field = 0; field < nFields; field++) {
                                    int col = fieldToCol.get(field);
                                    if (col >= 0) { //several may be -1
                                        if (fieldToCol.indexOf(col, 0) != field) //ensure none before it are the same
                                            throw new RuntimeException("Two fieldToCol=" + fieldToCol + 
                                                " have the same table column reference.");
                                    }
                                }

                                //ensure separators are set (to defaults)
                                if (tokenSeparator   == null) tokenSeparator = ",";
                                if (blockSeparator   == null) blockSeparator = " ";  //rowSeparator
                                if (decimalSeparator == null) decimalSeparator = ".";

                                //process the content (the results in one big block)
                                //  <om:result>2007-06-18T00:50:00Z,34.68,-72.66,0,24.3 ...
            //???how are Strings quoted?
                                int po = 0;
                                while (po < content.length() && String2.isWhite(content.charAt(po)))
                                    po++;
                                int nCols = table.nColumns();
                                while (po < content.length()) {

                                    //process a row of data
                                    int nRows1 = table.nRows() + 1;
                                    String rowValues[] = new String[nCols];
                                    for (int field = 0; field < nFields; field++) {
                                        String sep = field < nFields - 1? tokenSeparator : blockSeparator;
                                        int po2 = content.indexOf(sep, po + 1);
                                        if (po2 < 0) 
                                            po2 = content.length();
                                        String value = content.substring(po, po2);
                                        int col = fieldToCol.get(field);
                                        //String2.log("field=" + field + " col=" + col + " value=" + value);
                                        if (col >= 0) {
                                            //???deal with decimalSeparator for numeric Columns
                                            rowValues[col] = value;
                                        }                            
                                        po = po2 + sep.length();
                                        while (po < content.length() && String2.isWhite(content.charAt(po)))
                                            po++;
                                    }

                                    //add lat lon alt data
                                    //it's usually in the result fields, but not always
                                    if (rowValues[tableLonCol] == null)
                                        rowValues[tableLonCol] = tStationLonString;
                                    if (rowValues[tableLatCol] == null)
                                        rowValues[tableLatCol] = tStationLatString;
                                    if (rowValues[tableAltCol] == null)
                                        rowValues[tableAltCol] = tStationAltString;

                                    //add tStationID
                                    rowValues[tableStationIdCol] = tStationID;

                                    //does a row with identical LonLatAltTimeID exist in table?
                                    String tHash = rowValues[tableLonCol] + "," + rowValues[tableLatCol] + 
                                             "," + rowValues[tableAltCol] + "," + rowValues[tableTimeCol] +
                                             "," + rowValues[tableStationIdCol];
                                    int tRow = String2.parseInt((String)llatHash.get(tHash));
                                    if (tRow < Integer.MAX_VALUE) {
                                        //merge this data into that row
                                        for (int col = 0; col < nCols; col++) {
                                            String ts = rowValues[col];
                                            if (ts != null) {
                                                PrimitiveArray pa = table.getColumn(col);
                                                if (true || verbose) { 
                                                    //if there was an old value, ensure that old value = new value
                                                    //???leave this test in as insurance?
                                                    String tso = pa.getString(tRow);
                                                    pa.setString(tRow, ts);
                                                    ts = pa.getString(tRow); //setting a number changes it, e.g., 1 -> 1.0
                                                    if (tso.length() > 0 && !tso.equals(ts))
                                                        throw new RuntimeException("Error while merging row=" + tRow +
                                                            ": col=" + col + ", old=" + tso + " != new=" + ts);
                                                } else {
                                                    pa.setString(tRow, ts);
                                                }
                                            }
                                        }
                                    } else {
                                        //add this row
                                        for (int col = 0; col < nCols; col++) {
                                            String ts = rowValues[col];
                                            table.getColumn(col).addString(ts == null? "" : ts);
                                        }

                                        llatHash.put(tHash, "" + (table.nRows() - 1));
                                    }
                                }
                            }

                            //handle the error
                            if (error != null)
                                throw new RuntimeException(
                                    "Data source error on xml line #" + xmlReader.lineNumber() + 
                                    ": " + error);
                        }

                        //get the next tag
                        xmlReader.nextTag();
                        tags = xmlReader.allTags();
                    } while (!tags.startsWith("</"));

                    xmlReader.close();  
                } //end of non-ioosServer get chunk of data

                //writeToTableWriter
                if (reallyVerbose) String2.log("\nstation=" + tStationID + " tableNRows=" + table.nRows());
                //if (reallyReallyVerbose) String2.log("table=\n" + table);
                if (writeChunkToTableWriter(requestUrl, userDapQuery, table, tableWriter, false))
                    table = makeTable(tableDVI);
            } //end of obsProp loop

        } //end station loop

        //do the final writeToTableWriter
        writeChunkToTableWriter(requestUrl, userDapQuery, table, tableWriter, true);
        if (reallyVerbose) String2.log("  getDataForDapQuery done. TIME=" +
            (System.currentTimeMillis() - getTime)); 

    }

    /** Make a table with a column for each resultsVariable (with sourceNames) to hold source values.
     * @param tableDVI has the dvIndices for the columns of the table
     */
    private Table makeTable(IntArray tableDVI) throws Throwable {
        Table table = new Table();
        for (int col = 0; col < tableDVI.size(); col++) {
            int dvi = tableDVI.get(col);
            table.addColumn(dataVariables[dvi].sourceName(), 
                PrimitiveArray.factory(dataVariables[dvi].sourceDataTypeClass(), 128, false)); 
        }
        return table;
    }

private static String standardSummary = //from http://www.oostethys.org/ogc-oceans-interoperability-experiment 
"[Normally, the summary describes the dataset. Here, it describes \n" +
"the server.] \n" +
"The OCEANS IE -- formally approved as an OGC Interoperability \n" +
"Experiment in December 2006 -- engages data managers and scientists \n" +
"in the Ocean-Observing community to advance their understanding and \n" +
"application of various OGC specifications, solidify demonstrations \n" +
"for Ocean Science application areas, harden software \n" +
"implementations, and produce candidate OGC Best Practices documents \n" +
"that can be used to inform the broader ocean-observing community. \n" +
//"To achieve these goals, the OCEANS IE engages the OGC membership \n" +
//"to assure that any recommendations from the OCEANS IE will \n" +
//"properly leverage the OGC specifications. The OCEANS IE could \n" +
//"prompt Change Requests on OGC Specifications, which would be \n" +
//"provided to the OGC Technical Committee to influence the \n" +
//"underlying specifications. However, this IE will not develop \n" +
//"any new specifications, rather, participants will implement, \n" +
//"test and document experiences with existing specifications. \n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST \n" +
"include constraints for the longitude, latitude, time, and/or \n" +
"station_id variables. \n" +
"\n" +
"Initiators: SURA (lead), Texas A&M University, MBARI, GoMOOS and \n" +
"Unidata.\n" +
"\n" +
"Specific goals:\n" +
"* Compare Sensor Observation Service (SOS) from the OGC's Sensor \n" +
"  Web Enablement (SWE) initiative to the Web Feature Service (WFS) \n" +
"  as applied to ocean data in a variety of data formats including \n" +
"  text files, netCDF files, relational databases, and possibly \n" +
"  native sensor output; (see Experiment #1 for details)\n" +
"* Make use of semantic mediation via Semantic Web technologies to \n" +
"  allow plurality of identification for source types (platforms \n" +
"  and sensors) and phenomena types;\n" +
"* Test aggregation services and caching strategies to provide \n" +
"  efficient queries; \n" +
"* Explore possible enhancements of THREDDS server, so that THREDDS \n" +
"  resident data sources might be made available via SOS or WFS;";   //better summary? 


// IRIS - This SOS listed at 
//  http://www.oostethys.org/ogc-oceans-interoperability-experiment/experiment-1
//  (it may list others in future)
// Specifically http://demo.transducerml.org:8080/ogc/
//But the results are a tml reference. For now, don't get into that.
//WAIT!!! Can I request other results format???





    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosATemp(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosATemp");  

            //it was hard to find data. station advertises 1999+ for several composites.
            //but searching January's for each year found no data until 2006
            /* stopped working ~12/15/2008
            String2.log("\n*** EDDTableFromSOS nos AirTemperature test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317\"&time>=2006-01-01T00&time<=2006-01-01T01", 
                EDStatic.fullTestCacheDirectory, className + "_nosSosATemp", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, AirTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degree_C\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:00:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:06:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:12:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:18:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:24:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:30:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:36:00Z, -3.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:42:00Z, -4.0\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:54:00Z, -4.0\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T01:00:00Z, -4.1\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            */

            /* this stopped working 2009-01-12
            String2.log("\n*** EDDTableFromSOS nos AirTemperature test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450\"&time>=2008-09-01T00&time<=2008-09-01T01", 
//error=There is no Air Temperature sensor installed at station urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450 or dissemination has been stopped by CORMS.
                EDStatic.fullTestCacheDirectory, className + "_nosSosATemp", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, AirTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degree_C\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:06:00Z, 10.8\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:12:00Z, 10.7\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:18:00Z, 10.8\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:24:00Z, 10.7\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:30:00Z, 10.7\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:36:00Z, 10.8\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:42:00Z, 10.6\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:48:00Z, 10.4\n" +
"-160.502, 55.3367, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450, NaN, 2008-09-01T00:54:00Z, 10.5\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            */

            String2.log("\n*** EDDTableFromSOS nos AirTemperature test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null,  //1612340, NaN, 2008-10-26
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340\"&time>=2008-10-26T00&time<2008-10-26T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosATemp", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, AirTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degree_C\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:00:00Z, 24.9\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:06:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:12:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:18:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:24:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:30:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:36:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:42:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:48:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:54:00Z, 24.7\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS nos AirTemperature .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosATemp", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.36, 167.7362;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -14.28, 70.4;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range -3.6757152e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  AirTemperature {\n" +
"    Float64 colorBarMaximum 40.0;\n" +
"    Float64 colorBarMinimum -10.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String long_name \"Air Temperature\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#AirTemperature\";\n" +
"    String standard_name \"air_temperature\";\n" +
"    String units \"degree_C\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 167.7362;\n" +
"    Float64 geospatial_lat_max 70.4;\n" +
"    Float64 geospatial_lat_min -14.28;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 167.7362;\n" +
"    Float64 geospatial_lon_min -177.36;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //"-test"

            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

expected = 
"    Float64 Southernmost_Northing -14.28;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NOS SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have air temperature data.  ****These services are for testing and evaluation use only****\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.\";\n" +
"    String time_coverage_start \"1853-07-10T00:00:00Z\";\n" +
"    String title \"Station Data from the NOAA NOS SOS Server, EXPERIMENTAL - Air Temperature\";\n" +
"    Float64 Westernmost_Easting -177.36;\n" +
"  }\n" +
"}\n";
            int po = Math.max(0, results.indexOf(expected.substring(0, 30)));
            Test.ensureEqual(results.substring(po), expected, "RESULTS=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosATemp. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosATempAllStations(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosATemp");  

            //it was hard to find data. station advertises 1999+ for several composites.
            //but searching January's for each year found no data until 2006
            String2.log("\n*** EDDTableFromSOS nos AirTemperature test get all stations .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&time>=2008-10-26T00&time<=2008-10-26T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosATempAllStations", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, AirTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degree_C\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:00:00Z, 24.9\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:06:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:12:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:18:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:24:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:30:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:36:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:42:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:48:00Z, 24.8\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T00:54:00Z, 24.7\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340, NaN, 2008-10-26T01:00:00Z, 24.8\n" +
"-157.79, 21.4331, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612480, NaN, 2008-10-26T00:00:00Z, 22.9\n" +
"-157.79, 21.4331, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612480, NaN, 2008-10-26T00:06:00Z, 22.8\n";
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosATemp. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosATempStationList(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosATemp");  

            //it was hard to find data. station advertises 1999+ for several composites.
            //but searching January's for each year found no data until 2006
            String2.log("\n*** EDDTableFromSOS nos AirTemperature test get all stations .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, "longitude,latitude,station_id", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosATempStationList", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id\n" +
"degrees_east, degrees_north, \n" +
"-159.3561, 21.9544, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1611400\n" +
"-157.867, 21.3067, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612340\n" +
"-157.79, 21.4331, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1612480\n" +
"-156.4767, 20.895, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1615680\n" +
"-155.8294, 20.0366, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::1617433\n";
//...
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosATemp. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosBPres(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosBPres");  

            //it was hard to find data. station advertises 1999+ for several composites.
            //but searching January's for each year found no data until 2006
            /* stopped working ~12/15/2008
            String2.log("\n*** EDDTableFromSOS nos Pressure test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317\"&time>=2006-01-01T00&time<=2006-01-01T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosPressure", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, BarometricPressure\n" +
"degrees_east, degrees_north, , m, UTC, millibars\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:00:00Z, 1010.7\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:06:00Z, 1010.8\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:12:00Z, 1010.7\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:18:00Z, 1010.7\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:24:00Z, 1010.7\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:30:00Z, 1010.9\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:36:00Z, 1011.1\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:42:00Z, 1011.4\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:54:00Z, 1011.4\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T01:00:00Z, 1011.4\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            */

            String2.log("\n*** EDDTableFromSOS nos Pressure test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
//stopped working 2009-01-12  "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450\"&time>=2008-09-01T00&time<=2008-09-01T01", 
//error=There is no Barometric Pressure sensor installed at station urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9459450 or dissemination has been stopped by CORMS.
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094\"&time>=2008-09-01T00&time<=2008-09-01T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosPressure", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, BarometricPressure\n" +
"degrees_east, degrees_north, , m, UTC, millibars\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:00:00Z, 1011.3\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:06:00Z, 1011.2\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:12:00Z, 1011.2\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:18:00Z, 1011.1\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:24:00Z, 1011.1\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:30:00Z, 1011.1\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:36:00Z, 1011.0\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:42:00Z, 1010.9\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:48:00Z, 1010.9\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T00:54:00Z, 1010.9\n" +
"-164.065, 67.5767, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9491094, NaN, 2008-09-01T01:00:00Z, 1010.8\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS nos Pressure .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosPressure", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.36, 167.7362;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -14.28, 70.4;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range -3.6757152e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  BarometricPressure {\n" +
"    Float64 colorBarMaximum 1030.0;\n" +
"    Float64 colorBarMinimum 970.0;\n" +
"    String ioos_category \"Pressure\";\n" +
"    String long_name \"Barometric Pressure\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#BarometricPressure\";\n" +
"    String standard_name \"air_pressure\";\n" +
"    String units \"millibars\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 167.7362;\n" +
"    Float64 geospatial_lat_max 70.4;\n" +
"    Float64 geospatial_lat_min -14.28;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 167.7362;\n" +
"    Float64 geospatial_lon_min -177.36;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //-test
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosPressure. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosCond(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosCond");  

            //it was hard to find data. station advertises 1999+ for several composites.
            //but searching January's for each year found no data until 2006
            /* stopped working ~12/15/2008
            String2.log("\n*** EDDTableFromSOS nos Conductivity test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317\"&time>=2006-01-01T00&time<=2006-01-01T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosCond", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, Conductivity\n" +
"degrees_east, degrees_north, , m, UTC, mS cm-1\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:00:00Z, 2.53\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:06:00Z, 2.54\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:12:00Z, 2.55\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:18:00Z, 2.54\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:24:00Z, 2.54\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:30:00Z, 2.54\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:36:00Z, 2.53\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:42:00Z, 2.52\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:54:00Z, 2.71\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T01:00:00Z, 2.75\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            */

            String2.log("\n*** EDDTableFromSOS nos Conductivity test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680\"&time>=2008-09-01T00&time<=2008-09-01T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosCond", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, Conductivity\n" +
"degrees_east, degrees_north, , m, UTC, mS cm-1\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:00:00Z, 16.702\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:06:00Z, 16.671\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:12:00Z, 16.681\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:18:00Z, 16.693\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:24:00Z, 16.685\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:30:00Z, 16.714\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:36:00Z, 16.725\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:42:00Z, 16.74\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:48:00Z, 16.772\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T00:54:00Z, 16.809\n" +
"-76.5783, 39.2667, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8574680, NaN, 2008-09-01T01:00:00Z, 16.811\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS nos Conductivity .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosCond", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -94.985, -70.5633;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 29.48, 43.32;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range -2.2184928e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  Conductivity {\n" +
"    String ioos_category \"Salinity\";\n" +
"    String long_name \"Electrical Conductivity\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Conductivity\";\n" +
"    String standard_name \"sea_water_electrical_conductivity\";\n" +
"    String units \"mS cm-1\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting -70.5633;\n" +
"    Float64 geospatial_lat_max 43.32;\n" +
"    Float64 geospatial_lat_min 29.48;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max -70.5633;\n" +
"    Float64 geospatial_lon_min -94.985;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //-test
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosCond. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * CURRENTLY NOT WORKING. EDDTableFromSOS doesn't handled binned data.
     * Test NOS CO-OPS SOS (coops).
     *
     * @param doLongTest
     * @throws Throwable if trouble
     */
    public static void testNosSosCurrents(String datasetIdPrefix, boolean doLongTest) throws Throwable {
        String2.log("\ntestNosSosCurrents");
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        //see 6/12/08 email from Roy
        //   see web page:  http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos/
        //2008-10-29 I got Bad Gateway error (but it worked a few minutes later): 
        //Exception in thread "main" java.io.IOException: Server returned HTTP response code: 
        //502 for URL: http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos/SOS?service=SOS&request=GetCapabilities
        //results:
        //  n  ObservedProperty
        //---  --------------------------------------------------
        //  0  urn:ogc:def:property:OGC:currents
        //  1  urn:ogc:def:property:OGC:waterlevel
        //  2  urn:ogc:def:property:OGC:watertemperature
        //  3  urn:ogc:def:property:OGC:airtemperature
        //  4  urn:ogc:def:property:OGC:barometricpressure
        //  5  urn:ogc:def:property:OGC:wind
        //  6  urn:ogc:def:property:OGC:conductivity
        //  7  urn:ogc:def:property:OGC:salinity
        //String2.log(generateDatasetsXml(
        //    "http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos/SOS", false));
        //if (true) System.exit(0);

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosCurrents");  

        
            String2.log("\n*** EDDTableFromSOS test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "station_id,longitude,latitude,altitude,time,CurrentDirection,CurrentSpeed" +
                //36.95903 -76.01263
                "&longitude=-76.01263&latitude=36.95903&time>=2008-08-01T14&time<=2008-08-01T15", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nos_test1", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS NOS currents .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosCurrents", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -153.42212, -66.15883;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 17.86167, 61.27822;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 8.63001e+8, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  CurrentDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Currents\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Currents\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  CurrentSpeed {\n" +
"    Float64 colorBarMaximum 50.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Currents\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Currents\";\n" +
"    String standard_name \"sea_water_speed\";\n" +
"    String units \"cm s-1\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting -66.15883;\n" +
"    Float64 geospatial_lat_max 61.27822;\n" +
"    Float64 geospatial_lat_min 17.86167;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max -66.15883;\n" +
"    Float64 geospatial_lon_min -153.42212;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //-test
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

        if (true)
            throw new RuntimeException("EDDTableFromSOS can't handle binned data.");

        //expected error didn't occur!
        String2.getStringFromSystemIn("\n" + 
            MustBe.getStackTrace() + 
            "An expected error didn't occur at the above location.\n" + 
            "Press ^C to stop or Enter to continue..."); 

        
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nExpected error. EDDTableFromSOS can't handle binned data." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosSalinity(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosSalinity");  

            String2.log("\n*** EDDTableFromSOS nos Salinity .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosSalinity", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -94.985, -70.5633;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 29.48, 43.32;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n";
//"    Float64 actual_range -2.2184928e+9, NaN;\n" + //-2.1302784e+9
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

            //it was hard to find data. station advertises 1999+ for several composites.
            //but searching January's for each year found no data until 2006
//??? Does the error message need to be treated as simple no data?
//<ExceptionReport><Exception>atts=exceptionCode="NoApplicableCode", locator="8419317"
            /* stopped working ~12/15/2008
            String2.log("\n*** EDDTableFromSOS nos Salinity test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317\"&time>=2006-01-01T00&time<=2006-01-01T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosSalinity", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = //salinity is low because this is in estuary/river, not in ocean
"longitude, latitude, station_id, altitude, time, Salinity\n" +
"degrees_east, degrees_north, , m, UTC, PSU\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:00:00Z, 2.276\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:06:00Z, 2.286\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:12:00Z, 2.288\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:18:00Z, 2.286\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:24:00Z, 2.286\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:30:00Z, 2.286\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:36:00Z, 2.276\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:42:00Z, 2.253\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T00:54:00Z, 2.374\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2006-01-01T01:00:00Z, 2.404\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            */

            String2.log("\n*** EDDTableFromSOS nos Salinity test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                //8737048 requests stopped working ~2009-03-26 and no recent data
                //"&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8737048\"&time>=2008-09-01T00&time<=2008-09-01T01", 
                //so I switched to 8419317 on 2009-04-08
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317\"&time>=2009-04-05T00&time<=2009-04-05T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosSalinity", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = //salinity is low because this is in estuary/river, not in ocean
"longitude, latitude, station_id, altitude, time, Salinity\n" +
"degrees_east, degrees_north, , m, UTC, PSU\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:00:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:06:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:12:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:18:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:24:00Z, 4.254\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:30:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:36:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:42:00Z, 4.254\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:48:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T00:54:00Z, 4.245\n" +
"-70.5633, 43.32, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317, NaN, 2009-04-05T01:00:00Z, 4.245\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosSalinity." +
                "\nNOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosWind(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosWind");  

            // stopped working ~12/15/2008
            //    "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8419317\"&time>=2006-01-01T00&time<=2006-01-01T01", 
            //stopped working ~2009-03-26, and no recent data
            //    "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9461380\"&time>=2008-09-01T00&time<=2008-09-01T01", 

            String2.log("\n*** EDDTableFromSOS nos Wind test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756\"&time>=2009-04-06T00&time<=2009-04-06T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosWind", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WindSpeed, WindDirection, WindGust\n" +
"degrees_east, degrees_north, , m, UTC, cm s-1, degrees_true, cm s-1\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:00:00Z, 7.8, 79.0, 9.5\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:06:00Z, 7.4, 81.0, 9.5\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:12:00Z, 7.8, 80.0, 9.5\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:18:00Z, 8.1, 79.0, 10.1\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:24:00Z, 7.3, 79.0, 9.0\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:30:00Z, 7.6, 76.0, 8.9\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:36:00Z, 7.2, 82.0, 8.5\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:42:00Z, 8.1, 79.0, 9.2\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:48:00Z, 7.3, 80.0, 9.4\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T00:54:00Z, 8.0, 81.0, 9.4\n" +
"-165.43, 64.5, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9468756, NaN, 2009-04-06T01:00:00Z, 7.9, 80.0, 10.3\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS nos Wind .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosWind", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -176.632, 167.7362;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -14.28, 70.4;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range -3.6757152e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  WindSpeed {\n" +
"    Float64 colorBarMaximum 1500.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String standard_name \"wind_speed\";\n" +
"    String units \"cm s-1\";\n" +
"  }\n" +
"  WindDirection {\n" +
"    Float64 colorBarMaximum 3600.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String standard_name \"wind_from_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  WindGust {\n" +
"    Float64 colorBarMaximum 300.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String standard_name \"wind_speed_of_gust\";\n" +
"    String units \"cm s-1\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 167.7362;\n" +
"    Float64 geospatial_lat_max 70.4;\n" +
"    Float64 geospatial_lat_min -14.28;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 167.7362;\n" +
"    Float64 geospatial_lon_min -176.632;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //-test
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error " + datasetIdPrefix + "nosSosWind." +
                "\nNOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosWLevel(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosWLevel");  

            String2.log("\n*** EDDTableFromSOS nos wLevel .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosWLevel", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.36, 167.7362;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -14.28, 70.4;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range -3.6757152e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  WaterLevel {\n" +
"    Float64 colorBarMaximum 8000.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Sea Level\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#WaterLevel\";\n" +
"    String standard_name \"sea_floor_depth_below_sea_level\";\n" +
"    String units \"m\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 167.7362;\n" +
"    Float64 geospatial_lat_max 70.4;\n" +
"    Float64 geospatial_lat_min -14.28;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 167.7362;\n" +
"    Float64 geospatial_lon_min -177.36;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //-test
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

            
            String2.log("\n*** EDDTableFromSOS nos WLevel test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062\"&time>=2008-08-01T14:00&time<=2008-08-01T15:00", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosWLevel", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WaterLevel\n" +
"degrees_east, degrees_north, , m, UTC, m\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:00:00Z, 75.014\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:06:00Z, 75.014\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:12:00Z, 75.018\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:18:00Z, 75.015\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:24:00Z, 75.012\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:30:00Z, 75.012\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:36:00Z, 75.016\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:42:00Z, 75.018\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:48:00Z, 75.02\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T14:54:00Z, 75.019\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, -999.0, 2008-08-01T15:00:00Z, 75.017\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNosSosWTemp(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "nosSosWTemp");  

            String2.log("\n*** EDDTableFromSOS nos wtemp .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_nosSosWTemp", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.36, 167.7362;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -14.28, 70.4;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range -3.6757152e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  WaterTemperature {\n" +
"    Float64 colorBarMaximum 32.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#WaterTemperature\";\n" +
"    String standard_name \"sea_surface_temperature\";\n" +
"    String units \"degrees_C\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 167.7362;\n" +
"    Float64 geospatial_lat_max 70.4;\n" +
"    Float64 geospatial_lat_min -14.28;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 167.7362;\n" +
"    Float64 geospatial_lon_min -177.36;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://opendap.co-ops.nos.noaa.gov/ioos-dif-sos"; //-test
            Test.ensureEqual(results.substring(0, expected.length()), expected, "RESULTS=\n" + results);

            /* stopped working ~12/15/2008
            String2.log("\n*** EDDTableFromSOS nos WTemp test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062\"&time>=2008-08-01T14:00&time<2008-08-01T15:00", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosWTemp", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WaterTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degrees_C\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:00:00Z, 22.349\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:06:00Z, 22.4\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:12:00Z, 22.4\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:18:00Z, 22.4\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:24:00Z, 22.366\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:30:00Z, 22.4\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:36:00Z, 22.37\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:42:00Z, 22.379\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:48:00Z, 22.382\n" +
"-75.9345, 44.3311, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::8311062, NaN, 2008-08-01T14:54:00Z, 22.389\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            */

            String2.log("\n*** EDDTableFromSOS nos WTemp test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780\"&time>=2008-09-01T14:00&time<2008-09-01T15:00", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_nosSosWTemp", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WaterTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degrees_C\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:00:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:06:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:12:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:18:00Z, 11.7\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:24:00Z, 11.7\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:30:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:36:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:42:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:48:00Z, 11.6\n" +
"-124.322, 43.345, urn:x-noaa:def:station:NOAA.NOS.CO-OPS::9432780, NaN, 2008-09-01T14:54:00Z, 11.6\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     * Note: starting in 2009, there is a test server at http://sdftest.ndbc.noaa.gov/sos/ .
     * @throws Throwable if trouble
     */
    public static void testNdbcSosCurrents(String datasetIdPrefix, boolean doLongTest) throws Throwable {
        String2.log("\n*** testNdbcSosCurrents");
        testVerboseOn();
        EDDTable eddTable;               
        double tLon, tLat;
        String name, tName, results, expected, userDapQuery;
        Table table;
        String error = "";

        try { 
            //see 6/12/08 email from Roy
            //   see web page:  http://sdf.ndbc.noaa.gov/sos/
            //http://sdf.ndbc.noaa.gov/sos/server.php?request=GetObservation&service=SOS
            //  &offering=NDBC:46088&observedproperty=currents&responseformat=text/xml;
            //  schema=%22ioos/0.6.1%22&eventtime=2008-06-01T00:00Z/2008-06-02T00:00Z
            //request from  -70.14          43.53     1193961600            NaN     NDBC:44007
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosCurrents");  

            String2.log("\n*** EDDTableFromSOS ndbc currents .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_ndbc_test1", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -172.17, 165.0;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -8.0, 60.59;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 3.211488e+8, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  CurrentDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Currents\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Currents\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  CurrentSpeed {\n" +
"    Float64 colorBarMaximum 50.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Currents\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Currents\";\n" +
"    String standard_name \"sea_water_speed\";\n" +
"    String units \"cm s-1\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 165.0;\n" +
"    Float64 geospatial_lat_max 60.59;\n" +
"    Float64 geospatial_lat_min -8.0;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 165.0;\n" +
"    Float64 geospatial_lon_min -172.17;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/" + datasetIdPrefix + "ndbcSosCurrents.das\";\n" +
"    String infoUrl \"http://sdf.ndbc.noaa.gov/sos/\";\n" +
"    String institution \"NOAA NDBC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 60.59;\n" +
"    String sourceUrl \"http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\";\n" +
"    Float64 Southernmost_Northing -8.0;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NDBC SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have currents data.\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.  No more than thirty days of data can be requested.\";\n" +
"    String time_coverage_start \"1980-03-06T00:00:00Z\";\n" +
"    String title \"Buoy Data from the NOAA NDBC SOS Server - Currents\";\n" +
"    Float64 Westernmost_Easting -172.17;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            //test lon lat (numeric) = > <, and time > <
            String2.log("\n*** EDDTableFromSOS currents test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, "station_id,longitude,latitude,altitude,time,CurrentDirection,CurrentSpeed" +
                "&longitude=-87.94&latitude>=29.1&latitude<29.2&time>=2008-06-01T14:00&time<=2008-06-01T14:30", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbc_test1", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"station_id, longitude, latitude, altitude, time, CurrentDirection, CurrentSpeed\n" +
", degrees_east, degrees_north, m, UTC, degrees_true, cm s-1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -56.8, 2008-06-01T14:03:00Z, 83, 30.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -88.8, 2008-06-01T14:03:00Z, 96, 40.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -120.8, 2008-06-01T14:03:00Z, 96, 40.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -152.8, 2008-06-01T14:03:00Z, 96, 35.3\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -184.8, 2008-06-01T14:03:00Z, 89, 31.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -216.8, 2008-06-01T14:03:00Z, 90, 25.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -248.8, 2008-06-01T14:03:00Z, 74, 22.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -280.8, 2008-06-01T14:03:00Z, 65, 21.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -312.8, 2008-06-01T14:03:00Z, 71, 16.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -344.8, 2008-06-01T14:03:00Z, 96, 13.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -376.8, 2008-06-01T14:03:00Z, 110, 14.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -408.8, 2008-06-01T14:03:00Z, 121, 12.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -440.8, 2008-06-01T14:03:00Z, 111, 7.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -472.8, 2008-06-01T14:03:00Z, 6, 2.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -504.8, 2008-06-01T14:03:00Z, 17, 2.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -536.8, 2008-06-01T14:03:00Z, 37, 3.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -568.8, 2008-06-01T14:03:00Z, 124, 0.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -600.8, 2008-06-01T14:03:00Z, 147, 4.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -632.8, 2008-06-01T14:03:00Z, 167, 5.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -664.8, 2008-06-01T14:03:00Z, 150, 3.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -696.8, 2008-06-01T14:03:00Z, 317, 4.1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -728.8, 2008-06-01T14:03:00Z, 31, 2.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -760.8, 2008-06-01T14:03:00Z, 62, 3.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -792.8, 2008-06-01T14:03:00Z, 24, 4.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -824.8, 2008-06-01T14:03:00Z, 56, 4.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -856.8, 2008-06-01T14:03:00Z, 55, 2.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -888.8, 2008-06-01T14:03:00Z, 68, 4.6\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -920.8, 2008-06-01T14:03:00Z, 68, 5.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -952.8, 2008-06-01T14:03:00Z, 75, 4.3\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -984.8, 2008-06-01T14:03:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -1016.8, 2008-06-01T14:03:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -1048.8, 2008-06-01T14:03:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -56.8, 2008-06-01T14:23:00Z, 82, 27.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -88.8, 2008-06-01T14:23:00Z, 98, 33.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -120.8, 2008-06-01T14:23:00Z, 96, 38.6\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -152.8, 2008-06-01T14:23:00Z, 94, 35.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -184.8, 2008-06-01T14:23:00Z, 91, 30.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -216.8, 2008-06-01T14:23:00Z, 89, 24.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -248.8, 2008-06-01T14:23:00Z, 76, 22.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -280.8, 2008-06-01T14:23:00Z, 69, 19.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -312.8, 2008-06-01T14:23:00Z, 74, 15.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -344.8, 2008-06-01T14:23:00Z, 100, 14.3\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -376.8, 2008-06-01T14:23:00Z, 114, 14.3\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -408.8, 2008-06-01T14:23:00Z, 126, 12.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -440.8, 2008-06-01T14:23:00Z, 121, 7.1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -472.8, 2008-06-01T14:23:00Z, 22, 2.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -504.8, 2008-06-01T14:23:00Z, 17, 3.1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -536.8, 2008-06-01T14:23:00Z, 45, 3.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -568.8, 2008-06-01T14:23:00Z, 127, 2.1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -600.8, 2008-06-01T14:23:00Z, 160, 4.6\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -632.8, 2008-06-01T14:23:00Z, 166, 5.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -664.8, 2008-06-01T14:23:00Z, 142, 3.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -696.8, 2008-06-01T14:23:00Z, 314, 4.6\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -728.8, 2008-06-01T14:23:00Z, 48, 3.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -760.8, 2008-06-01T14:23:00Z, 9, 3.1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -792.8, 2008-06-01T14:23:00Z, 32, 5.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -824.8, 2008-06-01T14:23:00Z, 27, 5.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -856.8, 2008-06-01T14:23:00Z, 54, 2.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -888.8, 2008-06-01T14:23:00Z, 64, 2.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -920.8, 2008-06-01T14:23:00Z, 53, 5.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -952.8, 2008-06-01T14:23:00Z, 89, 4.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -984.8, 2008-06-01T14:23:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -1016.8, 2008-06-01T14:23:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -1048.8, 2008-06-01T14:23:00Z, 0, 0.0\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            //test station regex
            String2.log("\n*** EDDTableFromSOS Currents test get 2 stations from regex, 1 time, .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, "station_id,longitude,latitude,altitude,time,CurrentDirection,CurrentSpeed" +
                "&station_id=~\"(urn\\:x\\-noaa\\:def\\:station\\:noaa\\.nws\\.ndbc\\:\\:41035|urn\\:x\\-noaa\\:def\\:station\\:noaa\\.nws\\.ndbc\\:\\:42376)\"" +
                "&time>=2008-06-01T14:00&time<=2008-06-01T14:15", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbc_test1b", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
//Before revamping 2008-10, test returned values below.   NOW DIFFERENT!
//    "urn:x-noaa:def:station:noaa.nws.ndbc::41035, -77.28, 34.48, -1.6, 2008-06-01T14:00:00Z, 223, 3.3\n" +    now 74, 15.2
//    "urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -3.8, 2008-06-01T14:00:00Z, 206, 19.0\n";
"station_id, longitude, latitude, altitude, time, CurrentDirection, CurrentSpeed\n" +
", degrees_east, degrees_north, m, UTC, degrees_true, cm s-1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41035, -77.28, 34.48, -1.6, 2008-06-01T14:00:00Z, 74, 15.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -56.8, 2008-06-01T14:03:00Z, 83, 30.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -88.8, 2008-06-01T14:03:00Z, 96, 40.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -120.8, 2008-06-01T14:03:00Z, 96, 40.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -152.8, 2008-06-01T14:03:00Z, 96, 35.3\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -184.8, 2008-06-01T14:03:00Z, 89, 31.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -216.8, 2008-06-01T14:03:00Z, 90, 25.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -248.8, 2008-06-01T14:03:00Z, 74, 22.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -280.8, 2008-06-01T14:03:00Z, 65, 21.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -312.8, 2008-06-01T14:03:00Z, 71, 16.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -344.8, 2008-06-01T14:03:00Z, 96, 13.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -376.8, 2008-06-01T14:03:00Z, 110, 14.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -408.8, 2008-06-01T14:03:00Z, 121, 12.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -440.8, 2008-06-01T14:03:00Z, 111, 7.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -472.8, 2008-06-01T14:03:00Z, 6, 2.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -504.8, 2008-06-01T14:03:00Z, 17, 2.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -536.8, 2008-06-01T14:03:00Z, 37, 3.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -568.8, 2008-06-01T14:03:00Z, 124, 0.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -600.8, 2008-06-01T14:03:00Z, 147, 4.5\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -632.8, 2008-06-01T14:03:00Z, 167, 5.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -664.8, 2008-06-01T14:03:00Z, 150, 3.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -696.8, 2008-06-01T14:03:00Z, 317, 4.1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -728.8, 2008-06-01T14:03:00Z, 31, 2.9\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -760.8, 2008-06-01T14:03:00Z, 62, 3.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -792.8, 2008-06-01T14:03:00Z, 24, 4.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -824.8, 2008-06-01T14:03:00Z, 56, 4.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -856.8, 2008-06-01T14:03:00Z, 55, 2.8\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -888.8, 2008-06-01T14:03:00Z, 68, 4.6\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -920.8, 2008-06-01T14:03:00Z, 68, 5.2\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -952.8, 2008-06-01T14:03:00Z, 75, 4.3\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -984.8, 2008-06-01T14:03:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -1016.8, 2008-06-01T14:03:00Z, 0, 0.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::42376, -87.94, 29.16, -1048.8, 2008-06-01T14:03:00Z, 0, 0.0\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);


            //test station = 
            String2.log("\n*** EDDTableFromSOS Currents test get by station name, multi depths, .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, "station_id,longitude,latitude,altitude,time,CurrentDirection,CurrentSpeed" +
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::41012\"&time>=2008-06-01T00&time<=2008-06-01T01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbc_test2", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
    "station_id, longitude, latitude, altitude, time, CurrentDirection, CurrentSpeed\n" +
    ", degrees_east, degrees_north, m, UTC, degrees_true, cm s-1\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -5.0, 2008-06-01T00:00:00Z, 56, 6.3\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -7.0, 2008-06-01T00:00:00Z, 59, 14.8\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -9.0, 2008-06-01T00:00:00Z, 57, 20.6\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -11.0, 2008-06-01T00:00:00Z, 56, 22.2\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -13.0, 2008-06-01T00:00:00Z, 59, 25.0\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -15.0, 2008-06-01T00:00:00Z, 63, 27.6\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -17.0, 2008-06-01T00:00:00Z, 70, 31.8\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -19.0, 2008-06-01T00:00:00Z, 73, 33.0\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -21.0, 2008-06-01T00:00:00Z, 74, 33.8\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -23.0, 2008-06-01T00:00:00Z, 75, 33.5\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -25.0, 2008-06-01T00:00:00Z, 76, 32.8\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -27.0, 2008-06-01T00:00:00Z, 75, 31.5\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -29.0, 2008-06-01T00:00:00Z, 77, 28.9\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -31.0, 2008-06-01T00:00:00Z, 78, 25.4\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -33.0, 2008-06-01T00:00:00Z, 80, 23.0\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -5.0, 2008-06-01T01:00:00Z, 52, 14.9\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -7.0, 2008-06-01T01:00:00Z, 63, 23.8\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -9.0, 2008-06-01T01:00:00Z, 68, 28.9\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -11.0, 2008-06-01T01:00:00Z, 71, 31.5\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -13.0, 2008-06-01T01:00:00Z, 74, 32.5\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -15.0, 2008-06-01T01:00:00Z, 81, 31.9\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -17.0, 2008-06-01T01:00:00Z, 83, 31.3\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -19.0, 2008-06-01T01:00:00Z, 85, 31.3\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -21.0, 2008-06-01T01:00:00Z, 84, 32.0\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -23.0, 2008-06-01T01:00:00Z, 85, 30.7\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -25.0, 2008-06-01T01:00:00Z, 86, 29.6\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -27.0, 2008-06-01T01:00:00Z, 85, 28.2\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -29.0, 2008-06-01T01:00:00Z, 86, 26.9\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -31.0, 2008-06-01T01:00:00Z, 86, 25.2\n" +
    "urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -33.0, 2008-06-01T01:00:00Z, 85, 22.6\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            //long hard test 
            if (doLongTest) {
                String2.log("\n*** EDDTableFromSOS Currents test get data from many stations\n");
                tName = eddTable.makeNewFileForDapQuery(null, null, "station_id,longitude,latitude,altitude,time,CurrentDirection,CurrentSpeed" +
                    "&time>=2008-06-14T00&time<=2008-06-14T02&altitude=-25", 
                    EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbc_test3", ".csv"); 
                results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
                expected = 
"station_id, longitude, latitude, altitude, time, CurrentDirection, CurrentSpeed\n" +
", degrees_east, degrees_north, m, UTC, degrees_true, cm s-1\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -25.0, 2008-06-14T00:00:00Z, 93, 22.4\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -25.0, 2008-06-14T01:00:00Z, 96, 19.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41012, -80.55, 30.04, -25.0, 2008-06-14T02:00:00Z, 103, 19.7\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41036, -76.95, 34.21, -25.0, 2008-06-14T00:00:00Z, 170, 11.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41036, -76.95, 34.21, -25.0, 2008-06-14T01:00:00Z, 190, 11.0\n" +
"urn:x-noaa:def:station:noaa.nws.ndbc::41036, -76.95, 34.21, -25.0, 2008-06-14T02:00:00Z, 220, 9.0\n";
                Test.ensureEqual(results, expected, "RESULTS=\n" + results);
            }


            String2.log("\n*** EDDTableFromSOS Currents test display error in .png\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, "station_id,longitude,latitude,altitude,time,CurrentDirection,CurrentSpeed" +
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::invalid\"&time=2008-06-14T00", //&latitude%3E=60.5
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbc_testError", ".png"); 
            SSR.displayInBrowser("file://" + EDStatic.fullTestCacheDirectory + tName);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NDBC SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }


    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNdbcSosSalinity(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosSalinity");  

            String2.log("\n*** EDDTableFromSOS ndbc salinity .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_ndbcSosSalinity", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -151.719, -65.927;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 17.93, 59.603;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.1964564e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  Salinity {\n" +
"    Float64 colorBarMaximum 37.0;\n" +
"    Float64 colorBarMinimum 32.0;\n" +
"    String ioos_category \"Salinity\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Salinity\";\n" +
"    String standard_name \"sea_water_salinity\";\n" +
"    String units \"PSU\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting -65.927;\n" +
"    Float64 geospatial_lat_max 59.603;\n" +
"    Float64 geospatial_lat_min 17.93;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max -65.927;\n" +
"    Float64 geospatial_lon_min -151.719;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/" + datasetIdPrefix + "ndbcSosSalinity.das\";\n" +
"    String infoUrl \"http://sdf.ndbc.noaa.gov/sos/\";\n" +
"    String institution \"NOAA NDBC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 59.603;\n" +
"    String sourceUrl \"http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\";\n" +
"    Float64 Southernmost_Northing 17.93;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NDBC SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have salinity data.\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.  No more than thirty days of data can be requested.\";\n" +
"    String time_coverage_start \"2007-11-30T21:00:00Z\";\n" +
"    String title \"Buoy Data from the NOAA NDBC SOS Server - Salinity\";\n" +
"    Float64 Westernmost_Easting -151.719;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS Salinity test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::46013\"&time>=2008-08-01&time<2008-08-02", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosSalinity", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, Salinity\n" +
"degrees_east, degrees_north, , m, UTC, PSU\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T20:50:00Z, 33.89\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T22:50:00Z, 33.89\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NDBC SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This is a test to determine longest allowed time request.
     *
     * @throws Throwable if trouble
     */
    public static void testNdbcSosLongTime(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosSalinity");  

            String2.log("\n*** EDDTableFromSOS NDBC lon time, one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null,   
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::44038\"&time>=2007-08-01", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosSalinity", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, Salinity\n" +
"degrees_east, degrees_north, , m, UTC, PSU\n" +
"-66.55, 43.62, urn:x-noaa:def:station:noaa.nws.ndbc::44038, NaN, 2008-07-29T09:00:00Z, 31.7\n" +
"-66.55, 43.62, urn:x-noaa:def:station:noaa.nws.ndbc::44038, NaN, 2008-07-29T10:00:00Z, 31.7\n" +
"-66.55, 43.62, urn:x-noaa:def:station:noaa.nws.ndbc::44038, NaN, 2008-07-29T11:00:00Z, 31.7\n";
            Test.ensureEqual(results.substring(0, expected.length()), expected, 
                "RESULTS=\n" + results.substring(0, expected.length() + 200));

            //conclusion: NDBC doesn't seem to have a limit on time range,
            //but the data just doesn't go back very far.
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NDBC SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNdbcSosWLevel(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosWLevel");  

            String2.log("\n*** EDDTableFromSOS ndbc wLevel .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_ndbcSosWLevel", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -176.25, 178.27;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -46.92, 57.5;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.2095577e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  WaterLevel {\n" +
"    Float64 colorBarMaximum 8000.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Sea Level\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#WaterLevel\";\n" +
"    String standard_name \"sea_floor_depth_below_sea_level\";\n" +
"    String units \"m\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 178.27;\n" +
"    Float64 geospatial_lat_max 57.5;\n" +
"    Float64 geospatial_lat_min -46.92;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 178.27;\n" +
"    Float64 geospatial_lon_min -176.25;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/" + datasetIdPrefix + "ndbcSosWLevel.das\";\n" +
"    String infoUrl \"http://sdf.ndbc.noaa.gov/sos/\";\n" +
"    String institution \"NOAA NDBC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 57.5;\n" +
"    String sourceUrl \"http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\";\n" +
"    Float64 Southernmost_Northing -46.92;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NDBC SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have water level data.\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.  No more than thirty days of data can be requested.\";\n" +
"    String time_coverage_start \"2008-04-30T12:15:00Z\";\n" +
"    String title \"Buoy Data from the NOAA NDBC SOS Server - Water Level\";\n" +
"    Float64 Westernmost_Easting -176.25;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            
            String2.log("\n*** EDDTableFromSOS nos WLevel test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::55015\"&time>=2008-08-01T14:00&time<=2008-08-01T15:00", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosWLevel", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WaterLevel\n" +
"degrees_east, degrees_north, , m, UTC, m\n" +
"160.56, -46.92, urn:x-noaa:def:station:noaa.nws.ndbc::55015, NaN, 2008-08-01T14:00:00Z, 4944.303\n" +
"160.56, -46.92, urn:x-noaa:def:station:noaa.nws.ndbc::55015, NaN, 2008-08-01T14:15:00Z, 4944.215\n" +
"160.56, -46.92, urn:x-noaa:def:station:noaa.nws.ndbc::55015, NaN, 2008-08-01T14:30:00Z, 4944.121\n" +
"160.56, -46.92, urn:x-noaa:def:station:noaa.nws.ndbc::55015, NaN, 2008-08-01T14:45:00Z, 4944.025\n" +
"160.56, -46.92, urn:x-noaa:def:station:noaa.nws.ndbc::55015, NaN, 2008-08-01T15:00:00Z, 4943.93\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NOS SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNdbcSosWTemp(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosWTemp");  

            String2.log("\n*** EDDTableFromSOS ndbc wtemp .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_ndbcSosWTemp", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.75, 179.0;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 3.517, 60.8;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.1939616e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  WaterTemperature {\n" +
"    Float64 colorBarMaximum 32.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#WaterTemperature\";\n" +
"    String standard_name \"sea_surface_temperature\";\n" +
"    String units \"degrees_C\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 179.0;\n" +
"    Float64 geospatial_lat_max 60.8;\n" +
"    Float64 geospatial_lat_min 3.517;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 179.0;\n" +
"    Float64 geospatial_lon_min -177.75;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/" + datasetIdPrefix + "ndbcSosWTemp.das\";\n" +
"    String infoUrl \"http://sdf.ndbc.noaa.gov/sos/\";\n" +
"    String institution \"NOAA NDBC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 60.8;\n" +
"    String sourceUrl \"http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\";\n" +
"    Float64 Southernmost_Northing 3.517;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NDBC SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have water temperature data.\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.  No more than thirty days of data can be requested.\";\n" +
"    String time_coverage_start \"2007-11-02T00:00:00Z\";\n" +
"    String title \"Buoy Data from the NOAA NDBC SOS Server - Water Temperature\";\n" +
"    Float64 Westernmost_Easting -177.75;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS ndbc WTemp test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::46013\"&time>=2008-08-01T14:00&time<2008-08-01T20:00", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosWTemp", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WaterTemperature\n" +
"degrees_east, degrees_north, , m, UTC, degrees_C\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T14:50:00Z, 10.9\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T15:50:00Z, 10.9\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T16:50:00Z, 10.9\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T17:50:00Z, 10.9\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T18:50:00Z, 11.0\n" +
"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T19:50:00Z, 11.1\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NDBC SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNdbcSosWaves(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosWaves");  

            String2.log("\n*** EDDTableFromSOS ndbc waves .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_ndbcSosWaves", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.75, 179.0;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range -19.713, 60.8;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.1939616e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  SignificantWaveHeight {\n" +
"    Float64 colorBarMaximum 10.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_significant_height\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  DominantWavePeriod {\n" +
"    Float64 colorBarMaximum 20.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_period\";\n" +
"    String units \"s\";\n" +
"  }\n" +
"  AverageWavePeriod {\n" +
"    Float64 colorBarMaximum 20.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_period\";\n" +
"    String units \"s\";\n" +
"  }\n" +
"  SwellHeight {\n" +
"    Float64 colorBarMaximum 10.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_significant_height\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  SwellPeriod {\n" +
"    Float64 colorBarMaximum 20.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_period\";\n" +
"    String units \"s\";\n" +
"  }\n" +
"  WindWaveHeight {\n" +
"    Float64 colorBarMaximum 10.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_significant_height\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  WindWavePeriod {\n" +
"    Float64 colorBarMaximum 20.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_period\";\n" +
"    String units \"s\";\n" +
"  }\n" +
"  WaterTemperature {\n" +
"    Float64 colorBarMaximum 32.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Temperature\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_water_temperature\";\n" +
"    String units \"degrees_C\";\n" +
"  }\n" +
"  WaveDuration {\n" +
"    Float64 colorBarMaximum 20.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"s\";\n" +
"  }\n" +
"  CalculationMethod {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"  }\n" +
"  SamplingRate {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"Hz\";\n" +
"  }\n" +
"  NumberOfFrequencies {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"count\";\n" +
"  }\n" +
"  CenterFrequencies {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_wave_frequency\";\n" +
"    String units \"Hz\";\n" +
"  }\n" +
"  Bandwidths {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"Hz\";\n" +
"  }\n" +
"  SpectralEnergy {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_wave_variance_spectral_density\";\n" +
"    String units \"m2 Hz-1\";\n" +
"  }\n" +
"  MeanWaveDirectionPeakPeriod {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_to_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  SwellWaveDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_to_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  WindWaveDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_to_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  MeanWaveDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_to_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  PrincipalWaveDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_swell_wave_to_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  PolarCoordinateR1 {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  PolarCoordinateR2 {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  DirectionalWaveParameter {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  FourierCoefficientA1 {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_wave_variance_spectral_density\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  FourierCoefficientA2 {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_wave_variance_spectral_density\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  FourierCoefficientB1 {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_wave_variance_spectral_density\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  FourierCoefficientB2 {\n" +
"    String ioos_category \"Surface Waves\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Waves\";\n" +
"    String standard_name \"sea_surface_wave_variance_spectral_density\";\n" +
"    String units \"m\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 179.0;\n" +
"    Float64 geospatial_lat_max 60.8;\n" +
"    Float64 geospatial_lat_min -19.713;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 179.0;\n" +
"    Float64 geospatial_lon_min -177.75;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/" + datasetIdPrefix + "ndbcSosWaves.das\";\n" +
"    String infoUrl \"http://sdf.ndbc.noaa.gov/sos/\";\n" +
"    String institution \"NOAA NDBC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 60.8;\n" +
"    String sourceUrl \"http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\";\n" +
"    Float64 Southernmost_Northing -19.713;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NDBC SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have wave data.\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.  No more than thirty days of data can be requested.\";\n" +
"    String time_coverage_start \"2007-11-02T00:00:00Z\";\n" +
"    String title \"Buoy Data from the NOAA NDBC SOS Server - Waves\";\n" +
"    Float64 Westernmost_Easting -177.75;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            //test  station=    and superfluous string data > <
            String2.log("\n*** EDDTableFromSOS Waves test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::46013\"&time>=2008-08-01T14&time<=2008-08-01T17" +
                "&CalculationMethod>=\"Long\"&CalculationMethod<=\"Lonh\"", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosWaves", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, SignificantWaveHeight, DominantWavePeriod, AverageWavePeriod, SwellHeight, SwellPeriod, WindWaveHeight, WindWavePeriod, WaterTemperature, WaveDuration, CalculationMethod, SamplingRate, NumberOfFrequencies, CenterFrequencies, Bandwidths, SpectralEnergy, MeanWaveDirectionPeakPeriod, SwellWaveDirection, WindWaveDirection, MeanWaveDirection, PrincipalWaveDirection, PolarCoordinateR1, PolarCoordinateR2, DirectionalWaveParameter, FourierCoefficientA1, FourierCoefficientA2, FourierCoefficientB1, FourierCoefficientB2\n" +
"degrees_east, degrees_north, , m, UTC, m, s, s, m, s, m, s, degrees_C, s, , Hz, count, Hz, Hz, m2 Hz-1, degrees_true, degrees_true, degrees_true, degrees_true, degrees_true, degrees_true, degrees_true, degrees_true, m, m, m, m\n" +
//before 2008-10-29 (last week?), I think the DirectionalWaveParameters were different
//it changed again (~11am) vs earlier today (~9am)  2008-10-29
//see email to jeffDlB 2008-10-29
  "-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T14:50:00Z, 1.62, 14.81, 5.21, 1.12, 14.8, 1.17, 4.3, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.779157;0.372876;2.02274;0.714328;0.675131;0.296029;0.138154;0.605274;1.96737;1.16217;0.884235;0.462599;0.57436;0.504724;0.218129;0.38115;0.504237;0.45285;0.708456;0.626689;0.747685;0.883292;0.632856;0.448383;0.331531;0.123811;0.265022;0.214203;0.208534;0.21145;0.223251;0.114582;0.10544;0.130131;0.118191;0.0652535;0.0604571;0.0167055;0.0158453;0.00866108;0.00483522, 176.0, 176.0, 312.0, 158.0;336.0;188.0;11.0;78.0;263.0;189.0;176.0;196.0;212.0;249.0;182.0;267.0;292.0;299.0;306.0;290.0;299.0;304.0;294.0;301.0;304.0;320.0;314.0;311.0;303.0;312.0;317.0;315.0;307.0;316.0;314.0;305.0;317.0;306.0;311.0;310.0;303.0;294.0;308.0;298.0;302.0;303.0;113.0;127.0;113.0, 183.0;8.0;181.0;7.0;3.0;301.0;275.0;134.0;123.0;262.0;279.0;92.0;281.0;293.0;299.0;308.0;292.0;306.0;310.0;300.0;304.0;306.0;326.0;317.0;310.0;303.0;312.0;318.0;316.0;309.0;319.0;318.0;306.0;319.0;307.0;313.0;309.0;305.0;286.0;309.0;271.0;296.0;254.0;111.0;99.0;92.0, 0.212213;0.112507;0.304966;0.35902;0.254397;0.488626;0.263791;0.515939;0.462758;0.430386;0.497566;0.12097;0.497566;0.653071;0.826652;0.841777;0.702193;0.768824;0.797214;0.741445;0.797214;0.797214;0.826652;0.841777;0.857178;0.841777;0.938514;0.921652;0.905092;0.8118;0.826652;0.826652;0.78289;0.872861;0.905092;0.857178;0.88883;0.841777;0.768824;0.857178;0.677187;0.826652;0.629814;0.797214;0.677187;0.715041, 0.768824;0.78289;0.826652;0.497566;0.220049;0.263791;0.488626;0.372277;0.125437;0.386024;0.304966;0.278536;0.310546;0.365588;0.653071;0.66502;0.607386;0.488626;0.525379;0.430386;0.462758;0.446279;0.585756;0.544779;0.564896;0.534991;0.78289;0.768824;0.66502;0.43826;0.525379;0.462758;0.379088;0.715041;0.689577;0.585756;0.641337;0.544779;0.333904;0.554746;0.137339;0.564896;0.134872;0.372277;0.108501;0.340013, , , , , \n" +
//"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T14:50:00Z, 1.62, 14.81, 5.21, 1.12, 14.8, 1.17, 4.3, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.779157;0.372876;2.02274;0.714328;0.675131;0.296029;0.138154;0.605274;1.96737;1.16217;0.884235;0.462599;0.57436;0.504724;0.218129;0.38115;0.504237;0.45285;0.708456;0.626689;0.747685;0.883292;0.632856;0.448383;0.331531;0.123811;0.265022;0.214203;0.208534;0.21145;0.223251;0.114582;0.10544;0.130131;0.118191;0.0652535;0.0604571;0.0167055;0.0158453;0.00866108;0.00483522, 176.0, 176.0, 312.0, 158.0;336.0;188.0;11.0;78.0;263.0;189.0;176.0;196.0;212.0;249.0;182.0;267.0;292.0;299.0;306.0;290.0;299.0;304.0;294.0;301.0;304.0;320.0;314.0;311.0;303.0;312.0;317.0;315.0;307.0;316.0;314.0;305.0;317.0;306.0;311.0;310.0;303.0;294.0;308.0;298.0;302.0;303.0;113.0;127.0;113.0, 0.2;0.1;0.3;0.4;0.3;0.5;0.3;0.5;0.5;0.4;0.5;0.1;0.5;0.7;0.8;0.8;0.7;0.8;0.8;0.7;0.8;0.8;0.8;0.8;0.9;0.8;0.9;0.9;0.9;0.8;0.8;0.8;0.8;0.9;0.9;0.9;0.9;0.8;0.8;0.9;0.7;0.8;0.6;0.8;0.7;0.7, 0.212213;0.112507;0.304966;0.35902;0.254397;0.488626;0.263791;0.515939;0.462758;0.430386;0.497566;0.12097;0.497566;0.653071;0.826652;0.841777;0.702193;0.768824;0.797214;0.741445;0.797214;0.797214;0.826652;0.841777;0.857178;0.841777;0.938514;0.921652;0.905092;0.8118;0.826652;0.826652;0.78289;0.872861;0.905092;0.857178;0.88883;0.841777;0.768824;0.857178;0.677187;0.826652;0.629814;0.797214;0.677187;0.715041, 0.768824;0.78289;0.826652;0.497566;0.220049;0.263791;0.488626;0.372277;0.125437;0.386024;0.304966;0.278536;0.310546;0.365588;0.653071;0.66502;0.607386;0.488626;0.525379;0.430386;0.462758;0.446279;0.585756;0.544779;0.564896;0.534991;0.78289;0.768824;0.66502;0.43826;0.525379;0.462758;0.379088;0.715041;0.689577;0.585756;0.641337;0.544779;0.333904;0.554746;0.137339;0.564896;0.134872;0.372277;0.108501;0.340013, , , , , \n" +
//"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T14:50:00Z, 1.62, 14.81, 5.21, 1.12, 14.8, 1.17, 4.3, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.779157;0.372876;2.02274;0.714328;0.675131;0.296029;0.138154;0.605274;1.96737;1.16217;0.884235;0.462599;0.57436;0.504724;0.218129;0.38115;0.504237;0.45285;0.708456;0.626689;0.747685;0.883292;0.632856;0.448383;0.331531;0.123811;0.265022;0.214203;0.208534;0.21145;0.223251;0.114582;0.10544;0.130131;0.118191;0.0652535;0.0604571;0.0167055;0.0158453;0.00866108;0.00483522, 176.0, 176.0, 312.0, 158.0;336.0;188.0;11.0;78.0;263.0;189.0;176.0;196.0;212.0;249.0;182.0;267.0;292.0;299.0;306.0;290.0;299.0;304.0;294.0;301.0;304.0;320.0;314.0;311.0;303.0;312.0;317.0;315.0;307.0;316.0;314.0;305.0;317.0;306.0;311.0;310.0;303.0;294.0;308.0;298.0;302.0;303.0;113.0;127.0;113.0, 0.2;0.1;0.3;0.4;0.3;0.5;0.3;0.5;0.5;0.4;0.5;0.1;0.5;0.7;0.8;0.8;0.7;0.8;0.8;0.7;0.8;0.8;0.8;0.8;0.9;0.8;0.9;0.9;0.9;0.8;0.8;0.8;0.8;0.9;0.9;0.9;0.9;0.8;0.8;0.9;0.7;0.8;0.6;0.8;0.7;0.7, 0.212213;0.112507;0.304966;0.35902;0.254397;0.488626;0.263791;0.515939;0.462758;0.430386;0.497566;0.12097;0.497566;0.653071;0.826652;0.841777;0.702193;0.768824;0.797214;0.741445;0.797214;0.797214;0.826652;0.841777;0.857178;0.841777;0.938514;0.921652;0.905092;0.8118;0.826652;0.826652;0.78289;0.872861;0.905092;0.857178;0.88883;0.841777;0.768824;0.857178;0.677187;0.826652;0.629814;0.797214;0.677187;0.715041, 0.212213;0.112507;0.304966;0.35902;0.254397;0.488626;0.263791;0.515939;0.462758;0.430386;0.497566;0.12097;0.497566;0.653071;0.826652;0.841777;0.702193;0.768824;0.797214;0.741445;0.797214;0.797214;0.826652;0.841777;0.857178;0.841777;0.938514;0.921652;0.905092;0.8118;0.826652;0.826652;0.78289;0.872861;0.905092;0.857178;0.88883;0.841777;0.768824;0.857178;0.677187;0.826652;0.629814;0.797214;0.677187;0.715041, , , , , \n" +
  "-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T15:50:00Z, 1.52, 9.09, 4.98, 1.0, 9.1, 1.15, 5.6, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.244172;0.874391;0.950049;0.992051;0.292122;0.416385;0.116264;0.32567;1.0886;1.49577;0.707195;0.412901;0.383;0.336784;0.162325;0.507266;0.721374;0.521185;0.317616;0.580232;0.620904;0.720338;0.544952;0.400361;0.457406;0.340211;0.190368;0.295531;0.258054;0.13138;0.178793;0.207494;0.162191;0.0901461;0.101774;0.0468724;0.036226;0.0442694;0.0218615;0.0143249;0.00447678, 291.0, 291.0, 297.0, 245.0;180.0;16.0;1.0;157.0;253.0;192.0;221.0;234.0;193.0;171.0;182.0;331.0;297.0;291.0;287.0;304.0;296.0;295.0;307.0;306.0;297.0;309.0;309.0;310.0;301.0;321.0;300.0;296.0;315.0;295.0;305.0;311.0;311.0;312.0;312.0;311.0;309.0;305.0;310.0;317.0;304.0;303.0;125.0;132.0;122.0, 179.0;190.0;1.0;1.0;176.0;292.0;110.0;311.0;321.0;118.0;143.0;131.0;335.0;305.0;296.0;290.0;313.0;310.0;298.0;321.0;306.0;298.0;308.0;308.0;310.0;301.0;323.0;300.0;294.0;320.0;295.0;308.0;312.0;313.0;315.0;311.0;311.0;310.0;305.0;311.0;316.0;309.0;298.0;121.0;132.0;109.0, 0.153123;0.340013;0.288822;0.254397;0.372277;0.379088;0.400278;0.430386;0.299487;0.14767;0.228175;0.393087;0.534991;0.741445;0.78289;0.857178;0.702193;0.741445;0.797214;0.689577;0.857178;0.921652;0.905092;0.768824;0.905092;0.88883;0.921652;0.88883;0.872861;0.872861;0.872861;0.872861;0.921652;0.88883;0.857178;0.857178;0.88883;0.857178;0.78289;0.797214;0.728123;0.741445;0.826652;0.8118;0.872861;0.728123, 0.768824;0.575231;0.564896;0.689577;0.575231;0.497566;0.150371;0.259051;0.259051;0.728123;0.400278;0.393087;0.728123;0.689577;0.66502;0.677187;0.607386;0.400278;0.575231;0.534991;0.677187;0.8118;0.75501;0.365588;0.741445;0.741445;0.8118;0.689577;0.653071;0.653071;0.629814;0.618498;0.768824;0.689577;0.585756;0.596473;0.66502;0.629814;0.454444;0.415059;0.283632;0.216095;0.488626;0.488626;0.618498;0.236601, , , , , \n" +
//"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T15:50:00Z, 1.52, 9.09, 4.98, 1.0, 9.1, 1.15, 5.6, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.244172;0.874391;0.950049;0.992051;0.292122;0.416385;0.116264;0.32567;1.0886;1.49577;0.707195;0.412901;0.383;0.336784;0.162325;0.507266;0.721374;0.521185;0.317616;0.580232;0.620904;0.720338;0.544952;0.400361;0.457406;0.340211;0.190368;0.295531;0.258054;0.13138;0.178793;0.207494;0.162191;0.0901461;0.101774;0.0468724;0.036226;0.0442694;0.0218615;0.0143249;0.00447678, 291.0, 291.0, 297.0, 245.0;180.0;16.0;1.0;157.0;253.0;192.0;221.0;234.0;193.0;171.0;182.0;331.0;297.0;291.0;287.0;304.0;296.0;295.0;307.0;306.0;297.0;309.0;309.0;310.0;301.0;321.0;300.0;296.0;315.0;295.0;305.0;311.0;311.0;312.0;312.0;311.0;309.0;305.0;310.0;317.0;304.0;303.0;125.0;132.0;122.0, 0.2;0.3;0.3;0.3;0.4;0.4;0.4;0.4;0.3;0.1;0.2;0.4;0.5;0.7;0.8;0.9;0.7;0.7;0.8;0.7;0.9;0.9;0.9;0.8;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.8;0.8;0.7;0.7;0.8;0.8;0.9;0.7, 0.153123;0.340013;0.288822;0.254397;0.372277;0.379088;0.400278;0.430386;0.299487;0.14767;0.228175;0.393087;0.534991;0.741445;0.78289;0.857178;0.702193;0.741445;0.797214;0.689577;0.857178;0.921652;0.905092;0.768824;0.905092;0.88883;0.921652;0.88883;0.872861;0.872861;0.872861;0.872861;0.921652;0.88883;0.857178;0.857178;0.88883;0.857178;0.78289;0.797214;0.728123;0.741445;0.826652;0.8118;0.872861;0.728123, 0.768824;0.575231;0.564896;0.689577;0.575231;0.497566;0.150371;0.259051;0.259051;0.728123;0.400278;0.393087;0.728123;0.689577;0.66502;0.677187;0.607386;0.400278;0.575231;0.534991;0.677187;0.8118;0.75501;0.365588;0.741445;0.741445;0.8118;0.689577;0.653071;0.653071;0.629814;0.618498;0.768824;0.689577;0.585756;0.596473;0.66502;0.629814;0.454444;0.415059;0.283632;0.216095;0.488626;0.488626;0.618498;0.236601, , , , , \n" +
//"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T15:50:00Z, 1.52, 9.09, 4.98, 1.0, 9.1, 1.15, 5.6, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.244172;0.874391;0.950049;0.992051;0.292122;0.416385;0.116264;0.32567;1.0886;1.49577;0.707195;0.412901;0.383;0.336784;0.162325;0.507266;0.721374;0.521185;0.317616;0.580232;0.620904;0.720338;0.544952;0.400361;0.457406;0.340211;0.190368;0.295531;0.258054;0.13138;0.178793;0.207494;0.162191;0.0901461;0.101774;0.0468724;0.036226;0.0442694;0.0218615;0.0143249;0.00447678, 291.0, 291.0, 297.0, 245.0;180.0;16.0;1.0;157.0;253.0;192.0;221.0;234.0;193.0;171.0;182.0;331.0;297.0;291.0;287.0;304.0;296.0;295.0;307.0;306.0;297.0;309.0;309.0;310.0;301.0;321.0;300.0;296.0;315.0;295.0;305.0;311.0;311.0;312.0;312.0;311.0;309.0;305.0;310.0;317.0;304.0;303.0;125.0;132.0;122.0, 0.2;0.3;0.3;0.3;0.4;0.4;0.4;0.4;0.3;0.1;0.2;0.4;0.5;0.7;0.8;0.9;0.7;0.7;0.8;0.7;0.9;0.9;0.9;0.8;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.8;0.8;0.7;0.7;0.8;0.8;0.9;0.7, 0.153123;0.340013;0.288822;0.254397;0.372277;0.379088;0.400278;0.430386;0.299487;0.14767;0.228175;0.393087;0.534991;0.741445;0.78289;0.857178;0.702193;0.741445;0.797214;0.689577;0.857178;0.921652;0.905092;0.768824;0.905092;0.88883;0.921652;0.88883;0.872861;0.872861;0.872861;0.872861;0.921652;0.88883;0.857178;0.857178;0.88883;0.857178;0.78289;0.797214;0.728123;0.741445;0.826652;0.8118;0.872861;0.728123, 0.153123;0.340013;0.288822;0.254397;0.372277;0.379088;0.400278;0.430386;0.299487;0.14767;0.228175;0.393087;0.534991;0.741445;0.78289;0.857178;0.702193;0.741445;0.797214;0.689577;0.857178;0.921652;0.905092;0.768824;0.905092;0.88883;0.921652;0.88883;0.872861;0.872861;0.872861;0.872861;0.921652;0.88883;0.857178;0.857178;0.88883;0.857178;0.78289;0.797214;0.728123;0.741445;0.826652;0.8118;0.872861;0.728123, , , , , \n" +
  "-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T16:50:00Z, 1.49, 14.81, 5.11, 1.01, 14.8, 1.1, 4.8, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.724702;0.909481;2.34661;0.698133;0.516662;0.499779;0.284884;0.407779;0.94326;1.08406;0.29313;0.464502;0.346171;0.393304;0.327266;0.531525;0.423195;0.328752;0.332852;0.702979;0.627516;0.379029;0.603016;0.337529;0.385623;0.308393;0.266641;0.207837;0.0681764;0.212742;0.18737;0.138199;0.122643;0.130927;0.0889706;0.0656523;0.0608267;0.0359928;0.0115031;0.0100742;0.00469153, 175.0, 175.0, 309.0, 287.0;208.0;76.0;353.0;123.0;193.0;205.0;175.0;198.0;155.0;196.0;246.0;285.0;304.0;297.0;324.0;298.0;296.0;299.0;303.0;299.0;298.0;304.0;306.0;309.0;304.0;311.0;299.0;317.0;301.0;308.0;314.0;314.0;325.0;315.0;301.0;312.0;322.0;306.0;305.0;324.0;302.0;326.0;119.0;139.0;137.0, 350.0;193.0;12.0;13.0;171.0;135.0;151.0;161.0;162.0;158.0;143.0;301.0;313.0;303.0;304.0;321.0;320.0;303.0;302.0;306.0;299.0;300.0;307.0;305.0;311.0;302.0;316.0;299.0;317.0;299.0;308.0;317.0;320.0;346.0;313.0;304.0;312.0;327.0;305.0;306.0;331.0;299.0;333.0;115.0;139.0;143.0, 0.177024;0.224075;0.333904;0.393087;0.273532;0.310546;0.299487;0.534991;0.506669;0.43826;0.249826;0.327905;0.340013;0.8118;0.78289;0.585756;0.653071;0.741445;0.826652;0.826652;0.857178;0.797214;0.841777;0.857178;0.905092;0.88883;0.872861;0.921652;0.905092;0.88883;0.872861;0.872861;0.8118;0.728123;0.872861;0.905092;0.857178;0.8118;0.872861;0.826652;0.841777;0.826652;0.857178;0.78289;0.797214;0.8118, 0.596473;0.544779;0.488626;0.228175;0.316228;0.506669;0.479847;0.415059;0.372277;0.236601;0.228175;0.346234;0.479847;0.66502;0.534991;0.534991;0.288822;0.554746;0.728123;0.641337;0.728123;0.525379;0.653071;0.575231;0.768824;0.702193;0.618498;0.741445;0.741445;0.689577;0.629814;0.618498;0.430386;0.400278;0.629814;0.75501;0.629814;0.446279;0.641337;0.488626;0.585756;0.454444;0.618498;0.340013;0.454444;0.422653, , , , , \n";
//"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T16:50:00Z, 1.49, 14.81, 5.11, 1.01, 14.8, 1.1, 4.8, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.724702;0.909481;2.34661;0.698133;0.516662;0.499779;0.284884;0.407779;0.94326;1.08406;0.29313;0.464502;0.346171;0.393304;0.327266;0.531525;0.423195;0.328752;0.332852;0.702979;0.627516;0.379029;0.603016;0.337529;0.385623;0.308393;0.266641;0.207837;0.0681764;0.212742;0.18737;0.138199;0.122643;0.130927;0.0889706;0.0656523;0.0608267;0.0359928;0.0115031;0.0100742;0.00469153, 175.0, 175.0, 309.0, 287.0;208.0;76.0;353.0;123.0;193.0;205.0;175.0;198.0;155.0;196.0;246.0;285.0;304.0;297.0;324.0;298.0;296.0;299.0;303.0;299.0;298.0;304.0;306.0;309.0;304.0;311.0;299.0;317.0;301.0;308.0;314.0;314.0;325.0;315.0;301.0;312.0;322.0;306.0;305.0;324.0;302.0;326.0;119.0;139.0;137.0, 0.2;0.2;0.3;0.4;0.3;0.3;0.3;0.5;0.5;0.4;0.2;0.3;0.3;0.8;0.8;0.6;0.7;0.7;0.8;0.8;0.9;0.8;0.8;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.8;0.7;0.9;0.9;0.9;0.8;0.9;0.8;0.8;0.8;0.9;0.8;0.8;0.8, 0.177024;0.224075;0.333904;0.393087;0.273532;0.310546;0.299487;0.534991;0.506669;0.43826;0.249826;0.327905;0.340013;0.8118;0.78289;0.585756;0.653071;0.741445;0.826652;0.826652;0.857178;0.797214;0.841777;0.857178;0.905092;0.88883;0.872861;0.921652;0.905092;0.88883;0.872861;0.872861;0.8118;0.728123;0.872861;0.905092;0.857178;0.8118;0.872861;0.826652;0.841777;0.826652;0.857178;0.78289;0.797214;0.8118, 0.596473;0.544779;0.488626;0.228175;0.316228;0.506669;0.479847;0.415059;0.372277;0.236601;0.228175;0.346234;0.479847;0.66502;0.534991;0.534991;0.288822;0.554746;0.728123;0.641337;0.728123;0.525379;0.653071;0.575231;0.768824;0.702193;0.618498;0.741445;0.741445;0.689577;0.629814;0.618498;0.430386;0.400278;0.629814;0.75501;0.629814;0.446279;0.641337;0.488626;0.585756;0.454444;0.618498;0.340013;0.454444;0.422653, , , , , \n";
//"-123.32, 38.23, urn:x-noaa:def:station:noaa.nws.ndbc::46013, NaN, 2008-08-01T16:50:00Z, 1.49, 14.81, 5.11, 1.01, 14.8, 1.1, 4.8, NaN, NaN, Longuet-Higgins (1964), NaN, 46, 0.0325;0.0375;0.0425;0.0475;0.0525;0.0575;0.0625;0.0675;0.0725;0.0775;0.0825;0.0875;0.0925;0.1000;0.1100;0.1200;0.1300;0.1400;0.1500;0.1600;0.1700;0.1800;0.1900;0.2000;0.2100;0.2200;0.2300;0.2400;0.2500;0.2600;0.2700;0.2800;0.2900;0.3000;0.3100;0.3200;0.3300;0.3400;0.3500;0.3650;0.3850;0.4050;0.4250;0.4450;0.4650;0.4850, 0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0050;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0100;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200;0.0200, 0;0;0;0;0;0.724702;0.909481;2.34661;0.698133;0.516662;0.499779;0.284884;0.407779;0.94326;1.08406;0.29313;0.464502;0.346171;0.393304;0.327266;0.531525;0.423195;0.328752;0.332852;0.702979;0.627516;0.379029;0.603016;0.337529;0.385623;0.308393;0.266641;0.207837;0.0681764;0.212742;0.18737;0.138199;0.122643;0.130927;0.0889706;0.0656523;0.0608267;0.0359928;0.0115031;0.0100742;0.00469153, 175.0, 175.0, 309.0, 287.0;208.0;76.0;353.0;123.0;193.0;205.0;175.0;198.0;155.0;196.0;246.0;285.0;304.0;297.0;324.0;298.0;296.0;299.0;303.0;299.0;298.0;304.0;306.0;309.0;304.0;311.0;299.0;317.0;301.0;308.0;314.0;314.0;325.0;315.0;301.0;312.0;322.0;306.0;305.0;324.0;302.0;326.0;119.0;139.0;137.0, 0.2;0.2;0.3;0.4;0.3;0.3;0.3;0.5;0.5;0.4;0.2;0.3;0.3;0.8;0.8;0.6;0.7;0.7;0.8;0.8;0.9;0.8;0.8;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.9;0.8;0.7;0.9;0.9;0.9;0.8;0.9;0.8;0.8;0.8;0.9;0.8;0.8;0.8, 0.177024;0.224075;0.333904;0.393087;0.273532;0.310546;0.299487;0.534991;0.506669;0.43826;0.249826;0.327905;0.340013;0.8118;0.78289;0.585756;0.653071;0.741445;0.826652;0.826652;0.857178;0.797214;0.841777;0.857178;0.905092;0.88883;0.872861;0.921652;0.905092;0.88883;0.872861;0.872861;0.8118;0.728123;0.872861;0.905092;0.857178;0.8118;0.872861;0.826652;0.841777;0.826652;0.857178;0.78289;0.797214;0.8118, 0.177024;0.224075;0.333904;0.393087;0.273532;0.310546;0.299487;0.534991;0.506669;0.43826;0.249826;0.327905;0.340013;0.8118;0.78289;0.585756;0.653071;0.741445;0.826652;0.826652;0.857178;0.797214;0.841777;0.857178;0.905092;0.88883;0.872861;0.921652;0.905092;0.88883;0.872861;0.872861;0.8118;0.728123;0.872861;0.905092;0.857178;0.8118;0.872861;0.826652;0.841777;0.826652;0.857178;0.78289;0.797214;0.8118, , , , , \n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            //same test, but with  superfluous string data regex test
            String2.log("\n*** EDDTableFromSOS Waves test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::46013\"&time>=2008-08-01T14&time<=2008-08-01T17" +
                "&CalculationMethod=~\"(zztop|Long.*)\"", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosWaves", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error ndbcSosWaves. NDBC SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * This should work, but server is in flux so it often breaks.
     *
     * @throws Throwable if trouble
     */
    public static void testNdbcSosWind(String datasetIdPrefix) throws Throwable {
        testVerboseOn();
        EDDTable eddTable;               
        String name, tName, results, expected, userDapQuery;

        try { 
            eddTable = (EDDTable)oneFromDatasetXml(datasetIdPrefix + "ndbcSosWind");  

            String2.log("\n*** EDDTableFromSOS ndbc wind .das\n");
            String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);
            tName = eddTable.makeNewFileForDapQuery(null, null, "", EDStatic.fullTestCacheDirectory, 
                eddTable.className() + "_ndbcSosWind", ".das"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"Attributes {\n" +
" s {\n" +
"  longitude {\n" +
"    String _CoordinateAxisType \"Lon\";\n" +
"    Float64 actual_range -177.75, 179.0;\n" +
"    String axis \"X\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Longitude\";\n" +
"    String standard_name \"longitude\";\n" +
"    String units \"degrees_east\";\n" +
"  }\n" +
"  latitude {\n" +
"    String _CoordinateAxisType \"Lat\";\n" +
"    Float64 actual_range 14.36, 61.082;\n" +
"    String axis \"Y\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Latitude\";\n" +
"    String standard_name \"latitude\";\n" +
"    String units \"degrees_north\";\n" +
"  }\n" +
"  station_id {\n" +
"    String ioos_category \"Identifier\";\n" +
"    String long_name \"Station ID\";\n" +
"  }\n" +
"  altitude {\n" +
"    String _CoordinateAxisType \"Height\";\n" +
"    String _CoordinateZisPositive \"up\";\n" +
"    String axis \"Z\";\n" +
"    String ioos_category \"Location\";\n" +
"    String long_name \"Altitude\";\n" +
"    String positive \"up\";\n" +
"    String standard_name \"altitude\";\n" +
"    String units \"m\";\n" +
"  }\n" +
"  time {\n" +
"    String _CoordinateAxisType \"Time\";\n" +
"    Float64 actual_range 1.1939616e+9, NaN;\n" +
"    String axis \"T\";\n" +
"    String ioos_category \"Time\";\n" +
"    String long_name \"Time\";\n" +
"    String standard_name \"time\";\n" +
"    String time_origin \"01-JAN-1970 00:00:00\";\n" +
"    String units \"seconds since 1970-01-01T00:00:00Z\";\n" +
"  }\n" +
"  WindSpeed {\n" +
"    Float64 colorBarMaximum 15.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String standard_name \"wind_speed\";\n" +
"    String units \"m s-1\";\n" +
"  }\n" +
"  WindDirection {\n" +
"    Float64 colorBarMaximum 360.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String standard_name \"wind_from_direction\";\n" +
"    String units \"degrees_true\";\n" +
"  }\n" +
"  WindVerticalVelocity {\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String units \"m s-1\";\n" +
"  }\n" +
"  WindGust {\n" +
"    Float64 colorBarMaximum 30.0;\n" +
"    Float64 colorBarMinimum 0.0;\n" +
"    String ioos_category \"Wind\";\n" +
"    String observedProperty \"http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml#Winds\";\n" +
"    String standard_name \"wind_speed_of_gust\";\n" +
"    String units \"m s-1\";\n" +
"  }\n" +
" }\n" +
"  NC_GLOBAL {\n" +
"    String cdm_data_type \"Station\";\n" +
"    String Conventions \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
"    Float64 Easternmost_Easting 179.0;\n" +
"    Float64 geospatial_lat_max 61.082;\n" +
"    Float64 geospatial_lat_min 14.36;\n" +
"    String geospatial_lat_units \"degrees_north\";\n" +
"    Float64 geospatial_lon_max 179.0;\n" +
"    Float64 geospatial_lon_min -177.75;\n" +
"    String geospatial_lon_units \"degrees_east\";\n" +
"    String geospatial_vertical_positive \"up\";\n" +
"    String geospatial_vertical_units \"m\";\n" +
"    String history \"" + today + " http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\n" +
today + " " + EDStatic.erddapUrl + //in tests, always use non-https url
                "/tabledap/" + datasetIdPrefix + "ndbcSosWind.das\";\n" +
"    String infoUrl \"http://sdf.ndbc.noaa.gov/sos/\";\n" +
"    String institution \"NOAA NDBC\";\n" +
"    String license \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
"    Float64 Northernmost_Northing 61.082;\n" +
"    String sourceUrl \"http://sdf" + datasetIdPrefix + ".ndbc.noaa.gov/sos/server.php\";\n" +
"    Float64 Southernmost_Northing 14.36;\n" +
"    String standard_name_vocabulary \"CF-11\";\n" +
"    String summary \"The NOAA NDBC SOS server is part of the IOOS DIF SOS Project.  The stations in this dataset have wind data.\n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST include constraints for the longitude, latitude, time, and/or station_id variables.  No more than thirty days of data can be requested.\";\n" +
"    String time_coverage_start \"2007-11-02T00:00:00Z\";\n" +
"    String title \"Buoy Data from the NOAA NDBC SOS Server - Wind\";\n" +
"    Float64 Westernmost_Easting -177.75;\n" +
"  }\n" +
"}\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);

            String2.log("\n*** EDDTableFromSOS Wind test get one station .CSV data\n");
            tName = eddTable.makeNewFileForDapQuery(null, null, 
                "&station_id=\"urn:x-noaa:def:station:noaa.nws.ndbc::41004\"&time>=2008-08-01T00:00&time<=2008-08-01T04", 
                EDStatic.fullTestCacheDirectory, eddTable.className() + "_ndbcSosWind", ".csv"); 
            results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
            expected = 
"longitude, latitude, station_id, altitude, time, WindSpeed, WindDirection, WindVerticalVelocity, WindGust\n" +
"degrees_east, degrees_north, , m, UTC, m s-1, degrees_true, m s-1, m s-1\n" +
"-79.09, 32.5, urn:x-noaa:def:station:noaa.nws.ndbc::41004, NaN, 2008-08-01T00:50:00Z, 10.1, 229.0, NaN, 12.6\n" +
"-79.09, 32.5, urn:x-noaa:def:station:noaa.nws.ndbc::41004, NaN, 2008-08-01T01:50:00Z, 9.3, 232.0, NaN, 11.3\n" +
"-79.09, 32.5, urn:x-noaa:def:station:noaa.nws.ndbc::41004, NaN, 2008-08-01T02:50:00Z, 7.8, 237.0, NaN, 11.5\n" +
"-79.09, 32.5, urn:x-noaa:def:station:noaa.nws.ndbc::41004, NaN, 2008-08-01T03:50:00Z, 8.0, 236.0, NaN, 9.3\n";
            Test.ensureEqual(results, expected, "RESULTS=\n" + results);
        } catch (Throwable t) {
            String2.getStringFromSystemIn(MustBe.throwableToString(t) + 
                "\nUnexpected error. NDBC SOS Server is in flux." +
                "\nPress ^C to stop or Enter to continue..."); 
        }
    }

    /**
     * testOostethys should work.
     *
     * @throws Throwable if trouble
     */
    public static void testOostethys() throws Throwable {
        String2.log("\n*** testOostethys");
        testVerboseOn();
        double tLon, tLat;
        String name, tName, results, expected, userDapQuery;
        String error = "";
        String today = Calendar2.getCurrentISODateTimeStringLocal().substring(0, 10);

        EDDTable eddTable = (EDDTable)oneFromDatasetXml("gomoosBuoy"); //should work

        //getEmpiricalMinMax just do once
        //useful for SOS: get alt values
        //eddTable.getEmpiricalMinMax("2007-02-01", "2007-02-01", false, true);
        //if (true) System.exit(1);

        //test sos-server values
        String2.log("nOfferings=" + eddTable.sosNOfferings());
        String2.log(eddTable.sosOfferingName(0) + "  lon=" +
            eddTable.sosMinLongitude(0) + ", " +
            eddTable.sosMaxLongitude(0) + " lat=" +
            eddTable.sosMinLatitude(0) + ", " +
            eddTable.sosMaxLatitude(0) + " time=" +
            eddTable.sosMinTime(0) + ", " +
            eddTable.sosMaxTime(0));
        String2.log(String2.toCSVString(eddTable.sosObservedProperties()));

        Test.ensureEqual(eddTable.sosNOfferings(), 12, "");
        Test.ensureEqual(eddTable.sosOfferingName(0), "A01", "");
        Test.ensureEqual(eddTable.sosMinLongitude(0), -70.5683, "");  //2008-07-25 was .5655, pre 2008-10-09 was .5658, 2009-03-23 changed
        Test.ensureEqual(eddTable.sosMaxLongitude(0), -70.5683, "");
        Test.ensureEqual(eddTable.sosMinLatitude(0), 42.5223, "");  //2008-07-25 was 42.5232, pre 2008-10-09 was .5226, 2009-03-23 changed
        Test.ensureEqual(eddTable.sosMaxLatitude(0), 42.5223, "");
        Test.ensureEqual(eddTable.sosMinTime(0), 9.947196E8, "");
        Test.ensureEqual(eddTable.sosMaxTime(0), Double.NaN, "");
        Test.ensureEqual(String2.toCSVString(eddTable.sosObservedProperties()), 
            "http://marinemetadata.org/cf#sea_water_salinity, " +
            "http://marinemetadata.org/cf#sea_water_temperature", 
            "");

        userDapQuery = "longitude,latitude,altitude,time,station_id," +
            "sea_water_temperature,sea_water_salinity" +
            "&longitude>=-70&longitude<=-69&latitude>=43&latitude<=44" + 
            "&time>=2007-07-04T00:00&time<=2007-07-04T01:00";
        tName = eddTable.makeNewFileForDapQuery(null, null, userDapQuery, EDStatic.fullTestCacheDirectory, 
            eddTable.datasetID() + "_Data", ".nc"); 
        results = NcHelper.dumpString(EDStatic.fullTestCacheDirectory + tName, true);
        expected =
"netcdf gomoosBuoy_Data.nc {\n" +
" dimensions:\n" +
"   row = 20;\n" +
"   StringLengthForVariable4 = 3;\n" +
" variables:\n" +
"   double longitude(row=20);\n" +
"     :_CoordinateAxisType = \"Lon\";\n" +
"     :actual_range = -69.9877, -69.3578; // double\n" +
"     :axis = \"X\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Longitude\";\n" +
"     :standard_name = \"longitude\";\n" +
"     :units = \"degrees_east\";\n" +
"   double latitude(row=20);\n" +
"     :_CoordinateAxisType = \"Lat\";\n" +
"     :actual_range = 43.7148, 43.7813; // double\n" +
"     :axis = \"Y\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Latitude\";\n" +
"     :standard_name = \"latitude\";\n" +
"     :units = \"degrees_north\";\n" +
"   double altitude(row=20);\n" +
"     :_CoordinateAxisType = \"Height\";\n" +
"     :_CoordinateZisPositive = \"up\";\n" +
"     :actual_range = -50.0, -1.0; // double\n" +
"     :axis = \"Z\";\n" +
"     :ioos_category = \"Location\";\n" +
"     :long_name = \"Altitude\";\n" +
"     :positive = \"up\";\n" +
"     :standard_name = \"altitude\";\n" +
"     :units = \"m\";\n" +
"   double time(row=20);\n" +
"     :_CoordinateAxisType = \"Time\";\n" +
"     :actual_range = 1.1835072E9, 1.1835108E9; // double\n" +
"     :axis = \"T\";\n" +
"     :ioos_category = \"Time\";\n" +
"     :long_name = \"Time\";\n" +
"     :standard_name = \"time\";\n" +
"     :time_origin = \"01-JAN-1970 00:00:00\";\n" +
"     :units = \"seconds since 1970-01-01T00:00:00Z\";\n" +
"   char station_id(row=20, StringLengthForVariable4=3);\n" +
"     :ioos_category = \"Identifier\";\n" +
"     :long_name = \"Station ID\";\n" +
"   double sea_water_temperature(row=20);\n" +
"     :actual_range = 5.81699991226196, 13.8900003433228; // double\n" +
"     :colorBarMaximum = 32.0; // double\n" +
"     :colorBarMinimum = 0.0; // double\n" +
"     :ioos_category = \"Temperature\";\n" +
"     :long_name = \"Sea Water Temperature\";\n" +
"     :observedProperty = \"http://marinemetadata.org/cf#sea_water_temperature\";\n" +
"     :standard_name = \"sea_water_temperature\";\n" +
"     :units = \"degree_C\";\n" +
"   double sea_water_salinity(row=20);\n" +
"     :actual_range = 29.192590713501, 32.1141357421875; // double\n" +
"     :colorBarMaximum = 37.0; // double\n" +
"     :colorBarMinimum = 32.0; // double\n" +
"     :ioos_category = \"Salinity\";\n" +
"     :long_name = \"Sea Water Salinity\";\n" +
"     :observedProperty = \"http://marinemetadata.org/cf#sea_water_salinity\";\n" +
"     :standard_name = \"sea_water_salinity\";\n" +
"     :units = \"PSU\";\n" +
"\n" +
" :cdm_data_type = \"Station\";\n" +
" :Conventions = \"COARDS, CF-1.0, Unidata Dataset Discovery v1.0\";\n" +
" :Easternmost_Easting = -69.3578; // double\n" +
" :geospatial_lat_max = 43.7813; // double\n" +
" :geospatial_lat_min = 43.7148; // double\n" +
" :geospatial_lat_units = \"degrees_north\";\n" +
" :geospatial_lon_max = -69.3578; // double\n" +
" :geospatial_lon_min = -69.9877; // double\n" +
" :geospatial_lon_units = \"degrees_east\";\n" +
" :geospatial_vertical_max = -1.0; // double\n" +
" :geospatial_vertical_min = -50.0; // double\n" +
" :geospatial_vertical_positive = \"up\";\n" +
" :geospatial_vertical_units = \"m\";\n" +
" :history = \"" + today + " http://www.gomoos.org/cgi-bin/sos/oostethys_sos.cgi\n" +
today + " http://127.0.0.1:8080/cwexperimental/tabledap/gomoosBuoy.nc?longitude,latitude,altitude,time,station_id,sea_water_temperature,sea_water_salinity&longitude>=-70&longitude<=-69&latitude>=43&latitude<=44&time>=2007-07-04T00:00&time<=2007-07-04T01:00\";\n" +
" :id = \"gomoosBuoy_Data\";\n" +
" :infoUrl = \"http://www.oostethys.org/ogc-oceans-interoperability-experiment\";\n" +
" :institution = \"GoMOOS\";\n" +
" :license = \"The data may be used and redistributed for free but is not intended \n" +
"for legal use, since it may contain inaccuracies. Neither the data \n" +
"Contributor, ERD, NOAA, nor the United States Government, nor any \n" +
"of their employees or contractors, makes any warranty, express or \n" +
"implied, including warranties of merchantability and fitness for a \n" +
"particular purpose, or assumes any legal liability for the accuracy, \n" +
"completeness, or usefulness, of this information.\";\n" +
" :Northernmost_Northing = 43.7813; // double\n" +
" :observationDimension = \"row\";\n" +
" :sourceUrl = \"http://www.gomoos.org/cgi-bin/sos/oostethys_sos.cgi\";\n" +
" :Southernmost_Northing = 43.7148; // double\n" +
" :standard_name_vocabulary = \"CF-11\";\n" +
" :summary = \"[Normally, the summary describes the dataset. Here, it describes \n" +
"the server.] \n" +
"The OCEANS IE -- formally approved as an OGC Interoperability \n" +
"Experiment in December 2006 -- engages data managers and scientists \n" +
"in the Ocean-Observing community to advance their understanding and \n" +
"application of various OGC specifications, solidify demonstrations \n" +
"for Ocean Science application areas, harden software \n" +
"implementations, and produce candidate OGC Best Practices documents \n" +
"that can be used to inform the broader ocean-observing community. \n" +
"\n" +
"Because of the nature of SOS requests, requests for data MUST \n" +
"include constraints for the longitude, latitude, time, and/or \n" +
"station_id variables. \n" +
"\n" +
"Initiators: SURA (lead), Texas A&M University, MBARI, GoMOOS and \n" +
"Unidata.\n" +
"\n" +
"Specific goals:\n" +
"* Compare Sensor Observation Service (SOS) from the OGC's Sensor \n" +
"  Web Enablement (SWE) initiative to the Web Feature Service (WFS) \n" +
"  as applied to ocean data in a variety of data formats including \n" +
"  text files, netCDF files, relational databases, and possibly \n" +
"  native sensor output; (see Experiment #1 for details)\n" +
"* Make use of semantic mediation via Semantic Web technologies to \n" +
"  allow plurality of identification for source types (platforms \n" +
"  and sensors) and phenomena types;\n" +
"* Test aggregation services and caching strategies to provide \n" +
"  efficient queries; \n" +
"* Explore possible enhancements of THREDDS server, so that THREDDS \n" +
"  resident data sources might be made available via SOS or WFS;\";\n" +
" :time_coverage_end = \"2007-07-04T01:00:00Z\";\n" +
" :time_coverage_start = \"2007-07-04T00:00:00Z\";\n" +
" :title = \"Buoy Data from the GoMOOS SOS Server\";\n" +
" :Westernmost_Easting = -69.9877; // double\n" +
" data:\n" +
"longitude =\n" +
"  {-69.8891, -69.8891, -69.8891, -69.8891, -69.8891, -69.8891, -69.9877, -69.9877, -69.9877, -69.9877, -69.9877, -69.3578, -69.3578, -69.3578, -69.3578, -69.3578, -69.3578, -69.3578, -69.3578, -69.3578}\n" +
"latitude =\n" +
"  {43.7813, 43.7813, 43.7813, 43.7813, 43.7813, 43.7813, 43.7628, 43.7628, 43.7628, 43.7628, 43.7628, 43.7148, 43.7148, 43.7148, 43.7148, 43.7148, 43.7148, 43.7148, 43.7148, 43.7148}\n" +
"altitude =\n" +
"  {-1.0, -20.0, -35.0, -1.0, -20.0, -35.0, -1.0, -10.0, -1.0, -1.0, -10.0, -1.0, -2.0, -20.0, -50.0, -1.0, -1.0, -2.0, -20.0, -50.0}\n" +
"time =\n" +
"  {1.1835072E9, 1.1835072E9, 1.1835072E9, 1.1835108E9, 1.1835108E9, 1.1835108E9, 1.1835072E9, 1.1835072E9, 1.183509E9, 1.1835108E9, 1.1835108E9, 1.1835072E9, 1.1835072E9, 1.1835072E9, 1.1835072E9, 1.183509E9, 1.1835108E9, 1.1835108E9, 1.1835108E9, 1.1835108E9}\n" +
"station_id =\"D01\", \"D01\", \"D01\", \"D01\", \"D01\", \"D01\", \"D02\", \"D02\", \"D02\", \"D02\", \"D02\", \"E01\", \"E01\", \"E01\", \"E01\", \"E01\", \"E01\", \"E01\", \"E01\", \"E01\"\n" +
"sea_water_temperature =\n" +
"  {13.5, 9.28999996185303, 8.30000019073486, 13.5100002288818, 9.06999969482422, 8.51000022888184, 13.460000038147, 11.8699998855591, 13.4200000762939, 13.3699998855591, 11.6599998474121, 13.7600002288818, 13.7034998, 7.65000009536743, 5.84700012207031, 13.8900003433228, 13.8500003814697, 13.8292704, 7.57000017166138, 5.81699991226196}\n" +
"sea_water_salinity =\n" +
"  {29.3161296844482, 31.24924659729, NaN, 29.192590713501, 31.2218112945557, NaN, 29.9242057800293, 31.2601585388184, 29.9195117950439, 29.930046081543, 31.2559909820557, 31.1920852661133, NaN, 31.8228702545166, 32.1141357421875, 31.1868896484375, 31.1843872070312, NaN, 31.833927154541, 32.0988731384277}\n" +
"}\n";

//before 12/17/07 was -69.8876
//before 5/19/08 was -69.9879, 43.7617, changed again 2009-03-26
    Test.ensureEqual(results, expected, results);

        //data for mapExample  (no time)  just uses station table data
        tName = eddTable.makeNewFileForDapQuery(null, null, "longitude,latitude&longitude>-70", 
            EDStatic.fullTestCacheDirectory, eddTable.className() + "MapNT", ".csv");
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = //this changes a little periodically (NOT GOOD, stations have 1 point (which applies to historic data and changes!)
/*before 2009-01-12 this was
"longitude, latitude\n" +
"degrees_east, degrees_north\n" +
"-69.8891, 43.7813\n" +
"-69.9885, 43.7612\n" +
"-69.3578, 43.7148\n" +
"-68.998, 44.0548\n" +
"-68.1087, 44.1058\n" +
"-67.0123, 44.8893\n" +
"-66.5541, 43.6276\n" +
"-67.8798, 43.4907\n" +
"-65.9118, 42.3279\n";
before 2009-03-23 this was
"longitude, latitude\n" +
"degrees_east, degrees_north\n" +
"-69.8891, 43.7813\n" +
"-69.9885, 43.7612\n" +
"-69.3578, 43.7148\n" +
"-68.998, 44.0548\n" +
"-68.1085, 44.1057\n" +
"-67.0123, 44.8893\n" +
"-66.5541, 43.6276\n" +
"-67.8798, 43.4907\n" +
"-65.9118, 42.3279\n"; 
before 2009-03-26 was
"longitude, latitude\n" +
"degrees_east, degrees_north\n" +
"-69.8891, 43.7813\n" +
"-69.9885, 43.7612\n" +
"-69.3578, 43.7148\n" +
"-68.998, 44.0548\n" +
"-68.108, 44.1052\n" +
"-67.0123, 44.8893\n" +
"-66.5541, 43.6276\n" +
"-67.8798, 43.4907\n" +
"-65.9118, 42.3279\n";
"longitude, latitude\n" +
"degrees_east, degrees_north\n" +
"-69.8891, 43.7813\n" +
"-69.9877, 43.7628\n" +
"-69.3578, 43.7148\n" +
"-68.9982, 44.0555\n" +
"-68.108, 44.1052\n" +
"-67.0123, 44.8893\n" +
"-66.5541, 43.6276\n" +
"-67.8798, 43.4907\n" +
"-65.9118, 42.3279\n";*/
"longitude, latitude\n" +
"degrees_east, degrees_north\n" +
"-69.8891, 43.7813\n" +
"-69.9877, 43.7628\n" +
"-69.3578, 43.7148\n" +
"-68.9982, 44.0555\n" +
"-68.108, 44.1052\n" +
"-67.0123, 44.8893\n" +
"-66.5541, 43.6276\n" +
"-67.8798, 43.4907\n" +
"-65.9096, 42.3276\n";
       Test.ensureEqual(results, expected, "\nresults=\n" + results);  


        //data for mapExample (with time
        tName = eddTable.makeNewFileForDapQuery(null, null, "longitude,latitude&time=2007-12-11", 
            EDStatic.fullTestCacheDirectory, eddTable.className() + "MapWT", ".csv");
        results = new String((new ByteArray(EDStatic.fullTestCacheDirectory + tName)).toArray());
        //String2.log(results);
        expected = //changed a little on 2008-02-28, 2008-05-19, 2008-09-24, 2008-10-09 2009-01-12   -70.5658->-70.5655 42.5226->42.5232
           //change a little on 2009-03-26
"longitude, latitude\n" +
"degrees_east, degrees_north\n" +
"-70.5683, 42.5223\n" +
"-70.5683, 42.5223\n" +
"-70.5683, 42.5223\n" +
"-70.5683, 42.5223\n" +
"-70.4281, 43.1805\n" +
"-70.4281, 43.1805\n" +
"-70.4281, 43.1805\n" +
"-70.4281, 43.1805\n" +
"-70.0584, 43.5677\n" +
"-70.0584, 43.5677\n" +
"-70.0584, 43.5677\n" +
"-69.8891, 43.7813\n" +
"-69.8891, 43.7813\n" +
"-69.8891, 43.7813\n" +
"-69.8891, 43.7813\n" +
"-69.9877, 43.7628\n" +
"-69.9877, 43.7628\n" +
"-69.3578, 43.7148\n" +
"-69.3578, 43.7148\n" +
"-68.9982, 44.0555\n" +
"-68.9982, 44.0555\n" +
"-68.9982, 44.0555\n" +
"-68.9982, 44.0555\n" +
"-68.108, 44.1052\n" +
"-68.108, 44.1052\n" +
"-68.108, 44.1052\n" +
"-68.108, 44.1052\n" +
"-67.0123, 44.8893\n" +
"-67.0123, 44.8893\n" +
"-67.0123, 44.8893\n" +
"-65.9096, 42.3276\n" +
"-65.9096, 42.3276\n" +
"-65.9096, 42.3276\n" +
"-65.9096, 42.3276\n" +
"-65.9096, 42.3276\n" +
"-65.9096, 42.3276\n" +
"-65.9096, 42.3276\n";
        Test.ensureEqual(results, expected, "\nresults=\n" + results);  
    }

    //VAST
    //http://vast.uah.edu:8080/ows5-dev/weather?request=GetObservation&version=1.0&offering=WEATHER_DATA&observedProperty=WEATHER_OBSERVABLES&time=2004-04-01T05:00:00Z/2004-04-01T06:00:00Z&format=application/com-xml
    public static EDDTableFromSOS vastDataset() throws Throwable {
        return new EDDTableFromSOS(
            "uahWeather", //String tWebSiteID, String tDatasetID, 
            null, null,
            (new Attributes())
                .add("title",         "Weather Data from the UAH-VAST SOS Server")  //for demo: I emphasize source
                .add("summary",       "[standard]")
                .add("institution",   "UAH")
                .add("infoUrl",       "http://vast.uah.edu/joomla/index.php?option=com_content&task=view&id=58&Itemid=64")
                .add("cdm_data_type", CDM_STATION)
                .add("standard_name_vocabulary", FileNameUtility.getStandardNameVocabulary())
                .add("license",       "[standard]"),
            "longitude",
            "latitude",
            "altitude", 
                15, 15, //source alt min,max     from eddTable.getEmpiricalMinMax("2007-02-01", "2007-02-01", false, true); below
                1, //it deals with altitude 
            "time", EDVTimeStamp.ISO8601TZ_FORMAT,
            new Object[][]{  //dataVariables: sourceName, destName, type, addAttributes
                //unclear if source data values are doubles or floats; they aren't bruised
                {"temperature", "air_temperature", "double", (new Attributes())
                    .add("ioos_category", "Temperature")
                    .add("long_name", "Air Temperature")
                    .add("observedProperty", "WEATHER_OBSERVABLES") 
                    .add("standard_name", "air_temperature")
                    .add("units", "degree_C")},
                {"pressure", "air_pressure", "double", (new Attributes()) 
                    .add("ioos_category", "Other")
                    .add("long_name", "Air Pressure")
                    .add("observedProperty", "WEATHER_OBSERVABLES") 
                    .add("standard_name", "air_pressure")
                    .add("units", "hPa")},   //hPa is udunits for mbar 
                {"windSpeed", "wind_speed", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Wind Speed")
                    .add("observedProperty", "WEATHER_OBSERVABLES") 
                    .add("standard_name", "wind_speed")
                    .add("units", "km h-1")},
                {"windDirection", "wind_from_direction", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Wind From Direction")  //or wind_to_direction???
                    .add("observedProperty", "WEATHER_OBSERVABLES") 
                    .add("standard_name", "wind_from_direction")  //or wind_to_direction???
                    .add("units", "degrees_true")},
                }, 
            60, //int tReloadEveryNMinutes,
            "http://vast.uah.edu:8080/ows5-dev/weather", "WEATHER\\_DATA", false);
    }

    /**
     * testVast used to work BUT THEN STOPPED 2007-10-23.
     * PROBABLY SOME SMALL PROBLEM -- FIX IT WHEN SOS STABILIZES
     *
     * @throws Throwable if trouble
     */
    public static void testVast() throws Throwable {
        String2.log("\n*** testVast");
        testVerboseOn();

        //vast test 
        //info: http://vast.uah.edu/joomla/index.php?option=com_content&task=view&id=58&Itemid=64
        //example from web site:
        //  http://vast.uah.edu:8080/ows5-dev/weather?request=GetObservation&version=0.0.31
        //  &offering=WEATHER_DATA&time=2004-04-01T05:00:00Z/2004-04-01T06:00:00Z&format=application/com-xml
        EDDTable eddTable = vastDataset();

        //getEmpiricalMinMax just do once
        //useful for SOS: get alt values
        //eddTable.getEmpiricalMinMax("2007-02-01", "2007-02-01", false, true);
        //if (true) System.exit(1);

        /** needs work
        TableWriterAllInMemory tw = new TableWriterAllInMemory();
        eddTable.getDataForDapQuery("longitude,latitude,altitude,time,station_id," +
            "air_temperature,air_pressure,wind_speed,wind_from_direction" +
            "&time>=2004-04-01T06:58&time<=2004-04-01T06:59", tw);
        Table table = tw.cumulativeTable();
        String2.log("table=" + table);
        String results = table.toString();
        String expected = 
"    Row       longitude       latitude       altitude           time     station_id air_temperatur   air_pressure     wind_speed wind_from_dire\n" +
"      0           -86.6           34.7             15     1080802680   WEATHER_DATA           21.1        1059.38           29.6          137.3\n" +
"      1           -86.6           34.7             15     1080802690   WEATHER_DATA           21.1        1059.33           29.6          137.2\n" +
"      2           -86.6           34.7             15     1080802700   WEATHER_DATA           21.1        1059.28           29.6          137.2\n" +
"      3           -86.6           34.7             15     1080802710   WEATHER_DATA           21.1        1059.22           29.6          137.1\n" +
"      4           -86.6           34.7             15     1080802720   WEATHER_DATA             21        1059.17           29.6          137.1\n" +
"      5           -86.6           34.7             15     1080802730   WEATHER_DATA             21        1059.11           29.6            137\n" +
"      6           -86.6           34.7             15     1080802740   WEATHER_DATA             21        1059.06           29.6            137\n";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, 
            "\nfull results=" + results + "\nexpected=" + expected);
        */


    }

    //TAMU SOS via MapServer
    //is mentioned at http://www.oostethys.org/ogc-oceans-interoperability-experiment/experiment-1
    //info http://mapserver.gis.umn.edu/docs/howto/sos_server
    //Personal contact for person in charge: gerry.creager@tamu.edu
    //As of 2007-10-08 this may be not-really ready:
    //  See note at http://www.oostethys.org/development/meetings/meetings/ saying
    //  "It has been decided that SOS client testing will be done against the OOSTethys 
    //  reference server instead of MapServer's SOS because the OOSTethys server uses a newer 
    //  version of SOS. In addition, the SOS Map on OpenIOOS.org now makes use of the OOSTethys
    //  Registry to generate a list of servers to display.)". So I gather that you and the 
    //  OOSTethys group are basically waiting for a future version of MapServer and upgraded SOS support."
    //Creager said MapServer will be updated to SOS 1.0 sometime relatively soon.
    //http://sos-web.tamu.edu/sos-cgi/madis?service=sos&request=getcapabilities&version=0.1.2
    public static EDDTableFromSOS tamuDataset() throws Throwable {
        return new EDDTableFromSOS(
            "tamuWeather", //String tWebSiteID, String tDatasetID, 
            null, null,
            (new Attributes())
                .add("title",         "Weather Data from the TAMU MapServer SOS")  //for demo: I emphasize source
                .add("summary",       "[standard]")
                .add("institution",   "TAMU")
                .add("infoUrl",       "???")
                .add("cdm_data_type", CDM_STATION)
                .add("standard_name_vocabulary", FileNameUtility.getStandardNameVocabulary())
                .add("license",       "[standard]"),
            "longitude",
            "latitude",
            "depth", 
                Double.NaN, Double.NaN, //source alt min,max   from eddTable.getEmpiricalMinMax("2007-02-01", "2007-02-01", false, true); below
                -1, //it deals with depth
            "time", EDVTimeStamp.ISO8601TZ_FORMAT,
            new Object[][]{  //dataVariables: sourceName, destName, type, addAttributes
                {"airPressure", "air_pressure", "double", (new Attributes()) 
                    .add("ioos_category", "Other")
                    .add("long_name", "Air Pressure")
                    .add("observedProperty", "http://marinemetadata.org/cf#air_pressure_at_sea_level") 
                    .add("standard_name", "air_pressure")
                    .add("units", "hPa")},   //hPa is udunits for mbar 
                {"airTemperature", "air_temperature", "double", (new Attributes())
                    .add("ioos_category", "Temperature")
                    .add("long_name", "Air Temperature")
                    .add("observedProperty", "http://marinemetadata.org/cf#air_temperature") 
                    .add("standard_name", "air_temperature")
                    .add("units", "degree_C")},
                {"seaTemperature", "sea_surface_temperature", "double", (new Attributes())
                    .add("ioos_category", "Temperature")
                    .add("long_name", "Sea Surface Temperature")
                    .add("observedProperty", "http://marinemetadata.org/cf#sea_surface_temperature") 
                    .add("standard_name", "sea_surface_temperature")
                    .add("units", "degree_C")},
                {"wavePeriod", "sea_surface_wind_wave_period", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Sea Surface Wind Wave Period")
                    .add("observedProperty", "http://marinemetadata.org/cf#sea_surface_wind_wave_period") 
                    .add("standard_name", "sea_surface_wind_wave_period")
                    .add("units", "???")},
                {"waveHeight", "sea_surface_wind_wave_significant_height", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Sea Surface Wind Wave Significant Height")
                    .add("observedProperty", "http://marinemetadata.org/cf#sea_surface_wind_wave_significant_height") 
                    .add("standard_name", "sea_surface_wind_wave_significant_height")
                    .add("units", "m")},
                {"tendency_of_air_pressure", "tendency_of_air_pressure", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Tendency of Air Pressure")
                    .add("observedProperty", "http://marinemetadata.org/cf#tendency_of_air_pressure") 
                    .add("standard_name", "tendency_of_air_pressure")
                    .add("units", "???")},
                {"visibility_in_air", "visibility_in_air", "double", (new Attributes()) 
                    .add("ioos_category", "Other")
                    .add("long_name", "Wisibility in Air")
                    .add("observedProperty", "http://marinemetadata.org/cf#visibility_in_air") 
                    .add("standard_name", "visibility_in_air")
                    .add("units", "???")},
                {"tendency_of_air_pressure", "tendency_of_air_pressure", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Tendency of Air Pressure")
                    .add("observedProperty", "http://marinemetadata.org/cf#tendency_of_air_pressure") 
                    .add("standard_name", "tendency_of_air_pressure")
                    .add("units", "???")},
                {"windDirection", "wind_from_direction", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Wind From Direction")  //or wind_to_direction???
                    .add("observedProperty", "http://marinemetadata.org/cf#wind_from_direction") 
                    .add("standard_name", "wind_from_direction")  //or wind_to_direction???
                    .add("units", "degrees_true")},
                {"windSpeed", "wind_speed", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Wind Speed")
                    .add("observedProperty", "http://marinemetadata.org/cf#wind_speed") 
                    .add("standard_name", "wind_speed")
                    .add("units", "km h-1")},
                {"gustSpeed", "wind_speed_of_gust", "double", (new Attributes()) 
                    .add("ioos_category", "Wind")
                    .add("long_name", "Wind Speed of Gust")
                    .add("observedProperty", "http://marinemetadata.org/cf#wind_speed_of_gust") 
                    .add("standard_name", "wind_speed_of_gust")
                    .add("units", "km h-1")},
                //this one has odd name and similar to first variable
                {"airPressureAtSeaLevel", "air_pressure2", "double", (new Attributes()) 
                    .add("ioos_category", "Other")
                    .add("long_name", "Air Pressure at Sea Level")
                    .add("observedProperty", "http://marinemetadata.org/cf#demo_one_wk_ndbc_44029_air_pressure_at_sea_level") 
                    .add("standard_name", "air_pressure_at_sea_level")
                    .add("units", "hPa")},   //hPa is udunits for mbar 
                }, 
            60, //int tReloadEveryNMinutes,
            "http://sos-web.tamu.edu/sos-cgi/madis", ".*", false);
    }

    /**
     * testTamu not yet working
     *
     * @throws Throwable if trouble
     */
    public static void testTamu() throws Throwable {
        String2.log("\n*** testTamu");
        testVerboseOn();

        //not yet ready to for testing. wait for tamu to use newer version of MapServer.
        //original reference: http://www.oostethys.org/ogc-oceans-interoperability-experiment/experiment-1
        //example from web site:
        //  http://vast.uah.edu:8080/ows5-dev/weather?request=GetObservation&version=0.0.31
        //  &offering=WEATHER_DATA&time=2004-04-01T05:00:00Z/2004-04-01T06:00:00Z&format=application/com-xml
        EDDTable eddTable = tamuDataset();

        //getEmpiricalMinMax just do once
        //useful for SOS: get alt values
        //eddTable.getEmpiricalMinMax("2007-02-01", "2007-02-01", false, true);
        //if (true) System.exit(1);
        
        /* needs work
        TableWriterAllInMemory tw = new TableWriterAllInMemory();
        eddTable.getDataForDapQuery("longitude,latitude,altitude,time,station_id," +
            "air_pressure,air_temperature,sea_surface_temperature,sea_surface_wind_wave_period," +
            "sea_surface_wind_wave_significant_height,tendency_of_air_pressure," +
            "visibility_in_air,tendency_of_air_pressure,wind_from_direction,wind_speed," +
            "wind_speed_of_gust,wind_from_direction,air_pressure2" +
            "&time>=2007-07-04T06:00&time<=2007-07-04T07:00", tw);
        Table table = tw.cumulativeTable();
        String2.log("table=" + table);
        String results = table.toString();
        String expected = "...";
        Test.ensureEqual(results.substring(results.length() - expected.length()), expected, 
            "\nfull results=" + results + "\nexpected=" + expected);
        */
    }

    /** 
     * NOT FINISHED. But useful for what it does so far.
     * This generates a rough draft of the datasets.xml entry for an EDDTableFromSOS.
     * The XML can then be edited by hand and added to the datasets.xml file.
     *
     * @param tSourceUrl
     * @param sortColumnsByName
     * @throws Throwable if trouble
     */
    public static String generateDatasetsXml(String tSourceUrl, 
        boolean sortColumnsByName) throws Throwable {

        String2.log("EDDTableFromSos.generateDatasetsXml" +
            "\ntSourceUrl=" + tSourceUrl);
        StringBuffer sb = new StringBuffer();

        String tUrl = tSourceUrl + "?service=SOS&request=GetCapabilities"; 
            //* @param tSosVersion e.g., 1.0.0 or 0.0.31
            //&version=" + tSosVersion; //or don't specify version?
//        if (reallyVerbose) String2.log(SSR.getUrlResponseString(tUrl));
        SimpleXMLReader xmlReader = new SimpleXMLReader(SSR.getUrlInputStream(tUrl));
        xmlReader.nextTag();
        String tags = xmlReader.allTags();
        String sosPrefix = "";
        boolean ioosServer = false;
        if (xmlReader.tag(0).equals("Capabilities")) {
            sosPrefix = "";              //ioosServer
            ioosServer = true;
        } else if (xmlReader.tag(0).equals("sos:Capabilities")) {
            sosPrefix = "sos:"; //oostethys
        } else {
            xmlReader.close();
            throw new RuntimeException("First SOS capabilities tag=" + 
                tags + " should have been <Capabilities> or <sos:Capabilities>.");
        }
        String sosVersion = xmlReader.attributeValue("version"); //e.g., "0.0.31"
        if (verbose) String2.log("  sosPrefix=" + sosPrefix + 
            "\n  getCapabilities version=" + sosVersion); 
        String offeringTag = "<" + 
            sosPrefix + "Capabilities><" + 
            sosPrefix + "Contents><" + 
            sosPrefix + "ObservationOfferingList><" + 
            sosPrefix + "ObservationOffering>";
        String offeringEndTag = "<" + 
            sosPrefix + "Capabilities><" + 
            sosPrefix + "Contents><" + 
            sosPrefix + "ObservationOfferingList></" + 
            sosPrefix + "ObservationOffering>";

        //gather station info and uniqueObservedProperties
        StringArray stationIDs = new StringArray();
        StringArray stationHasObsProp = new StringArray();
        StringArray uniqueObsProp = new StringArray();

        String tStationID = "";
        StringBuffer tStationObsPropList = new StringBuffer();

        do {
            //process the tags
            //String2.log("tags=" + tags + xmlReader.content());

            if (tags.startsWith(offeringTag)) {
                String endOfTag = tags.substring(offeringTag.length());
                String content = xmlReader.content();
                String error = null;
//String2.log("endOfTag=" + endOfTag + xmlReader.content());


/* separate phenomena
            <sos:ObservationOffering xmlns:xlink="http://www.w3.org/1999/xlink" gml:id="A01">
				<gml:description> Latest data from Mooring A01 from the Gulf of Maine Ocean Observing System (GoMOOS), located in the Gulf of Maine Massachusetts Bay </gml:description>
				<gml:name>A01</gml:name>
				<gml:boundedBy>
					<gml:Envelope>
						<gml:lowerCorner srsName="urn:ogc:def:crs:EPSG:6.5:4329">42.5277 -70.5665</gml:lowerCorner>
						<gml:upperCorner srsName="urn:ogc:def:crs:EPSG:6.5:4329">42.5277 -70.5665</gml:upperCorner>
					</gml:Envelope>
				</gml:boundedBy>
				<sos:time>
					<gml:TimePeriod gml:id="AVAILABLE_OFFERING_TIME">
						<gml:beginPosition>2001-07-10T06:30:00Z</gml:beginPosition>
						<gml:endPosition indeterminatePosition="now"/>
						<gml:timeInterval unit="hour">1</gml:timeInterval>
					</gml:TimePeriod>
				</sos:time>
				<sos:procedure xlink:href="urn:gomoos.org:source.mooring#A01"/>
                <sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_temperature"/>
                <sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_salinity"/>
				<sos:featureOfInterest xlink:href="urn:something:bodyOfWater"/>
				<sos:responseFormat>application/com-xml</sos:responseFormat>
   			</sos:ObservationOffering>
*/
/* or compositePhenomenon...
                <sos:observedProperty>
                    <swe:CompositePhenomenon gml:id="WEATHER_OBSERVABLES">
                        <gml:name>Weather measurements</gml:name>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#AirTemperature"/>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#AtmosphericPressure"/>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#WindSpeed"/>
                        <swe:component xlink:href="http://vast.uah.edu/dictionary/phenomena.xml#WindDirection"/>
                    </swe:CompositePhenomenon>
                </sos:observedProperty>
*/

                if (endOfTag.equals("")) {
                    //String2.log("startTag");
                    tStationID = "";
                    tStationObsPropList.setLength(0);

                } else if (endOfTag.equals("</gml:name>")) { //ioosServer and OOSTethys have this
                    //e.g., 44004
                    tStationID = content;

                    if (ioosServer) { 
                        //remove 
                        String pre = ":station:";
                        int prePo = tStationID.indexOf(pre);
                        if (prePo >= 0)
                            tStationID = tStationID.substring(prePo + pre.length());
                    }

                } else if (endOfTag.equals("<" + sosPrefix + "observedProperty>")) {
                    //handle non-composite observedProperty
                    //<sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_temperature"/>
                    String tXlink = xmlReader.attributeValue("xlink:href"); //without quotes
                    if (tXlink != null) {
                        int opPo = uniqueObsProp.indexOf(tXlink);   
                        if (opPo < 0) {
                            opPo = uniqueObsProp.size();
                            uniqueObsProp.add(tXlink);
                        }
                        while (tStationObsPropList.length() <= opPo)
                            tStationObsPropList.append(' ');
                        tStationObsPropList.setCharAt(opPo, 'X');
                    }

                } else if (endOfTag.equals("<" + sosPrefix + "observedProperty><swe:CompositePhenomenon>")) {
                    //handle composite observedProperty
                    //<swe:CompositePhenomenon gml:id="WEATHER_OBSERVABLES">
                    String tXlink = xmlReader.attributeValue("gml:id"); //without quotes
                    if (tXlink != null) {
                        int opPo = uniqueObsProp.indexOf(tXlink);   
                        if (opPo < 0) {
                            opPo = uniqueObsProp.size();
                            uniqueObsProp.add(tXlink);
                        }
                        while (tStationObsPropList.length() <= opPo)
                            tStationObsPropList.append(' ');
                        tStationObsPropList.setCharAt(opPo, 'X');
                    }
                }

                //handle the error
                if (error != null)
                    throw new RuntimeException(
                        "Error on capabilities xml line #" + xmlReader.lineNumber() + 
                        " stationID=" + tStationID + ": " + error);

            //end of a station
            } else if (tags.startsWith(offeringEndTag)) {
                //String2.log("endTag");
                if (tStationID.length() > 0 && 
                    tStationObsPropList.indexOf("X") >= 0) {
                    stationIDs.add(tStationID);
                    stationHasObsProp.add(tStationObsPropList.toString());
                } else {
                    String2.log("Invalid Station id=" + tStationID + 
                        " tStationObsPropList='" + tStationObsPropList + "'");
                }
                tStationID = "";
                tStationObsPropList.setLength(0);
            }

            //get the next tag
            xmlReader.nextTag();
            tags = xmlReader.allTags();
        } while (!tags.startsWith("</"));

        xmlReader.close();      

        //*** basically, make a Table which has the dataset's info
/*        Table table = new Table();

        //get global attributes and ensure required entries are present (perhaps as "???")
        OpendapHelper.getAttributes(das, "GLOBAL", table.globalAttributes());
        addDummyRequiredGlobalAttributesForDatasetsXml(table.globalAttributes(), null);

        //make a column for each var for each observedProperty
        //table.addColumn(table.nColumns(), varName, new StringArray(), atts);

        //sort the column names?
        if (sortColumnsByName)
            table.sortColumnsByName();

        //write the information
        sb.append(directionsForDatasetsXml());
        sb.append(
"    <dataset type=\"EDDTableFromDapSequence\" datasetID=\"???\">\n" +
"        <sourceUrl>" + url + "</sourceUrl>\n" +
"        <reloadEveryNMinutes>???" + DEFAULT_RELOAD_EVERY_N_MINUTES + "</reloadEveryNMinutes>\n" +
"        <altitudeMetersPerSourceUnit>???1</altitudeMetersPerSourceUnit>\n");
        sb.append(attsForDatasetsXml(table.globalAttributes(), "        "));

        sb.append(variablesForDatasetsXml(table, "dataVariable", false));
        sb.append(
"    </dataset>\n");
*/
        //write the station/obsProp info
        int longestStationID = Math.max(7, stationIDs.maxStringLength());
        int longestHasProp = stationHasObsProp.maxStringLength(); 
        sb.append("\n" + 
            String2.left("  n  Station", longestStationID) + "  Has ObservedProperty\n" +
            "---  " + String2.makeString('-', longestStationID) +
                "  0123456789012345678901234567890123456789\n");
        for (int si = 0; si < stationIDs.size(); si++) {
            sb.append(
                String2.right("" + si, 3) + "  " +
                String2.left(stationIDs.get(si), longestStationID) + "  " + 
                stationHasObsProp.get(si) + "\n");
        }

        //list the props
        sb.append("\n  n  ObservedProperty\n" +
                    "---  --------------------------------------------------\n");
        for (int op = 0; op < uniqueObsProp.size(); op++) 
            sb.append(String2.right("" + op, 3) + "  " + uniqueObsProp.get(op) + "\n");

        return sb.toString();
    }


    /** 
     * NOT FINISHED. NOT ACTIVE.
     * This parses a phenomenon dictionary and generates a HashMap
     * with key=phenomenonURI, value=StringArray of non-composite phenomena
     * (variables) that it is comprised of.
     *
     * @param url
     * @param hashMap the hashMap to which with phenomena will be added
     * @throws Throwable if trouble
     */
    public static void getPhenomena(String url, HashMap hashMap) throws Throwable {

        String2.log("EDDTableFromSOS.getPhenomena" +
            "\nurl=" + url);
        StringBuffer sb = new StringBuffer();

        SimpleXMLReader xmlReader = new SimpleXMLReader(SSR.getUrlInputStream(url));
        xmlReader.nextTag();
        String tags = xmlReader.allTags();
        String sosPrefix = "";
        boolean ioosServer = false;
        String startTag = "<gml:Dictionary>";
        String endTag   = "</gml:Dictionary>";

        if (!tags.equals(startTag)) {
            xmlReader.close();
            throw new RuntimeException("First PhenomenaDictionary tag=" + 
                tags + " should have been " + startTag);
        }

        String codeSpace = null;
        String tID = null;
        StringArray tComponents = null;

        do {
            //process the tags
            //String2.log("tags=" + tags + xmlReader.content());

            String endOfTag = tags.substring(startTag.length());
            String content = xmlReader.content();
            String error = null;
//String2.log("endOfTag=" + endOfTag + xmlReader.content());


/* separate phenomena
                <sos:observedProperty xlink:href="http://marinemetadata.org/cf#sea_water_temperature"/>

<gml:Dictionary ...>
    <gml:identifier codeSpace="urn:x-noaa:ioos:def:phenomenonNames">PhenomenaDictionary</gml:identifier>
    ...
    */
            if (endOfTag.equals("<gml:identifier>")) {
                codeSpace = xmlReader.attributeValue("codeSpace"); 
                if (verbose) String2.log("  codeSpace=" + codeSpace);

            //phenomenon
            /*  <gml:definitionMember >
                    <swe:Phenomenon gml:id="WaterTemperature">
                        <gml:description>Temperature of the water.</gml:description>
                        <gml:identifier codeSpace="urn:x-noaa:ioos:def:phenomenonNames">WaterTemperature</gml:identifier>
                    </swe:Phenomenon>
                </gml:definitionMember> 
            */
            } else if (endOfTag.equals("<gml:definitionMember><swe:Phenomenon>")) {
                tID = xmlReader.attributeValue("gml:id");
                if (tID == null)
                    xmlReader.throwException("<swe:Phenomenon> tag has no gml:id.");
                tComponents = new StringArray();
                tComponents.add(codeSpace + "#" + tID);
                hashMap.put(codeSpace + "#" + tID, tComponents); 


            //compositePhenomenon
            /*  <gml:definitionMember >
                    <swe:CompositePhenomenon gml:id="Winds" dimension="4">
                        <gml:description>Wind origin direction and speed.</gml:description>
                        <gml:identifier codeSpace="urn:x-noaa:ioos:def:phenomenonNames">Winds</gml:identifier>
                        <swe:base xlink:href="#MinimumWinds"/>
                        <swe:component xlink:href="#WindGust"/>
                    </swe:CompositePhenomenon>
                </gml:definitionMember> 
            */
            } else if (endOfTag.equals("<gml:definitionMember><swe:CompositePhenomenon>")) {
                tID = xmlReader.attributeValue("gml:id"); 
                tComponents = new StringArray();

            } else if (endOfTag.equals("<gml:definitionMember><swe:CompositePhenomenon><swe:base>") ||
                       endOfTag.equals("<gml:definitionMember><swe:CompositePhenomenon><swe:component>")) {
                String href = xmlReader.attributeValue("xlink:href"); 
                if (href == null)
                    String2.log("WARNING: on XML line #" + xmlReader.lineNumber() + 
                        ": " + endOfTag + " doesn't have an xlink:href.");
                else {
                    //get referenced item's components
                    href = (href.startsWith("#")? codeSpace : "") + href;
                    StringArray tsa = (StringArray)hashMap.get(href);
                    if (tsa == null) 
                        xmlReader.throwException(href + " isn't already defined in this document " +
                            "(Bob's assumption is that components of composite will be already defined).");
                    tComponents.append(tsa);
                }

            } else if (endOfTag.equals("<gml:definitionMember></swe:CompositePhenomenon>")) {
                hashMap.put(codeSpace + "#" + tID, tComponents);
            }

            //get the next tag
            xmlReader.nextTag();
            tags = xmlReader.allTags();
        } while (!tags.startsWith("</gml:Dictionary>"));

        xmlReader.close();      

    }

    /**
     * NOT FINISHED. NOT ACTIVE.
     * This tests getPhenomena.
     */
    public static void testGetPhenomena() throws Throwable {
        String2.log("testGetPhenomena");
        testVerboseOn();
        HashMap hashMap = new HashMap();
        getPhenomena("http://www.csc.noaa.gov/ioos/schema/IOOS-DIF/IOOS/0.6.1/dictionaries/phenomenaDictionary.xml", 
            hashMap);
        String sar[] = String2.toStringArray(hashMap.keySet().toArray());
        for (int i = 0; i < sar.length; i++) 
            String2.log(sar[i] + "\n" + ((StringArray)hashMap.get(sar[i])).toNewlineString() + "\n");

    }

    /**
     * This runs all of the tests for the ndbc test server.
     */
    public static void testNdbcTestServer() throws Throwable {
        String2.log("\n****************** EDDTableFromSos.testTestServer() *****************\n");
        testVerboseOn();

        testNdbcSosCurrents("test", false);  //doLongTest
        testNdbcSosLongTime("test");
        testNdbcSosSalinity("test");  
        testNdbcSosWLevel("test");  
        testNdbcSosWTemp("test");  
        testNdbcSosWaves("test");
        testNdbcSosWind("test");
    }

    /**
     * This runs all of the tests for the nos test server.
     */
    public static void testNosTestServer() throws Throwable {
        String2.log("\n****************** EDDTableFromSos.testTestServer() *****************\n");
        testVerboseOn();

        testNosSosATemp("test");
        //testNosSosATempAllStations("test"); //long test
        testNosSosATempStationList("test");
        testNosSosBPres("test");
        testNosSosCond("test");
        testNosSosSalinity("test");
        testNosSosWLevel("test");  
        testNosSosWTemp("test");  
        testNosSosWind("test");
    }

    /**
     * This runs all of the tests for this class.
     */
    public static void test() throws Throwable {
        String2.log("\n****************** EDDTableFromSos.test() *****************\n");
        testVerboseOn();

        // usually run
        testOostethys();

        testNdbcSosCurrents("", false);  //doLongTest
        testNdbcSosLongTime("");
        testNdbcSosSalinity("");  
        testNdbcSosWLevel("");  
        testNdbcSosWTemp("");  
        testNdbcSosWaves("");
        testNdbcSosWind("");

        testNosSosATemp("");
        //testNosSosATempAllStations(""); //long test
        testNosSosATempStationList("");
        testNosSosBPres("");
        testNosSosCond("");
        testNosSosSalinity("");
        testNosSosWLevel("");  
        testNosSosWTemp("");  
        testNosSosWind("");
        // */

 
        //*** usually not run
        //testNdbcTestServer();
        //testGetPhenomena();
        //String2.log(generateDatasetsXml("http://sdf.ndbc.noaa.gov/sos/server.php", "1.0.0", false));
        //testVast(); //not working;
        //testTamu(); //not working;

        //testNosSosCurrents(false);  //doLongTest   NOT FINISHED. EDDTableFromSOS can't handle binned data. 
    }
}
