/* 
 * TableWriterAll Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.array.Attributes;
import com.cohort.array.DoubleArray;
import com.cohort.array.FloatArray;
import com.cohort.array.IntArray;
import com.cohort.array.PrimitiveArray;
import com.cohort.array.ShortArray;
import com.cohort.array.StringArray;
import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.Math2;
import com.cohort.util.MustBe;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.EDV;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.Arrays;

/**
 * TableWriterAll provides a way to write a table to a series of 
 * DataOutputStreams (one per column) 
 * in chunks so that the whole table is available but doesn't have to be in memory 
 * at one time.
 * This is used by EDDTable.
 *
 * <p>This is different from most TableWriters in that finish() doesn't 
 * write the data anywhere (to an outputStream or to another tableWriter), 
 * it just makes all of the data available.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2007-08-23
 */
public class TableWriterAll extends TableWriter {

    //set by constructor
    protected String dir;
    protected String fileNameNoExt;

    //set firstTime
    //POLICY: because this procedure may be used in more than one thread,
    //do work on unique temp files names using randomInt, then rename to proper file name.
    //If procedure fails half way through, there won't be a half-finished file.
    protected int randomInt = Math2.random(Integer.MAX_VALUE);
    protected DataOutputStream[] columnStreams;
    protected long totalNRows = 0; 

    protected Table cumulativeTable; //set by writeAllAndFinish, if used

    /**
     * The constructor.
     *
     * @param tDir a private cache directory for storing the intermediate files,
     *    usually EDStatic.fullCacheDirectory + datasetID + "/"
     * @param tFileNameNoExt is the fileName without dir or extension (used as basis for temp files)
     */
    public TableWriterAll(String tDir, String tFileNameNoExt) {
        super(null);
        dir = File2.addSlash(tDir);
        fileNameNoExt = tFileNameNoExt;
    }


    /**
     * This adds the current contents of table (a chunk of data) to the columnStreams.
     * This calls ensureCompatible each time it is called.
     * If this is the first time this is called, this does first time things
     *   (e.g., open the columnStreams).
     * The number of columns, the column names, and the types of columns 
     *   must be the same each time this is called.
     *
     * <p>The table should have missing values stored as destinationMissingValues
     * or destinationFillValues.
     * This implementation doesn't change them.
     *
     * @param table  with destinationValues
     * @throws Throwable if trouble
     */
    public void writeSome(Table table) throws Throwable {
        if (table.nRows() == 0) 
            return;

        //ensure the table's structure is the same as before
        boolean firstTime = columnNames == null;
        ensureCompatible(table);

        //do firstTime stuff
        int nColumns = table.nColumns();
        if (firstTime) {
            columnStreams = new DataOutputStream[nColumns];
            for (int col = 0; col < nColumns; col++) {
                columnStreams[col] = new DataOutputStream(new BufferedOutputStream(
                    new FileOutputStream(
                    dir + fileNameNoExt + "." + randomInt + "." + columnNames[col])));
            }
        }

        //do everyTime stuff
        //write the data
        for (int col = 0; col < nColumns; col++) 
            table.getColumn(col).writeDos(columnStreams[col]);
        totalNRows += table.nRows();
    }

    
    /**
     * This writes any end-of-file info to the stream and flushes the stream.
     *
     * @throws Throwable if trouble (e.g., EDStatic.THERE_IS_NO_DATA if there is no data)
     */
    public void finish() throws Throwable {
        //check for EDStatic.THERE_IS_NO_DATA
        if (columnStreams == null)
            throw new SimpleException(EDStatic.THERE_IS_NO_DATA);
        for (int col = 0; col < columnStreams.length; col++) {
            //close the stream
            columnStreams[col].close();
        }
        columnStreams = null;

        //diagnostic
        if (verbose)
            String2.log("TableWriterAll done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");
    }

    /**
     * If caller has the entire table, use this instead of repeated writeSome() + finish().
     *
     * @throws Throwable if trouble (e.g., EDStatic.THERE_IS_NO_DATA if there is no data)
     */
    public void writeAllAndFinish(Table tCumulativeTable) throws Throwable {
        //no need to write to files then reconstruct the cumulativeTable later
        cumulativeTable = tCumulativeTable;
        totalNRows = cumulativeTable.nRows();
        ensureCompatible(cumulativeTable);
    }

    /**
     * Call this after finish() to get a PrimitiveArray with all of the data for one of the columns.
     * Since this may be a large object, destroy this immediately when done.
     * Call this after finish() is called as part of getting the results.
     * 
     * <p>Missing values are still represented as destinationMissingValue or
     * destinationFillValue.
     * Use pa.table.convertToStandardMissingValues() if NaNs are needed.
     *
     * @param col   0..
     * @return a PrimitiveArray with all of the data for one of the columns.
     * @throws Throwable if trouble  (e.g., totalNRows > Integer.MAX_VALUE)
     */
    public PrimitiveArray column(int col) throws Throwable {
        //get it from cumulativeTable
        if (cumulativeTable != null)
            return cumulativeTable.getColumn(col);

        //get it from DOSFile
        if (totalNRows >= Integer.MAX_VALUE)
            throw new SimpleException(EDStatic.thereIsTooMuchData);
        PrimitiveArray pa = PrimitiveArray.factory(columnType(col), (int)totalNRows, false);
        DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(
            dir + fileNameNoExt + "." + randomInt + "." + columnNames[col])));
        pa.readDis(dis, (int)totalNRows);
        dis.close();
        return pa;
    }

    /**
     * This returns the total number of rows.
     * Call this after finish() is called as part of getting the results.
     *
     * @return the total number of rows.
     */
    public long nRows() {return totalNRows;}

    /** 
     * Call this after finish() to assemble the cumulative table.
     * This checks ensureMemoryAvailable.
     * This doesn't delete the temp files.
     *
     * <p>For the TableWriterAllWithMetadata, this has the updated metadata.
     */
    public Table cumulativeTable() throws Throwable {
        //is it available from writeAllAndFinish
        if (cumulativeTable != null)
            return cumulativeTable;

        //make cumulativeTable
        Table table = makeEmptyTable();

        //ensure memory available    too bad this is after all data is gathered
        int nColumns = nColumns();
        EDStatic.ensureMemoryAvailable(nColumns * nRows() * table.estimatedBytesPerRow(),
            fileNameNoExt);

        //actually get the data
        for (int col = 0; col < nColumns; col++) 
            table.setColumn(col, column(col));
        return table;
    }

    /**
     * This deletes the columnStreams files and cumulativeTable (if any).
     * This won't throw an exception.
     *
     * <p>It isn't essential that the user call this.
     * It will be called automatically then java garbage collector calls finalize.
     * And/or the cache cleaning system will do this in ~1 hour if the caller doesn't.

     */
    public void releaseResources() {
        try {
            cumulativeTable = null;

            if (columnNames == null)
                return;
            int nColumns = nColumns();
            for (int col = 0; col < nColumns; col++) {
                File2.delete(dir + fileNameNoExt + "." + randomInt + "." + columnNames[col]);
            }
        } catch (Throwable t) {
            String2.log(MustBe.throwableToString(t));
        }
    }
    
    /** 
     * Users of this class shouldn't call this -- use releaseResources() instead.
     * Java calls this when an object is no longer used, just before garbage collection. 
     * 
     */
    protected void finalize() throws Throwable {
        releaseResources();
        super.finalize();
    }


}



