/* 
 * TableWriterSeparatedValue Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.dataset;

import com.cohort.util.Calendar2;
import com.cohort.util.SimpleException;
import com.cohort.util.String2;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.pointdata.Table;
import gov.noaa.pfel.erddap.util.EDStatic;
import gov.noaa.pfel.erddap.variable.EDV;

import java.io.BufferedWriter;
import java.io.OutputStreamWriter;

/**
 * TableWriterSeparatedValue provides a way to write a table to comma or
 * tab separated value ASCII 
 * outputStream in chunks so that the whole table doesn't have to be in memory 
 * at one time.
 * This is used by EDDTable.
 * The outputStream isn't obtained until the first call to writeSome().
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2007-08-24
 */
public class TableWriterSeparatedValue extends TableWriter {

    //set by constructor
    protected String separator;
    protected boolean quoted;
    protected boolean writeUnits;

    //set by firstTime
    protected boolean isTimeStamp[];
    protected boolean isString[];
    protected BufferedWriter writer;

    /**
     * The constructor.
     *
     * @param tOutputStreamSource  the source of an outputStream that receives the 
     *     results, usually already buffered.
     *     The ouputStream is not procured until there is data to be written.
     * @param tSeparator  usually a tab or a comma
     * @param tQuoted if true, if a String value has a double quote or comma, 
     *    the value will be written in double quotes
     *    and internal double quotes become two double quotes.
     *    In any case, newline characters are replaced by char #166 (pipe with gap).
     */
    public TableWriterSeparatedValue(OutputStreamSource tOutputStreamSource,
        String tSeparator, boolean tQuoted, boolean tWriteUnits) {

        super(tOutputStreamSource);
        separator = tSeparator;
        quoted = tQuoted;
        writeUnits = tWriteUnits;
    }

    /**
     * This adds the current contents of table (a chunk of data) to the OutputStream.
     * This calls ensureCompatible each time it is called.
     * If this is the first time this is called, this does first time things
     *   (e.g., call OutputStreamSource.outputStream() and write file header).
     * The number of columns, the column names, and the types of columns 
     *   must be the same each time this is called.
     *
     * <p>The table should have missing values stored as destinationMissingValues
     * or destinationFillValues.
     * This implementation converts them to NaNs.
     *
     * @param table with destinationValues
     * @throws Throwable if trouble
     */
    public void writeSome(Table table) throws Throwable {
        if (table.nRows() == 0) 
            return;

        //ensure the table's structure is the same as before
        boolean firstTime = columnNames == null;
        ensureCompatible(table);

        //do firstTime stuff
        int nColumns = table.nColumns();
        if (firstTime) {
            isTimeStamp = new boolean[nColumns];
            for (int col = 0; col < nColumns; col++) {
                String u = table.columnAttributes(col).getString("units");
                isTimeStamp[col] = u != null && u.equals(EDV.TIME_UNITS);
            }

            //write the header
            writer = new BufferedWriter(new OutputStreamWriter(
                outputStreamSource.outputStream("UTF-8"), "UTF-8"));

            //write the column names   
            isString = new boolean[nColumns];
            for (int col = 0; col < nColumns; col++) {
                isString[col] = table.getColumn(col).getElementType() == String.class;
                writer.write(quote(quoted, table.getColumnName(col)));
                writer.write(col == nColumns - 1? "\n" : separator);
            }

            //write the units   
            if (writeUnits) {
                for (int col = 0; col < nColumns; col++) {
                    String s = table.columnAttributes(col).getString("units");
                    if (s == null) s = "";
                    if (isTimeStamp[col])
                        s = "UTC"; //no longer true: "seconds since 1970-01-01..."
                    writer.write(quote(quoted, s));
                    writer.write(col == nColumns - 1? "\n" : separator);
                }
            }
        }

        //*** do everyTime stuff
        convertToStandardMissingValues(table);  //NaNs; not the method in Table, so metadata is unchanged

        //write the data
        int nRows = table.nRows();
        for (int row = 0; row < nRows; row++) {
            for (int col = 0; col < nColumns; col++) {
                if (isTimeStamp[col]) {
                    double d = table.getDoubleData(col, row);
                    String s = Double.isNaN(d)? "" : Calendar2.epochSecondsToIsoStringT(d) + "Z";
                    writer.write(s);
                } else if (isString[col]) {
                    writer.write(quote(quoted, table.getStringData(col, row)));
                } else {
                    String s = table.getStringData(col, row);
                    writer.write(s.length() == 0? "NaN" : s);
                }
                writer.write(col == nColumns -1? "\n" : separator);
            }
        }       

        //ensure it gets to user right away
        if (nRows > 1) //some callers work one row at a time; avoid excessive flushing
            writer.flush(); 

    }

    
    /**
     * This writes any end-of-file info to the stream and flush the stream.
     *
     * @throws Throwable if trouble (e.g., EDStatic.THERE_IS_NO_DATA if there is no data)
     */
    public void finish() throws Throwable {
        //check for EDStatic.THERE_IS_NO_DATA
        if (writer == null)
            throw new SimpleException(EDStatic.THERE_IS_NO_DATA);

        writer.flush(); //essential

        //diagnostic
        if (verbose)
            String2.log("TableWriterSeparatedValue done. TIME=" + 
                (System.currentTimeMillis() - time) + "\n");

    }

    //this deals with special characters (see tQuoted definition above)
    private static String quote(boolean quoted, String s) {
        //this is Bob's unprecedented solution to dealing with newlines
        // is char #166, so distinct from pipe, char #124
        s = String2.replaceAll(s, '\n', ''); 
        if (quoted) {
            if (s.indexOf('"') >= 0 || s.indexOf(',') >= 0) 
                s = "\"" + String2.replaceAll(s, "\"", "\"\"") + "\"";

        }
        return s;
    }

    /**
     * This is a convenience method to write an entire table in one step.
     *
     * @throws Throwable if trouble  (no columns is trouble; no rows is not trouble)
     */
    public static void writeAllAndFinish(Table table, OutputStreamSource outputStreamSource, 
        String separator, boolean quoted, boolean writeUnits) throws Throwable {

        TableWriterSeparatedValue twsv = new TableWriterSeparatedValue(
            outputStreamSource, separator, quoted, writeUnits);
        twsv.writeAllAndFinish(table);
    }

}



