/* 
 * HtmlWidgets Copyright 2007, NOAA.
 * See the LICENSE.txt file in this file's directory.
 */
package gov.noaa.pfel.erddap.util;

import com.cohort.util.Calendar2;
import com.cohort.util.File2;
import com.cohort.util.String2;
import com.cohort.util.XML;

import gov.noaa.pfel.coastwatch.util.SSR;

import java.awt.Color;
import java.io.FileWriter;
import java.io.Writer;

/**
 * HtmlWidgets has methods to simplify creation of widgets in
 * HTML forms.
 *
 * <p>"name" vs. "id" - for widgets, "name" is the required attributes.
 * "id" is not required and causes problems.
 *
 * @author Bob Simons (bob.simons@noaa.gov) 2007-09-04
 */
public class HtmlWidgets {

    /** Leave this set to false here, but change within a program if desired. */
    public static boolean debugMode = false;

    /** This is the standard start of an HTML 4.01 document up to and including the 'head' tag. */
    public final static String DOCTYPE_HTML_TRANSITIONAL = 
        "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" \n" +
        "  \"http://www.w3.org/TR/html4/loose.dtd\"> \n" +
        "<html>\n" +
        "<head>\n";

   /** 
    * This is the standard start of an XHTML 1.0 document up to and including the 'head' tag.
    * This is specific to UTF-8. 
    * See comments about making xhtml render properly in browsers:
    *   Must use UTF-8; remove ?xml header; use mime type text/html;
    *   http://www.w3.org/TR/xhtml1/#guidelines
    * <br>But it is more important that it be proper xml -- so leave xml tag in.
    * <br>More useful clues from http://www.w3.org/MarkUp/Forms/2003/xforms-for-html-authors
    *   which is xhtml that renders correctly.
    *   It needs head/meta tags and head/style tags. (see below)
    *   And use xhmtl validator at http://validator.w3.org/#validate_by_input for testing.
    * <br>In OutputStreamFromHttpResponse, don't use response.setHeader("Content-Disposition","attachment;filename="
    *   for xhtml files.
    */
    public final static String DOCTYPE_XHTML_TRANSITIONAL = 
        "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" +
        "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"\n" +
        "  \"http://www.w3.org/TR/xhtml1/xhtml1-transitional.dtd\">\n" +
        "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n" + //from www.w3.org documents -- it's visible in all browsers!
        "<head>\n" +
        "  <meta http-equiv=\"content-type\" content=\"text/html; charset=UTF-8\" />\n";// +

    public final static String SANS_SERIF_STYLE = 
        "font-family:Arial,Helvetica,sans-serif; font-size:85%;"; //color:#FFFFCC
    
    public final static int BUTTONS_0n = -1, BUTTONS_1 = -2, BUTTONS_100 = -8,
        BUTTONS_1000 = -16;
   
    /** One line: white,grays,black, then one rainbow. */
    public final static String PALETTE17[] = {
        "FFFFFF", "CCCCCC", "999999", "666666", "000000", 
        "FF0000", "FF9900", "FFFF00", "99FF00", "00FF00", "00FF99", 
        "00FFFF", "0099FF", "0000FF", "9900FF", "FF00FF", "FF99FF"};

    /** This will display a message to the user if JavaScript is not supported
     * or disabled. Last verified 2007-10-11. */
    public static String ifJavaScriptDisabled =
        "<noscript><p><font color=\"red\"><b>This web page works better if JavaScript is enabled.</b> Please:\n" +
        "<br>1) Enable JavaScript in your browser:\n" +
        "<br>&nbsp;&nbsp; &bull; Windows\n" +  
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Firefox: select \"Tools : Options : Content : Enable JavaScript : OK\"\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Internet Explorer: select \n" +  //true for IE 6 and 7 
        "   \"Tools : Internet Options : Security : Internet : Custom level :\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; Sripting/ Active Scripting : Enable : OK : OK\"\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Opera: select \"Tools : Quick Preferences : Enable JavaScript\"\n" +
        "<br>&nbsp;&nbsp; &bull; Mac OS X\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Firefox: select \"Firefox : Preferences : Content : Enable JavaScript : OK\"\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Internet Explorer (this is an out-of-date browser -- please consider switching): \n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; select \"Explorer : Preferences : Web Content : Enable Scripting : OK\"\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Opera: select \"Opera : Quick Preferences : Enable JavaScript\"\n" +
        "<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp; &bull; Safari: select \"Safari : Preferences : Security : Enable JavaScript\"\n" + 
        "<br>2) Reload this web page.\n" +
        "<br>&nbsp;</font>\n" +
        "</noscript>\n";

    /**
     * If you want to use Tip for big html tooltips (see below),
     * include this in the 'body' of a web page (preferably right after the body tag).
     * General info: http://www.walterzorn.com/tooltip/tooltip_e.htm (LGPL License -- so give credit).
     * By making this a reference to a file (not inline text), the .js file is 
     * downloaded once to the user's browser, then cached if Expired header is set.
     * 
     * <p>I last downloaded wz_tooltip.js on 2009-06-19 ver. 5.31.
     * In wz_tooltip.js, I changed default FontSize from 8pt to 10pt
     * config. FontSize		= '10pt' 	// E.g. '9pt' or '12px' - unit is mandatory
     *
     * @param dir a public directory from the web page's point of view (e.g., EDStatic.imageDirUrl) 
     *   where wz_tooltip.js can be found.
     */
    public static String htmlTooltipScript(String dir) {
        return
        "<!-- Big HTML tooltips are generated with wz_tooltip from \n" +
        "    http://www.walterzorn.com/tooltip/tooltip_e.htm (LGPL license) -->\n" +
        "<script type=\"text/javascript\" src=\"" + dir + "wz_tooltip.js\"></script>\n" +
        "\n";
    }

    /** 
     * This is the width and height of the sliderLeft.gif, sliderCenter.gif, 
     * and sliderRight.gif images in imageDirUrl.
     * sliderBg.gif must have sliderThumbHeight.
     */
    public static int sliderThumbWidth = 15;
    public static int sliderThumbHeight = 17;

    /**
     * This returns the html to include Walter Zorn's image dragDrop script 
     * for the 'body' of a web page (preferably right after the body tag).
     * General info: http://www.walterzorn.com/dragdrop/dragdrop_e.htm (LGPL License -- so give credit).
     * By making this a reference to a file (not inline text), the .js file is 
     * downloaded once to the user's browser, then cached if Expired header is set.
     * 
     * @param dir a public directory from the web page's point of view (e.g., EDStatic.imageDirUrl()) 
     *   where wz_dragdrop.js can be found.
     */
    public static String dragDropScript(String dir) {
        return
        "<!-- Drag and Drop is performed by wz_dragdrop from\n" +
        "     http://www.walterzorn.com/dragdrop/dragdrop_e.htm (LGPL license) -->\n" +
        "<script type=\"text/javascript\" src=\"" + dir + "wz_dragdrop.js\"></script>\n" +
        "\n";
    }

    /** This is a standalone javascript which does minimal percent encoding of a string,
     *  similar to SSR.minimalPercentEncode.
     */
    public static String PERCENT_ENCODE_JS =
        "<script type=\"text/javascript\"> \n" +
        "function percentEncode(s) { \n" + //browser handles other chars, but needs help with + & " ', and thus %
        "  var s2=\"\";\n" +
        "  for (var i = 0; i < s.length; i++) {\n" +
        "    var ch=s.charAt(i);\n" +
        "    if (ch == \"%\") s2+=\"%25\";\n" +
        "    else if (ch == \"&\") s2+=\"%26\";\n" +     //to distinguish & in value in &id=value
        "    else if (ch == \"\\\"\") s2+=\"%22\";\n" +  //avoids trouble with urls in javascript 
        "    else if (ch == \"'\") s2+=\"%27\";\n" +     //avoids trouble with urls in javascript
        "    else if (ch == \"+\") s2+=\"%2B\";\n" +     //avoid trouble with +
        "    else if (ch == \" \") s2+=\"%20\";\n" +     //safer than +
        "    else s2+=ch;\n" +
        "  }\n" +
        "  return s2;\n" +
        "}\n" +
        "</script>\n";


    // *********************** set by constructor  ****************************

    //these can be changed as needed while creating a form
    public String style;
    public boolean htmlTooltips;
    public String imageDirUrl;

    //this is set by beginForm
    public String formName;

    /** If true, pressing Enter in a textField submits the form (default = false). */
    public boolean enterTextSubmitsForm = false;

    /** The default constructor. */
    public HtmlWidgets() {
        this("", false, "");
    }

    /** A constructor that lets you add 
     * style information (e.g., "")
     * and specify the number of items to be shown at a time by select() widgets.
     *
     * @param tStyle sets the style for widgets that need a style in addition to  
     *    the current document style, e.g., SANS_SERIF_STLYE or "" if none
     * @param tHtmlTooltips if true, tooltip text can be any HTML text of any length.
     *     If true, caller must have called htmlTooltipScript above.
     *     If false, tooltips are plain text and should be (less than 60 char on some browsers).
     * @param tImageDirUrl the public url for the image dir which has the
     *     arrow, p... and m... .gif files for the 'select' buttons
     *     (or null or "" if not needed).
     */
    public HtmlWidgets(String tStyle, boolean tHtmlTooltips, String tImageDirUrl) {
        style = tStyle;
        htmlTooltips = tHtmlTooltips;
        imageDirUrl = tImageDirUrl;
    }

    /**
     * This changes the style held by this class and returns the old style.
     * @param newStyle
     * @return oldStyle
     */    
    public String setStyle(String newStyle) {
        String oldStyle = style;
        style = newStyle;
        return oldStyle;
    }
 
    /** This is mostly used internally; it returns the formatted style. */
    public String completeStyle() {
        return completeStyle(style);
    }

    /** This is mostly used internally; it returns the formatted style. */
    public String completeStyle(String tStyle) {
        //note space at beginning and end
        return tStyle == null || tStyle.length() == 0? "" : 
            "\n      style=\"" + XML.encodeAsXML(tStyle) + "\" ";
    }

    /**
     * This is mostly used internally; it returns the formatted tooltip.
     *
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     */
    public String completeTooltip(String tooltip) {
        if (tooltip == null || tooltip.length() == 0)
            return ""; 
        //note space at beginning and end
        return htmlTooltips?
            "\n      " + htmlTooltip(tooltip) + " " :
            "\n      title=\"" + XML.encodeAsXML(tooltip) + "\" ";
    }

    /**
     * This creates the HTML code for the beginning of a form.
     *
     * @param name  make it unique for this page
     * @param method usually GET or POST
     * @param url  the url to which the form information will be submitted
     * @param other other attributes  (or "" if none)
     * @return the HTML code for the beginning of a form.
     */
    public String beginForm(String name, String method, String url, String other) {
        formName = name;
        return 
            "<form name=\"" + name + "\" method=\"" + method + "\"\n" +
            "  action=\"" + url + "\" " + 
            other + " >\n";
    }

    /**
     * This creates the HTML code for the end of a form.
     *
     * @return the HTML code for the end of a form.
     */
    public String endForm() {
        return "</form>\n";
    }

    /** 
     * This creates the HTML code to begin a table.
     *
     * @param lineWidth  width of lines (in pixels, 0 = none)
     * @param cellPadding 0 for really tight, 2 for comfortable
     * @param other attributes e.g., "width=\"2%\" bgcolor=\"" + bgColor + "\" style=\"...\"" 
     * @return the HTML code to begin a table.
     */
    public String beginTable(int lineWidth, int cellPadding, String other) { 
        return 
            "  <table border=\"" + lineWidth + "\" " + other + 
            " cellspacing=\"0\" cellpadding=\"" + cellPadding + "\">\n";
    }

    /** 
     * This creates the HTML code to end a table.
     *
     * @return the HTML code to end a table.
     */
    public String endTable() {
        return "  </table>\n";
    }

    /**
     * This creates the HTML code for a button.
     * "labelText" is displayed and name=labelText is returned if clicked
     *
     * @param type "submit" or "button".  
     *    "submit" buttons submit the form to the form's url.
     *    "button" buttons do "other" things (via 'other').
     * @param name of the widget (name=labelText is returned when the form is submitted)
     *    (or null or "" if you don't need anything submitted with form)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param labelText the plain text label visible to the user 
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for a button.
     */
    public String button(String type, String name, String tooltip, String labelText, 
        String other) {
        return 
            //note old-style "input" tag
            "    <input type=\"" + type + "\"" +
            (name != null && name.length() > 0? " name=\"" + name + "\"" : "") +                
            " value=\"" + XML.encodeAsXML(labelText) + "\"" +
            " " + other + 
            completeStyle() +
            completeTooltip(tooltip)+ 
            " >\n"; 
    }

    /**
     * This creates the HTML code for the button.
     * The labelTextHtml is displayed and name=labelText is returned if clicked
     *
     * @param type "submit" or "button".  
     *    "submit" buttons submit the form to the form's url.
     *    "button" buttons do "other" things (via 'other').
     * @param name of the widget (name=labelText is returned when the form is submitted)
     *    (or null or "" if you don't need anything submitted with form)
     * @param value is the value returned when the form is submitted (name=value)
     *    (or null or "" if you don't need anything submitted with form)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param labelHtml the html which will label the button
     *     (e.g.,an img tag)
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for the button.
     */
    public String htmlButton(String type, String name, String value, String tooltip, 
        String labelHtml, String other) {
        return 
            //note new-style "button" tag
            "    <button type=\"" + type + "\"" +
            (name  != null && name.length()  > 0? " name=\""  + name + "\"" : "") +                
            (value != null && value.length() > 0? " value=\"" + XML.encodeAsXML(value) + "\"" : "") +
            " " + other + 
            completeStyle() +
            completeTooltip(tooltip)+ 
            " >" + labelHtml + "</button>\n"; 
    }

    /**
     * This creates the HTML code for a checkbox.
     * This also makes a "hidden" widget called "previous_"+name with
     *    value=[checked] since un-checked checkboxes don't return info 
     *    with the form.
     *
     * @param name the name of the widget (if checked, name=value is returned when the form is submitted)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param checked the initial state of the checkbox
     * @param value (sometimes just "true")
     * @param rightLabelHtml "" if none
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for a checkbox.
     */
    public String checkbox(String name, String tooltip, 
        boolean checked, String value, String rightLabelHtml, String other) {

        return 
            "    <input type=\"checkbox\" name=\"" + name + "\"" +                
            " value=\"" + XML.encodeAsXML(value) + "\"" + 
            completeTooltip(tooltip) +
            (checked? "\n        checked=\"checked\"" : "") +
            " " + other + " > " + rightLabelHtml + "\n" +

            //add hidden so processRequest can distinguish notChecked from 
            //first time user (default)
            hidden("previous_" + name, "" + checked);
    }

    /**
     * This create the HTML code for a comment.
     *
     * @param commentText
     * @return the HTML code for a comment.
     */
    public String comment(String comment) {
        return "    <!-- " + XML.encodeAsXML(comment) + " -->\n"; 
    }

    /**
     * This creates the HTML code for a hidden widget.
     *
     * @param name the name of the widget (name=value is returned when the form is submitted)
     * @param value is the value of this attribute
     * @return the HTML code for a hidden widget.
     */
    public String hidden(String name, String value) {
        return 
            "    <input type=\"hidden\" name=\"" + name + "\"" +                
            " value=\"" + XML.encodeAsXML(value) + "\" >\n";
    }


    /**
     * This creates the HTML code for a series of radio buttons.
     *
     * @param name the name of the widget (name=optionText is returned when the form is submitted)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param onARow  if true, the radio buttons will all be put on one row.
     *    Otherwise, they are put on separate lines separated by a linebreak.
     * @param optionTexts which provides the plain text to be displayed for each 
     *   of the options (special characters will be html-encoded).
     * @param selected the index of the selected item or -1 if none
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for a series of radio buttons.
     */
    public String radioButtons(String name, String tooltip, boolean onARow,
        String optionTexts[], int selected, String other) {

        StringBuffer sb = new StringBuffer();
        String br = onARow? "    " : "    <br>";
        for (int i = 0; i < optionTexts.length; i++) {
            sb.append(i == 0? "    " : br);              
            sb.append(radioButton(name, tooltip, optionTexts[i], i == selected, other));
        }
        return sb.toString();
    }

    /**
     * This creates the HTML code for one radio button.
     *
     * @param name the name of the widget (name=optionText is returned when the form is submitted)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param optionText which provides the plain text to be displayed for this option.
     * @param selected indicates if this item is selected
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for one radio button.
     */
    public String radioButton(String name, String tooltip, 
        String optionText, boolean selected, String other) {

        String s = XML.encodeAsXML(optionText);
        return 
            //<span> avoids check box and value being separated by newline when lots of options
            "<span style=\"white-space: nowrap;\"><input type=\"radio\" name=\"" + name + "\"" +                
            " value=\"" + s + "\"" + 
            (selected? " checked" : "") +
            completeTooltip(tooltip) +
            " " + other + " >" + s + "</span>\n";
    }

    /**
     * This creates the HTML code to display 17 radio buttons with colored backgrounds.
     *
     * @return the HTML code to display 17 radio buttons with colored backgrounds.
     */
    public String color17(String htmlLabel, String name, String tooltip, 
        int selected, String other) {
        return color(PALETTE17, 17, htmlLabel, name, tooltip, selected, other);
    }

    /**
     * This creates the HTML code to display radio buttons with colored backgrounds.
     *
     * @param htmlLabel is an optional label (e.g., "Color: "),
     *   or use "" for nothing.
     * @param colors e.g., PALETTE17, each must be an RRGGBB value
     * @param perRow e.g., 17
     * @param selected can be -1 if none should be selected initially
     * @return the HTML code to display radio buttons with colored backgrounds.
     */
    public String color(String[] colors, int perRow, 
        String htmlLabel, String name, String tooltip, int selected, String other) {

        StringBuffer sb = new StringBuffer();
        sb.append(
            "<table cellspacing=\"0\" cellpadding=\"1\">\n" +
            "  <tr>\n");
        boolean hasLabel = htmlLabel != null && htmlLabel.length() > 0;
        if (hasLabel)
            sb.append("    <td nowrap>" + htmlLabel + "</td>\n");
        int inRow = 0;
        for (int i = 0; i < colors.length; i++) {
            //a radio button with the appropriate background color
            String checked = i == selected? " checked" : "";
            sb.append(
                "    <td bgcolor=\"#" + colors[i] + "\">" +
                    "<input type=\"radio\" name=\"" + name + 
                    "\" value=\"" + colors[i] + "\"" + checked + " " +
                    completeTooltip(tooltip) +
                    " " + other + "></td>\n");

            //new row?
            if (++inRow % perRow == 0 && i != colors.length - 1)
                sb.append(
                    "  </tr>\n" +
                    "  <tr>\n" +
                    (hasLabel? "    <td>&nbsp;</td>\n" : ""));
        }
        sb.append(
            "  </tr>\n" +
            "</table>\n");
        return sb.toString();
    }

    /**
     * This creates the HTML code for a list in a box or a dropdown list.
     * If nRows &lt; 0, this uses a table to bring the elements close together,
     * and so may need to be in an enclosing table if you want other items
     * on same line.
     *
     * @param name the name of the widget (name=value is returned when the form is submitted)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param nRows &gt; 1 (e.g. 8) to display many options,
     *   <br>1=1 row with no buttons, 
     *   <br>nRows is negative implies some combination of 
     *     BUTTONS_0n, BUTTONS_1, BUTTONS_100, BUTTONS_1000
     * @param options which provides the plain text to be displayed for each of the options.
     * @param selected the index of the selected item or -1 if none
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for a list in a box or a dropdown list.
     */
    public String select(String name, String tooltip, int nRows,
        String options[], int selected, String other) {

        StringBuffer sb = new StringBuffer();
        int nOptions = options.length;
        if (selected < 0 || selected >= nOptions)
            selected = -1;
        if (nRows > nOptions)
            nRows = nOptions;

        //if buttons visible, put select and buttons in a table (to pack close together)
        if (nRows < 0 && nOptions > 1) {
            sb.append(beginTable(0, 0, "width=\"2%\"")); 
            sb.append("      <tr><td>");//td for <select>
        }

        //the main 'select' widget
        sb.append("\n    <select name=\"" + name + "\" size=\"" + Math.max(1, nRows) + "\"" +
            completeTooltip(tooltip) +
            completeStyle() + 
            " " + other + " >\n");

        String spacer = nOptions < 20? "      " : ""; //save space if lots of options
        for (int i = 0; i < nOptions; i++) {
            //Win IE 7 needs 'value' to be explicitly set for (name).value to work in JavaScript
            //and for value to be returned when form is submitted.
            //I tried 'label' to be used as label, text, and value; didn't work.
            sb.append(spacer + "<option" + 
                (i == selected? " selected=\"selected\"" : "") + 
                ">" + XML.encodeAsXML(options[i]) + "</option>\n"); 
        }
        sb.append("    </select>\n");

        //the buttons
        if (nRows < 0 && nOptions > 1) {
            sb.append("        </td>\n"); //end of <select>'s td

            //        selectButton(name, nRows, test,    nOptions, minNOptions, incr,            img,             tooltip);
            sb.append(selectButton(name, nRows, BUTTONS_0n,   nOptions, 2,    Integer.MIN_VALUE, "arrowLL.gif",   "Select the first item."));
            sb.append(selectButton(name, nRows, BUTTONS_1000, nOptions, 1100, -1000,             "minus1000.gif", "Jump back 1000 items."));
            sb.append(selectButton(name, nRows, BUTTONS_100,  nOptions, 110,  -100,              "minus100.gif",  "Jump back 100 items."));
            sb.append(selectButton(name, nRows, BUTTONS_1,    nOptions, 2,    -1,                "minus.gif",     "Select the previous item."));
            sb.append(selectButton(name, nRows, BUTTONS_1,    nOptions, 2,    1,                 "plus.gif",      "Select the next item."));
            sb.append(selectButton(name, nRows, BUTTONS_100,  nOptions, 110,  100,               "plus100.gif",   "Jump forward 100 items."));
            sb.append(selectButton(name, nRows, BUTTONS_1000, nOptions, 1100, 1000,              "plus1000.gif",  "Jump forward 1000 items."));
            sb.append(selectButton(name, nRows, BUTTONS_0n,   nOptions, 2,    Integer.MAX_VALUE, "arrowRR.gif",   "Select the last item."));

            sb.append(
                "\n        </tr>" +
                "\n      </table>\n");
        }
        return sb.toString();
    }

    /**
     * This creates the HTML to make a button to assist select().
     * This is used by select() to make the e.g., -100, +100 buttons.
     *
     * @param imageName e.g., plus100.gif
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @return the HTML to make a button to assist select().
     */
    private String selectButton(String name, int nRows, int test, 
        int nOptions, int minNOptions, int incr, String imageName, String tooltip) {

        StringBuffer sb = new StringBuffer();
        if ((-nRows & -test) != 0 && nOptions >= minNOptions) {
            String si = formName + "." + name + ".selectedIndex";
            String rhs = "0";
            if (incr == Integer.MIN_VALUE)      {}  //stay 0
            else if (incr < 0)                  rhs  = "Math.max(0," + si + "-" + -incr + ")";
            else if (incr == Integer.MAX_VALUE) rhs  = "" + (nOptions-1); 
            else if (incr > 0)                  rhs  = "Math.min(" + (nOptions-1) + "," + si + "+" + incr + ")";
            sb.append(
                "<td>" +
                "<img src=\"" + imageDirUrl + imageName + "\"\n" +
                "  " + completeTooltip(tooltip) +
                "  alt=\"" + 
                    (nRows == Integer.MIN_VALUE? "|&lt;" : nRows == Integer.MAX_VALUE? "&gt;|" : "" + nRows) + "\"" +
                    " onMouseUp=\"" + si + "=" + rhs + ";\" >\n" + //works much better than onClick and onDblClick
                "</td>\n");
        }
        return sb.toString();
    }


    /**
     * This create the HTML code for a textField.
     * If user pressed Enter, enterTextSubmitsForm determines if the form is submitted.
     *
     * <p>You can make the edges disappear with 
     * <tt>style=\"border:0px; background:" + bgColor + "\" </tt>
     *
     * <p>You can make the text bold with <tt>style=\"font-weight:bold;"\" </tt>
     *
     * @param name the name of the widget (name=newValue is returned when the form is submitted)
     * @param tooltip If htmlTooltips is true, this is already html.
     *     If it is false, this is plain text.  Or "" if no tooltip.
     * @param fieldLength  the size of the field, in mspaces(?), e.g., 10.
     *     If the fieldLength <= 0, no 'size' value is specified in the html.
     * @param maxLength    usually 255
     * @param initialTextValue the initial text value, or "" if none
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the HTML code for a textField.
     */
    public String textField(String name, String tooltip, 
        int fieldLength, int maxLength, String initialTextValue, String other) {

        return 
            "    <input type=\"text\" name=\"" + name + "\"" +                
            " value=\"" + XML.encodeAsXML(initialTextValue) + "\"" +
            //"\n      onkeypress=\"return !enter(event);\"" + 
            //supress Enter->submit
            //was the keypress event's keycode 'Enter'?
            //see http://www.mredkj.com/tutorials/validate.html
            (enterTextSubmitsForm? "" : 
                "\n      onkeypress='" +  //supress submission
                    " var key = window.event? event.keyCode : event.which? event.which : 0; " + //'?' deals with IE vs. Netscape vs. ? 
                    " return key != 13;' \n") +
            completeTooltip(tooltip) +
            completeStyle() +              
            "\n      " +
            (fieldLength > 0? "size=\"" + fieldLength + "\" " : "") + 
            "maxlength=\"" + maxLength + "\"" +
            " " + other + " >\n";
    }

    /**
     * This creates the html to draw an image (e.g., question mark) that has a big html tooltip.
     * htmlTooltipScript (see above) must be already in the document.
     * See tip().
     *
     * @param imageRef the reference for the question mark image (e.g., EDStatic.imageDirUrl() + EDD.questionMarkImageFile)
     * @param html  the html tooltip text, e.g., "Hi,<br>there!".
     *     It needs explicit br tags to set window width correctly.
     *     For plain text, use XML.encodeAsPreXML(plainText, 100).
     * @param other e.g., "onclick=\"pleaseWait();\""
     * @return the html to draw an image (e.g., question mark) that has a big html tooltip.
     */
    public static String htmlTooltipImage(String imageRef, String html, String other) {
         return "<img src=\"" + imageRef + "\" alt=\"\"" + htmlTooltip(html) + " " + other + ">\n";
    }


    /**
     * This creates the onmouseover and onmouseout html to add a big html tooltip to any widget.
     * htmlTooltipScript (see above) must be already in the document.
     *
     * @param html  the html tooltip text, e.g., "Hi,<br>there!".
     *     It needs explicit br tags to set window width correctly.
     *     For plain text, generate html from XML.encodeAsPreXML(plainText, 100).
     * @return the onmouseover and onmouseout html to add a big html tooltip to any widget.
     */
    public static String htmlTooltip(String html) {
         if (html == null || html.length() == 0)
             return "";

         //example from http://www.walterzorn.com/tooltip/tooltip_e.htm:
         //<a href="index.htm" onmouseover="Tip('Text with <img src=\"pics/image.jpg\" width=\"60\">image.')" onmouseout="UnTip()"> Homepage</a>
         //tooltip = XML.encodeAsXML(tooltip); //didn't work
         StringBuffer sb = new StringBuffer();
         int n = html.length();
         for (int i = 0; i < n; i++) {
             char ch = html.charAt(i);
             if      (ch == '\\') sb.append("\\\\");
             else if (ch == '\"') sb.append("&quot;");
             else if (ch == '\'') sb.append("&#039;");
             else if (ch == '\n') sb.append(' '); //causes problem: quotes not closed at end of line
             else sb.append(ch);
         }
         String2.replaceAll(sb, "&#039;", "\\&#039;");
         String2.replaceAll(sb, "  ", "&nbsp;&nbsp;");
         return " onmouseover=\"Tip('" + sb.toString() + "')\" onmouseout=\"UnTip()\" "; //with space at beginning and end
    }

    /** 
     * This returns the html for a white image which takes up the specified space.
     *
     * @param width in pixels
     * @param height in pixels
     * @param other other html parameters, e.g., "align=\"left\""
     */
    public String spacer(int width, int height, String other) {
        return
            "    <img src=\"" + imageDirUrl + "spacer.gif\" " +
                "width=\"" + width + "\" height=\"" + height + "\" " + other + " alt=\"\">\n";
    }

    /**
     * This returns the html for the images which make up a 1-thumb slider.
     * See also sliderScript.
     *
     * <p>The source images must be in imageDirUrl.
     *   sliderCenter.gif must have width=15, height=17.
     *   sliderBg.gif must have height = 17.
     *
     * @param sliderNumber e.g., 0.. n-1.
     *    In the HTML, the images will be named 
     *    sliderLeft[sliderNumber], sliderBg[sliderNumber].
     * @param bgWidth the width of the track, in pixels  (e.g., 500).
     *    All sliders on a form must use the same bgWidth.
     * @param other   e.g., align="left"
     * @return the html to describe a 1-thumb slider.
     */
    public String slider(int sliderNumber, int bgWidth, String other) {
        return //order is important
            "      <img name=\"sliderLeft" + sliderNumber + "\" src=\"" + imageDirUrl + "sliderCenter.gif\" \n" +
            "        width=\"" + sliderThumbWidth + "\" height=\"" + sliderThumbHeight + "\" " + other + " alt=\"\">\n" +
            "      <img name=\"sliderBg"   + sliderNumber + "\" src=\"" + imageDirUrl + "sliderBg.gif\" \n" +
            "        width=\"" + bgWidth          + "\" height=\"" + sliderThumbHeight + "\" " + other + " alt=\"\">\n";
    }

    /**
     * This returns the html for the images which make up a 2-thumb slider.
     * See also sliderScript.
     *
     * <p>The source images must be in imageDirUrl.
     *   sliderLeft.gif and sliderRight.gif must have width=15, height=17.
     *   sliderBg.gif must have height = 17.
     *
     * @param sliderNumber e.g., 0.. n-1.
     *    In the HTML, the images will be named 
     *    sliderLeft[sliderNumber], sliderBg[sliderNumber], sliderRight[sliderNumber].
     * @param bgWidth the width of the track, in pixels  (e.g., 500)
     * @param other   e.g., align="left"
     * @return the html to describe a 2-thumb slider.
     */
    public String dualSlider(int sliderNumber, int bgWidth, String other) {
        return //order is important
            "      <img name=\"sliderLeft"  + sliderNumber + "\" src=\"" + imageDirUrl + "sliderLeft.gif\" \n" +
            "        width=\"" + sliderThumbWidth + "\" height=\"" + sliderThumbHeight + "\" " + other + " alt=\"\">\n" +
            "      <img name=\"sliderBg"    + sliderNumber + "\" src=\"" + imageDirUrl + "sliderBg.gif\" \n" +
            "        width=\"" + bgWidth          + "\" height=\"" + sliderThumbHeight + "\" " + other + " alt=\"\">\n" +
            "      <img name=\"sliderRight" + sliderNumber + "\" src=\"" + imageDirUrl + "sliderRight.gif\" \n" +
            "        width=\"" + sliderThumbWidth + "\" height=\"" + sliderThumbHeight + "\" " + other + " alt=\"\">\n";
    }


    /**
     * This returns the html for the javascript which controls a series of sliders.
     * <br>dragDropScript() must be right after the 'body' tag.
     * <br>Use slider() or dualSlider() repeatedly in the form.
     * <br>Put sliderScript() right before the '/body' tag.
     * <br>See documentation in http://www.walterzorn.com/dragdrop/commands_e.htm
     *   and http://www.walterzorn.com/dragdrop/api_e.htm .
     *
     * @param fromTextFields the names of all of the 'from' textFields, 
     *    preceded by their formName + "." (e.g., "f1.from0", "f1.from1").
     *    1-thumb sliders connect to this textField only.
     *    For no slider, there should still be a placeholder entry (e.g., "").
     * @param toTextFields the names of all of the 'to' textFields , 
     *    preceded by their formName + "." (e.g., "f1.to0", "f1.to1").
     *    For no slider or 1-thumb sliders, there should still be a placeholder entry (e.g., "").
     * @param nThumbs is the number of thumbs for each slider
     *    (0=inactive, 1, or 2)
     * @param userValuesCsvs a csv list of values for each slider.
     *    <br>If a given nThumbs == 0, its csv is irrelevant.
     *    <br>Each list can be any length (at least 1), but only up to bgWidth+1 will be used.
     * @param initFromPositions the initial values for the left (or only) sliders.
     *    <br>If a given slider is inactive, there should still be a placeholder value.
     *    <br>These are usually calculated as Math.roung(index * bgWidth / (nIndices-1.0))
     * @param initToPositions the initial values for the right sliders.
     *    <br>If a given slider is inactive or has 1-thumb, there should still be a placeholder value.
     *    <br>These are usually calculated as Math.roung(index * bgWidth / (nIndices-1.0))
     * @param bgWidth the width of the track, in pixels  (e.g., 500).
     *    All sliders on a form must use the same bgWidth.
     * @return the html for the javascript which controls a series of sliders.
     * @throws RuntimeException if trouble
     */
    public String sliderScript(String fromTextFields[], String toTextFields[],
        int nThumbs[], 
        String userValuesCsvs[], int initFromPositions[], int initToPositions[], int bgWidth) {

        int nSliders = nThumbs.length;
        if (nSliders == 0)
            return "";
        if (nSliders != fromTextFields.length)
            throw new RuntimeException(
                "nThumbs.length=" + nSliders + " != fromTextFields.length=" + fromTextFields.length);
        if (nSliders != toTextFields.length)
            throw new RuntimeException(
                "nThumbs.length=" + nSliders + " != toTextFields.length=" + toTextFields.length);
        if (nSliders != userValuesCsvs.length)
            throw new RuntimeException(
                "nThumbs.length=" + nSliders + " != userValuesCsvs.length=" + userValuesCsvs.length);
        if (nSliders != initFromPositions.length)
            throw new RuntimeException(
                "nThumbs.length=" + nSliders + " != initFromPositions.length=" + initFromPositions.length);
        if (nSliders != initToPositions.length)
            throw new RuntimeException(
                "nThumbs.length=" + nSliders + " != initToPositions.length=" + initToPositions.length);
        StringBuffer sb = new StringBuffer();
        sb.append(
"<script type=\"text/javascript\"> \n" +
"<!--\n" +
"var fromTextFields = [");
for (int s = 0; s < nSliders; s++) 
    sb.append(String2.toJson(fromTextFields[s]) + (s < nSliders-1? ", " : "];\n"));
sb.append(
"var toTextFields = [");
for (int s = 0; s < nSliders; s++) 
    sb.append(String2.toJson(toTextFields[s]) + (s < nSliders-1? ", " : "];\n"));
sb.append(
"var userValues = [\n");
for (int s = 0; s < nSliders; s++) 
    sb.append("  [" + userValuesCsvs[s] + "]" + (s < nSliders-1? ",\n" : "];\n"));
sb.append(
"var initFromPositions = [\n");
for (int s = 0; s < nSliders; s++) 
    sb.append(initFromPositions[s] + (s < nSliders-1? ", " : "];\n"));
sb.append(
"var initToPositions = [\n");
for (int s = 0; s < nSliders; s++) 
    sb.append(initToPositions[s] + (s < nSliders-1? ", " : "];\n"));
sb.append(
"\n");

//SET_DHTML
sb.append(
"SET_DHTML(CURSOR_DEFAULT, NO_ALT, SCROLL, \n");
for (int s = 0; s < nSliders; s++) {
    if (nThumbs[s] > 0)  sb.append(
"  \"sliderBg"    + s + "\"+NO_DRAG,\n" +
"  \"sliderLeft"  + s + "\"+HORIZONTAL+MAXOFFLEFT+0,\n");
    if (nThumbs[s] == 2) sb.append(
"  \"sliderRight" + s + "\"+HORIZONTAL+MAXOFFRIGHT+" + bgWidth + ",\n");
}
sb.setLength(sb.length() - 2); //remove ,\n
sb.append(");\n" +
"\n" +
"var el = dd.elements;\n" +
"\n");

//log      
sb.append(
"function log(msg) {\n");
if (debugMode) sb.append(
"  if (typeof(console) != \"undefined\") console.log(msg);\n"); //for debugging only
sb.append(
"}\n" +
"\n");

//toUserValue
sb.append(
"function toUserValue(which, val) {\n" +
"  var nuv = userValues[which].length;\n" +
"  var index = Math.floor((val * nuv) / " + bgWidth + ");\n" +
"  return userValues[which][Math.min(Math.max(0, index), (nuv - 1))];\n" +
"};\n" +
"\n");

//updateUI
sb.append(
"function updateUI(left, which) {\n" +
"  log(\"left=\" + left + \" which=\" + which); \n" +
//"  //get the widgets\n" +
"  var leftS  = eval(\"el.sliderLeft\" + which);  \n" +
"  var rightS = eval(\"el.sliderRight\" + which);\n" +
"  if (left) {\n" +
"    var val = leftS.x - leftS.defx;\n" +
"    log(\"x=\" + leftS.x + \" val=\" + val);  //for development only\n" +
"    var fromW  = eval(\"document.\" + fromTextFields[which]);\n" +
"    fromW.value = toUserValue(which, val);\n" +
"    if (typeof(rightS) != \"undefined\") rightS.maxoffl = -val;\n" + //positive is to the left!
"  } else {\n" +
"    var val = rightS.x - rightS.defx;\n" +
"    log(\"x=\" + rightS.x + \" val=\" + val); \n" +
"    var toW    = eval(\"document.\" + toTextFields[which]);\n" +
"    toW.value = toUserValue(which, val);\n" +
"    if (typeof(leftS) != \"undefined\") leftS.maxoffr = val;\n" +
"  }\n" +
"};\n" +
"\n");

//my_DragFunc
sb.append(
"function my_DragFunc() {\n" +
"  try {\n");
for (int s = 0; s < nSliders; s++) {
    if (nThumbs[s] > 0)  sb.append(
"    if (dd.obj.name == 'sliderLeft"  + s + "') updateUI(true, " + s + ");\n");
    if (nThumbs[s] > 1)  sb.append(
"    if (dd.obj.name == 'sliderRight" + s + "') updateUI(false, " + s + ");\n");
}
sb.append(
"  } catch (ex) {\n" +
"    " + (debugMode? "alert" : "log") + "(ex.toString());\n" +
"  }\n" +
"}\n" +
"\n");

//initSlider    //do last
sb.append(
"function initSlider(which) {\n" +
"  var bgS    = eval(\"el.sliderBg\" + which);  \n" +
"  var leftS  = eval(\"el.sliderLeft\" + which);  \n" +
"  var rightS = eval(\"el.sliderRight\" + which);\n" +
" \n" +
"  var oneThumb = typeof(rightS) == \"undefined\";\n" +
"  leftS.setZ(bgS.z+1); \n" +
"  bgS.addChild(\"sliderLeft\" + which); \n" +
"  leftS.defx = bgS.x - Math.round(" + sliderThumbWidth + " / (oneThumb? 2 : 1));\n" +
"  leftS.moveTo(leftS.defx + initFromPositions[which], bgS.y); \n" +
"  leftS.maxoffr = initToPositions[which];\n" +
"\n" +
"  if (!oneThumb) {\n" +
"    rightS.setZ(bgS.z+1); \n" +
"    bgS.addChild(\"sliderRight\" + which); \n" +
"    rightS.defx = bgS.x;\n" +
"    rightS.moveTo(rightS.defx + initToPositions[which], bgS.y); \n" +
"    rightS.maxoffl = -initFromPositions[which];\n" +  //positive is to the left!
"  }\n" +
"}\n" +
"\n" +
"try {\n");
for (int s = 0; s < nSliders; s++) {
    if (nThumbs[s] > 0)  sb.append(
"  initSlider(" + s + ");\n");
}
sb.append(
"} catch (ex) {\n" +
"  " + (debugMode? "alert" : "log") + "(ex.toString());\n" +
"}\n" +

"\n" +
"//-->\n" +
"</script>\n" +
"\n");
        return sb.toString();
    }
    
    /**
     * This makes a test document and displays it in the browser.
     */
    public static void test() throws Throwable {
        debugMode = true;
        String fullName = SSR.getTempDirectory() + "TestHtmlWidgets.html";
        String imageDir = "file://c:/programs/tomcat/webapps/cwexperimental/images/";
        File2.delete(fullName);
        Writer writer = new FileWriter(fullName);
        boolean tHtmlTooltips = true;
        HtmlWidgets widgets = new HtmlWidgets("", tHtmlTooltips, 
            "http://coastwatch.pfeg.noaa.gov/erddap/images/"); //SANS_SERIF_STYLE);
        writer.write(
            DOCTYPE_HTML_TRANSITIONAL +
            "<head>\n" +
            "  <title>Test Html Widgets</title>\n" +
            "</head>\n" +
            "<body" + 
            " style=\"" + SANS_SERIF_STYLE + "\"" +
            ">\n" +
            htmlTooltipScript(imageDir) +
            dragDropScript(imageDir) +
            "<h1>Test Html Widgets</h1>\n" +
            ifJavaScriptDisabled);
        writer.write("This is some text in the SANS_SERIF_STYLE.\n");
        writer.write(htmlTooltipImage(imageDir + "QuestionMark.jpg", "Hi <b>bold</b>\n<br>there \\/'\"&amp;&brvbar;!", ""));
        String formName = "f1";
        writer.write(widgets.beginForm(formName, "GET", "testUrl", ""));
        writer.write(widgets.checkbox("checkboxName", "checkboxTooltip literal: &gt;!", true, "checkboxValue", "rightLabel", ""));
        writer.write(widgets.comment("This is a comment."));
        String options[] = new String[1200];
        for (int i = 0; i < 1200; i++)
            options[i] = "option" + i;
        writer.write("<br>\n");
        writer.write(widgets.radioButtons("radioName", "radioTooltip literal: &gt;!", true, new String[]{"apple", "banana", "cucumber"}, 1, ""));
        writer.write(widgets.select("dropdownName", "dropdownTooltip literal: &gt;!", BUTTONS_0n + BUTTONS_1 + BUTTONS_100 + BUTTONS_1000, options, 37, ""));
        writer.write(widgets.select("selectName", "selectTooltip literal: &gt;!", 6, options, 137, ""));
        writer.write("<br>\n");
        writer.write(widgets.textField("textFieldName", "textFieldTooltip literal: &gt;!", 10, 255, "initialValue", ""));
        writer.write("<br>\n");
        writer.write(widgets.textField("textFieldName2", "", 20, 255, "has big tooltip", 
            htmlTooltip("Hi <b>bold</b>!\n<br>There \\/'\"&amp;&brvbar;!")));
        writer.write("<br>\n");
        writer.write(widgets.hidden("testHiddenName", "hiddenValue"));
        //make a button that looks like a submit button but doesn't submit the form
        writer.write(widgets.button("button", "submit1Name", "submit tooltip  literal: &gt;!", "Submit", "onclick=\"alert('alert!');\""));
        writer.write(widgets.color17("Color: ", "color", "Pick a color.", 2, ""));
        writer.write("some other text\n");
         
        //sliders
        String fromNames[] = {"f1.from0", "f1.from1", "", "f1.from3", "f1.from4", "f1.from5"};
        String toNames[] = {"f1.to0", "", "", "f1.to3", "f1.to4", "f1.to5"};
        int nThumbs[] = {2, 1, 0, 2, 2, 1};
        String userValuesCsvs[] = {
            "\"a\",\"b\",\"c\",\"d\"",
            "\"B\",\"o\",\"b\"",
            null,
            "1,2,4,8,16,32,64,128",
            "\"only\"",
            "\"only\""};
        int initFromPositions[] = {200, 100, 0,   0,   0, 100};
        int initToPositions[]   = {300, 500, 0, 500, 500, 500};
        int bgWidth = 500;
//see sliderScript below

        writer.write(
"<table border=\"0\" cellspacing=\"0\" cellpadding=\"0\">\n" +
"\n" +
"<tr><td>Raw values: \n" +
"Min: " + widgets.textField("from0", "", 10, 10, "sliderFrom0", "") + "\n" +
"Max: " + widgets.textField("to0",   "", 10, 10, "sliderTo0",   "") + "\n" +
"</td></tr>\n" +
"<tr><td>\n" +
widgets.dualSlider(0, bgWidth, "align=\"left\"") +
"</td></tr>\n" +
"\n" +

"<tr><td>Raw values: \n" +
"Min: " + widgets.textField("from1", "", 10, 10, "sliderFrom1", "") + "\n" +
"</td></tr>\n" +
"\n" +

"<tr><td>\n" +
widgets.slider(1, bgWidth, "align=\"left\"") +
"</td></tr>\n" +
"\n" +

"<tr><td>Raw values: \n" +
"Min: " + widgets.textField("from3", "", 10, 10, "sliderFrom3", "") + "\n" +
"Max: " + widgets.textField("to3",   "", 10, 10, "sliderTo3", "") + "\n" +
"</td></tr>\n" +
"<tr><td>\n" +
widgets.dualSlider(3, bgWidth, "align=\"left\"") +
"</td></tr>\n" +


"<tr><td>Raw values: \n" +
"Min: " + widgets.textField("from4", "", 10, 10, "sliderFrom4", "") + "\n" +
"Max: " + widgets.textField("to4",   "", 10, 10, "sliderTo4", "") + "\n" +
"</td></tr>\n" +
"<tr><td>\n" +
widgets.dualSlider(4, bgWidth, "align=\"left\"") +
"</td></tr>\n" +

"<tr><td>Raw values: \n" +
"Min: " + widgets.textField("from5", "", 10, 10, "sliderFrom5", "") + "\n" +
"</td></tr>\n" +
"<tr><td>\n" +
widgets.slider(5, bgWidth, "align=\"left\"") +
"</td></tr>\n" +

"\n" +
"<tr><td>\n" +
"Something Else\n" +
"</td></tr>\n" +
"</table>\n");


        writer.write(widgets.endForm());        
        //writer.write("<script src=\"http://gmodules.com/ig/ifr?url=http://hosting.gmodules.com/ig/gadgets/file/104311141343481305007/erddap_windvectors_qsSQ_8day.xml&amp;synd=open&amp;w=280&amp;h=345&amp;title=ERDDAP%3A+Wind+Vectors%2C+QuikSCAT%2C+Science+Quality+(8+Day+Composite)&amp;border=http%3A%2F%2Fgmodules.com%2Fig%2Fimages%2F&amp;output=js\"></script>\n");
        //writer.write("<script src=\"http://gmodules.com/ig/ifr?url=http://hosting.gmodules.com/ig/gadgets/file/104311141343481305007/erddap_sst_goes_8day.xml&amp;synd=open&amp;output=js\"></script>\n");
        //writer.write("<img src=\"http://coastwatch.pfeg.noaa.gov/erddap/griddap/erdGAssta8day.png?sst[(last)][(0.0)][(22):(50)][(225):(255)]%26.draw=surface%26.vars=longitude|latitude|sst%26.colorBar=Rainbow|C|Linear|8|32|\" alt=\"ERDDAP: SST, GOES Imager (8 Day Composite)\">\n");

        writer.write(widgets.sliderScript(fromNames, toNames, nThumbs, userValuesCsvs, 
            initFromPositions, initToPositions, bgWidth));

        writer.write(
            "</body>\n" +
            "</html>\n");
        writer.close();


        SSR.displayInBrowser("file://" + fullName);


    }


}



